/*****************************************************************************
 * This code is part of Art Gallery Solver (AGSol) Package, which aims the 
 * resolution of the Art Gallery Problem With Point Guards.
 *
 * This software version (1.0.2) has been tested under and is compatible with 
 * CGAL 3.9 and GLPK 4.52.
 *
 * Authors:
 *  Davi Colli Tozoni - davi.tozoni@gmail.com
 *  Marcelo Castilho Couto - coutomarcelo@gmail.com
 *
 * AGSol Concept and Design: 
 *  Davi Colli Tozoni, Marcelo Castilho Couto, Pedro Jussieu de Rezende & Cid 
 * Carvalho de Souza.
 * 
 * Other information can be found at:
 *  http://www.ic.unicamp.br/~cid/Problem-instances/Art-Gallery/index.html
 *
 * --
 * 
 *  This program is free software: you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program.  If not, see <http://www.gnu.org/licenses/>.
 *      
 ****************************************************************************/


#include "polAlgorithms.h"
#include "PolygonWithHolesExt.h"
#include <iostream>

#include <CGAL/partition_2.h>
#include <CGAL/Polygon_set_2.h>
#include <CGAL/ch_akl_toussaint.h>

using namespace std;

typedef CGAL::Polygon_set_2<Extended_kernel, std::list<Point> > PolygonSet;

/**
 * Computes visibility polygons for polygons with holes.  The idea is to use
 * the same algorithm for the class PolygonExt (without holes) and then remove
 * the shadows generated by each hole.
 */
PolygonExt PolygonWithHolesExt::getVisibility(Point z) {

    RT xmin = (* this->outer_boundary().left_vertex()).x();
    RT xmax = (* this->outer_boundary().right_vertex()).x();
    RT ymax = (* this->outer_boundary().top_vertex()).y();
    RT ymin = (* this->outer_boundary().bottom_vertex()).y();
    
    Polygon pbox;
    pbox.push_back(Point(xmin - 1000 * (xmax - xmin), ymin - 1000 * (ymax - ymin)));
    pbox.push_back(Point(xmin - 1000 * (xmax - xmin), ymax + 1000 * (ymax - ymin)));
    pbox.push_back(Point(xmax + 1000 * (xmax - xmin), ymax + 1000 * (ymax - ymin)));
    pbox.push_back(Point(xmax + 1000 * (xmax - xmin), ymin - 1000 * (ymax - ymin)));

    PolygonExt visZ = (PolygonExt(this->outer_boundary())).getVisibility(z);

    PolygonSet visFinalZ(visZ);

    PolygonWithHoles::Hole_const_iterator holeIt = this->holes_begin();
    for(; holeIt != this->holes_end(); ++holeIt) {

        Polygon holeItPol(*holeIt);
        
        if (! holeItPol.is_counterclockwise_oriented()) {
            holeItPol.reverse_orientation();
        }


        if (!(CGAL::do_intersect (visZ, holeItPol)))
            continue;
        
        std::list<Polygon> convexPartitions;
        CGAL::approx_convex_partition_2(holeItPol.vertices_begin(), holeItPol.vertices_end(), std::back_inserter(convexPartitions));

        std::list<Polygon>::const_iterator it;
        for(it = convexPartitions.begin(); it != convexPartitions.end(); ++it) {

            std::list<Point> blockedPointsZ;

            Polygon::Vertex_iterator vertIt = it->vertices_begin();
            for(;vertIt != it->vertices_end(); ++vertIt) {
                blockedPointsZ.push_back(*vertIt);
                Point wZ = pointOnInfDir(z, (*vertIt), pbox);
                blockedPointsZ.push_back(wZ);
            }

            std::list<Point> chZ;

            CGAL::ch_akl_toussaint(blockedPointsZ.begin(), blockedPointsZ.end(), std::back_inserter(chZ));

            visFinalZ.difference(Polygon(chZ.begin(),chZ.end()));
        }
    }

    std::list<PolygonWithHoles> visPolsZ;
    visFinalZ.polygons_with_holes(std::back_inserter(visPolsZ));

    PolygonExt ret;

    std::list<PolygonWithHoles>::const_iterator polsIt;
    for (polsIt = visPolsZ.begin(); polsIt != visPolsZ.end(); ++polsIt) {
        ret = polsIt->outer_boundary();
        break;
    }

    if (! ret.is_counterclockwise_oriented()) {
        ret.reverse_orientation();
    }
    
    return cleanPol(ret);
}


/**
 * Computes the area of a polygon with holes by removing the area of the
 * boundary polygon and subtracting the area of the holes.
 */
RT PolygonWithHolesExt::area() {
    RT ret = this->outer_boundary().area();

    PolygonWithHoles::Hole_const_iterator holeIt = this->holes_begin();
    for(; holeIt != this->holes_end(); ++holeIt) {

        Polygon holeItPol(*holeIt);
        ret = ret - holeItPol.area();
    }

    return ret;
}

/**
 * Computes the total number of vertices of the polygon with holes.
 */
int PolygonWithHolesExt::size() {
    int ret = this->outer_boundary().size();

    PolygonWithHoles::Hole_const_iterator holeIt = this->holes_begin();
    for(; holeIt != this->holes_end(); ++holeIt) {

        Polygon holeItPol(*holeIt);
        ret = ret + holeItPol.size();
    }

    return ret;
}
