/* Stencil semistencil codes for stencilprobe test
 * Raúl de la Cruz (delacruz@bsc.es)
 * Mauricio Araya-Polo (mauricio.araya@bsc.es)
 * Barcelona Supercomputing Center (BSC)
 *
 * These stencils are based on Chombo's heattut code.
 * (Only the spatial differential operator is computed)
 * Five different lengths of stencils are implemented:
 *
 *   * 7pt stencil (1pt / Original Chombo's heattut)
 *   * 13pt stencil (2pt length per direction)
 *   * 25pt stencil (4pt length per direction)
 *   * 43pt stencil (7pt length per direction)
 *   * 85pt stencil (14pt length per direction)
 *
 * All the points in the stencil are weighted with
 * specific coefficients (cl[z|x|y] coeffs in common.h)
 * (l is the order of the stencil and z, x, y the axies).
 * The operation applied to the domain is the following:
 *
 *                 Y
 *      clz *        O cly    Anext[i,j,k]   = c00 *  A0[i  ,j  ,k  ] +
 *          ·      ·                           c1z * (A0[i+1,j  ,k  ] + A0[i-1,j  ,k  ]) +
 *      c2z *     * c2y                        ···
 *          |   /                              clz * (A0[i+l,j  ,k  ] + A0[i-l,j  ,k  ]) +
 *      c1z *  * c1y                           c1x *  A0[i  ,j+1,k  ] +
 *      c00 |/  c1x c2x clx                    ···
 *          O---*---*···O                      clx *  A0[i  ,j+l,k  ] +
 *  [i,j,k] |            X                     c1y *  A0[i  ,j  ,k+1] +
 *          * c1z                              ···
 *          |                                  cly *  A0[i  ,j  ,k+l]
 *          * c2z
 *          ·                 Anext[i,j+l,k] = c1x *  A0[i  ,j+l-1,k    ] +
 *          * clz                              ···
 *        Z                                    clx *  A0[i  ,j    ,k    ]
 *
 *  * -> Point read           Anext[i,j,k+l] = c1y *  A0[i  ,j    ,k+l-1] +
 *  O -> Point read                            ···
 *       & written                             clx *  A0[i  ,j    ,k    ]
 *
 * Notice that seismic nomenclature is used for these codes:
 * (Z unit-stride dimension, X and Y largest stride dimensions)
 *
 * Three versions for each stencil are implemented:
 *  * Fussion: 1 loop  (by default)
 *  * Fission: 2 loops (OPTS=-DFISSION_2LOOPS)
 *  * Fission: 3 loops (OPTS=-DFISSION_3LOOPS)
 *
 * This semistencil algorithm is applying semi-computation
 * technique only on two of three dimensions (X and Y).
 * When applied on three axis the performance is bad,
 * use OPTS=-DSEMI_ALL to run semi-stencil on three axis.
 *
 * First version November 2009
 *
 * Changelog
 * - February 2011: support for domains smaller than STENCIL sizes
 * - January 2013: Intel pragmas for autovectorization
 * - June 2013: Initial SSE/AVX support for Sandy Bridge (only STENCIL=4)
 *     Use: -DSSE and -DAVX respectively
 * - July 2013: Temporal/non-temporal stores for SSE/AVX
 *     Use: -DSTREAM to active non-temporal stores (cache-bypass)
 * - August 2013: Intel MIC implementation with mm512 intrinsics
 *     Use: -DMIC
 *
 */


#ifndef _SEMI_H_
#define _SEMI_H_


#ifdef SSE
# include <xmmintrin.h>
#endif

#if defined(AVX) || defined(MIC)
# include <immintrin.h>
#endif


#if (!defined(SSE) && !defined(AVX) && !defined(MIC)) // Scalar or auto-vector generated by compiler

/* FORWARD & BACKWARD on each direction for Semi-stencil computations */
# define FORWARDYZERO( p3, p2, length ) FORWARDYZERO_ ## length( p3, p2 )
# define FORWARDXZERO( p3, p2, length ) FORWARDXZERO_ ## length( p3, p2 )
# define FORWARDZZERO( p3, p2, length ) FORWARDZZERO_ ## length( p3, p2 )

# define FORWARDY( p3, p2, length ) FORWARDY_ ## length( p3, p2 )
# define FORWARDX( p3, p2, length ) FORWARDX_ ## length( p3, p2 )
# define FORWARDZ( p3, p2, length ) FORWARDZ_ ## length( p3, p2 )

# define BACKWARDY( p3, p2, length ) BACKWARDY_ ## length( p3, p2 )
# define BACKWARDX( p3, p2, length ) BACKWARDX_ ## length( p3, p2 )
# define BACKWARDZ( p3, p2, length ) BACKWARDZ_ ## length( p3, p2 )

/* BODY for Semi-stencil in YX directions - GOOD! */
# define BODYZ( p3, p2, length )     BODYZ_ ## length( p3, p2 )

/* BODY for Semi-stencil in all directions (ZYX) - Not good */
# define BODYZ_ALL( p3, p2, length ) BODYZ_ALL_ ## length( p3, p2 )

/* BODYs for fission in 2 loops */
# define BODYZ1_2LOOPS( p3, p2, length )     BODYZ1_2LOOPS_ ## length( p3, p2 )
# define BODYZ2_2LOOPS( p3, p2, length )     BODYZ2_2LOOPS_ ## length( p3, p2 )

/* BODYs for fission in 3 loops */
# define BODYZ1_3LOOPS( p3, p2, length )     BODYZ1_3LOOPS_ ## length ( p3, p2 )
# define BODYZ2_3LOOPS( p3, p2, length )     BODYZ2_3LOOPS_ ## length ( p3, p2 )
# define BODYZ3_3LOOPS( p3, p2, length )     BODYZ3_3LOOPS_ ## length ( p3, p2 )

# define IINC 1

#elif defined(SSE) // 2 elements per vector (128 bits)

# error "SSE version not yet implemented!"

# define IINC 2

#elif defined(AVX) // 4 elements per vector (256 bits)

/* FORWARD & BACKWARD on each direction for Semi-stencil computations */
# define FORWARDYZERO( p3, p2, length ) FORWARDYZERO_AVX_ ## length( p3, p2 )
# define FORWARDXZERO( p3, p2, length ) FORWARDXZERO_AVX_ ## length( p3, p2 )
# define FORWARDZZERO( p3, p2, length ) FORWARDZZERO_AVX_ ## length( p3, p2 )

# define FORWARDY( p3, p2, length ) FORWARDY_AVX_ ## length( p3, p2 )
# define FORWARDX( p3, p2, length ) FORWARDX_AVX_ ## length( p3, p2 )
# define FORWARDZ( p3, p2, length ) FORWARDZ_AVX_ ## length( p3, p2 )

# define BACKWARDY( p3, p2, length ) BACKWARDY_AVX_ ## length( p3, p2 )
# define BACKWARDX( p3, p2, length ) BACKWARDX_AVX_ ## length( p3, p2 )
# define BACKWARDZ( p3, p2, length ) BACKWARDZ_AVX_ ## length( p3, p2 )

/* BODY for Semi-stencil in YX directions - GOOD! */
# define BODYZ( p3, p2, length )     BODYZ_AVX_ ## length( p3, p2 )

/* BODY for Semi-stencil in all directions (ZYX) - Not good */
# define BODYZ_ALL( p3, p2, length ) BODYZ_ALL_AVX_ ## length( p3, p2 )

/* BODYs for fission in 2 loops */
# define BODYZ1_2LOOPS( p3, p2, length )     BODYZ1_2LOOPS_AVX_ ## length( p3, p2 )
# define BODYZ2_2LOOPS( p3, p2, length )     BODYZ2_2LOOPS_AVX_ ## length( p3, p2 )

/* BODYs for fission in 3 loops */
# define BODYZ1_3LOOPS( p3, p2, length )     BODYZ1_3LOOPS_AVX_ ## length ( p3, p2 )
# define BODYZ2_3LOOPS( p3, p2, length )     BODYZ2_3LOOPS_AVX_ ## length ( p3, p2 )
# define BODYZ3_3LOOPS( p3, p2, length )     BODYZ3_3LOOPS_AVX_ ## length ( p3, p2 )

# define IINC 4

# ifndef STREAM
#   define _mm256_stream_pd _mm256_store_pd
# endif

# elif defined(MIC)

/* FORWARD & BACKWARD on each direction for Semi-stencil computations */
# define FORWARDYZERO( p3, p2, length ) FORWARDYZERO_MIC_ ## length( p3, p2 )
# define FORWARDXZERO( p3, p2, length ) FORWARDXZERO_MIC_ ## length( p3, p2 )
# define FORWARDZZERO( p3, p2, length ) FORWARDZZERO_MIC_ ## length( p3, p2 )

# define FORWARDY( p3, p2, length ) FORWARDY_MIC_ ## length( p3, p2 )
# define FORWARDX( p3, p2, length ) FORWARDX_MIC_ ## length( p3, p2 )
# define FORWARDZ( p3, p2, length ) FORWARDZ_MIC_ ## length( p3, p2 )

# define BACKWARDY( p3, p2, length ) BACKWARDY_MIC_ ## length( p3, p2 )
# define BACKWARDX( p3, p2, length ) BACKWARDX_MIC_ ## length( p3, p2 )
# define BACKWARDZ( p3, p2, length ) BACKWARDZ_MIC_ ## length( p3, p2 )

/* BODY for Semi-stencil in YX directions - GOOD! */
# define BODYZ( p3, p2, length )     BODYZ_MIC_ ## length( p3, p2 )

/* BODY for Semi-stencil in all directions (ZYX) - Not good */
# define BODYZ_ALL( p3, p2, length ) BODYZ_ALL_MIC_ ## length( p3, p2 )

/* BODYs for fission in 2 loops */
# define BODYZ1_2LOOPS( p3, p2, length )     BODYZ1_2LOOPS_MIC_ ## length( p3, p2 )
# define BODYZ2_2LOOPS( p3, p2, length )     BODYZ2_2LOOPS_MIC_ ## length( p3, p2 )

/* BODYs for fission in 3 loops */
# define BODYZ1_3LOOPS( p3, p2, length )     BODYZ1_3LOOPS_MIC_ ## length ( p3, p2 )
# define BODYZ2_3LOOPS( p3, p2, length )     BODYZ2_3LOOPS_MIC_ ## length ( p3, p2 )
# define BODYZ3_3LOOPS( p3, p2, length )     BODYZ3_3LOOPS_MIC_ ## length ( p3, p2 )

# define IINC 8

# ifndef STREAM
#   define _mm512_storenr_pd _mm512_store_pd
# endif

#endif


/* 7pt stencil */
#define FORWARDYZERO_1( p3, p2 )        \
      p3[Index3D(nx, ny, i, j, k+1)] =  \
        c1y * p2[Index3D(nx, ny, i, j,   k)];

#define FORWARDXZERO_1( p3, p2 )                                        \
      p3[Index3D(nx, ny, i, j+1, k)] = p3[Index3D(nx, ny, i, j+1, k)] + \
        c1x * p2[Index3D(nx, ny, i,   j, k)];

#define FORWARDZZERO_1( p3, p2 )                                        \
      p3[Index3D(nx, ny, i+1, j, k)] = p3[Index3D(nx, ny, i+1, j, k)] + \
        c1z * p2[Index3D(nx, ny,   i, j, k)];

#define FORWARDY_1( p3, p2 )           \
      p3[Index3D(nx, ny, i, j, k+1)] = \
        c1y * p2c;

#define FORWARDX_1( p3, p2 )                                            \
      p3[Index3D(nx, ny, i, j+1, k)] = p3[Index3D(nx, ny, i, j+1, k)] + \
        c1x * p2c;

#define FORWARDZ_1( p3, p2 )                                            \
      p3[Index3D(nx, ny, i+1, j, k)] = p3[Index3D(nx, ny, i+1, j, k)] + \
        c1z * p2c;

#define BACKWARDY_1( p3, p2 )                                       \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1y * p2[Index3D(nx, ny, i, j, k+1)];

#define BACKWARDX_1( p3, p2 )                                       \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1x * p2[Index3D(nx, ny, i, j+1, k)];

#define BACKWARDZ_1( p3, p2 )                                       \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1z * p2[Index3D(nx, ny, i+1, j, k)];

#define BODYZ_1( p3, p2 )                                           \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)];                        \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  +                                                \
        c1z * p2[Index3D(nx, ny, i-1, j, k)] +                      \
        c1z * p2[Index3D(nx, ny, i+1, j, k)] +                      \
                                                                    \
        c1x * p2j1 +                                                \
                                                                    \
        c1y * p2k1;

#define BODYZ_ALL_1( p3, p2 )                                       \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
      p2i1 = p2[Index3D(nx, ny, i+1, j, k)];                        \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)];                        \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  +                                                \
        c1z * p2i1 +                                                \
                                                                    \
        c1x * p2j1 +                                                \
                                                                    \
        c1y * p2k1;

#define BODYZ1_2LOOPS_1( p3, p2 )                                   \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  +                                                \
                                                                    \
        c1z * p2[Index3D(nx, ny, i-1, j, k)] +                      \
        c1z * p2[Index3D(nx, ny, i+1, j, k)] +                      \
                                                                    \
        c1y * p2k1;

#define BODYZ2_2LOOPS_1( p3, p2 )                                   \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1x * p2j1;

#define BODYZ1_3LOOPS_1( p3, p2 )                                   \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  +                                                \
                                                                    \
        c1z * p2[Index3D(nx, ny, i-1, j, k)] +                      \
        c1z * p2[Index3D(nx, ny, i+1, j, k)];

#define BODYZ2_3LOOPS_1( p3, p2 )                                   \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1y * p2k1;

#define BODYZ3_3LOOPS_1( p3, p2 )                                   \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1x * p2j1;



/* 13pt stencil */
#define FORWARDYZERO_2( p3, p2 )                \
      p3[Index3D(nx, ny, i, j, k+2)] =          \
        c2y * p2[Index3D(nx, ny, i, j,   k)] +  \
        c1y * p2[Index3D(nx, ny, i, j, k+1)];

#define FORWARDXZERO_2( p3, p2 )                                        \
      p3[Index3D(nx, ny, i, j+2, k)] = p3[Index3D(nx, ny, i, j+2, k)] + \
        c2x * p2[Index3D(nx, ny, i,   j, k)] +                          \
        c1x * p2[Index3D(nx, ny, i, j+1, k)];

#define FORWARDZZERO_2( p3, p2 )                                        \
      p3[Index3D(nx, ny, i+2, j, k)] = p3[Index3D(nx, ny, i+2, j, k)] + \
        c2z * p2[Index3D(nx, ny,   i, j, k)] +                          \
        c1z * p2[Index3D(nx, ny, i+1, j, k)];

#define FORWARDY_2( p3, p2 )            \
      p3[Index3D(nx, ny, i, j, k+2)] =  \
        c2y * p2c  +                    \
        c1y * p2k1;

#define FORWARDX_2( p3, p2 )                                            \
      p3[Index3D(nx, ny, i, j+2, k)] = p3[Index3D(nx, ny, i, j+2, k)] + \
        c2x * p2c  +                                                    \
        c1x * p2j1;

#define FORWARDZ_2( p3, p2 )                                            \
      p3[Index3D(nx, ny, i+2, j, k)] = p3[Index3D(nx, ny, i+2, j, k)] + \
        c2z * p2c  +                                                    \
        c1z * p2i1;

#define BACKWARDY_2( p3, p2 )                                       \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1y * p2[Index3D(nx, ny, i, j, k+1)] +                      \
        c2y * p2[Index3D(nx, ny, i, j, k+2)];

#define BACKWARDX_2( p3, p2 )                                       \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1x * p2[Index3D(nx, ny, i, j+1, k)] +                      \
        c2x * p2[Index3D(nx, ny, i, j+2, k)];

#define BACKWARDZ_2( p3, p2 )                                       \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1z * p2[Index3D(nx, ny, i+1, j, k)] +                      \
        c2z * p2[Index3D(nx, ny, i+2, j, k)];

#define BODYZ_2( p3, p2 )                                           \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)];                        \
      p2j2 = p2[Index3D(nx, ny, i, j+2, k)];                        \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)];                        \
      p2k2 = p2[Index3D(nx, ny, i, j, k+2)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  +                                                \
        c2z * p2[Index3D(nx, ny, i-2, j, k)] +                      \
        c2z * p2[Index3D(nx, ny, i+2, j, k)] +                      \
        c1z * p2[Index3D(nx, ny, i-1, j, k)] +                      \
        c1z * p2[Index3D(nx, ny, i+1, j, k)] +                      \
                                                                    \
        c2x * p2j2 +                                                \
        c1x * p2j1 +                                                \
                                                                    \
        c2y * p2k2 +                                                \
        c1y * p2k1;

#define BODYZ_ALL_2( p3, p2 )                                       \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
      p2i1 = p2[Index3D(nx, ny, i+1, j, k)];                        \
      p2i2 = p2[Index3D(nx, ny, i+2, j, k)];                        \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)];                        \
      p2j2 = p2[Index3D(nx, ny, i, j+2, k)];                        \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)];                        \
      p2k2 = p2[Index3D(nx, ny, i, j, k+2)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  +                                                \
        c2z * p2i2 +                                                \
        c1z * p2i1 +                                                \
                                                                    \
        c2x * p2j2 +                                                \
        c1x * p2j1 +                                                \
                                                                    \
        c2y * p2k2 +                                                \
        c1y * p2k1;

#define BODYZ1_2LOOPS_2( p3, p2 )                                   \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)];                        \
      p2k2 = p2[Index3D(nx, ny, i, j, k+2)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  +                                                \
                                                                    \
        c2z * p2[Index3D(nx, ny, i-2, j, k)] +                      \
        c2z * p2[Index3D(nx, ny, i+2, j, k)] +                      \
        c1z * p2[Index3D(nx, ny, i-1, j, k)] +                      \
        c1z * p2[Index3D(nx, ny, i+1, j, k)] +                      \
                                                                    \
        c2y * p2k2 +                                                \
        c1y * p2k1;

#define BODYZ2_2LOOPS_2( p3, p2 )                                   \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)];                        \
      p2j2 = p2[Index3D(nx, ny, i, j+2, k)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c2x * p2j2 +                                                \
        c1x * p2j1;

#define BODYZ1_3LOOPS_2( p3, p2 )                                   \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  +                                                \
                                                                    \
        c2z * p2[Index3D(nx, ny, i-2, j, k)] +                      \
        c2z * p2[Index3D(nx, ny, i+2, j, k)] +                      \
        c1z * p2[Index3D(nx, ny, i-1, j, k)] +                      \
        c1z * p2[Index3D(nx, ny, i+1, j, k)];

#define BODYZ2_3LOOPS_2( p3, p2 )                                   \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)];                        \
      p2k2 = p2[Index3D(nx, ny, i, j, k+2)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c2y * p2k2 +                                                \
        c1y * p2k1;

#define BODYZ3_3LOOPS_2( p3, p2 )                                   \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)];                        \
      p2j2 = p2[Index3D(nx, ny, i, j+2, k)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c2x * p2j2 +                                                \
        c1x * p2j1;



/* 25pt stencil */
#define FORWARDYZERO_4( p3, p2 )                \
      p3[Index3D(nx, ny, i, j, k+4)] =          \
        c4y * p2[Index3D(nx, ny, i, j,   k)] +  \
        c3y * p2[Index3D(nx, ny, i, j, k+1)] +  \
        c2y * p2[Index3D(nx, ny, i, j, k+2)] +  \
        c1y * p2[Index3D(nx, ny, i, j, k+3)];


#define FORWARDYZERO_AVX_4( p3, p2 )                                         \
          __m256d A0zce = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k  )] ); \
          __m256d A0yp1 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+1)] ); \
          __m256d A0yp2 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+2)] ); \
          __m256d A0yp3 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+3)] ); \
\
          __m256d tempy = _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0zce, c4yv ),   \
                                                        _mm256_mul_pd( A0yp1, c3yv )),  \
                                         _mm256_add_pd( _mm256_mul_pd( A0yp2, c2yv ),   \
                                                        _mm256_mul_pd( A0yp3, c1yv ))); \
\
          _mm256_stream_pd( &p3[Index3D(nx, ny, i, j, k+4)], tempy );


#define FORWARDYZERO_MIC_4( p3, p2 )                                         \
          __m512d A0zce = _mm512_load_pd( &p2[Index3D(nx, ny, i, j, k  )] ); \
          __m512d A0yp1 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j, k+1)] ); \
          __m512d A0yp2 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j, k+2)] ); \
          __m512d A0yp3 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j, k+3)] ); \
\
          __m512d tempy1 = _mm512_mul_pd( A0zce, c4yv ); \
          __m512d tempy2 = _mm512_mul_pd( A0yp1, c3yv ); \
                  tempy1 = _mm512_fmadd_pd( A0yp2, c2yv, tempy1 ); \
                  tempy2 = _mm512_fmadd_pd( A0yp3, c1yv, tempy2 ); \
\
                  tempy1 = _mm512_add_pd( tempy1, tempy2 ); \
\
          _mm512_storenr_pd( &p3[Index3D(nx, ny, i, j, k+4)], tempy1 );


#define FORWARDXZERO_4( p3, p2 )                                        \
      p3[Index3D(nx, ny, i, j+4, k)] = p3[Index3D(nx, ny, i, j+4, k)] + \
        c4x * p2[Index3D(nx, ny, i,   j, k)] +                          \
        c3x * p2[Index3D(nx, ny, i, j+1, k)] +                          \
        c2x * p2[Index3D(nx, ny, i, j+2, k)] +                          \
        c1x * p2[Index3D(nx, ny, i, j+3, k)];


#define FORWARDXZERO_AVX_4( p3, p2 )                                         \
          __m256d Anextxv = _mm256_load_pd( &p3[Index3D(nx, ny, i, j+4, k)] );  \
\
          __m256d A0zce = _mm256_load_pd( &p2[Index3D(nx, ny, i, j  , k)] ); \
          __m256d A0xp1 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+1, k)] ); \
          __m256d A0xp2 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+2, k)] ); \
          __m256d A0xp3 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+3, k)] ); \
\
          __m256d tempx = _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0zce, c4xv ),   \
                                                        _mm256_mul_pd( A0xp1, c3xv )),  \
                                         _mm256_add_pd( _mm256_mul_pd( A0xp2, c2xv ),   \
                                                        _mm256_mul_pd( A0xp3, c1xv ))); \
\
          Anextxv = _mm256_add_pd( Anextxv, tempx );                            \
          _mm256_store_pd( &p3[Index3D(nx, ny, i, j+4, k)], Anextxv );


#define FORWARDXZERO_MIC_4( p3, p2 )                                         \
          __m512d Anextxv = _mm512_load_pd( &p3[Index3D(nx, ny, i, j+4, k)] );  \
\
          __m512d A0zce = _mm512_load_pd( &p2[Index3D(nx, ny, i, j  , k)] ); \
          __m512d A0xp1 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j+1, k)] ); \
          __m512d A0xp2 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j+2, k)] ); \
          __m512d A0xp3 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j+3, k)] ); \
\
          __m512d tempx1 = _mm512_mul_pd( A0zce, c4xv ); \
          __m512d tempx2 = _mm512_mul_pd( A0xp1, c3xv ); \
                  tempx1 = _mm512_fmadd_pd( A0xp2, c2xv, tempx1 ); \
                  tempx2 = _mm512_fmadd_pd( A0xp3, c1xv, tempx2 ); \
\
                  tempx1 = _mm512_add_pd( tempx1, tempx2 ); \
                  Anextxv = _mm512_add_pd( Anextxv, tempx1 ); \
\
          _mm512_store_pd( &p3[Index3D(nx, ny, i, j+4, k)], Anextxv );


#define FORWARDZZERO_4( p3, p2 )                                        \
      p3[Index3D(nx, ny, i+4, j, k)] = p3[Index3D(nx, ny, i+4, j, k)] + \
        c4z * p2[Index3D(nx, ny,   i, j, k)] +                          \
        c3z * p2[Index3D(nx, ny, i+1, j, k)] +                          \
        c2z * p2[Index3D(nx, ny, i+2, j, k)] +                          \
        c1z * p2[Index3D(nx, ny, i+3, j, k)];

#define FORWARDY_4( p3, p2 )            \
      p3[Index3D(nx, ny, i, j, k+4)] =  \
        c4y * p2c  +                    \
        c3y * p2k1 +                    \
        c2y * p2k2 +                    \
        c1y * p2k3;


#define FORWARDY_AVX_4( p3, p2 )        \
          tempy = _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0zce, c4yv ),   \
                                                _mm256_mul_pd( A0yp1, c3yv )),  \
                                 _mm256_add_pd( _mm256_mul_pd( A0yp2, c2yv ),   \
                                                _mm256_mul_pd( A0yp3, c1yv ))); \
\
          _mm256_stream_pd( &p3[Index3D(nx, ny, i, j, k+4)], tempy );


#define FORWARDY_MIC_4( p3, p2 )        \
          __m512d tempy1 = _mm512_mul_pd( A0zce, c4yv ); \
          __m512d tempy2 = _mm512_mul_pd( A0yp1, c3yv ); \
                  tempy1 = _mm512_fmadd_pd( A0yp2, c2yv, tempy1 ); \
                  tempy2 = _mm512_fmadd_pd( A0yp3, c1yv, tempy2 ); \
\
                  tempy1 = _mm512_add_pd( tempy1, tempy2 ); \
\
          _mm512_storenr_pd( &p3[Index3D(nx, ny, i, j, k+4)], tempy1 );


#define FORWARDX_4( p3, p2 )                                            \
      p3[Index3D(nx, ny, i, j+4, k)] = p3[Index3D(nx, ny, i, j+4, k)] + \
        c4x * p2c  +                                                    \
        c3x * p2j1 +                                                    \
        c2x * p2j2 +                                                    \
        c1x * p2j3;


#define FORWARDX_AVX_4( p3, p2 )                                                \
          __m256d Anextxv = _mm256_load_pd( &p3[Index3D(nx, ny, i, j+4, k)] );  \
\
          tempx = _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0zce, c4xv ),   \
                                                _mm256_mul_pd( A0xp1, c3xv )),  \
                                 _mm256_add_pd( _mm256_mul_pd( A0xp2, c2xv ),   \
                                                _mm256_mul_pd( A0xp3, c1xv ))); \
\
          Anextxv = _mm256_add_pd( Anextxv, tempx );                            \
          _mm256_store_pd( &p3[Index3D(nx, ny, i, j+4, k)], Anextxv );


#define FORWARDX_MIC_4( p3, p2 )                                                \
          __m512d Anextxv = _mm512_load_pd( &p3[Index3D(nx, ny, i, j+4, k)] );  \
\
          __m512d tempx1 = _mm512_mul_pd( A0zce, c4xv ); \
          __m512d tempx2 = _mm512_mul_pd( A0xp1, c3xv ); \
                  tempx1 = _mm512_fmadd_pd( A0xp2, c2xv, tempx1 ); \
                  tempx2 = _mm512_fmadd_pd( A0xp3, c1xv, tempx2 ); \
\
                  tempx1 = _mm512_add_pd( tempx1, tempx2 ); \
                  Anextxv = _mm512_add_pd( Anextxv, tempx1 ); \
\
          _mm512_store_pd( &p3[Index3D(nx, ny, i, j+4, k)], Anextxv );


#define FORWARDZ_4( p3, p2 )                                            \
      p3[Index3D(nx, ny, i+4, j, k)] = p3[Index3D(nx, ny, i+4, j, k)] + \
        c4z * p2c  +                                                    \
        c3z * p2i1 +                                                    \
        c2z * p2i2 +                                                    \
        c1z * p2i3;

#define BACKWARDY_4( p3, p2 )                                       \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1y * p2[Index3D(nx, ny, i, j, k+1)] +                      \
        c2y * p2[Index3D(nx, ny, i, j, k+2)] +                      \
        c3y * p2[Index3D(nx, ny, i, j, k+3)] +                      \
        c4y * p2[Index3D(nx, ny, i, j, k+4)];

#define BACKWARDX_4( p3, p2 )                                       \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1x * p2[Index3D(nx, ny, i, j+1, k)] +                      \
        c2x * p2[Index3D(nx, ny, i, j+2, k)] +                      \
        c3x * p2[Index3D(nx, ny, i, j+3, k)] +                      \
        c4x * p2[Index3D(nx, ny, i, j+4, k)];

#define BACKWARDZ_4( p3, p2 ) \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1z * p2[Index3D(nx, ny, i+1, j, k)] +                      \
        c2z * p2[Index3D(nx, ny, i+2, j, k)] +                      \
        c3z * p2[Index3D(nx, ny, i+3, j, k)] +                      \
        c4z * p2[Index3D(nx, ny, i+4, j, k)];

#define BODYZ_4( p3, p2 ) \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];                        \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)];                        \
      p2j2 = p2[Index3D(nx, ny, i, j+2, k)];                        \
      p2j3 = p2[Index3D(nx, ny, i, j+3, k)];                        \
      p2j4 = p2[Index3D(nx, ny, i, j+4, k)];                        \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)];                        \
      p2k2 = p2[Index3D(nx, ny, i, j, k+2)];                        \
      p2k3 = p2[Index3D(nx, ny, i, j, k+3)];                        \
      p2k4 = p2[Index3D(nx, ny, i, j, k+4)];                        \
                                                                    \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  +                                                \
        c4z * p2[Index3D(nx, ny, i-4, j, k)] +                      \
        c4z * p2[Index3D(nx, ny, i+4, j, k)] +                      \
        c3z * p2[Index3D(nx, ny, i-3, j, k)] +                      \
        c3z * p2[Index3D(nx, ny, i+3, j, k)] +                      \
        c2z * p2[Index3D(nx, ny, i-2, j, k)] +                      \
        c2z * p2[Index3D(nx, ny, i+2, j, k)] +                      \
        c1z * p2[Index3D(nx, ny, i-1, j, k)] +                      \
        c1z * p2[Index3D(nx, ny, i+1, j, k)] +                      \
                                                                    \
        c4x * p2j4 +                                                \
        c3x * p2j3 +                                                \
        c2x * p2j2 +                                                \
        c1x * p2j1 +                                                \
                                                                    \
        c4y * p2k4 +                                                \
        c3y * p2k3 +                                                \
        c2y * p2k2 +                                                \
        c1y * p2k1;

#define BODYZ_AVX_4( p3, p2 ) \
          __m256d A0zm4 = _mm256_load_pd( &p2[Index3D(nx, ny, i-4, j, k)] ); \
          __m256d A0zce = _mm256_load_pd( &p2[Index3D(nx, ny, i,   j, k)] ); \
          __m256d A0zp4 = _mm256_load_pd( &p2[Index3D(nx, ny, i+4, j, k)] ); \
          __m256d A0xp1 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+1, k)] ); \
          __m256d A0xp2 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+2, k)] ); \
          __m256d A0xp3 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+3, k)] ); \
          __m256d A0xp4 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+4, k)] ); \
          __m256d A0yp1 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+1)] ); \
          __m256d A0yp2 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+2)] ); \
          __m256d A0yp3 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+3)] ); \
          __m256d A0yp4 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+4)] ); \
\
          __m256d A0zm2 = _mm256_permute2f128_pd( A0zm4, A0zce, 0x21 ); /* 100001 */ \
          __m256d A0zm3 = _mm256_shuffle_pd( A0zm4, A0zm2, 0x5 );       /* 0101   */ \
          __m256d A0zm1 = _mm256_shuffle_pd( A0zm2, A0zce, 0x5 );       /* 0101   */ \
          __m256d A0zp2 = _mm256_permute2f128_pd( A0zce, A0zp4, 0x21 ); /* 100001 */ \
          __m256d A0zp1 = _mm256_shuffle_pd( A0zce, A0zp2, 0x5 );       /* 0101   */ \
          __m256d A0zp3 = _mm256_shuffle_pd( A0zp2, A0zp4, 0x5 );       /* 0101   */ \
\
          __m256d Anextv = _mm256_load_pd( &p3[Index3D(nx, ny, i, j, k)] ); \
\
          __m256d tempz = _mm256_add_pd( _mm256_mul_pd( A0zce, c00v ), \
                                         _mm256_add_pd( _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0zm1, c1zv ),     \
                                                                                      _mm256_mul_pd( A0zm2, c2zv )),    \
                                                                       _mm256_add_pd( _mm256_mul_pd( A0zm3, c3zv ),     \
                                                                                      _mm256_mul_pd( A0zm4, c4zv ))),   \
                                                        _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0zp1, c1zv ),     \
                                                                                      _mm256_mul_pd( A0zp2, c2zv )),    \
                                                                       _mm256_add_pd( _mm256_mul_pd( A0zp3, c3zv ),     \
                                                                                      _mm256_mul_pd( A0zp4, c4zv ))))); \
\
          __m256d tempx = _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0xp1, c1xv ),    \
                                                        _mm256_mul_pd( A0xp2, c2xv )),   \
                                         _mm256_add_pd( _mm256_mul_pd( A0xp3, c3xv ),    \
                                                        _mm256_mul_pd( A0xp4, c4xv )));  \
\
          __m256d tempy = _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0yp1, c1yv ), \
                                                        _mm256_mul_pd( A0yp2, c2yv )), \
                                         _mm256_add_pd( _mm256_mul_pd( A0yp3, c3yv ), \
                                                        _mm256_mul_pd( A0yp4, c4yv ))); \
\
          Anextv = _mm256_add_pd( _mm256_add_pd( tempz, tempx ), _mm256_add_pd( tempy, Anextv ) ); \
          _mm256_store_pd( &p3[Index3D(nx, ny, i, j, k)], Anextv );


#define BODYZ_MIC_4( p3, p2 ) \
          __m512d A0zm8 = _mm512_load_pd( &p2[Index3D(nx, ny, i-8, j, k)] ); \
          __m512d A0zce = _mm512_load_pd( &p2[Index3D(nx, ny, i,   j, k)] ); \
          __m512d A0zp8 = _mm512_load_pd( &p2[Index3D(nx, ny, i+8, j, k)] ); \
          __m512d A0xp1 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j+1, k)] ); \
          __m512d A0xp2 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j+2, k)] ); \
          __m512d A0xp3 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j+3, k)] ); \
          __m512d A0xp4 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j+4, k)] ); \
          __m512d A0yp1 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j, k+1)] ); \
          __m512d A0yp2 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j, k+2)] ); \
          __m512d A0yp3 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j, k+3)] ); \
          __m512d A0yp4 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j, k+4)] ); \
\
          __m512d A0zm4 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zce, (__m512i)A0zm8, 8  ); \
          __m512d A0zm3 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zce, (__m512i)A0zm8, 10 ); \
          __m512d A0zm2 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zce, (__m512i)A0zm8, 12 ); \
          __m512d A0zm1 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zce, (__m512i)A0zm8, 14 ); \
          __m512d A0zp1 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zp8, (__m512i)A0zce, 2  ); \
          __m512d A0zp2 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zp8, (__m512i)A0zce, 4  ); \
          __m512d A0zp3 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zp8, (__m512i)A0zce, 6  ); \
          __m512d A0zp4 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zp8, (__m512i)A0zce, 8  ); \
\
          __m512d Anextv = _mm512_load_pd( &p3[Index3D(nx, ny, i, j, k)] ); \
\
          __m512d temp1 = _mm512_mul_pd( A0zce, c00v ); \
          __m512d temp2 = _mm512_mul_pd( A0zm1, c1zv ); \
                  temp1 = _mm512_fmadd_pd( A0zm2, c2zv, temp1 ); \
                  temp2 = _mm512_fmadd_pd( A0zm3, c3zv, temp2 ); \
                  temp1 = _mm512_fmadd_pd( A0zm4, c4zv, temp1 ); \
                  temp2 = _mm512_fmadd_pd( A0zp1, c1zv, temp2 ); \
                  temp1 = _mm512_fmadd_pd( A0zp2, c2zv, temp1 ); \
                  temp2 = _mm512_fmadd_pd( A0zp3, c3zv, temp2 ); \
                  temp1 = _mm512_fmadd_pd( A0zp4, c4zv, temp1 ); \
\
                  temp1 = _mm512_fmadd_pd( A0xp1, c1xv, temp1 ); \
                  temp2 = _mm512_fmadd_pd( A0xp2, c2xv, temp2 ); \
                  temp1 = _mm512_fmadd_pd( A0xp3, c3xv, temp1 ); \
                  temp2 = _mm512_fmadd_pd( A0xp4, c4xv, temp2 ); \
\
                  temp1 = _mm512_fmadd_pd( A0yp1, c1yv, temp1 ); \
                  temp2 = _mm512_fmadd_pd( A0yp2, c2yv, temp2 ); \
                  temp1 = _mm512_fmadd_pd( A0yp3, c3yv, temp1 ); \
                  temp2 = _mm512_fmadd_pd( A0yp4, c4yv, temp2 ); \
\
          Anextv = _mm512_add_pd( _mm512_add_pd( temp1, temp2 ), Anextv ); \
          _mm512_store_pd( &p3[Index3D(nx, ny, i, j, k)], Anextv );


#define BODYZ_ALL_4( p3, p2 )                \
      p2c  = p2[Index3D(nx, ny, i,   j, k)]; \
      p2i1 = p2[Index3D(nx, ny, i+1, j, k)]; \
      p2i2 = p2[Index3D(nx, ny, i+2, j, k)]; \
      p2i3 = p2[Index3D(nx, ny, i+3, j, k)]; \
      p2i4 = p2[Index3D(nx, ny, i+4, j, k)]; \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)]; \
      p2j2 = p2[Index3D(nx, ny, i, j+2, k)]; \
      p2j3 = p2[Index3D(nx, ny, i, j+3, k)]; \
      p2j4 = p2[Index3D(nx, ny, i, j+4, k)]; \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)]; \
      p2k2 = p2[Index3D(nx, ny, i, j, k+2)]; \
      p2k3 = p2[Index3D(nx, ny, i, j, k+3)]; \
      p2k4 = p2[Index3D(nx, ny, i, j, k+4)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  + \
        c4z * p2i4 + \
        c3z * p2i3 + \
        c2z * p2i2 + \
        c1z * p2i1 + \
\
        c4x * p2j4 + \
        c3x * p2j3 + \
        c2x * p2j2 + \
        c1x * p2j1 + \
\
        c4y * p2k4 + \
        c3y * p2k3 + \
        c2y * p2k2 + \
        c1y * p2k1;

#define BODYZ1_2LOOPS_4( p3, p2 )            \
      p2c  = p2[Index3D(nx, ny, i,   j, k)]; \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)]; \
      p2k2 = p2[Index3D(nx, ny, i, j, k+2)]; \
      p2k3 = p2[Index3D(nx, ny, i, j, k+3)]; \
      p2k4 = p2[Index3D(nx, ny, i, j, k+4)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  + \
\
        c4z * p2[Index3D(nx, ny, i-4, j, k)] + \
        c4z * p2[Index3D(nx, ny, i+4, j, k)] + \
        c3z * p2[Index3D(nx, ny, i-3, j, k)] + \
        c3z * p2[Index3D(nx, ny, i+3, j, k)] + \
        c2z * p2[Index3D(nx, ny, i-2, j, k)] + \
        c2z * p2[Index3D(nx, ny, i+2, j, k)] + \
        c1z * p2[Index3D(nx, ny, i-1, j, k)] + \
        c1z * p2[Index3D(nx, ny, i+1, j, k)] + \
\
        c4y * p2k4 + \
        c3y * p2k3 + \
        c2y * p2k2 + \
        c1y * p2k1;

#define BODYZ2_2LOOPS_4( p3, p2 )            \
      p2c  = p2[Index3D(nx, ny, i,   j, k)]; \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)]; \
      p2j2 = p2[Index3D(nx, ny, i, j+2, k)]; \
      p2j3 = p2[Index3D(nx, ny, i, j+3, k)]; \
      p2j4 = p2[Index3D(nx, ny, i, j+4, k)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c4x * p2j4 + \
        c3x * p2j3 + \
        c2x * p2j2 + \
        c1x * p2j1;


#define BODYZ1_2LOOPS_AVX_4( p3, p2 )               \
          __m256d A0zm4 = _mm256_load_pd( &p2[Index3D(nx, ny, i-4, j, k)] ); \
          __m256d A0zce = _mm256_load_pd( &p2[Index3D(nx, ny, i,   j, k)] ); \
          __m256d A0zp4 = _mm256_load_pd( &p2[Index3D(nx, ny, i+4, j, k)] ); \
\
          __m256d A0yp1 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+1)] ); \
          __m256d A0yp2 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+2)] ); \
          __m256d A0yp3 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+3)] ); \
          __m256d A0yp4 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+4)] ); \
\
          __m256d A0zm2 = _mm256_permute2f128_pd( A0zm4, A0zce, 0x21 ); /* 100001 */ \
          __m256d A0zm3 = _mm256_shuffle_pd( A0zm4, A0zm2, 0x5 );       /* 0101   */ \
          __m256d A0zm1 = _mm256_shuffle_pd( A0zm2, A0zce, 0x5 );       /* 0101   */ \
          __m256d A0zp2 = _mm256_permute2f128_pd( A0zce, A0zp4, 0x21 ); /* 100001 */ \
          __m256d A0zp1 = _mm256_shuffle_pd( A0zce, A0zp2, 0x5 );       /* 0101   */ \
          __m256d A0zp3 = _mm256_shuffle_pd( A0zp2, A0zp4, 0x5 );       /* 0101   */ \
\
          __m256d Anextv = _mm256_load_pd( &p3[Index3D(nx, ny, i, j, k)] ); \
\
          __m256d tempz = _mm256_add_pd( _mm256_mul_pd( A0zce, c00v ), \
                                         _mm256_add_pd( _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0zm1, c1zv ),     \
                                                                                      _mm256_mul_pd( A0zm2, c2zv )),    \
                                                                       _mm256_add_pd( _mm256_mul_pd( A0zm3, c3zv ),     \
                                                                                      _mm256_mul_pd( A0zm4, c4zv ))),   \
                                                        _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0zp1, c1zv ),     \
                                                                                      _mm256_mul_pd( A0zp2, c2zv )),    \
                                                                       _mm256_add_pd( _mm256_mul_pd( A0zp3, c3zv ),     \
                                                                                      _mm256_mul_pd( A0zp4, c4zv ))))); \
\
          __m256d tempy = _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0yp1, c1yv ), \
                                                        _mm256_mul_pd( A0yp2, c2yv )), \
                                         _mm256_add_pd( _mm256_mul_pd( A0yp3, c3yv ), \
                                                        _mm256_mul_pd( A0yp4, c4yv ))); \
\
          Anextv = _mm256_add_pd( tempz, _mm256_add_pd( tempy, Anextv ) ); \
          _mm256_store_pd( &p3[Index3D(nx, ny, i, j, k)], Anextv );


#define BODYZ2_2LOOPS_AVX_4( p3, p2 )            \
          __m256d A0zce = _mm256_load_pd( &p2[Index3D(nx, ny, i,   j, k)] ); \
          __m256d A0xp1 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+1, k)] ); \
          __m256d A0xp2 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+2, k)] ); \
          __m256d A0xp3 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+3, k)] ); \
          __m256d A0xp4 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+4, k)] ); \
\
          __m256d Anextv = _mm256_load_pd( &p3[Index3D(nx, ny, i, j, k)] ); \
\
          __m256d tempx = _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0xp1, c1xv ),    \
                                                        _mm256_mul_pd( A0xp2, c2xv )),   \
                                         _mm256_add_pd( _mm256_mul_pd( A0xp3, c3xv ),    \
                                                        _mm256_mul_pd( A0xp4, c4xv )));  \
\
          Anextv = _mm256_add_pd( tempx, Anextv ); \
          _mm256_store_pd( &p3[Index3D(nx, ny, i, j, k)], Anextv );


#define BODYZ1_2LOOPS_MIC_4( p3, p2 )               \
          __m512d A0zm8 = _mm512_load_pd( &p2[Index3D(nx, ny, i-8, j, k)] ); \
          __m512d A0zce = _mm512_load_pd( &p2[Index3D(nx, ny, i,   j, k)] ); \
          __m512d A0zp8 = _mm512_load_pd( &p2[Index3D(nx, ny, i+8, j, k)] ); \
\
          __m512d A0yp1 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j, k+1)] ); \
          __m512d A0yp2 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j, k+2)] ); \
          __m512d A0yp3 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j, k+3)] ); \
          __m512d A0yp4 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j, k+4)] ); \
\
          __m512d A0zm4 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zce, (__m512i)A0zm8, 8  ); \
          __m512d A0zm3 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zce, (__m512i)A0zm8, 10 ); \
          __m512d A0zm2 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zce, (__m512i)A0zm8, 12 ); \
          __m512d A0zm1 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zce, (__m512i)A0zm8, 14 ); \
          __m512d A0zp1 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zp8, (__m512i)A0zce, 2  ); \
          __m512d A0zp2 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zp8, (__m512i)A0zce, 4  ); \
          __m512d A0zp3 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zp8, (__m512i)A0zce, 6  ); \
          __m512d A0zp4 = (__m512d)_mm512_alignr_epi32( (__m512i)A0zp8, (__m512i)A0zce, 8  ); \
\
          __m512d Anextv = _mm512_load_pd( &p3[Index3D(nx, ny, i, j, k)] ); \
\
          __m512d temp1 = _mm512_mul_pd( A0zce, c00v ); \
          __m512d temp2 = _mm512_mul_pd( A0zm1, c1zv ); \
                  temp1 = _mm512_fmadd_pd( A0zm2, c2zv, temp1 ); \
                  temp2 = _mm512_fmadd_pd( A0zm3, c3zv, temp2 ); \
                  temp1 = _mm512_fmadd_pd( A0zm4, c4zv, temp1 ); \
                  temp2 = _mm512_fmadd_pd( A0zp1, c1zv, temp2 ); \
                  temp1 = _mm512_fmadd_pd( A0zp2, c2zv, temp1 ); \
                  temp2 = _mm512_fmadd_pd( A0zp3, c3zv, temp2 ); \
                  temp1 = _mm512_fmadd_pd( A0zp4, c4zv, temp1 ); \
\
                  temp1 = _mm512_fmadd_pd( A0yp1, c1yv, temp1 ); \
                  temp2 = _mm512_fmadd_pd( A0yp2, c2yv, temp2 ); \
                  temp1 = _mm512_fmadd_pd( A0yp3, c3yv, temp1 ); \
                  temp2 = _mm512_fmadd_pd( A0yp4, c4yv, temp2 ); \
\
          Anextv = _mm512_add_pd( _mm512_add_pd( temp1, temp2 ), Anextv ); \
          _mm512_store_pd( &p3[Index3D(nx, ny, i, j, k)], Anextv );


#define BODYZ2_2LOOPS_MIC_4( p3, p2 )            \
          __m512d A0zce = _mm512_load_pd( &p2[Index3D(nx, ny, i,   j, k)] ); \
          __m512d A0xp1 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j+1, k)] ); \
          __m512d A0xp2 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j+2, k)] ); \
          __m512d A0xp3 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j+3, k)] ); \
          __m512d A0xp4 = _mm512_load_pd( &p2[Index3D(nx, ny, i, j+4, k)] ); \
\
          __m512d Anextv = _mm512_load_pd( &p3[Index3D(nx, ny, i, j, k)] ); \
\
          __m512d temp1 = _mm512_mul_pd( A0xp1, c1xv ); \
          __m512d temp2 = _mm512_mul_pd( A0xp2, c2xv ); \
                  temp1 = _mm512_fmadd_pd( A0xp3, c3xv, temp1 ); \
                  temp2 = _mm512_fmadd_pd( A0xp4, c4xv, temp2 ); \
\
          Anextv = _mm512_add_pd( _mm512_add_pd( temp1, temp2 ), Anextv ); \
          _mm512_store_pd( &p3[Index3D(nx, ny, i, j, k)], Anextv );


#define BODYZ1_3LOOPS_4( p3, p2 )            \
      p2c  = p2[Index3D(nx, ny, i,   j, k)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  + \
\
        c4z * p2[Index3D(nx, ny, i-4, j, k)] + \
        c4z * p2[Index3D(nx, ny, i+4, j, k)] + \
        c3z * p2[Index3D(nx, ny, i-3, j, k)] + \
        c3z * p2[Index3D(nx, ny, i+3, j, k)] + \
        c2z * p2[Index3D(nx, ny, i-2, j, k)] + \
        c2z * p2[Index3D(nx, ny, i+2, j, k)] + \
        c1z * p2[Index3D(nx, ny, i-1, j, k)] + \
        c1z * p2[Index3D(nx, ny, i+1, j, k)];

#define BODYZ2_3LOOPS_4( p3, p2 )            \
      p2c  = p2[Index3D(nx, ny, i, j,   k)]; \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)]; \
      p2k2 = p2[Index3D(nx, ny, i, j, k+2)]; \
      p2k3 = p2[Index3D(nx, ny, i, j, k+3)]; \
      p2k4 = p2[Index3D(nx, ny, i, j, k+4)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c4y * p2k4 + \
        c3y * p2k3 + \
        c2y * p2k2 + \
        c1y * p2k1;

#define BODYZ3_3LOOPS_4( p3, p2 )            \
      p2c  = p2[Index3D(nx, ny, i,   j, k)]; \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)]; \
      p2j2 = p2[Index3D(nx, ny, i, j+2, k)]; \
      p2j3 = p2[Index3D(nx, ny, i, j+3, k)]; \
      p2j4 = p2[Index3D(nx, ny, i, j+4, k)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c4x * p2j4 + \
        c3x * p2j3 + \
        c2x * p2j2 + \
        c1x * p2j1;


#define BODYZ1_3LOOPS_AVX_4( p3, p2 )               \
          __m256d A0zm4 = _mm256_load_pd( &p2[Index3D(nx, ny, i-4, j, k)] ); \
          __m256d A0zce = _mm256_load_pd( &p2[Index3D(nx, ny, i,   j, k)] ); \
          __m256d A0zp4 = _mm256_load_pd( &p2[Index3D(nx, ny, i+4, j, k)] ); \
\
          __m256d A0zm2 = _mm256_permute2f128_pd( A0zm4, A0zce, 0x21 ); /* 100001 */ \
          __m256d A0zm3 = _mm256_shuffle_pd( A0zm4, A0zm2, 0x5 );       /* 0101   */ \
          __m256d A0zm1 = _mm256_shuffle_pd( A0zm2, A0zce, 0x5 );       /* 0101   */ \
          __m256d A0zp2 = _mm256_permute2f128_pd( A0zce, A0zp4, 0x21 ); /* 100001 */ \
          __m256d A0zp1 = _mm256_shuffle_pd( A0zce, A0zp2, 0x5 );       /* 0101   */ \
          __m256d A0zp3 = _mm256_shuffle_pd( A0zp2, A0zp4, 0x5 );       /* 0101   */ \
\
          __m256d Anextv = _mm256_load_pd( &p3[Index3D(nx, ny, i, j, k)] ); \
\
          __m256d tempz = _mm256_add_pd( _mm256_mul_pd( A0zce, c00v ), \
                                         _mm256_add_pd( _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0zm1, c1zv ),     \
                                                                                      _mm256_mul_pd( A0zm2, c2zv )),    \
                                                                       _mm256_add_pd( _mm256_mul_pd( A0zm3, c3zv ),     \
                                                                                      _mm256_mul_pd( A0zm4, c4zv ))),   \
                                                        _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0zp1, c1zv ),     \
                                                                                      _mm256_mul_pd( A0zp2, c2zv )),    \
                                                                       _mm256_add_pd( _mm256_mul_pd( A0zp3, c3zv ),     \
                                                                                      _mm256_mul_pd( A0zp4, c4zv ))))); \
\
          Anextv = _mm256_add_pd( tempz, Anextv ); \
          _mm256_store_pd( &p3[Index3D(nx, ny, i, j, k)], Anextv );


#define BODYZ2_3LOOPS_AVX_4( p3, p2 )            \
          __m256d A0zce = _mm256_load_pd( &p2[Index3D(nx, ny, i, j,   k)] ); \
          __m256d A0yp1 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+1)] ); \
          __m256d A0yp2 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+2)] ); \
          __m256d A0yp3 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+3)] ); \
          __m256d A0yp4 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j, k+4)] ); \
\
          __m256d Anextv = _mm256_load_pd( &p3[Index3D(nx, ny, i, j, k)] ); \
\
          __m256d tempy = _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0yp1, c1yv ), \
                                                        _mm256_mul_pd( A0yp2, c2yv )), \
                                         _mm256_add_pd( _mm256_mul_pd( A0yp3, c3yv ), \
                                                        _mm256_mul_pd( A0yp4, c4yv ))); \
\
          Anextv = _mm256_add_pd( tempy, Anextv ); \
          _mm256_store_pd( &p3[Index3D(nx, ny, i, j, k)], Anextv );


#define BODYZ3_3LOOPS_AVX_4( p3, p2 )            \
          __m256d A0zce = _mm256_load_pd( &p2[Index3D(nx, ny, i,   j, k)] ); \
          __m256d A0xp1 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+1, k)] ); \
          __m256d A0xp2 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+2, k)] ); \
          __m256d A0xp3 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+3, k)] ); \
          __m256d A0xp4 = _mm256_load_pd( &p2[Index3D(nx, ny, i, j+4, k)] ); \
\
          __m256d Anextv = _mm256_load_pd( &p3[Index3D(nx, ny, i, j, k)] ); \
\
          __m256d tempx = _mm256_add_pd( _mm256_add_pd( _mm256_mul_pd( A0xp1, c1xv ),    \
                                                        _mm256_mul_pd( A0xp2, c2xv )),   \
                                         _mm256_add_pd( _mm256_mul_pd( A0xp3, c3xv ),    \
                                                        _mm256_mul_pd( A0xp4, c4xv )));  \
\
          Anextv = _mm256_add_pd( tempx, Anextv ); \
          _mm256_store_pd( &p3[Index3D(nx, ny, i, j, k)], Anextv );



/* 43pt stencil */
#define FORWARDYZERO_7( p3, p2 )               \
      p3[Index3D(nx, ny, i, j, k+7)] =         \
        c7y * p2[Index3D(nx, ny, i, j,   k)] + \
        c6y * p2[Index3D(nx, ny, i, j, k+1)] + \
        c5y * p2[Index3D(nx, ny, i, j, k+2)] + \
        c4y * p2[Index3D(nx, ny, i, j, k+3)] + \
        c3y * p2[Index3D(nx, ny, i, j, k+4)] + \
        c2y * p2[Index3D(nx, ny, i, j, k+5)] + \
        c1y * p2[Index3D(nx, ny, i, j, k+6)];

#define FORWARDXZERO_7( p3, p2 )                                        \
      p3[Index3D(nx, ny, i, j+7, k)] = p3[Index3D(nx, ny, i, j+7, k)] + \
        c7x * p2[Index3D(nx, ny, i,   j, k)] + \
        c6x * p2[Index3D(nx, ny, i, j+1, k)] + \
        c5x * p2[Index3D(nx, ny, i, j+2, k)] + \
        c4x * p2[Index3D(nx, ny, i, j+3, k)] + \
        c3x * p2[Index3D(nx, ny, i, j+4, k)] + \
        c2x * p2[Index3D(nx, ny, i, j+5, k)] + \
        c1x * p2[Index3D(nx, ny, i, j+6, k)];

#define FORWARDZZERO_7( p3, p2 )                                        \
      p3[Index3D(nx, ny, i+7, j, k)] = p3[Index3D(nx, ny, i+7, j, k)] + \
        c7z * p2[Index3D(nx, ny,   i, j, k)] + \
        c6z * p2[Index3D(nx, ny, i+1, j, k)] + \
        c5z * p2[Index3D(nx, ny, i+2, j, k)] + \
        c4z * p2[Index3D(nx, ny, i+3, j, k)] + \
        c3z * p2[Index3D(nx, ny, i+4, j, k)] + \
        c2z * p2[Index3D(nx, ny, i+5, j, k)] + \
        c1z * p2[Index3D(nx, ny, i+6, j, k)];

#define FORWARDY_7( p3, p2 )           \
      p3[Index3D(nx, ny, i, j, k+7)] = \
        c7y * p2c  + \
        c6y * p2k1 + \
        c5y * p2k2 + \
        c4y * p2k3 + \
        c3y * p2k4 + \
        c2y * p2k5 + \
        c1y * p2k6;

#define FORWARDX_7( p3, p2 )                                            \
      p3[Index3D(nx, ny, i, j+7, k)] = p3[Index3D(nx, ny, i, j+7, k)] + \
        c7x * p2c  + \
        c6x * p2j1 + \
        c5x * p2j2 + \
        c4x * p2j3 + \
        c3x * p2j4 + \
        c2x * p2j5 + \
        c1x * p2j6;

#define FORWARDZ_7( p3, p2 )                                            \
      p3[Index3D(nx, ny, i+7, j, k)] = p3[Index3D(nx, ny, i+7, j, k)] + \
        c7z * p2c  + \
        c6z * p2i1 + \
        c5z * p2i2 + \
        c4z * p2i3 + \
        c3z * p2i4 + \
        c2z * p2i5 + \
        c1z * p2i6;

#define BACKWARDY_7( p3, p2 )                                       \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1y * p2[Index3D(nx, ny, i, j, k+1)] + \
        c2y * p2[Index3D(nx, ny, i, j, k+2)] + \
        c3y * p2[Index3D(nx, ny, i, j, k+3)] + \
        c4y * p2[Index3D(nx, ny, i, j, k+4)] + \
        c5y * p2[Index3D(nx, ny, i, j, k+5)] + \
        c6y * p2[Index3D(nx, ny, i, j, k+6)] + \
        c7y * p2[Index3D(nx, ny, i, j, k+7)];

#define BACKWARDX_7( p3, p2 )                                       \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1x * p2[Index3D(nx, ny, i, j+1, k)] + \
        c2x * p2[Index3D(nx, ny, i, j+2, k)] + \
        c3x * p2[Index3D(nx, ny, i, j+3, k)] + \
        c4x * p2[Index3D(nx, ny, i, j+4, k)] + \
        c5x * p2[Index3D(nx, ny, i, j+5, k)] + \
        c6x * p2[Index3D(nx, ny, i, j+6, k)] + \
        c7x * p2[Index3D(nx, ny, i, j+7, k)];

#define BACKWARDZ_7( p3, p2 )                                       \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1z * p2[Index3D(nx, ny, i+1, j, k)] + \
        c2z * p2[Index3D(nx, ny, i+2, j, k)] + \
        c3z * p2[Index3D(nx, ny, i+3, j, k)] + \
        c4z * p2[Index3D(nx, ny, i+4, j, k)] + \
        c5z * p2[Index3D(nx, ny, i+5, j, k)] + \
        c6z * p2[Index3D(nx, ny, i+6, j, k)] + \
        c7z * p2[Index3D(nx, ny, i+7, j, k)];

#define BODYZ_7( p3, p2 )                    \
      p2c  = p2[Index3D(nx, ny, i,   j, k)]; \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)]; \
      p2j2 = p2[Index3D(nx, ny, i, j+2, k)]; \
      p2j3 = p2[Index3D(nx, ny, i, j+3, k)]; \
      p2j4 = p2[Index3D(nx, ny, i, j+4, k)]; \
      p2j5 = p2[Index3D(nx, ny, i, j+5, k)]; \
      p2j6 = p2[Index3D(nx, ny, i, j+6, k)]; \
      p2j7 = p2[Index3D(nx, ny, i, j+7, k)]; \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)]; \
      p2k2 = p2[Index3D(nx, ny, i, j, k+2)]; \
      p2k3 = p2[Index3D(nx, ny, i, j, k+3)]; \
      p2k4 = p2[Index3D(nx, ny, i, j, k+4)]; \
      p2k5 = p2[Index3D(nx, ny, i, j, k+5)]; \
      p2k6 = p2[Index3D(nx, ny, i, j, k+6)]; \
      p2k7 = p2[Index3D(nx, ny, i, j, k+7)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  + \
        c7z * p2[Index3D(nx, ny, i-7, j, k)] + \
        c7z * p2[Index3D(nx, ny, i+7, j, k)] + \
        c6z * p2[Index3D(nx, ny, i-6, j, k)] + \
        c6z * p2[Index3D(nx, ny, i+6, j, k)] + \
        c5z * p2[Index3D(nx, ny, i-5, j, k)] + \
        c5z * p2[Index3D(nx, ny, i+5, j, k)] + \
        c4z * p2[Index3D(nx, ny, i-4, j, k)] + \
        c4z * p2[Index3D(nx, ny, i+4, j, k)] + \
        c3z * p2[Index3D(nx, ny, i-3, j, k)] + \
        c3z * p2[Index3D(nx, ny, i+3, j, k)] + \
        c2z * p2[Index3D(nx, ny, i-2, j, k)] + \
        c2z * p2[Index3D(nx, ny, i+2, j, k)] + \
        c1z * p2[Index3D(nx, ny, i-1, j, k)] + \
        c1z * p2[Index3D(nx, ny, i+1, j, k)] + \
\
        c7x * p2j7 + \
        c6x * p2j6 + \
        c5x * p2j5 + \
        c4x * p2j4 + \
        c3x * p2j3 + \
        c2x * p2j2 + \
        c1x * p2j1 + \
\
        c7y * p2k7 + \
        c6y * p2k6 + \
        c5y * p2k5 + \
        c4y * p2k4 + \
        c3y * p2k3 + \
        c2y * p2k2 + \
        c1y * p2k1;

#define BODYZ_ALL_7( p3, p2 )                \
      p2c  = p2[Index3D(nx, ny, i,   j, k)]; \
      p2i1 = p2[Index3D(nx, ny, i+1, j, k)]; \
      p2i2 = p2[Index3D(nx, ny, i+2, j, k)]; \
      p2i3 = p2[Index3D(nx, ny, i+3, j, k)]; \
      p2i4 = p2[Index3D(nx, ny, i+4, j, k)]; \
      p2i5 = p2[Index3D(nx, ny, i+5, j, k)]; \
      p2i6 = p2[Index3D(nx, ny, i+6, j, k)]; \
      p2i7 = p2[Index3D(nx, ny, i+7, j, k)]; \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)]; \
      p2j2 = p2[Index3D(nx, ny, i, j+2, k)]; \
      p2j3 = p2[Index3D(nx, ny, i, j+3, k)]; \
      p2j4 = p2[Index3D(nx, ny, i, j+4, k)]; \
      p2j5 = p2[Index3D(nx, ny, i, j+5, k)]; \
      p2j6 = p2[Index3D(nx, ny, i, j+6, k)]; \
      p2j7 = p2[Index3D(nx, ny, i, j+7, k)]; \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)]; \
      p2k2 = p2[Index3D(nx, ny, i, j, k+2)]; \
      p2k3 = p2[Index3D(nx, ny, i, j, k+3)]; \
      p2k4 = p2[Index3D(nx, ny, i, j, k+4)]; \
      p2k5 = p2[Index3D(nx, ny, i, j, k+5)]; \
      p2k6 = p2[Index3D(nx, ny, i, j, k+6)]; \
      p2k7 = p2[Index3D(nx, ny, i, j, k+7)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  + \
        c7z * p2i7 + \
        c6z * p2i6 + \
        c5z * p2i5 + \
        c4z * p2i4 + \
        c3z * p2i3 + \
        c2z * p2i2 + \
        c1z * p2i1 + \
\
        c7x * p2j7 + \
        c6x * p2j6 + \
        c5x * p2j5 + \
        c4x * p2j4 + \
        c3x * p2j3 + \
        c2x * p2j2 + \
        c1x * p2j1 + \
\
        c7y * p2k7 + \
        c6y * p2k6 + \
        c5y * p2k5 + \
        c4y * p2k4 + \
        c3y * p2k3 + \
        c2y * p2k2 + \
        c1y * p2k1;

#define BODYZ1_2LOOPS_7( p3, p2 )            \
      p2c  = p2[Index3D(nx, ny, i,   j, k)]; \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)]; \
      p2k2 = p2[Index3D(nx, ny, i, j, k+2)]; \
      p2k3 = p2[Index3D(nx, ny, i, j, k+3)]; \
      p2k4 = p2[Index3D(nx, ny, i, j, k+4)]; \
      p2k5 = p2[Index3D(nx, ny, i, j, k+5)]; \
      p2k6 = p2[Index3D(nx, ny, i, j, k+6)]; \
      p2k7 = p2[Index3D(nx, ny, i, j, k+7)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  + \
\
        c7z * p2[Index3D(nx, ny, i-7, j, k)] + \
        c7z * p2[Index3D(nx, ny, i+7, j, k)] + \
        c6z * p2[Index3D(nx, ny, i-6, j, k)] + \
        c6z * p2[Index3D(nx, ny, i+6, j, k)] + \
        c5z * p2[Index3D(nx, ny, i-5, j, k)] + \
        c5z * p2[Index3D(nx, ny, i+5, j, k)] + \
        c4z * p2[Index3D(nx, ny, i-4, j, k)] + \
        c4z * p2[Index3D(nx, ny, i+4, j, k)] + \
        c3z * p2[Index3D(nx, ny, i-3, j, k)] + \
        c3z * p2[Index3D(nx, ny, i+3, j, k)] + \
        c2z * p2[Index3D(nx, ny, i-2, j, k)] + \
        c2z * p2[Index3D(nx, ny, i+2, j, k)] + \
        c1z * p2[Index3D(nx, ny, i-1, j, k)] + \
        c1z * p2[Index3D(nx, ny, i+1, j, k)] + \
\
        c7y * p2k7 + \
        c6y * p2k6 + \
        c5y * p2k5 + \
        c4y * p2k4 + \
        c3y * p2k3 + \
        c2y * p2k2 + \
        c1y * p2k1;

#define BODYZ2_2LOOPS_7( p3, p2 )            \
      p2c  = p2[Index3D(nx, ny, i,   j, k)]; \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)]; \
      p2j2 = p2[Index3D(nx, ny, i, j+2, k)]; \
      p2j3 = p2[Index3D(nx, ny, i, j+3, k)]; \
      p2j4 = p2[Index3D(nx, ny, i, j+4, k)]; \
      p2j5 = p2[Index3D(nx, ny, i, j+5, k)]; \
      p2j6 = p2[Index3D(nx, ny, i, j+6, k)]; \
      p2j7 = p2[Index3D(nx, ny, i, j+7, k)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c7x * p2j7 + \
        c6x * p2j6 + \
        c5x * p2j5 + \
        c4x * p2j4 + \
        c3x * p2j3 + \
        c2x * p2j2 + \
        c1x * p2j1;

#define BODYZ1_3LOOPS_7( p3, p2 )            \
      p2c  = p2[Index3D(nx, ny, i,   j, k)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  + \
\
        c7z * p2[Index3D(nx, ny, i-7, j, k)] + \
        c7z * p2[Index3D(nx, ny, i+7, j, k)] + \
        c6z * p2[Index3D(nx, ny, i-6, j, k)] + \
        c6z * p2[Index3D(nx, ny, i+6, j, k)] + \
        c5z * p2[Index3D(nx, ny, i-5, j, k)] + \
        c5z * p2[Index3D(nx, ny, i+5, j, k)] + \
        c4z * p2[Index3D(nx, ny, i-4, j, k)] + \
        c4z * p2[Index3D(nx, ny, i+4, j, k)] + \
        c3z * p2[Index3D(nx, ny, i-3, j, k)] + \
        c3z * p2[Index3D(nx, ny, i+3, j, k)] + \
        c2z * p2[Index3D(nx, ny, i-2, j, k)] + \
        c2z * p2[Index3D(nx, ny, i+2, j, k)] + \
        c1z * p2[Index3D(nx, ny, i-1, j, k)] + \
        c1z * p2[Index3D(nx, ny, i+1, j, k)];

#define BODYZ2_3LOOPS_7( p3, p2 )            \
      p2c  = p2[Index3D(nx, ny, i,   j, k)]; \
      p2k1 = p2[Index3D(nx, ny, i, j, k+1)]; \
      p2k2 = p2[Index3D(nx, ny, i, j, k+2)]; \
      p2k3 = p2[Index3D(nx, ny, i, j, k+3)]; \
      p2k4 = p2[Index3D(nx, ny, i, j, k+4)]; \
      p2k5 = p2[Index3D(nx, ny, i, j, k+5)]; \
      p2k6 = p2[Index3D(nx, ny, i, j, k+6)]; \
      p2k7 = p2[Index3D(nx, ny, i, j, k+7)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c7y * p2k7 + \
        c6y * p2k6 + \
        c5y * p2k5 + \
        c4y * p2k4 + \
        c3y * p2k3 + \
        c2y * p2k2 + \
        c1y * p2k1;

#define BODYZ3_3LOOPS_7( p3, p2 )            \
      p2c  = p2[Index3D(nx, ny, i,   j, k)]; \
      p2j1 = p2[Index3D(nx, ny, i, j+1, k)]; \
      p2j2 = p2[Index3D(nx, ny, i, j+2, k)]; \
      p2j3 = p2[Index3D(nx, ny, i, j+3, k)]; \
      p2j4 = p2[Index3D(nx, ny, i, j+4, k)]; \
      p2j5 = p2[Index3D(nx, ny, i, j+5, k)]; \
      p2j6 = p2[Index3D(nx, ny, i, j+6, k)]; \
      p2j7 = p2[Index3D(nx, ny, i, j+7, k)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c7x * p2j7 + \
        c6x * p2j6 + \
        c5x * p2j5 + \
        c4x * p2j4 + \
        c3x * p2j3 + \
        c2x * p2j2 + \
        c1x * p2j1;



/* 85pt stencil */
#define FORWARDYZERO_14( p3, p2 )                \
      p3[Index3D(nx, ny, i, j, k+14)] =          \
        c14y * p2[Index3D(nx, ny, i, j,   k)] +  \
        c13y * p2[Index3D(nx, ny, i, j, k+1)] +  \
        c12y * p2[Index3D(nx, ny, i, j, k+2)] +  \
        c11y * p2[Index3D(nx, ny, i, j, k+3)] +  \
        c10y * p2[Index3D(nx, ny, i, j, k+4)] +  \
        c9y  * p2[Index3D(nx, ny, i, j, k+5)] +  \
        c8y  * p2[Index3D(nx, ny, i, j, k+6)] +  \
        c7y  * p2[Index3D(nx, ny, i, j, k+7)] +  \
        c6y  * p2[Index3D(nx, ny, i, j, k+8)] +  \
        c5y  * p2[Index3D(nx, ny, i, j, k+9)] +  \
        c4y  * p2[Index3D(nx, ny, i, j, k+10)] + \
        c3y  * p2[Index3D(nx, ny, i, j, k+11)] + \
        c2y  * p2[Index3D(nx, ny, i, j, k+12)] + \
        c1y  * p2[Index3D(nx, ny, i, j, k+13)];

#define FORWARDXZERO_14( p3, p2 )                                         \
      p3[Index3D(nx, ny, i, j+14, k)] = p3[Index3D(nx, ny, i, j+14, k)] + \
        c14x * p2[Index3D(nx, ny, i,   j, k)] +  \
        c13x * p2[Index3D(nx, ny, i, j+1, k)] +  \
        c12x * p2[Index3D(nx, ny, i, j+2, k)] +  \
        c11x * p2[Index3D(nx, ny, i, j+3, k)] +  \
        c10x * p2[Index3D(nx, ny, i, j+4, k)] +  \
        c9x  * p2[Index3D(nx, ny, i, j+5, k)] +  \
        c8x  * p2[Index3D(nx, ny, i, j+6, k)] +  \
        c7x  * p2[Index3D(nx, ny, i, j+7, k)] +  \
        c6x  * p2[Index3D(nx, ny, i, j+8, k)] +  \
        c5x  * p2[Index3D(nx, ny, i, j+9, k)] +  \
        c4x  * p2[Index3D(nx, ny, i, j+10, k)] + \
        c3x  * p2[Index3D(nx, ny, i, j+11, k)] + \
        c2x  * p2[Index3D(nx, ny, i, j+12, k)] + \
        c1x  * p2[Index3D(nx, ny, i, j+13, k)];

#define FORWARDZZERO_14( p3, p2 )                                         \
      p3[Index3D(nx, ny, i+14, j, k)] = p3[Index3D(nx, ny, i+14, j, k)] + \
        c14z * p2[Index3D(nx, ny,   i, j, k)] +  \
        c13z * p2[Index3D(nx, ny, i+1, j, k)] +  \
        c12z * p2[Index3D(nx, ny, i+2, j, k)] +  \
        c11z * p2[Index3D(nx, ny, i+3, j, k)] +  \
        c10z * p2[Index3D(nx, ny, i+4, j, k)] +  \
        c9z  * p2[Index3D(nx, ny, i+5, j, k)] +  \
        c8z  * p2[Index3D(nx, ny, i+6, j, k)] +  \
        c7z  * p2[Index3D(nx, ny, i+7, j, k)] +  \
        c6z  * p2[Index3D(nx, ny, i+8, j, k)] +  \
        c5z  * p2[Index3D(nx, ny, i+9, j, k)] +  \
        c4z  * p2[Index3D(nx, ny, i+10, j, k)] + \
        c3z  * p2[Index3D(nx, ny, i+11, j, k)] + \
        c2z  * p2[Index3D(nx, ny, i+12, j, k)] + \
        c1z  * p2[Index3D(nx, ny, i+13, j, k)];

#define FORWARDY_14( p3, p2 )           \
      p3[Index3D(nx, ny, i, j, k+14)] = \
        c14y * p2c  +  \
        c13y * p2k1 +  \
        c12y * p2k2 +  \
        c11y * p2k3 +  \
        c10y * p2k4 +  \
        c9y  * p2k5 +  \
        c8y  * p2k6 +  \
        c7y  * p2k7 +  \
        c6y  * p2k8 +  \
        c5y  * p2k9 +  \
        c4y  * p2k10 + \
        c3y  * p2k11 + \
        c2y  * p2k12 + \
        c1y  * p2k13;

#define FORWARDX_14( p3, p2 )                                             \
      p3[Index3D(nx, ny, i, j+14, k)] = p3[Index3D(nx, ny, i, j+14, k)] + \
        c14x * p2c  +  \
        c13x * p2j1 +  \
        c12x * p2j2 +  \
        c11x * p2j3 +  \
        c10x * p2j4 +  \
        c9x  * p2j5 +  \
        c8x  * p2j6 +  \
        c7x  * p2j7 +  \
        c6x  * p2j8 +  \
        c5x  * p2j9 +  \
        c4x  * p2j10 + \
        c3x  * p2j11 + \
        c2x  * p2j12 + \
        c1x  * p2j13;

#define FORWARDZ_14( p3, p2 )                                             \
      p3[Index3D(nx, ny, i+14, j, k)] = p3[Index3D(nx, ny, i+14, j, k)] + \
        c14z * p2c  +  \
        c13z * p2i1 +  \
        c12z * p2i2 +  \
        c11z * p2i3 +  \
        c10z * p2i4 +  \
        c9z  * p2i5 +  \
        c8z  * p2i6 +  \
        c7z  * p2i7 +  \
        c6z  * p2i8 +  \
        c5z  * p2i9 +  \
        c4z  * p2i10 + \
        c3z  * p2i11 + \
        c2z  * p2i12 + \
        c1z  * p2i13;

#define BACKWARDY_14( p3, p2 )                                      \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1y  * p2[Index3D(nx, ny, i, j, k+1)] +  \
        c2y  * p2[Index3D(nx, ny, i, j, k+2)] +  \
        c3y  * p2[Index3D(nx, ny, i, j, k+3)] +  \
        c4y  * p2[Index3D(nx, ny, i, j, k+4)] +  \
        c5y  * p2[Index3D(nx, ny, i, j, k+5)] +  \
        c6y  * p2[Index3D(nx, ny, i, j, k+6)] +  \
        c7y  * p2[Index3D(nx, ny, i, j, k+7)] +  \
        c8y  * p2[Index3D(nx, ny, i, j, k+8)] +  \
        c9y  * p2[Index3D(nx, ny, i, j, k+9)] +  \
        c10y * p2[Index3D(nx, ny, i, j, k+10)] + \
        c11y * p2[Index3D(nx, ny, i, j, k+11)] + \
        c12y * p2[Index3D(nx, ny, i, j, k+12)] + \
        c13y * p2[Index3D(nx, ny, i, j, k+13)] + \
        c14y * p2[Index3D(nx, ny, i, j, k+14)];

#define BACKWARDX_14( p3, p2 )                                      \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1x  * p2[Index3D(nx, ny, i, j+1, k)] +  \
        c2x  * p2[Index3D(nx, ny, i, j+2, k)] +  \
        c3x  * p2[Index3D(nx, ny, i, j+3, k)] +  \
        c4x  * p2[Index3D(nx, ny, i, j+4, k)] +  \
        c5x  * p2[Index3D(nx, ny, i, j+5, k)] +  \
        c6x  * p2[Index3D(nx, ny, i, j+6, k)] +  \
        c7x  * p2[Index3D(nx, ny, i, j+7, k)] +  \
        c8x  * p2[Index3D(nx, ny, i, j+8, k)] +  \
        c9x  * p2[Index3D(nx, ny, i, j+9, k)] +  \
        c10x * p2[Index3D(nx, ny, i, j+10, k)] + \
        c11x * p2[Index3D(nx, ny, i, j+11, k)] + \
        c12x * p2[Index3D(nx, ny, i, j+12, k)] + \
        c13x * p2[Index3D(nx, ny, i, j+13, k)] + \
        c14x * p2[Index3D(nx, ny, i, j+14, k)];

#define BACKWARDZ_14( p3, p2 )                                      \
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c1z  * p2[Index3D(nx, ny, i+1, j, k)] +  \
        c2z  * p2[Index3D(nx, ny, i+2, j, k)] +  \
        c3z  * p2[Index3D(nx, ny, i+3, j, k)] +  \
        c4z  * p2[Index3D(nx, ny, i+4, j, k)] +  \
        c5z  * p2[Index3D(nx, ny, i+5, j, k)] +  \
        c6z  * p2[Index3D(nx, ny, i+6, j, k)] +  \
        c7z  * p2[Index3D(nx, ny, i+7, j, k)] +  \
        c8z  * p2[Index3D(nx, ny, i+8, j, k)] +  \
        c9z  * p2[Index3D(nx, ny, i+9, j, k)] +  \
        c10z * p2[Index3D(nx, ny, i+10, j, k)] + \
        c11z * p2[Index3D(nx, ny, i+11, j, k)] + \
        c12z * p2[Index3D(nx, ny, i+12, j, k)] + \
        c13z * p2[Index3D(nx, ny, i+13, j, k)] + \
        c14z * p2[Index3D(nx, ny, i+14, j, k)];

#define BODYZ_14( p3, p2 )                     \
      p2c   = p2[Index3D(nx, ny, i,   j, k)];  \
      p2j1  = p2[Index3D(nx, ny, i, j+1, k)];  \
      p2j2  = p2[Index3D(nx, ny, i, j+2, k)];  \
      p2j3  = p2[Index3D(nx, ny, i, j+3, k)];  \
      p2j4  = p2[Index3D(nx, ny, i, j+4, k)];  \
      p2j5  = p2[Index3D(nx, ny, i, j+5, k)];  \
      p2j6  = p2[Index3D(nx, ny, i, j+6, k)];  \
      p2j7  = p2[Index3D(nx, ny, i, j+7, k)];  \
      p2j8  = p2[Index3D(nx, ny, i, j+8, k)];  \
      p2j9  = p2[Index3D(nx, ny, i, j+9, k)];  \
      p2j10 = p2[Index3D(nx, ny, i, j+10, k)]; \
      p2j11 = p2[Index3D(nx, ny, i, j+11, k)]; \
      p2j12 = p2[Index3D(nx, ny, i, j+12, k)]; \
      p2j13 = p2[Index3D(nx, ny, i, j+13, k)]; \
      p2j14 = p2[Index3D(nx, ny, i, j+14, k)]; \
      p2k1  = p2[Index3D(nx, ny, i, j, k+1)];  \
      p2k2  = p2[Index3D(nx, ny, i, j, k+2)];  \
      p2k3  = p2[Index3D(nx, ny, i, j, k+3)];  \
      p2k4  = p2[Index3D(nx, ny, i, j, k+4)];  \
      p2k5  = p2[Index3D(nx, ny, i, j, k+5)];  \
      p2k6  = p2[Index3D(nx, ny, i, j, k+6)];  \
      p2k7  = p2[Index3D(nx, ny, i, j, k+7)];  \
      p2k8  = p2[Index3D(nx, ny, i, j, k+8)];  \
      p2k9  = p2[Index3D(nx, ny, i, j, k+9)];  \
      p2k10 = p2[Index3D(nx, ny, i, j, k+10)]; \
      p2k11 = p2[Index3D(nx, ny, i, j, k+11)]; \
      p2k12 = p2[Index3D(nx, ny, i, j, k+12)]; \
      p2k13 = p2[Index3D(nx, ny, i, j, k+13)]; \
      p2k14 = p2[Index3D(nx, ny, i, j, k+14)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  +                             \
        c14z * p2[Index3D(nx, ny, i-14, j, k)] + \
        c14z * p2[Index3D(nx, ny, i+14, j, k)] + \
        c13z * p2[Index3D(nx, ny, i-13, j, k)] + \
        c13z * p2[Index3D(nx, ny, i+13, j, k)] + \
        c12z * p2[Index3D(nx, ny, i-12, j, k)] + \
        c12z * p2[Index3D(nx, ny, i+12, j, k)] + \
        c11z * p2[Index3D(nx, ny, i-11, j, k)] + \
        c11z * p2[Index3D(nx, ny, i+11, j, k)] + \
        c10z * p2[Index3D(nx, ny, i-10, j, k)] + \
        c10z * p2[Index3D(nx, ny, i+10, j, k)] + \
        c9z  * p2[Index3D(nx, ny, i-9, j, k)] +  \
        c9z  * p2[Index3D(nx, ny, i+9, j, k)] +  \
        c8z  * p2[Index3D(nx, ny, i-8, j, k)] +  \
        c8z  * p2[Index3D(nx, ny, i+8, j, k)] +  \
        c7z  * p2[Index3D(nx, ny, i-7, j, k)] +  \
        c7z  * p2[Index3D(nx, ny, i+7, j, k)] +  \
        c6z  * p2[Index3D(nx, ny, i-6, j, k)] +  \
        c6z  * p2[Index3D(nx, ny, i+6, j, k)] +  \
        c5z  * p2[Index3D(nx, ny, i-5, j, k)] +  \
        c5z  * p2[Index3D(nx, ny, i+5, j, k)] +  \
        c4z  * p2[Index3D(nx, ny, i-4, j, k)] +  \
        c4z  * p2[Index3D(nx, ny, i+4, j, k)] +  \
        c3z  * p2[Index3D(nx, ny, i-3, j, k)] +  \
        c3z  * p2[Index3D(nx, ny, i+3, j, k)] +  \
        c2z  * p2[Index3D(nx, ny, i-2, j, k)] +  \
        c2z  * p2[Index3D(nx, ny, i+2, j, k)] +  \
        c1z  * p2[Index3D(nx, ny, i-1, j, k)] +  \
        c1z  * p2[Index3D(nx, ny, i+1, j, k)] +  \
\
        c14x * p2j14 + \
        c13x * p2j13 + \
        c12x * p2j12 + \
        c11x * p2j11 + \
        c10x * p2j10 + \
        c9x  * p2j9 +  \
        c8x  * p2j8 +  \
        c7x  * p2j7 +  \
        c6x  * p2j6 +  \
        c5x  * p2j5 +  \
        c4x  * p2j4 +  \
        c3x  * p2j3 +  \
        c2x  * p2j2 +  \
        c1x  * p2j1 +  \
\
        c14y * p2k14 + \
        c13y * p2k13 + \
        c12y * p2k12 + \
        c11y * p2k11 + \
        c10y * p2k10 + \
        c9y  * p2k9 +  \
        c8y  * p2k8 +  \
        c7y  * p2k7 +  \
        c6y  * p2k6 +  \
        c5y  * p2k5 +  \
        c4y  * p2k4 +  \
        c3y  * p2k3 +  \
        c2y  * p2k2 +  \
        c1y  * p2k1;

#define BODYZ_ALL_14( p3, p2 )                 \
      p2c   = p2[Index3D(nx, ny, i,   j, k)];  \
      p2i1  = p2[Index3D(nx, ny, i+1, j, k)];  \
      p2i2  = p2[Index3D(nx, ny, i+2, j, k)];  \
      p2i3  = p2[Index3D(nx, ny, i+3, j, k)];  \
      p2i4  = p2[Index3D(nx, ny, i+4, j, k)];  \
      p2i5  = p2[Index3D(nx, ny, i+5, j, k)];  \
      p2i6  = p2[Index3D(nx, ny, i+6, j, k)];  \
      p2i7  = p2[Index3D(nx, ny, i+7, j, k)];  \
      p2i8  = p2[Index3D(nx, ny, i+8, j, k)];  \
      p2i9  = p2[Index3D(nx, ny, i+9, j, k)];  \
      p2i10 = p2[Index3D(nx, ny, i+10, j, k)]; \
      p2i11 = p2[Index3D(nx, ny, i+11, j, k)]; \
      p2i12 = p2[Index3D(nx, ny, i+12, j, k)]; \
      p2i13 = p2[Index3D(nx, ny, i+13, j, k)]; \
      p2i14 = p2[Index3D(nx, ny, i+14, j, k)]; \
      p2j1  = p2[Index3D(nx, ny, i, j+1, k)];  \
      p2j2  = p2[Index3D(nx, ny, i, j+2, k)];  \
      p2j3  = p2[Index3D(nx, ny, i, j+3, k)];  \
      p2j4  = p2[Index3D(nx, ny, i, j+4, k)];  \
      p2j5  = p2[Index3D(nx, ny, i, j+5, k)];  \
      p2j6  = p2[Index3D(nx, ny, i, j+6, k)];  \
      p2j7  = p2[Index3D(nx, ny, i, j+7, k)];  \
      p2j8  = p2[Index3D(nx, ny, i, j+8, k)];  \
      p2j9  = p2[Index3D(nx, ny, i, j+9, k)];  \
      p2j10 = p2[Index3D(nx, ny, i, j+10, k)]; \
      p2j11 = p2[Index3D(nx, ny, i, j+11, k)]; \
      p2j12 = p2[Index3D(nx, ny, i, j+12, k)]; \
      p2j13 = p2[Index3D(nx, ny, i, j+13, k)]; \
      p2j14 = p2[Index3D(nx, ny, i, j+14, k)]; \
      p2k1  = p2[Index3D(nx, ny, i, j, k+1)];  \
      p2k2  = p2[Index3D(nx, ny, i, j, k+2)];  \
      p2k3  = p2[Index3D(nx, ny, i, j, k+3)];  \
      p2k4  = p2[Index3D(nx, ny, i, j, k+4)];  \
      p2k5  = p2[Index3D(nx, ny, i, j, k+5)];  \
      p2k6  = p2[Index3D(nx, ny, i, j, k+6)];  \
      p2k7  = p2[Index3D(nx, ny, i, j, k+7)];  \
      p2k8  = p2[Index3D(nx, ny, i, j, k+8)];  \
      p2k9  = p2[Index3D(nx, ny, i, j, k+9)];  \
      p2k10 = p2[Index3D(nx, ny, i, j, k+10)]; \
      p2k11 = p2[Index3D(nx, ny, i, j, k+11)]; \
      p2k12 = p2[Index3D(nx, ny, i, j, k+12)]; \
      p2k13 = p2[Index3D(nx, ny, i, j, k+13)]; \
      p2k14 = p2[Index3D(nx, ny, i, j, k+14)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  +   \
        c14z * p2i14 + \
        c13z * p2i13 + \
        c12z * p2i12 + \
        c11z * p2i11 + \
        c10z * p2i10 + \
        c9z  * p2i9 +  \
        c8z  * p2i8 +  \
        c7z  * p2i7 +  \
        c6z  * p2i6 +  \
        c5z  * p2i5 +  \
        c4z  * p2i4 +  \
        c3z  * p2i3 +  \
        c2z  * p2i2 +  \
        c1z  * p2i1 +  \
\
        c14x * p2j14 + \
        c13x * p2j13 + \
        c12x * p2j12 + \
        c11x * p2j11 + \
        c10x * p2j10 + \
        c9x  * p2j9 +  \
        c8x  * p2j8 +  \
        c7x  * p2j7 +  \
        c6x  * p2j6 +  \
        c5x  * p2j5 +  \
        c4x  * p2j4 +  \
        c3x  * p2j3 +  \
        c2x  * p2j2 +  \
        c1x  * p2j1 +  \
\
        c14y * p2k14 + \
        c13y * p2k13 + \
        c12y * p2k12 + \
        c11y * p2k11 + \
        c10y * p2k10 + \
        c9y  * p2k9 +  \
        c8y  * p2k8 +  \
        c7y  * p2k7 +  \
        c6y  * p2k6 +  \
        c5y  * p2k5 +  \
        c4y  * p2k4 +  \
        c3y  * p2k3 +  \
        c2y  * p2k2 +  \
        c1y  * p2k1;

#define BODYZ1_2LOOPS_14( p3, p2 )             \
      p2c   = p2[Index3D(nx, ny, i,   j, k)];  \
      p2k1  = p2[Index3D(nx, ny, i, j, k+1)];  \
      p2k2  = p2[Index3D(nx, ny, i, j, k+2)];  \
      p2k3  = p2[Index3D(nx, ny, i, j, k+3)];  \
      p2k4  = p2[Index3D(nx, ny, i, j, k+4)];  \
      p2k5  = p2[Index3D(nx, ny, i, j, k+5)];  \
      p2k6  = p2[Index3D(nx, ny, i, j, k+6)];  \
      p2k7  = p2[Index3D(nx, ny, i, j, k+7)];  \
      p2k8  = p2[Index3D(nx, ny, i, j, k+8)];  \
      p2k9  = p2[Index3D(nx, ny, i, j, k+9)];  \
      p2k10 = p2[Index3D(nx, ny, i, j, k+10)]; \
      p2k11 = p2[Index3D(nx, ny, i, j, k+11)]; \
      p2k12 = p2[Index3D(nx, ny, i, j, k+12)]; \
      p2k13 = p2[Index3D(nx, ny, i, j, k+13)]; \
      p2k14 = p2[Index3D(nx, ny, i, j, k+14)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  + \
\
        c14z * p2[Index3D(nx, ny, i-14, j, k)] + \
        c14z * p2[Index3D(nx, ny, i+14, j, k)] + \
        c13z * p2[Index3D(nx, ny, i-13, j, k)] + \
        c13z * p2[Index3D(nx, ny, i+13, j, k)] + \
        c12z * p2[Index3D(nx, ny, i-12, j, k)] + \
        c12z * p2[Index3D(nx, ny, i+12, j, k)] + \
        c11z * p2[Index3D(nx, ny, i-11, j, k)] + \
        c11z * p2[Index3D(nx, ny, i+11, j, k)] + \
        c10z * p2[Index3D(nx, ny, i-10, j, k)] + \
        c10z * p2[Index3D(nx, ny, i+10, j, k)] + \
        c9z  * p2[Index3D(nx, ny, i-9, j, k)] +  \
        c9z  * p2[Index3D(nx, ny, i+9, j, k)] +  \
        c8z  * p2[Index3D(nx, ny, i-8, j, k)] +  \
        c8z  * p2[Index3D(nx, ny, i+8, j, k)] +  \
        c7z  * p2[Index3D(nx, ny, i-7, j, k)] +  \
        c7z  * p2[Index3D(nx, ny, i+7, j, k)] +  \
        c6z  * p2[Index3D(nx, ny, i-6, j, k)] +  \
        c6z  * p2[Index3D(nx, ny, i+6, j, k)] +  \
        c5z  * p2[Index3D(nx, ny, i-5, j, k)] +  \
        c5z  * p2[Index3D(nx, ny, i+5, j, k)] +  \
        c4z  * p2[Index3D(nx, ny, i-4, j, k)] +  \
        c4z  * p2[Index3D(nx, ny, i+4, j, k)] +  \
        c3z  * p2[Index3D(nx, ny, i-3, j, k)] +  \
        c3z  * p2[Index3D(nx, ny, i+3, j, k)] +  \
        c2z  * p2[Index3D(nx, ny, i-2, j, k)] +  \
        c2z  * p2[Index3D(nx, ny, i+2, j, k)] +  \
        c1z  * p2[Index3D(nx, ny, i-1, j, k)] +  \
        c1z  * p2[Index3D(nx, ny, i+1, j, k)] +  \
\
        c14y * p2k14 + \
        c13y * p2k13 + \
        c12y * p2k12 + \
        c11y * p2k11 + \
        c10y * p2k10 + \
        c9y  * p2k9 +  \
        c8y  * p2k8 +  \
        c7y  * p2k7 +  \
        c6y  * p2k6 +  \
        c5y  * p2k5 +  \
        c4y  * p2k4 +  \
        c3y  * p2k3 +  \
        c2y  * p2k2 +  \
        c1y  * p2k1;

#define BODYZ2_2LOOPS_14( p3, p2 )             \
      p2c   = p2[Index3D(nx, ny, i,   j, k)];  \
      p2j1  = p2[Index3D(nx, ny, i, j+1, k)];  \
      p2j2  = p2[Index3D(nx, ny, i, j+2, k)];  \
      p2j3  = p2[Index3D(nx, ny, i, j+3, k)];  \
      p2j4  = p2[Index3D(nx, ny, i, j+4, k)];  \
      p2j5  = p2[Index3D(nx, ny, i, j+5, k)];  \
      p2j6  = p2[Index3D(nx, ny, i, j+6, k)];  \
      p2j7  = p2[Index3D(nx, ny, i, j+7, k)];  \
      p2j8  = p2[Index3D(nx, ny, i, j+8, k)];  \
      p2j9  = p2[Index3D(nx, ny, i, j+9, k)];  \
      p2j10 = p2[Index3D(nx, ny, i, j+10, k)]; \
      p2j11 = p2[Index3D(nx, ny, i, j+11, k)]; \
      p2j12 = p2[Index3D(nx, ny, i, j+12, k)]; \
      p2j13 = p2[Index3D(nx, ny, i, j+13, k)]; \
      p2j14 = p2[Index3D(nx, ny, i, j+14, k)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c14x * p2j14 + \
        c13x * p2j13 + \
        c12x * p2j12 + \
        c11x * p2j11 + \
        c10x * p2j10 + \
        c9x  * p2j9 +  \
        c8x  * p2j8 +  \
        c7x  * p2j7 +  \
        c6x  * p2j6 +  \
        c5x  * p2j5 +  \
        c4x  * p2j4 +  \
        c3x  * p2j3 +  \
        c2x  * p2j2 +  \
        c1x  * p2j1;

#define BODYZ1_3LOOPS_14( p3, p2 )           \
      p2c  = p2[Index3D(nx, ny, i,   j, k)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c00 * p2c  + \
\
        c14z * p2[Index3D(nx, ny, i-14, j, k)] + \
        c14z * p2[Index3D(nx, ny, i+14, j, k)] + \
        c13z * p2[Index3D(nx, ny, i-13, j, k)] + \
        c13z * p2[Index3D(nx, ny, i+13, j, k)] + \
        c12z * p2[Index3D(nx, ny, i-12, j, k)] + \
        c12z * p2[Index3D(nx, ny, i+12, j, k)] + \
        c11z * p2[Index3D(nx, ny, i-11, j, k)] + \
        c11z * p2[Index3D(nx, ny, i+11, j, k)] + \
        c10z * p2[Index3D(nx, ny, i-10, j, k)] + \
        c10z * p2[Index3D(nx, ny, i+10, j, k)] + \
        c9z  * p2[Index3D(nx, ny, i-9, j, k)] +  \
        c9z  * p2[Index3D(nx, ny, i+9, j, k)] +  \
        c8z  * p2[Index3D(nx, ny, i-8, j, k)] +  \
        c8z  * p2[Index3D(nx, ny, i+8, j, k)] +  \
        c7z  * p2[Index3D(nx, ny, i-7, j, k)] +  \
        c7z  * p2[Index3D(nx, ny, i+7, j, k)] +  \
        c6z  * p2[Index3D(nx, ny, i-6, j, k)] +  \
        c6z  * p2[Index3D(nx, ny, i+6, j, k)] +  \
        c5z  * p2[Index3D(nx, ny, i-5, j, k)] +  \
        c5z  * p2[Index3D(nx, ny, i+5, j, k)] +  \
        c4z  * p2[Index3D(nx, ny, i-4, j, k)] +  \
        c4z  * p2[Index3D(nx, ny, i+4, j, k)] +  \
        c3z  * p2[Index3D(nx, ny, i-3, j, k)] +  \
        c3z  * p2[Index3D(nx, ny, i+3, j, k)] +  \
        c2z  * p2[Index3D(nx, ny, i-2, j, k)] +  \
        c2z  * p2[Index3D(nx, ny, i+2, j, k)] +  \
        c1z  * p2[Index3D(nx, ny, i-1, j, k)] +  \
        c1z  * p2[Index3D(nx, ny, i+1, j, k)];

#define BODYZ2_3LOOPS_14( p3, p2 )             \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];   \
      p2k1  = p2[Index3D(nx, ny, i, j, k+1)];  \
      p2k2  = p2[Index3D(nx, ny, i, j, k+2)];  \
      p2k3  = p2[Index3D(nx, ny, i, j, k+3)];  \
      p2k4  = p2[Index3D(nx, ny, i, j, k+4)];  \
      p2k5  = p2[Index3D(nx, ny, i, j, k+5)];  \
      p2k6  = p2[Index3D(nx, ny, i, j, k+6)];  \
      p2k7  = p2[Index3D(nx, ny, i, j, k+7)];  \
      p2k8  = p2[Index3D(nx, ny, i, j, k+8)];  \
      p2k9  = p2[Index3D(nx, ny, i, j, k+9)];  \
      p2k10 = p2[Index3D(nx, ny, i, j, k+10)]; \
      p2k11 = p2[Index3D(nx, ny, i, j, k+11)]; \
      p2k12 = p2[Index3D(nx, ny, i, j, k+12)]; \
      p2k13 = p2[Index3D(nx, ny, i, j, k+13)]; \
      p2k14 = p2[Index3D(nx, ny, i, j, k+14)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c14y * p2k14 + \
        c13y * p2k13 + \
        c12y * p2k12 + \
        c11y * p2k11 + \
        c10y * p2k10 + \
        c9y  * p2k9 +  \
        c8y  * p2k8 +  \
        c7y  * p2k7 +  \
        c6y  * p2k6 +  \
        c5y  * p2k5 +  \
        c4y  * p2k4 +  \
        c3y  * p2k3 +  \
        c2y  * p2k2 +  \
        c1y  * p2k1;

#define BODYZ3_3LOOPS_14( p3, p2 )             \
      p2c  = p2[Index3D(nx, ny, i,   j, k)];   \
      p2j1  = p2[Index3D(nx, ny, i, j+1, k)];  \
      p2j2  = p2[Index3D(nx, ny, i, j+2, k)];  \
      p2j3  = p2[Index3D(nx, ny, i, j+3, k)];  \
      p2j4  = p2[Index3D(nx, ny, i, j+4, k)];  \
      p2j5  = p2[Index3D(nx, ny, i, j+5, k)];  \
      p2j6  = p2[Index3D(nx, ny, i, j+6, k)];  \
      p2j7  = p2[Index3D(nx, ny, i, j+7, k)];  \
      p2j8  = p2[Index3D(nx, ny, i, j+8, k)];  \
      p2j9  = p2[Index3D(nx, ny, i, j+9, k)];  \
      p2j10 = p2[Index3D(nx, ny, i, j+10, k)]; \
      p2j11 = p2[Index3D(nx, ny, i, j+11, k)]; \
      p2j12 = p2[Index3D(nx, ny, i, j+12, k)]; \
      p2j13 = p2[Index3D(nx, ny, i, j+13, k)]; \
      p2j14 = p2[Index3D(nx, ny, i, j+14, k)]; \
\
      p3[Index3D(nx, ny, i, j, k)] = p3[Index3D(nx, ny, i, j, k)] + \
        c14x * p2j14 + \
        c13x * p2j13 + \
        c12x * p2j12 + \
        c11x * p2j11 + \
        c10x * p2j10 + \
        c9x  * p2j9 +  \
        c8x  * p2j8 +  \
        c7x  * p2j7 +  \
        c6x  * p2j6 +  \
        c5x  * p2j5 +  \
        c4x  * p2j4 +  \
        c3x  * p2j3 +  \
        c2x  * p2j2 +  \
        c1x  * p2j1;




/**************************/
/* Semi-stencil Skeletons */
/**************************/

#define SEMISTENCIL_1LOOP( Anext, A0, STENCIL,      \
                           xs, xe, ys, ye, zs, ze ) \
    /* Head Y */                                    \
    for (k=zs-STENCIL; k<MIN(zs,ze-STENCIL); k++)   \
    {                                               \
      for (j=ys; j<ye; j++)                         \
      {                                             \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                   \
        {                                           \
          FORWARDYZERO( Anext, A0, STENCIL )        \
        }                                           \
      }                                             \
    }                                               \
                                                    \
    /* Body Y */                                    \
    for (k=zs; k<ze-STENCIL; k++)                   \
    {                                               \
      /* Head X */                                  \
      for (j=ys-STENCIL; j<MIN(ys,ye-STENCIL); j++) \
      {                                             \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                   \
        {                                           \
          FORWARDXZERO( Anext, A0, STENCIL )        \
        }                                           \
      }                                             \
                                                    \
      /* Body X */                                  \
      for (j=ys; j<ye-STENCIL; j++)                 \
      {                                             \
        /* Body Z */                                \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                   \
        {                                           \
          BODYZ( Anext, A0, STENCIL )               \
          FORWARDY( Anext, A0, STENCIL )            \
          FORWARDX( Anext, A0, STENCIL )            \
        }                                           \
      }                                             \
                                                    \
      /* Tail X */                                  \
      for (j=MAX(ye-STENCIL,ys); j<ye; j++)         \
      {                                             \
        /* Body Z */                                \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                   \
        {                                           \
          BODYZ( Anext, A0, STENCIL )               \
          FORWARDY( Anext, A0, STENCIL )            \
        }                                           \
      }                                             \
    }                                               \
                                                    \
    /* Tail Y */                                    \
    for (k=MAX(ze-STENCIL,zs); k<ze; k++)           \
    {                                               \
      /* Head X */                                  \
      for (j=ys-STENCIL; j<MIN(ys,ye-STENCIL); j++) \
      {                                             \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                   \
        {                                           \
          FORWARDXZERO( Anext, A0, STENCIL )        \
        }                                           \
      }                                             \
                                                    \
      /* Body X */                                  \
      for (j=ys; j<ye-STENCIL; j++)                 \
      {                                             \
        /* Body Z */                                \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                   \
        {                                           \
          BODYZ( Anext, A0, STENCIL )               \
          FORWARDX( Anext, A0, STENCIL )            \
        }                                           \
      }                                             \
                                                    \
      /* Tail X */                                  \
      for (j=MAX(ye-STENCIL,ys); j<ye; j++)         \
      {                                             \
        /* Body Z */                                \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                   \
        {                                           \
          BODYZ( Anext, A0, STENCIL )               \
        }                                           \
      }                                             \
    }


#define SEMISTENCIL_ALL( Anext, A0, STENCIL,          \
                         xs, xe, ys, ye, zs, ze )     \
    /* Head Y */                                      \
    for (k=zs-STENCIL; k<MIN(zs,ze-STENCIL); k++)     \
    {                                                 \
      for (j=ys; j<ye; j++)                           \
      {                                               \
        for (i=xs; i<xe; i++)                         \
        {                                             \
          FORWARDYZERO( Anext, A0, STENCIL )          \
        }                                             \
      }                                               \
    }                                                 \
                                                      \
    /* Body Y */                                      \
    for (k=zs; k<ze-STENCIL; k++)                     \
    {                                                 \
      /* Head X */                                    \
      for (j=ys-STENCIL; j<MIN(ys,ye-STENCIL); j++)   \
      {                                               \
        for (i=xs; i<xe; i++)                         \
        {                                             \
          FORWARDXZERO( Anext, A0, STENCIL )          \
        }                                             \
      }                                               \
                                                      \
      /* Body X */                                    \
      for (j=ys; j<ye-STENCIL; j++)                   \
      {                                               \
        /* Head Z */                                  \
        for (i=xs-STENCIL; i<MIN(xs,xe-STENCIL); i++) \
        {                                             \
          FORWARDZZERO( Anext, A0, STENCIL )          \
        }                                             \
                                                      \
        /* Body Z */                                  \
        for (i=xs; i<xe-STENCIL; i++)                 \
        {                                             \
          BODYZ_ALL( Anext, A0, STENCIL )             \
          FORWARDZ( Anext, A0, STENCIL )              \
          FORWARDY( Anext, A0, STENCIL )              \
          FORWARDX( Anext, A0, STENCIL )              \
        }                                             \
                                                      \
        /* Tail Z */                                  \
        for (i=MAX(xe-STENCIL,xs); i<xe; i++)         \
        {                                             \
          BODYZ_ALL( Anext, A0, STENCIL )             \
          FORWARDY( Anext, A0, STENCIL )              \
          FORWARDX( Anext, A0, STENCIL )              \
        }                                             \
      }                                               \
                                                      \
      /* Tail X */                                    \
      for (j=MAX(ye-STENCIL,ys); j<ye; j++)           \
      {                                               \
        /* Head Z */                                  \
        for (i=xs-STENCIL; i<MIN(xs,xe-STENCIL); i++) \
        {                                             \
          FORWARDZZERO( Anext, A0, STENCIL )          \
        }                                             \
                                                      \
        /* Body Z */                                  \
        for (i=xs; i<xe-STENCIL; i++)                 \
        {                                             \
          BODYZ_ALL( Anext, A0, STENCIL )             \
          FORWARDZ( Anext, A0, STENCIL )              \
          FORWARDY( Anext, A0, STENCIL )              \
        }                                             \
                                                      \
        /* Tail Z */                                  \
        for (i=MAX(xe-STENCIL,xs); i<xe; i++)         \
        {                                             \
          BODYZ_ALL( Anext, A0, STENCIL )             \
          FORWARDY( Anext, A0, STENCIL )              \
        }                                             \
      }                                               \
    }                                                 \
                                                      \
    /* Tail Y */                                      \
    for (k=MAX(ze-STENCIL,zs); k<ze; k++)             \
    {                                                 \
      /* Head X */                                    \
      for (j=ys-STENCIL; j<MIN(ys,ye-STENCIL); j++)   \
      {                                               \
        for (i=xs; i<xe; i++)                         \
        {                                             \
          FORWARDXZERO( Anext, A0, STENCIL )          \
        }                                             \
      }                                               \
                                                      \
      /* Body X */                                    \
      for (j=ys; j<ye-STENCIL; j++)                   \
      {                                               \
        /* Head Z */                                  \
        for (i=xs-STENCIL; i<MIN(xs,xe-STENCIL); i++) \
        {                                             \
          FORWARDZZERO( Anext, A0, STENCIL )          \
        }                                             \
                                                      \
        /* Body Z */                                  \
        for (i=xs; i<xe-STENCIL; i++)                 \
        {                                             \
          BODYZ_ALL( Anext, A0, STENCIL )             \
          FORWARDZ( Anext, A0, STENCIL )              \
          FORWARDX( Anext, A0, STENCIL )              \
        }                                             \
                                                      \
        /* Tail Z */                                  \
        for (i=MAX(xe-STENCIL,xs); i<xe; i++)         \
        {                                             \
          BODYZ_ALL( Anext, A0, STENCIL )             \
          FORWARDX( Anext, A0, STENCIL )              \
        }                                             \
      }                                               \
                                                      \
      /* Tail X */                                    \
      for (j=MAX(ye-STENCIL,ys); j<ye; j++)           \
      {                                               \
        /* Head Z */                                  \
        for (i=xs-STENCIL; i<MIN(xs,xe-STENCIL); i++) \
        {                                             \
          FORWARDZZERO( Anext, A0, STENCIL )          \
        }                                             \
                                                      \
        /* Body Z */                                  \
        for (i=xs; i<xe-STENCIL; i++)                 \
        {                                             \
          BODYZ_ALL( Anext, A0, STENCIL )             \
          FORWARDZ( Anext, A0, STENCIL )              \
        }                                             \
                                                      \
        /* Tail Z */                                  \
        for (i=MAX(xe-STENCIL,xs); i<xe; i++)         \
        {                                             \
          BODYZ_ALL( Anext, A0, STENCIL )             \
        }                                             \
      }                                               \
    }


#define SEMISTENCIL_FISSION_2LOOPS( Anext, A0, STENCIL, \
                               xs, xe, ys, ye, zs, ze ) \
    /* Head Y */                                        \
    for (k=zs-STENCIL; k<MIN(zs,ze-STENCIL); k++)       \
    {                                                   \
      for (j=ys; j<ye; j++)                             \
      {                                                 \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          FORWARDYZERO( Anext, A0, STENCIL )            \
        }                                               \
      }                                                 \
    }                                                   \
                                                        \
    /* Body Y */                                        \
    for (k=zs; k<ze-STENCIL; k++)                       \
    {                                                   \
      /* Head X */                                      \
      for (j=ys-STENCIL; j<MIN(ys,ye-STENCIL); j++)     \
      {                                                 \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          FORWARDXZERO( Anext, A0, STENCIL )            \
        }                                               \
      }                                                 \
                                                        \
      /* Body X */                                      \
      for (j=ys; j<ye-STENCIL; j++)                     \
      {                                                 \
        /* Body Z */                                    \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ1_2LOOPS( Anext, A0, STENCIL )           \
          FORWARDY( Anext, A0, STENCIL )                \
        }                                               \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ2_2LOOPS( Anext, A0, STENCIL )           \
          FORWARDX( Anext, A0, STENCIL )                \
        }                                               \
      }                                                 \
                                                        \
      /* Tail X */                                      \
      for (j=MAX(ye-STENCIL,ys); j<ye; j++)             \
      {                                                 \
        /* Body Z */                                    \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ1_2LOOPS( Anext, A0, STENCIL )           \
          FORWARDY( Anext, A0, STENCIL )                \
        }                                               \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ2_2LOOPS( Anext, A0, STENCIL )           \
        }                                               \
      }                                                 \
    }                                                   \
                                                        \
    /* Tail Y */                                        \
    for (k=MAX(ze-STENCIL,zs); k<ze; k++)               \
    {                                                   \
      /* Head X */                                      \
      for (j=ys-STENCIL; j<MIN(ys,ye-STENCIL); j++)     \
      {                                                 \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          FORWARDXZERO( Anext, A0, STENCIL )            \
        }                                               \
      }                                                 \
                                                        \
      /* Body X */                                      \
      for (j=ys; j<ye-STENCIL; j++)                     \
      {                                                 \
        /* Body Z */                                    \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ1_2LOOPS( Anext, A0, STENCIL )           \
        }                                               \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ2_2LOOPS( Anext, A0, STENCIL )           \
          FORWARDX( Anext, A0, STENCIL )                \
        }                                               \
      }                                                 \
                                                        \
      /* Tail X */                                      \
      for (j=MAX(ye-STENCIL,ys); j<ye; j++)             \
      {                                                 \
        /* Body Z */                                    \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ1_2LOOPS( Anext, A0, STENCIL )           \
        }                                               \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ2_2LOOPS( Anext, A0, STENCIL )           \
        }                                               \
      }                                                 \
    }


#define SEMISTENCIL_FISSION_3LOOPS( Anext, A0, STENCIL, \
                               xs, xe, ys, ye, zs, ze ) \
    /* Head Y */                                        \
    for (k=zs-STENCIL; k<MIN(zs,ze-STENCIL); k++)       \
    {                                                   \
      for (j=ys; j<ye; j++)                             \
      {                                                 \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          FORWARDYZERO( Anext, A0, STENCIL )            \
        }                                               \
      }                                                 \
    }                                                   \
                                                        \
    /* Body Y */                                        \
    for (k=zs; k<ze-STENCIL; k++)                       \
    {                                                   \
      /* Head X */                                      \
      for (j=ys-STENCIL; j<ys; j++)                     \
      {                                                 \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          FORWARDXZERO( Anext, A0, STENCIL )            \
        }                                               \
      }                                                 \
                                                        \
      /* Body X */                                      \
      for (j=ys; j<ye-STENCIL; j++)                     \
      {                                                 \
        /* Body Z */                                    \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ1_3LOOPS( Anext, A0, STENCIL )           \
        }                                               \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ2_3LOOPS( Anext, A0, STENCIL )           \
          FORWARDY( Anext, A0, STENCIL )                \
        }                                               \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ3_3LOOPS( Anext, A0, STENCIL )           \
          FORWARDX( Anext, A0, STENCIL )                \
        }                                               \
      }                                                 \
                                                        \
      /* Tail X */                                      \
      for (j=MAX(ye-STENCIL,ys); j<ye; j++)             \
      {                                                 \
        /* Body Z */                                    \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ1_3LOOPS( Anext, A0, STENCIL )           \
        }                                               \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ2_3LOOPS( Anext, A0, STENCIL )           \
          FORWARDY( Anext, A0, STENCIL )                \
        }                                               \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ3_3LOOPS( Anext, A0, STENCIL )           \
        }                                               \
      }                                                 \
    }                                                   \
                                                        \
    /* Tail Y */                                        \
    for (k=MAX(ze-STENCIL,zs); k<ze; k++)               \
    {                                                   \
      /* Head X */                                      \
      for (j=ys-STENCIL; j<MIN(ys,ye-STENCIL); j++)     \
      {                                                 \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          FORWARDXZERO( Anext, A0, STENCIL )            \
        }                                               \
      }                                                 \
                                                        \
      /* Body X */                                      \
      for (j=ys; j<ye-STENCIL; j++)                     \
      {                                                 \
        /* Body Z */                                    \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ1_3LOOPS( Anext, A0, STENCIL )           \
        }                                               \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ2_3LOOPS( Anext, A0, STENCIL )           \
        }                                               \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ3_3LOOPS( Anext, A0, STENCIL )           \
          FORWARDX( Anext, A0, STENCIL )                \
        }                                               \
      }                                                 \
                                                        \
      /* Tail X */                                      \
      for (j=MAX(ye-STENCIL,ys); j<ye; j++)             \
      {                                                 \
        /* Body Z */                                    \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ1_3LOOPS( Anext, A0, STENCIL )           \
        }                                               \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ2_3LOOPS( Anext, A0, STENCIL )           \
        }                                               \
_Pragma("ivdep") \
_Pragma("vector always") \
        for (i=xs; i<xe; i+=IINC)                       \
        {                                               \
          BODYZ3_3LOOPS( Anext, A0, STENCIL )           \
        }                                               \
      }                                                 \
    }


/* Add as many entries as versions */
#if !defined(SEMI_ALL)
# if defined(FISSION_2LOOPS)
#   define SEMISTENCIL_( ... ) SEMISTENCIL_FISSION_2LOOPS( __VA_ARGS__ )
# elif defined(FISSION_3LOOPS)
#   define SEMISTENCIL_( ... ) SEMISTENCIL_FISSION_3LOOPS( __VA_ARGS__ )
# else
#   define SEMISTENCIL_( ... ) SEMISTENCIL_1LOOP( __VA_ARGS__ )
# endif
#else
# define SEMISTENCIL_( ... ) SEMISTENCIL_ALL( __VA_ARGS__ )
#endif


#endif /* _SEMI_H_ */

