      PROGRAM SFIG3
*
*     This code is part of a package for solving rank deficient least
*     squares problems, written by:
*     ==================================================================
*     L. Foster                   and   R. Kommu
*     Department of Mathematics         Department of Physics
*     San Jose State University         San Jose State University
*     San Jose, CA 95192                San Jose, CA 95192
*     foster@math.sjsu.edu              rkommu@email.sjsu.edu
*     ==================================================================
*     03/05/2004
*
*  Purpose
*  =======
*
*  xFIG3 is a program that calculates four ratios used by LAPACK
*  as evidence that a least squares routine is correct.  The ratios
*  can be use to create plots similar to those in Figure 3 in Foster
*  and Kommu's paper.  The program calculates these ratios
*  for the  routines xGELSD, xGELSY, and xGELSZ
*  for a set of matrices described by a short data file.
*  This input file specifies the dimensions m and n
*  of the matrices, the ranks of the matrices, scaling
*  of the matrices (normal, up or down), the number of different
*  random matrices to run for this matrix size, rank and scaling and
*  a seed used, along with m, n and the rank, to initialize a random
*  number generator.  The matrices are generated by the routine
*  xQRT25 which is identical to LAPACK's xQRT15 except in xQRT25
*  an arbitrary rank can be specified whereas xQRT15 allows only
*  two rank choices. xQRT15 is used to generate matrices in LAPACK's
*  testing and timing routines.
*
*  A sample input file is:
*
*  The first line is for comments and is not used.
*  100 100   1  1 10 1212
*  100 100  20  1 10 1212
*  100 100  40  1 10 1212  Each line includes:
*  100 100  60  1 10 1212  M N RANK SCALING(1,2,3) N0._REPITITIONS SEED
*  100 100  80  1 10 1212  The final line should be:
*  100 100 100  1 10 1212    0 0 0 0 0 0
*    0 0 0 0 0 0
*
*  This file will generate 100 by 100 matrices with ranks of 20, 40,
*  60, 80, and 100.  The matrices will be scaled normally.
*  For each matrix size, rank and scaling factor 10 random matrices will
*  be run  where the random number seed used to initiate the random
*  number generator is set at 1212.  Note that the
*  fourth number in each line specifies the scaling factor as
*  defined in xQRT15 and xQRT25.  A scale factor of 1 is normal
*  scaling, 2 is scaled up, and 3 is scaled down. The routine xQRT25
*  uses a random number generator which is initialized using the
*  seed value as well m, n and the rank. The line of 0's is used to
*  signal the end of data.
*
*  The program writes the results to files sfig3_12.out, sfig3_14.out,
*  sfig3_16.out, and sfig3_17.out. Each of the files contains one of the
*  ratios described above for the set of matrices specified by
*  the input file.  The ratios in each file are:
*
*  sfig3_12.out:  || s - svlues ||/(||svlues||*eps*max(m,n)) where
*       s indicates the first rank singular values of the original
*       matrix A, svlues are the calculated singular values of the upper
*       triangular matrix that result from the complete orthogonal
*       decomposition of A and eps is machine epsilon.  For xGELSY
*       and xGELSZ these ratios are calculated by LAPACK's xQRT12.
*
*  sfig3_14.out:  A ratio testing that x is the row space of A.  The
*       ratio is calculated by LAPACK's xQRT14.  The comments in
*       xQRT14 describe the  ratio in more detail.
*
*  sfig3_16.out: norm(B - A*X) / ( max(m,n) * norm(A) * norm(X) * eps )
*       as calulated by LAPACK xQRT16.
*
*  sfig3_17.out:     || r' A ||/(||A||*||B||*max(m,n,nrhs)*eps) where
*       r = B - A x.  This ratio is calculated by LAPACK's xQRT17.
*
*  According to the LAPACK guidelines all the ratios should be less
*  than 30.  A Matlab program plotfig3.m can be used to read these
*  files and produce a plot similar to Figure 3.
*
*  Internal Parameters
*  ===================
*
*  LDA     INTEGER
*          The leading dimension of the A matrix.
*
*  LDB     INTEGER
*          The leading dimension of the B matrix.
*
*  LEVEL   INTEGER
*          Input and output level.
*          Level = 0: No monitor output except for errors. File
*                     output.
*          Level = 1: Some monitor output. File output.
*          Level = 2: Periodic monitor output.  File output.
*          Level = 3: Prompts for interactive input.
*                     Periodic monitor output. File output.
*
*  LWORK   INTEGER
*          The allocated size of the work space WORK.  LWORK is set to
*          (LDA + NRHS)*(LDA + 2) + 2*LDA  which is required by
*          the documentation in xQRT14 and xQRT12.
*
*  MAXIC   INTEGER
*          The maximum number of runs.  The allocated number of rows
*          in the arrays acc12, acc14, acc16, and acc17.
*
*  NRHS    INTEGER
*          The number of columns in B.
*
      INTEGER            LDA, LDB, LEVEL, LWORK, MAXIC, NRHS
      PARAMETER          ( LEVEL = 2 )
      PARAMETER          ( LDA = 500, LDB = 500, MAXIC = 10000,
     $                   NRHS = 1 )
      PARAMETER          ( LWORK = ( LDA+NRHS )*( LDA+2 )+2*LDA )
*
*
      INTEGER            I, IC, INFO, IREP, ISAME, ISEEDT, CRANK, J, M,
     $                   MNMIN, N, NOUT1, NOUT2, NOUT3, NOUT4,
     $                   NREP, RANK, SCALE
      INTEGER            IPVT( LDA )
      INTEGER            ISEED( 4 )
      INTEGER            IWORK( LWORK )
*
      CHARACTER*80       LINE
*
      REAL               EPS, NORMA, NORMB, ONE, SUM, RCOND, ZERO
      REAL               DMAX12, DMAX16, DMAX14, DMAX17
      REAL               YMAX12, YMAX16, YMAX14, YMAX17
      REAL               ZMAX12, ZMAX16, ZMAX14, ZMAX17
      REAL               A( LDA, LDA ), A0( LDA, LDA ), B( LDB, NRHS ),
     $                   B0( LDB, NRHS ), C( LDB, NRHS ),
     $                   X( LDB, NRHS ), XY( LDB, NRHS )
      REAL               WORK( LWORK )
      REAL               S( LDA ), SD( LDA )
      REAL               ACC12( MAXIC, 3 ), ACC14( MAXIC, 3 ),
     $                   ACC16( MAXIC, 3 ), ACC17( MAXIC, 3 )
*
*
*     .. External Functions ..
      REAL               SASUM, SLAMCH, SQRT12, SQRT14, SQRT17
      EXTERNAL           SASUM, SLAMCH, SQRT12, SQRT14, SQRT17
*     ..
*     .. External Subroutines ..
      EXTERNAL           SAXPY, SGELSD, SGELSY, SGELSZ, SQRT16, SQRT25
*     ..
*     .. Intrinsic Functions ..
      INTRINSIC          REAL, MAX, SQRT
*     ..
*     Executable statements
*
      IF( LEVEL.NE.0 ) THEN
         WRITE( *, * )'xFIG3 is running'
      END IF
*
      IC = 0
      ISAME = 0
      ONE = 1.0E0
      ZERO = 0.0E0
      EPS = SLAMCH( 'Epsilon' )
*
*     Threshold for rank estimation
*
      RCOND = SQRT( EPS ) - ( SQRT( EPS )-EPS ) / 2
*
      IF( LEVEL.EQ.0 .OR. LEVEL.EQ.1 .OR. LEVEL.EQ.2 ) THEN
*        The first line of the input file is for comments.
*        Read and ignore first line.
         READ( *, FMT = '( A80 )' )LINE
      END IF
*
*
   10 CONTINUE
*     Top of loop
*
      IF( LEVEL.EQ.3 ) THEN
*        Prompt for keyboard input.
         WRITE( *, * )
         WRITE( *, * )'ENTER ROWS, COLS, RANK, SCALE(1-3),',
     $      ' NREP AND SEED   (ALL 0 TO QUIT)'
         READ( *, * )M, N, RANK, SCALE, NREP, ISEEDT
      ELSE
         READ( *, * )M, N, RANK, SCALE, NREP, ISEEDT
      END IF
      IF( M.GT.LDA .OR. N.GT.LDA ) THEN
         WRITE( *, * )
         WRITE( *, * )'M OR N IS TOO LARGE, INCREASE LDA.'
*        Exit loop
         GO TO 280
      END IF
      IF( M.EQ.0 ) THEN
*           Exit loop when M < 0
         GO TO 280
      END IF
      MNMIN = N
      IF( M.LE.N )
     $   MNMIN = M
*     Initialize seed for random number generator
      ISEED( 1 ) = ISEEDT
      ISEED( 2 ) = 10000*M + N
      ISEED( 3 ) = RANK
      ISEED( 4 ) = 13
*
      DO 260 IREP = 1, NREP
*
*        Generate a test matrix
*
         CALL SQRT25( SCALE, RANK, M, N, NRHS, A0, LDA, B0, LDB, S,
     $                NORMA, NORMB, ISEED, WORK, LWORK )
         IC = IC + 1
         IF( IC.GT.MAXIC ) THEN
            WRITE( *, * )
            WRITE( *, * )'THE MAXIMIUM NUMBER OF RUNS ',
     $         'HAS BEEN REACHED. INCREASE MAXIC.'
            GO TO 280
         END IF
*
*        ******* xGELSD **********************************
*
         DO 40 I = 1, M
            DO 20 J = 1, N
               A( I, J ) = A0( I, J )
   20       CONTINUE
            DO 30 J = 1, NRHS
               B( I, J ) = B0( I, J )
   30       CONTINUE
   40    CONTINUE
         DO 50 J = 1, N
            IPVT( J ) = 0
   50    CONTINUE
*
         CALL SGELSD( M, N, NRHS, A, LDA, B, LDB, SD, RCOND, CRANK,
     $                WORK, LWORK, IWORK, INFO )
         DO 70 I = 1, N
            DO 60 J = 1, NRHS
               X( I, J ) = B( I, J )
   60       CONTINUE
   70    CONTINUE
*
*        Test:  Compute relative error in sing. values
*
         IF( RANK.GT.0 ) THEN
            CALL SAXPY( MNMIN, -ONE, S, 1, SD, 1 )
            ACC12( IC, 1 ) = SASUM( MNMIN, SD, 1 ) /
     $                       SASUM( MNMIN, S, 1 ) /
     $                       ( EPS*REAL( MNMIN ) )
         ELSE
            ACC12( IC, 1 ) = ZERO
         END IF
*
*        Test:  Compute error in solution
*
         DO 90 I = 1, M
            DO 80 J = 1, NRHS
               B( I, J ) = B0( I, J )
   80       CONTINUE
   90    CONTINUE
*
         CALL SQRT16( 'No transpose', M, N, NRHS, A0, LDA, X, LDB, B,
     $                LDB, WORK, ACC16( IC, 1 ) )
*
*        Test:  Check norm of r'*A
*
         ACC17( IC, 1 ) = ZERO
         IF( M.GT.CRANK )
     $      ACC17( IC, 1 ) = SQRT17( 'No transpose', 1, M, N, NRHS, A0,
     $                       LDA, X, LDB, B0, LDB, C, WORK, LWORK )
*
*        Test:  Check if x is in the rowspace of A
*
         ACC14( IC, 1 ) = ZERO
         IF( N.GT.CRANK )
     $      ACC14( IC, 1 ) = SQRT14( 'No transpose', M, N, NRHS, A0,
     $                       LDA, X, LDB, WORK, LWORK )
*
*        ******* xGELSY **********************************
*
         DO 120 I = 1, M
            DO 100 J = 1, N
               A( I, J ) = A0( I, J )
  100       CONTINUE
            DO 110 J = 1, NRHS
               B( I, J ) = B0( I, J )
  110       CONTINUE
  120    CONTINUE
         DO 130 J = 1, N
            IPVT( J ) = 0
  130    CONTINUE
*
         CALL SGELSY( M, N, NRHS, A, LDA, B, LDB, IPVT, RCOND, CRANK,
     $                WORK, LWORK, INFO )
         DO 150 I = 1, N
            DO 140 J = 1, NRHS
               X( I, J ) = B( I, J )
               XY( I, J ) = X( I, J )
  140       CONTINUE
  150    CONTINUE
*
*        Test :  Compute relative error in sing. values of T11
*
         ACC12( IC, 2 ) = SQRT12( CRANK, CRANK, A, LDA, S, WORK, LWORK )
*
*        Test:  Compute error in solution
*
         DO 170 I = 1, M
            DO 160 J = 1, NRHS
               B( I, J ) = B0( I, J )
  160       CONTINUE
  170    CONTINUE
*
         CALL SQRT16( 'No transpose', M, N, NRHS, A0, LDA, X, LDB, B,
     $                LDB, WORK, ACC16( IC, 2 ) )
*
*        Test:  Check norm of r'*A
*
         ACC17( IC, 2 ) = ZERO
         IF( M.GT.CRANK )
     $      ACC17( IC, 2 ) = SQRT17( 'No transpose', 1, M, N, NRHS, A0,
     $                       LDA, X, LDB, B0, LDB, C, WORK, LWORK )
*
*        Test:  Check if x is in the rowspace of A
*
         ACC14( IC, 2 ) = ZERO
         IF( N.GT.CRANK )
     $      ACC14( IC, 2 ) = SQRT14( 'No transpose', M, N, NRHS, A0,
     $                       LDA, X, LDB, WORK, LWORK )
*
*        ******* xGELSZ **********************************
*
         DO 200 I = 1, M
            DO 180 J = 1, N
               A( I, J ) = A0( I, J )
  180       CONTINUE
            DO 190 J = 1, NRHS
               B( I, J ) = B0( I, J )
  190       CONTINUE
  200    CONTINUE
         DO 210 J = 1, N
            IPVT( J ) = 0
  210    CONTINUE
*
         CALL SGELSZ( M, N, NRHS, A, LDA, B, LDB, IPVT, RCOND, CRANK,
     $                WORK, LWORK, INFO )
         SUM = 0
         DO 230 I = 1, N
            DO 220 J = 1, NRHS
               X( I, J ) = B( I, J )
               SUM = SUM + ABS( X( I, J )-XY( I, J ) )
  220       CONTINUE
  230    CONTINUE
         IF( SUM.EQ.ZERO )
     $      ISAME = ISAME + 1
*
*        Test :  Compute relative error in sing. values of T11
*
         ACC12( IC, 3 ) = SQRT12( CRANK, CRANK, A, LDA, S, WORK, LWORK )
*
*        Test:  Compute error in solution
*
         DO 250 I = 1, M
            DO 240 J = 1, NRHS
               B( I, J ) = B0( I, J )
  240       CONTINUE
  250    CONTINUE
*
         CALL SQRT16( 'No transpose', M, N, NRHS, A0, LDA, X, LDB, B,
     $                LDB, WORK, ACC16( IC, 3 ) )
*
*        Test:  Check norm of r'*A
*
         ACC17( IC, 3 ) = ZERO
         IF( M.GT.CRANK )
     $      ACC17( IC, 3 ) = SQRT17( 'No transpose', 1, M, N, NRHS, A0,
     $                       LDA, X, LDB, B0, LDB, C, WORK, LWORK )
*
*        Test:  Check if x is in the rowspace of A
*
         ACC14( IC, 3 ) = ZERO
         IF( N.GT.CRANK )
     $      ACC14( IC, 3 ) = SQRT14( 'No transpose', M, N, NRHS, A0,
     $                       LDA, X, LDB, WORK, LWORK )
*
  260 CONTINUE
*
      IF( LEVEL.EQ.2 .OR. LEVEL.EQ.3 ) THEN
*        Monitor output.
         DMAX16 = 0
         DMAX12 = 0
         DMAX17 = 0
         DMAX14 = 0
         YMAX16 = 0
         YMAX12 = 0
         YMAX17 = 0
         YMAX14 = 0
         ZMAX16 = 0
         ZMAX12 = 0
         ZMAX17 = 0
         ZMAX14 = 0
         DO 270 I = IC - NREP + 1, IC
            DMAX16 = MAX( DMAX16, ACC16( I, 1 ) )
            DMAX12 = MAX( DMAX12, ACC12( I, 1 ) )
            DMAX17 = MAX( DMAX17, ACC17( I, 1 ) )
            DMAX14 = MAX( DMAX14, ACC14( I, 1 ) )
            YMAX16 = MAX( YMAX16, ACC16( I, 2 ) )
            YMAX12 = MAX( YMAX12, ACC12( I, 2 ) )
            YMAX17 = MAX( YMAX17, ACC17( I, 2 ) )
            YMAX14 = MAX( YMAX14, ACC14( I, 2 ) )
            ZMAX16 = MAX( ZMAX16, ACC16( I, 3 ) )
            ZMAX12 = MAX( ZMAX12, ACC12( I, 3 ) )
            ZMAX17 = MAX( ZMAX17, ACC17( I, 3 ) )
            ZMAX14 = MAX( ZMAX14, ACC14( I, 3 ) )
  270    CONTINUE
         WRITE( *, * )
         WRITE( *, * )'For rank = ', INT( RANK ), '  max of ',
     $      INT( NREP ), ' ratios calculated by'
         WRITE( *, * )'            xQRT16         xQRT12',
     $      '         xQRT17         xQRT14'
         WRITE( *, * )'xGELSD: ', REAL( DMAX16 ), REAL( DMAX12 ),
     $      REAL( DMAX17 ), REAL( DMAX14 )
         WRITE( *, * )'xGELSY: ', REAL( YMAX16 ), REAL( YMAX12 ),
     $      REAL( YMAX17 ), REAL( YMAX14 )
         WRITE( *, * )'xGELSZ: ', REAL( ZMAX16 ), REAL( ZMAX12 ),
     $      REAL( ZMAX17 ), REAL( ZMAX14 )
      END IF
*
      GO TO 10
*     End of loop
  280 CONTINUE
*
*     Calculate the maximum of the four ratios for the three methods
      DMAX16 = 0
      DMAX12 = 0
      DMAX17 = 0
      DMAX14 = 0
      YMAX16 = 0
      YMAX12 = 0
      YMAX17 = 0
      YMAX14 = 0
      ZMAX16 = 0
      ZMAX12 = 0
      ZMAX17 = 0
      ZMAX14 = 0
      DO 290 I = 1, IC
         DMAX16 = MAX( DMAX16, ACC16( I, 1 ) )
         DMAX12 = MAX( DMAX12, ACC12( I, 1 ) )
         DMAX17 = MAX( DMAX17, ACC17( I, 1 ) )
         DMAX14 = MAX( DMAX14, ACC14( I, 1 ) )
         YMAX16 = MAX( YMAX16, ACC16( I, 2 ) )
         YMAX12 = MAX( YMAX12, ACC12( I, 2 ) )
         YMAX17 = MAX( YMAX17, ACC17( I, 2 ) )
         YMAX14 = MAX( YMAX14, ACC14( I, 2 ) )
         ZMAX16 = MAX( ZMAX16, ACC16( I, 3 ) )
         ZMAX12 = MAX( ZMAX12, ACC12( I, 3 ) )
         ZMAX17 = MAX( ZMAX17, ACC17( I, 3 ) )
         ZMAX14 = MAX( ZMAX14, ACC14( I, 3 ) )
  290 CONTINUE
*
      NOUT1 = 12
      OPEN( NOUT1, FILE = 'fig3_12.out' )
      NOUT2 = 14
      OPEN( NOUT2, FILE = 'fig3_14.out' )
      NOUT3 = 16
      OPEN( NOUT3, FILE = 'fig3_16.out' )
      NOUT4 = 17
      OPEN( NOUT4, FILE = 'fig3_17.out' )
*
      DO 300 I = 1, IC
         WRITE( NOUT1, * )REAL( ACC12( I, 1 ) ), REAL( ACC12( I, 2 ) ),
     $      REAL( ACC12( I, 3 ) )
  300 CONTINUE
*
      DO 310 I = 1, IC
         WRITE( NOUT2, * )REAL( ACC14( I, 1 ) ), REAL( ACC14( I, 2 ) ),
     $      REAL( ACC14( I, 3 ) )
  310 CONTINUE
*
      DO 320 I = 1, IC
         WRITE( NOUT3, * )REAL( ACC16( I, 1 ) ), REAL( ACC16( I, 2 ) ),
     $      REAL( ACC16( I, 3 ) )
  320 CONTINUE
*
      DO 330 I = 1, IC
         WRITE( NOUT4, * )REAL( ACC17( I, 1 ) ), REAL( ACC17( I, 2 ) ),
     $      REAL( ACC17( I, 3 ) )
  330 CONTINUE
*
      IF( LEVEL.NE.0 ) THEN
         WRITE( *, * )
         WRITE( *, * )'The xGELSY and xGELSZ solutions were identical',
     $      ' for ', 100*REAL( ISAME ) / REAL( IC )
         WRITE( *, * )'     percent of the ', IC, ' runs.'
*
         WRITE( *, * )
         WRITE( *, * )'The maxima of ', IC, ' ratios calculated by'
         WRITE( *, * )'            xQRT16         xQRT12         xQRT17'
     $      , '         xQRT14'
         WRITE( *, * )'xGELSD: ', REAL( DMAX16 ), REAL( DMAX12 ),
     $      REAL( DMAX17 ), REAL( DMAX14 )
         WRITE( *, * )'xGELSY: ', REAL( YMAX16 ), REAL( YMAX12 ),
     $      REAL( YMAX17 ), REAL( YMAX14 )
         WRITE( *, * )'xGELSZ: ', REAL( ZMAX16 ), REAL( ZMAX12 ),
     $      REAL( ZMAX17 ), REAL( ZMAX14 )
*
         WRITE( *, * )
         WRITE( *, * )'The results have been written to files ',
     $      'fig3_12.out'
         WRITE( *, * )'     fig3_14.out, fig3_16.out and ',
     $      'fig3_17.out.'
         WRITE( *, * )'Run plotfig3 from Matlab to plot the ratios.'
         WRITE( *, * )
      END IF
      END
