package RIB::RepositoryCreator;

use strict;
use RIB::Util;
use RIB::Parser;
use File::Copy;
use File::Path;
use HTML::Entities;

sub new {
  my $proto = shift;
  my $class = ref($proto) || $proto;
  my $self  = {};
  bless ($self, $class);
  return $self;
}


sub createRepository {
  my $self = shift;
  my $config_xml = shift;
  my $name = shift;
  my $contact = shift;
  my $password = shift;
  my $util = shift;

  my $rp = new RIB::Parser;
  my $ribdir = $util->RibDir;
  my $riburl = $util->RibUrl;
  my $sth = undef;
  my $newdir = '';
  my $RelAttrNameMaxSize = 50;

  ##############################################
  # weed out characters that might cause problems
  $config_xml =~ s/[^\w\-\~\`\!\@\#\$\%\^\&\*\(\)\_\+\-\=\{\}\|\[\]\\\;\'\:\"\,\.\/\<\>\?\\\s]//g;


  ##############################################
  # get new handle
  my $repo_handle = $util->getSequence("repositories");

  ##############################################
  # parse configuration
  unless ($rp->parse_config($config_xml)) {
    return ($rp->errormsg);
  }

  # make sure configuration is valid
  unless (@{$rp->classes}) {
    return "Invalid configuration - no class declarations could be found.";
  }
  foreach (@{$rp->classes}) {
    if ($_->name() !~ /^[A-Za-z0-9\_]+$/) {
      return "Invalid class name in configuration : &quot;"
                 . $_->name
                 . "&quot;. Legal characters for class names are in the ranges "
                 . "A-Z, a-z, and 0-9. Underscore (_) characters are "
                 . "also acceptable.";
    }
  }

  ##############################################
  # create the directory structure for this repository
  $newdir = "$ribdir/docRoot/" .$repo_handle;
  if (-e $newdir) {
    return ("Directory $newdir already exists");
  }
  unless (mkdir($newdir, 0755)) {
    return("can't create directory $newdir : $!");
  }
  unless (mkpath("$newdir/files", 0, 0755)) {
    rmtree ($newdir,0,1);
    return("can't create directory $newdir/files : $!");
  }
  unless (open(CONFIG,">$newdir/config.xml")) {
    rmtree ($newdir,0,1);
    return("can't create $newdir/config.xml : $!");
  }
  print CONFIG $config_xml;
  close CONFIG;
  
  
  ##############################################
  # Copy in default HTML templates
  opendir (DIR, "$ribdir/conf") || return("Can't opendir $ribdir/conf : $!");
  foreach my $file (grep(/\.html$/,readdir(DIR))) {
    my $template = '';
    unless (open (INPUT, "$ribdir/conf/$file")) {
      rmtree ($newdir,0,1);
      return("can't read $ribdir/conf/$file : $!");
    }
    my $tmp=$/;
    undef ($/);
    $template = <INPUT>;
    $/ = $tmp;
    close (INPUT);
    unless (open (TOP, ">$newdir/$file")) {
      my $error = $!; # in case rmtree resets $!
      rmtree ($newdir,0,1);
      return("can't open $newdir/$file for writing : $error");
    }
    print TOP $template;
    close TOP;
  }
  closedir(DIR);
  
  ##############################################
  # configure the database for this repository
  
  my $class;
  my $primary_class = $rp->classes->[0];
  lc($primary_class->name) eq 'rigobject' and $primary_class = $rp->classes->[1];
  my $primary_attribute = $primary_class->attributes->[0];
  
  # insert record into repositories table
  $sth = $util->dbh->prepare("INSERT INTO repositories (name, handle, password, "
                     . "contact, primary_class, primary_attribute) VALUES ("
                     . $util->dbh->quote($name)
                     . ",$repo_handle,"
                     . $util->dbh->quote($password)
                     . ","
                     . $util->dbh->quote($contact)
                     . ",'"
                     . $primary_class->name
                     . "','"
                     . $primary_attribute->name
                     . "')");
  unless ($sth->execute) {
    rmtree ($newdir,0,1);
    return($DBI::errstr);
  }

  my @created_tables = ();
  
  # create a table for each class
  foreach $class (@{$rp->classes}) {
    next if lc($class->name) eq 'rigobject';
    my $table_name = $repo_handle . "_" . $class->name;
    my $create_query = "CREATE TABLE $table_name ( ";
    my $field;
    foreach $field (@{$class->relationships}) {
      unless ($field->name =~ /^[A-Za-z][A-Za-z0-9]*\.[A-Za-z][A-Za-z0-9]*$/) {
        rmtree ($newdir,0,1);
        foreach (@created_tables) { $util->dbh->do("drop table $_"); }
        $util->dbh->do("DELETE FROM repositories WHERE handle=$repo_handle");
        return("Can't use &quot;"
             . $field->name
             . "&quot; for a relationship name in class &quot;"
             . $class->name
             . "&quot; because it is not of the proper form. ");
      }
      if (length($field->name) > $RelAttrNameMaxSize) {
        rmtree ($newdir,0,1);
        foreach (@created_tables) { $util->dbh->do("drop table $_"); }
        $util->dbh->do("DELETE FROM repositories WHERE handle=$repo_handle");
        return("Can't use &quot;"
             . $field->name
             . "&quot; for a field name in class &quot;"
             . $class->name
             . "&quot; because it is too long. The maximum size "
             . "for a field name is "
             . $RelAttrNameMaxSize);
      }
      my $fieldname = $field->name;
      $fieldname =~ s/\./\_/;
      $create_query .= " $fieldname\_ TEXT,";
    }
    foreach $field (@{$class->attributes}) {
      unless ($field->name =~ /^[A-Za-z][A-Za-z0-9]*$/) {
        rmtree ($newdir,0,1);
        foreach (@created_tables) { $util->dbh->do("drop table $_"); }
        $util->dbh->do("DELETE FROM repositories WHERE handle=$repo_handle");
        return("Can't use &quot;"
                   . $field->name
                   . "&quot; for an attribute name in class &quot;"
                   . $class->name
                   . "&quot; because it is not of the proper form. "
                   . "Attribute names must consist entirely of letters and "
                   . "numbers (no spaces) and must begin with a letter.");
      }
      if (length($field->name) > $RelAttrNameMaxSize) {
        rmtree ($newdir,0,1);
        foreach (@created_tables) { $util->dbh->do("drop table $_"); }
        $util->dbh->do("DELETE FROM repositories WHERE handle=$repo_handle");
        return("Can't use &quot;"
                   . $field->name
                   . "&quot; for a field name in class &quot;"
                   . $class->name
                   . "&quot; because it is too long. The maximum size "
                   . "for a field name is "
                   . $RelAttrNameMaxSize);
      }
      $create_query .= " " . $field->name . "_ TEXT,";
    }
    $create_query .= " handle BIGINT UNSIGNED,"
                  .  " extends BIGINT UNSIGNED,"
                  .  " last_modified TIMESTAMP,"
                  .  " created TIMESTAMP,"
                  .  " approved TINYINT)";
                       # entends field used when a row has more than
                       # one entry for a field. For example, if a
                       # table has an 'Abstract' field then if someone
                       # creates an object that has two Abstracts
                       # then all except the first must be stored
                       # in an extra row. extends will contain
                       # the handle of the row that "owns" this one.
    unless ($util->dbh->do($create_query)) {
      my $error = $DBI::errstr;
      $util->dbh->do("delete from repositories where handle=$repo_handle");
      foreach (@created_tables) { $util->dbh->do("drop table $_"); }
      rmtree ($newdir,0,1);
      return($error);
    }
    # now create a row in the sequences table for this table
    unless ($util->dbh->do("INSERT INTO sequences (table_name,_seq) "
                   . "VALUES ('$table_name', 1)")) {
      my $error = $DBI::errstr;
      $util->dbh->do("delete from repositories where handle=$repo_handle");
      foreach (@created_tables) { $util->dbh->do("drop table $_"); }
      rmtree ($newdir,0,1);
      return($error);
    }
    push (@created_tables,$table_name);
  }
  
  ####################################################
  # create the interoperation tables
  my $create_query = undef;
  
  $create_query = "CREATE TABLE ${repo_handle}\_interop_registry ( name TEXT, url TEXT, handle BIGINT UNSIGNED, checkpoint TIMESTAMP, last_attempt TIMESTAMP, last_success TIMESTAMP, last_failure TIMESTAMP, update_interval INT UNSIGNED, log TEXT)";
  unless ($util->dbh->do($create_query)) {
      my $error = $DBI::errstr;
      $util->dbh->do("delete from repositories where handle=$repo_handle");
      foreach (@created_tables) { $util->dbh->do("drop table $_"); }
      rmtree ($newdir,0,1);
      return($error);
  }
  push (@created_tables,$repo_handle."_interop_registry");

  my $c = $rp->getClass($primary_class->name);
  $create_query = "CREATE TABLE $repo_handle\_interop_objects (";
  foreach my $field (@{$c->attributes},@{$c->relationships}) {
    my $fieldname = $field->name;
    $fieldname =~ s/\./_/;
    $create_query .= "$fieldname\_ TEXT, ";
  }
  $create_query .= "url TEXT, owner_handle BIGINT UNSIGNED, "
          . "last_modified TIMESTAMP, created TIMESTAMP)";
  unless ($util->dbh->do($create_query)) {
      my $error = $DBI::errstr;
      $util->dbh->do("delete from repositories where handle=$repo_handle");
      foreach (@created_tables) { $util->dbh->do("drop table $_"); }
      rmtree ($newdir,0,1);
      return($error);
  }
  push (@created_tables,$repo_handle."_interop_objects");

  return $repo_handle;
}

1;
