#include <stdio.h>
#include <string.h>
#include "ab-post.h"

#define MAXEVENTS 2000   /* Max total # of events */
#define MAXWORDSIZE 35   /* Max size of comp/arch/process name */
#define MAXSUCC 25       /* Max number of dag successors */
#define MAXPRED 25       /* Max number of dag predecessors */
#define MAXPROC 50       /* Max number of processes */
#define MAXPEND 50       /* Max number of temporarily unmatched send/rcvs */
#define NONE (-1)


/* The routines in this file should construct the causality dag, 
*  given an array of event structures that has been read in.
*  It is assumed that all the events at a particular process are
*  correctly ordered.
*/

/* The structure that will hold an event.  
   Assumptions:
   * Component & machine names are of size <= MAXWORDSIZE
   * Depending on the event type, some fields may be missing. 
*/

/* Revision 1.0: 1993/07/24 swee
 * Added stuff needed by barrier call
*/
typedef int bool;

struct event {
  long event_sec;
  long event_usec;
  int event_type;
  char event_comp[MAXWORDSIZE];  
  int event_pid;
  int event_rc;
  char dest_comp[MAXWORDSIZE];
  char dest_mach[MAXWORDSIZE];
  int dest_pid;
  int msg_type;
  int msg_length;
  int msg_serial;
  int msg_count;
  long abmon_sec;
  long abmon_usec;
  int numsucc;
  int succ[MAXSUCC];
  int numpred;
  int pred[MAXPRED];
  int data; /* used by xab-post routines */
  char name[MAXWORDSIZE];  /* used by barrier */
  int number;
};



extern struct event *read_trf();



/* Compress a list of integers so that all NONE's appear at the
*  end.
*/
compress_list(list,size)
int *list;
int size;
{
  int p,nones;

  p=0;
  nones=0;
  
  while (p<size) {
    if (list[p]==NONE)
      nones++;
    else if (nones>0) {
      list[p-nones]=list[p];
      list[p]=NONE;
    }
    p++;
  }
}

/* Add an element to an integer list which is terminated by NONE values.
*  Return code 0 if successful, 1 if list already full.
*/

int add_to_list(list,value,size)
int *list;
int value;
int size;
{
  int i;

  i=0;

  while((list[i]!=NONE) && (i<size))
    i++;

  if (i<size) {
    list[i]=value;
    return(0);
  }
  else
    return(1);
}






/* Add NONE as a predecessor of an event, indicating it is the first
*  event at that process.
*/

add_none_pred(ev,index)
struct event *ev;
int index;
{
  if (ev[index].numpred==MAXPRED) {
    printf("ERROR: too many predecessors at node %d. \n",index);
    exit(1);
  }

  ev[index].pred[ev[index].numpred++]=NONE;
}

/* Add NONE as a successor of an event, indicating it is the last event
*  at that process.
*/

add_none_succ(ev,index)
struct event *ev;
int index;
{
  if (ev[index].numsucc==MAXSUCC) {
    printf("ERROR: too many successors at node %d. \n",index);
    exit(1);
  }

  ev[index].succ[ev[index].numsucc++]=NONE;
}




/*  Add links between events "parent" and "child", showing that
*   parent precedes child in the dag.
*/

add_links(ev,parent,child)
struct event *ev;
int parent;
int child;
{

  if (ev[parent].numsucc==MAXSUCC) {
    printf("ERROR: too many successors at node %d. \n",parent);
    exit(1);
  }
  if (ev[child].numpred==MAXPRED) {
    printf("ERROR: too many predecessors at node %d. \n",child);
    exit(1);
  }

  ev[parent].succ[ev[parent].numsucc++]=child;
  ev[child].pred[ev[child].numpred++]=parent;

}





/* Make an event be pointed to by the previous event at its process,
*  and vice versa.
*  The array last_event is not assumed to be in any particular order,
*  though it must contain only one event at each process.  
*/

set_standard_ptrs(ev,index,last_event)
struct event *ev;
int index;
int last_event[MAXPROC+1];
{
  int j=0;

  while (((strcmp(ev[last_event[j]].event_comp,ev[index].event_comp)!=0) ||
	 (ev[last_event[j]].event_pid != ev[index].event_pid)) &&
	 (last_event[j] != NONE) )
    j++;

  if (j==MAXPROC+1) {
    printf("ERROR: too many processes. \n");
    exit(1);
  }
  else if (last_event[j]==NONE) {
    last_event[j]=index;
    add_none_pred(ev,index);
  }
  else {
    add_links(ev,last_event[j],index);
    last_event[j]=index;
  }

  /* if the event is initiate_done, copy some info to corresponding
  *  initiate to make things simpler later on.
  */

  if (ev[index].event_type==XAB_initiate_done) {
    j=0;
    while (ev[ev[index].pred[j]].event_type != XAB_initiate)
	j++;
    ev[ev[index].pred[j]].dest_pid=ev[index].dest_pid;
  }

}




/* Set dag links correctly to signify a '[v]send'-- if the message's
*  receive has not been found yet, just add the new message to the
*  send list.
*/

check_send(ev,index,pending_sends,pending_rcvs)
struct event *ev;
int index;
int *pending_sends;
int *pending_rcvs;
{
  int i=0;

  while ((pending_rcvs[i] != NONE) && 
	 !((strcmp(ev[pending_rcvs[i]].event_comp,ev[index].dest_comp)==0) &&
	   (strcmp(ev[pending_rcvs[i]].dest_comp,ev[index].event_comp)==0) &&
	   (ev[pending_rcvs[i]].event_pid==ev[index].dest_pid) &&
	   (ev[pending_rcvs[i]].dest_pid==ev[index].event_pid) &&
	   (ev[pending_rcvs[i]].msg_serial==ev[index].msg_serial)))
    i++;

  if (pending_rcvs[i]==NONE) {
    if (add_to_list(pending_sends,index,MAXPEND)!=0) {
      printf("Error: too many pending sends at node %d. \n",index);
      exit(1);
    }
  }
  else {
    add_links(ev,index,pending_rcvs[i]);
    pending_rcvs[i]=NONE;
    compress_list(pending_rcvs,MAXPEND);
  }
}
/* Set dag links correctly to signify a broadcast-- match with any
*  corresponding receives, and save in the broadcast list.  
*  (Broadcast == send to pid -1).
*/

check_broadcast(ev,index,pending_bcasts,pending_rcvs)
struct event *ev;
int index;
int *pending_bcasts;
int *pending_rcvs;
{
  int i=0;

  while (pending_rcvs[i] != NONE) { 
    if ((strcmp(ev[pending_rcvs[i]].event_comp,ev[index].dest_comp)==0) &&
	(strcmp(ev[pending_rcvs[i]].dest_comp,ev[index].event_comp)==0) &&
	(ev[pending_rcvs[i]].dest_pid==ev[index].event_pid) &&
	(ev[pending_rcvs[i]].msg_serial==ev[index].msg_serial)) {
      add_links(ev,index,pending_rcvs[i]);
      pending_rcvs[i]=NONE;
      compress_list(pending_rcvs,MAXPEND);
    }
    i++;
  }
      
  if (add_to_list(pending_bcasts,index,MAXPEND)!=0) {
	  printf("Error: too many pending broadcasts at node %d. \n",index);
	  exit(1);
	}
       
}


      



/* Set dag links correctly to signify a '[v]rcv_done'-- if the message's
*  send has not been found yet, just add the new message to the
*  receive list.
*/

check_rcv(ev,index,pending_sends,pending_bcasts,pending_rcvs)
struct event *ev;
int index;
int *pending_sends;
int *pending_bcasts;
int *pending_rcvs;
{
  int i=0;

  while ((pending_sends[i] != NONE) && 
	 !((strcmp(ev[pending_sends[i]].dest_comp,ev[index].event_comp)==0) &&
	   (strcmp(ev[pending_sends[i]].event_comp,ev[index].dest_comp)==0) &&
	   (ev[pending_sends[i]].dest_pid==ev[index].event_pid) &&
	   (ev[pending_sends[i]].event_pid==ev[index].dest_pid) &&
	   (ev[pending_sends[i]].msg_serial==ev[index].msg_serial)))
    i++;

  if (pending_sends[i]!=NONE)  {
    add_links(ev,pending_sends[i],index);
    pending_sends[i]=NONE;
    compress_list(pending_sends,MAXPEND);
  }
  else {

    i=0;
    while ((pending_bcasts[i] != NONE) && 
	 !((strcmp(ev[pending_bcasts[i]].dest_comp,ev[index].event_comp)==0)&&
	   (strcmp(ev[pending_bcasts[i]].event_comp,ev[index].dest_comp)==0)&&
	   (ev[pending_bcasts[i]].event_pid==ev[index].dest_pid) &&
	   (ev[pending_bcasts[i]].msg_serial==ev[index].msg_serial)))
    i++;

    if (pending_bcasts[i] == NONE)
      {
	if (add_to_list(pending_rcvs,index,MAXPEND)!=0) {
	  printf("Error: too many pending rcvs at node %d. \n",index);
	  exit(1);
	}
      }
    else
      add_links(ev,pending_bcasts[i],index);
  }
}

/* Set dag links correctly to signify a 'initiate'-- if the process's
*  enroll has not been found yet, just add the new message to the
*  initiate list. 
*  Note that set_standard_ptrs should have added destination pid to the
*  initiate node.
*/

check_initiate(ev,index,pending_inits,pending_enrolls)
struct event *ev;
int index;
int *pending_inits;
int *pending_enrolls;
{
  int i=0;

  while ((pending_enrolls[i] != NONE) && 
	 !((strcmp(ev[pending_enrolls[i]].event_comp,ev[index].dest_comp)==0)
	   && (ev[pending_enrolls[i]].event_pid==ev[index].dest_pid)))
    i++;

  if (pending_enrolls[i]==NONE) {
    if (add_to_list(pending_inits,index,MAXPROC)!=0) {
      printf("Error: too many pending enrolls at node %d. \n",index);
      exit(1);
    }
  }
  else {
    add_links(ev,index,pending_enrolls[i]);
    pending_enrolls[i]=NONE;
    compress_list(pending_enrolls,MAXPROC);
  }
}


/* Set dag links correctly to signify a 'enroll'-- if the process's
*  enroll has not been found yet, just add the new message to the
*  initiate list.
*  Note that set_standard_ptrs should have added component pid to the
*  initiate node.
*/

check_enroll(ev,index,pending_inits,pending_enrolls)
struct event *ev;
int index;
int *pending_inits;
int *pending_enrolls;
{
  int i=0;

  while ((pending_inits[i] != NONE) && 
	 !((strcmp(ev[pending_inits[i]].dest_comp,ev[index].event_comp)==0)
	   && (ev[pending_inits[i]].dest_pid==ev[index].event_pid)))
    i++;

  if (pending_inits[i]==NONE) {
    if (add_to_list(pending_enrolls,index,MAXPROC)!=0) {
      printf("Error: too many pending enrolls at node %d. \n",index);
      exit(1);
    }
  }
  else {
    add_links(ev,pending_inits[i],index);
    pending_inits[i]=NONE;
    compress_list(pending_inits,MAXPROC);
  }
}




/* Given an array of events, construct the dag:
*  Returns the total number of events involved.
*  Note that most of the arrays have one extra element at the
*  end to store a NONE sentinel.
*/

int make_dag(ev)
struct event *ev;
{
  int i,nproc,nevent;
  int last_event[MAXPROC+1];
  int pending_sends[MAXPEND+1],pending_bcasts[MAXPEND+1],pending_rcvs[MAXPEND+1];
  int pending_inits[MAXPROC+1],pending_enrolls[MAXPROC+1];

  for (i=0;i<=MAXPROC;i++) {
    last_event[i]=NONE;
    pending_inits[i]=NONE;
    pending_enrolls[i]=NONE;
  }  
  for (i=0;i<=MAXPEND;i++) {
    pending_sends[i]=NONE;
    pending_rcvs[i]=NONE;
    pending_bcasts[i]=NONE;
  }

  i=0;

  while (ev[i].event_type != NONE) {
    set_standard_ptrs(ev,i,last_event);
    i++;
  }

  i=0;

  while (ev[i].event_type != NONE) {
  
    switch (ev[i].event_type) {
      
    case XAB_snd: case XAB_vsnd:
      if (ev[i].dest_pid != (-1))
	check_send(ev,i,pending_sends,pending_rcvs);
      else
	check_broadcast(ev,i,pending_bcasts,pending_rcvs);
      break;
    case XAB_rcv_done: case XAB_vrcv_done:
      check_rcv(ev,i,pending_sends,pending_bcasts,pending_rcvs);
      break;
    case XAB_initiate:
      check_initiate(ev,i,pending_inits,pending_enrolls);
      break;
    case XAB_enroll:
      check_enroll(ev,i,pending_inits,pending_enrolls);
      break;
    default:
      break;
    }

    i++;
  }

  nevent=i;

  i=0;
  nproc=0;
  while(last_event[i] != (-1)) {
    add_none_succ(ev,last_event[i++]);
    nproc++;
  }

  return(nevent);
}

/* print info about an event: 
*/

print_event(ev,index)
struct event *ev;
int index;
{
  if (index==NONE) 
    printf("* null event * \n");
  else {
    printf("EVENT #%d:\n",index);
    printf("type: %d \n",ev[index].event_type);
    printf("time: %d.%d \n",ev[index].event_sec,ev[index].event_usec);
    printf("process: %s %d \n",ev[index].event_comp,ev[index].event_pid);
  }
}




/* Test routine to manually check for correctness of dag.
*  Useful for debugging.
*/

dag_test(ev)
struct event *ev;
{
  int i=0,j;

  while (i!=(-1)) {
    printf("Enter # of event to check: \n");
    scanf("%d",&i);

    printf("-------------------\n");
    print_event(ev,i);
    printf("---preceded by:\n");
    for (j=0;j<ev[i].numpred;j++)
      print_event(ev,ev[i].pred[j]);
    printf("---succeeded by:\n");
    for (j=0;j<ev[i].numsucc;j++)
      print_event(ev,ev[i].succ[j]);
  }
}













