function [MtFE, CtFE, T0] = femT_assemFE(T0, nref)
% function [MtFE, CtFE, TF] = femT_assemFE(TE, nref)
%
%   Computes the temporal finite element matrices
%   MtFE and CtFE as described in Sec 6.1 of
%
%       R. Andreev
%       Space-time discretization of the heat equation
%       Numerical Algorithms, 2014
%       (see README.txt for precise reference)
%
%   The output 1d mesh TF is 
%   the uint8(nref)-times uniformly refined version of
%   the input 1d mesh TE

%   R. Andreev, 2012.10.22

	nref = uint8(nref); % converts logical to integer explicitly
	assert(isscalar(nref) && (nref >= 0) && isinteger(nref), 'nref should be a nonnegative integer');

	K = length(T0);
	if (nref == 0)
		MtFE = spdiags(diff(T0)' * [1/2 1/2], 0:1, K-1, K);
		CtFE = diff(speye(K));
		return
	end
	
	StE = sparse(interp1(1:K, eye(K), 1:(1/2):K));

%	 % Alternative sparse computation of StE:
%	 K = 2*(K-1) + 1;
%	 StE = spdiags(repmat([1 2 1]/2, [K 1]), -1:1, K, K);
%	 StE = StE(:, 1:2:end);

	[MtFEs, CtFEs, T0] = femT_assemFE((StE * T0')', nref-1);

	MtFE = MtFEs * StE; 
	CtFE = CtFEs * StE;
end
