function [X, W, R] = QR_GauLeg(R, I)
% function [X, W, R] = QR_GauLeg(R, I)
%
%   Gauss-Legendre quadrature for the interval (I(1),I(2))
%   on R nodes X with corresponding weights W
%   
%   If R is specified, but I is not,
%   returns a function handle that accepts I

%   andreevr, ETHZ, 2009.11.11
%                   2012.10.16
%
%   Implements the algorithm based 
%   on eigenvalue decomposition due to
%
%       G.H. Golub and J.A. Welsch, 
%       Calculation of Gauss quadrature rules
%       Math. Comp. 23 (1969), 221-230.

	assert((1 <= R) && (R <= 2^10), 'Expect 1 <= R <= 2^10 for the number of nodes R');

	if (nargin == 1); X = @(J)(QR_GauLeg(R, J)); return; end

	assert(numel(I) == 2, 'Expect input I to be a real vector of the form I = [a, b]');
	I = [I(1) I(2)];

	j = diag(1 ./ sqrt(4 - [1:R-1].^-2), 1); 
	[V, D] = eig(j + j');
	X = mean(I) + diag(D)' * (diff(I)/2);
	W = 2 * abs(V(1,:)).^2 * (diff(I)/2);
end
