function [x,y] = ssquad_mgr(domain_id,nnodes,func_id,quadtype_id,fn_prefix)
% SSQUAD_MGR manages evaluation of the complete quadrature algorithm
% SSQUAD_MGR(domain_id,nnodes,func_id,quadtype_id,fn_prefix) demonstrates 
% convergence of the high dimensional quadrature rules from the article 
% "Quadrature Algorithms for High Dimensional Singular Integrands on 
% Simplices" , NUMERICAL ALGORITHMS, 70 (2015), pp. 847-874.
% The quadrature is computed with the aid of the coordinate transformations 
% realized by the routine SSQUADTRF and is used for numerical integration 
% of the isotropic singularity ||z||^alpha, alpha = -2*d+k+1/pi where 
% d is the space dimension and k is the dimension of the intersection of
% two simplices
%
% Input parameters:
% domain_id   domain identification in the form 10*d+k where d is the space 
%             dimension and k is the dimension of the intersection of two
%             simplices 
% nnodes      number of quadrature nodes per dimension
% func_id     identification of the integrand
% quadtype_id identification of the quadrature rule (see QUADRC for the 
%             description)
% fn_prefix   prefix of the MAT file to save the output
%
% Logs are written in log.dat 
%
% The output is written in the files which names are built of input
% parameters fn_prefix and domain_id

% Alexey Chernov, University of Bonn, Germany (corresponding author)
% Current affiliation: University of Oldenburg, Germany
% Email: alexey.chernov@uni-oldenburg.de
%
% Christoph Schwab, ETH Zurich, Switzerland
% Email: schwab@sam.math.ethz.ch
%
% Tobias von Petersdorff, University of Maryland, College Park, USA
% Email: tvp@math.umd.edu
%
% Last revised July 20, 2013

logfile = fopen('log.txt','a');
fprintf(logfile,...
    '\n\n\nstart: type=[%s], nnodes=[%s], func_id=%d, quadtype_id=%d\n',...
                  num2str(domain_id),num2str(nnodes),func_id,quadtype_id);
fclose(logfile);

fn = strcat(fn_prefix,num2str(domain_id)); % set the name of the output file
for i_nnodes=1:length(nnodes)     
    vertexlist = initvert(domain_id);
    d =         min(size(vertexlist));
    k = 2*d+1 - max(size(vertexlist));
    
    switch func_id % initialize the integrand
        case 1     % radial algebtaic singularity: 
                   % ||z||^alpha, alpha = -2*d+k+1/pi
            HalfSingOrder = (-2*d+k+1/pi)/2;
            F = @(x,y,z) sum(z.^2,2).^HalfSingOrder;
        otherwise
            error('Unknown func_id');
    end
    
    % computing the quadrature
    [Q,nfeval] = eval_ssquad(F,vertexlist,quadtype_id,...
                nnodes(i_nnodes),2*nnodes(i_nnodes),2*HalfSingOrder);
            
    % write the output
    x(i_nnodes)=nfeval; y(i_nnodes)=Q; save(fn,'x','y','-ASCII','-double');
    
    % write the log
    logfile = fopen('log.txt','a');
    fprintf(logfile,...
        'd=%d, k=%2d, nnodes=%2d of %2d: nfeval = %14d, Q = %1.15f \n',...
         d,k,nnodes(i_nnodes),nnodes(end),nfeval,Q);
    fclose(logfile);
end

end

