function [t,wt] = subquadw(j,k,d,t,wt,nr) 
% SUBQUADW evaluation of the basic subquadrature rule with monomial weights
% [t,wt] = SUBQUADW(j,k,d,t,wt) realizes Steps 8, 7, 6 of the algorithm 
% from the article "Quadrature Algorithms for High Dimensional Singular 
% Integrands on Simplices", NUMERICAL ALGORITHMS, 70 (2015), pp. 847-874
% when monomial weights in regular directions are incorporated in 
% the quadrature weights (i.e. for Gauss-Jacobi quadrature in regular 
% directions). 
%
% Input parameters:
% j      index of the basic subquadrature rule
% k      dimension of the intersection of two simplices
% d      spatial dimension
% [t,wt] node(s) and weight(s) of a quadrature rule in [0,1]^{2d}, where
%        the first component t(:,1) corresponds to the singular variable
%
% Output parameters:
% [t,wt] node(s) and weight(s) of the basic subquadrature

% Alexey Chernov, University of Bonn, Germany (corresponding author)
% Current affiliation: University of Oldenburg, Germany
% Email: alexey.chernov@uni-oldenburg.de
%
% Christoph Schwab, ETH Zurich, Switzerland
% Email: schwab@sam.math.ethz.ch
%
% Tobias von Petersdorff, University of Maryland, College Park, USA
% Email: tvp@math.umd.edu
%
% Last revised May 7, 2014

% initialization of the index range of the variables
s1 = 1;    
a  = 2:d-k;
s2 =   d-k+1;
b  =   d-k+2:2*d-2*k;
p  =         2*d-2*k+1:2*d-2*k+j;    
s3 =                   2*d-2*k+j+1;	
q  =                   2*d-2*k+j+2:2*d-k;
utNj=                               2*d-k+1:2*d-k+j;
ut  =                               2*d-k+1:2*d;    

% generation of the Gauss-Jacobi quadrature with integer order weights 
% in regular directions and realization of Step 8 of the algorithm
wt = wt.*t(:,1).^(2*d-k-1);
if(k==0)       
    [r2,w2] = gjquad(0,d-1,nr);
    [t,wt] = tensquad(t,wt,r2,w2);
    wt = [wt; wt];   
    t(:,2) = t(:,1).*t(:,2);      t = [t; t(:,[2 1])];        
    clear r2 w2;
elseif(k<d)   
    [r2,w2] = gjquad(0,d-k-1,nr);
    [r3,w3] = gjquad(0,  k-1,nr);
    [t,wt] = tensquad(t,wt,r2,w2);    [r,wr] = tensquad(t,wt,r3,w3); 
    wr = [wr; wr];
    r(:,2) = r(:,1).*r(:,2);            r(:,3) = r(:,1).*r(:,3);
    r = [r; r(:,[2 1 3])];      
    
    [t,wt] = tensquad(t,wt,r2,w2);
    t(:,2) = t(:,1).*t(:,2);            t(:,3) = t(:,1).*t(:,3);
    
    wt = [wr; wt];
    t = [r; t(:,[2 3 1])];      
        
    clear r2 w2 r3 w3 r wr;    
end
    
if(d-k-1>0) % handling a,b
    [r,wr]=gjquadrs(d-k-1,nr);    
    [t,wt]=tensquad(t,wt,r,wr);   [t,wt]=tensquad(t,wt,r,wr);
    clear r wr;
    % reordering the variables for further convenience
    if(k==0)
        t(:,[a(1),s2]) = t(:,[s2,a(1)]); 
    elseif(k<d)
        if(length(a)==1)
            t(:,[2,3,4,5]) = t(:,[4,2,5,3]);
        else
            t(:,[2,3,d-k+1,d-k+2]) = t(:,[d-k+1,d-k+2,2,3]);
            t(:,[d-k+2,2*d-2*k+1]) = t(:,[2*d-2*k+1,d-k+2]); 
        end
    end
end
if(j>0)     % handling p
    [r,wr]=gjquadrs(j,nr);
    [t,wt]=tensquad(t,wt,r,wr);
    clear r wr;
    t(:,[p(1) s3]) = t(:,[s3 p(1)]);     
end


if(k-j-1>0) % handling q
    [r,wr]=gjquadrs(k-j-1,nr);
    [t,wt]=tensquad(t,wt,r,wr);
    clear r wr;
end  
if(k>0)     % handling ut
    [r,wr]=gjquadrs(k,nr);
    [t,wt]=tensquad(t,wt,r,wr); 
    clear r wr;       
end

% step 7
if(k<d-1)  % vanishing a-b coordinates for k=d-1, k=d     
    % a-s1 coordinates
    sigma = 1 - sum(t(:,a),2);
    t(:,a) = bsxfun(@times,t(:,s1),t(:,a));
    t(:,s1) = sigma.*t(:,s1);
    % b-s2 coordinates
    sigma = 1 - sum(t(:,b),2);
    t(:,b) = bsxfun(@times,t(:,s2),t(:,b));
    t(:,s2) = sigma.*t(:,s2);   
end

if(k>0)  % vanishing p-q-s3 coordinates for k=0
    % p-q-s3 coordinates

    if(j>0)
        t(:,p) = -bsxfun(@times,t(:,p),t(:,s3));       
    end
    if(j<k-1) 
        sigma   = 1 - sum(t(:,q),2);
        t(:,q) = bsxfun(@times,t(:,s3),t(:,q));
        t(:,s3) = sigma.*t(:,s3);
    end
% step 6: tilde u coordinates  
    sigma = 1 - sum(t(:,[s3 q]),2);
    wt    = wt.*(sigma.^k); 
    
    t(:,ut) = bsxfun(@times,t(:,ut),sigma);
    if(j>0)
        t(:,utNj)    = t(:,utNj) - t(:,p);
    end
end
