function [t,wt] = subquad(j,k,d,t,wt)  
% SUBQUAD evaluation of the basic subquadrature rule (steps 8,7,6)
% [t,wt] = SUBQUAD(j,k,d,t,wt) realizes Steps 8, 7, 6 of the algorithm 
% from the article "Quadrature Algorithms for High Dimensional Singular 
% Integrands on Simplices", NUMERICAL ALGORITHMS, 70 (2015), pp. 847-874 
% when monomial weights in regular directions are not incorporated in 
% the quadrature weights (as e.g. for Gauss-Legendre quadrature in regular 
% directions). Depending on k and d SUBQUAD calls CPSCUBE in the right mode
% to realize Step 8 of the algorithm. SUBQUAD calls C2S
%
% Input parameters:
% j      index of the basic subquadrature rule
% k      dimension of the intersection of two simplices
% d      spatial dimension
% [t,wt] node(s) and weight(s) of a quadrature rule in [0,1]^{2d}, where
%        the first component t(:,1) corresponds to the singular variable
%
% Output parameters:
% [t,wt] node(s) and weight(s) of the basic subquadrature

% Alexey Chernov, University of Bonn, Germany (corresponding author)
% Current affiliation: University of Oldenburg, Germany
% Email: alexey.chernov@uni-oldenburg.de
%
% Christoph Schwab, ETH Zurich, Switzerland
% Email: schwab@sam.math.ethz.ch
%
% Tobias von Petersdorff, University of Maryland, College Park, USA
% Email: tvp@math.umd.edu
%
% Last revised May 7, 2014

% initialization of the index range of the variables
s1 = 1;    
a  = 2:d-k;
s2 =   d-k+1;
b  =   d-k+2:2*d-2*k;
p  =         2*d-2*k+1:2*d-2*k+j;    
s3 =                   2*d-2*k+j+1;	
q  =                   2*d-2*k+j+2:2*d-k;
utNj=                               2*d-k+1:2*d-k+j;
ut  =                               2*d-k+1:2*d;    

% realization of step 8
if(k==0)       
    [t,wt] = cpscube([s1,s2],t,wt); 
elseif(k<d)
    [t,wt] = cpscube([s1,s2,s3],t,wt);    
elseif(k==d)
    t(:,[1,j+1]) = t(:,[j+1,1]);
end

% realization of step 7
if(k<d-1)  % vanishing a-b coordinates for k=d-1, k=d     
    % a-s1 coordinates
    [t(:,a),wt] = c2s(t(:,a),wt);
    wt = wt.*(t(:,s1).^(d-k-1));
    sigma = 1 - sum(t(:,a),2);
    t(:,a) = bsxfun(@times,t(:,s1),t(:,a));
    t(:,s1) = sigma.*t(:,s1);
    % b-s2 coordinates
    [t(:,b),wt] = c2s(t(:,b),wt);
    wt = wt.*(t(:,s2).^(d-k-1));
    sigma = 1 - sum(t(:,b),2);
    t(:,b) = bsxfun(@times,t(:,s2),t(:,b));
    t(:,s2) = sigma.*t(:,s2);   
end

if(k>0)  % vanishing p-q-s3 coordinates for k=0
    % p-q-s3 coordinates

    wt = wt.*(t(:,s3).^(k-1));
    if(j>0)
        [t(:,p),wt] = c2s(t(:,p),wt);
        t(:,p) = -bsxfun(@times,t(:,p),t(:,s3));       
    end
    if(j<k-1) 
        [t(:,q),wt] = c2s(t(:,q),wt);
        sigma   = 1 - sum(t(:,q),2);
        t(:,q) = bsxfun(@times,t(:,s3),t(:,q));
        t(:,s3) = sigma.*t(:,s3);
    end
% realization of step 6: tilde u coordinates  
    [t(:,ut),wt] = c2s(t(:,ut),wt);  
    sigma = 1 - sum(t(:,[s3 q]),2);
    wt    = wt.*(sigma.^k); 
    
    t(:,ut) = bsxfun(@times,t(:,ut),sigma);
    if(j>0)
        t(:,utNj)    = t(:,utNj) - t(:,p);
    end
end
