function [x,y,z,w]=ssquadtrfw(k,d,t,wt,A1,v10,A2,v20,nr)
% SSQUADTRFW coordinate transformation for sing. quadrature with Jacobi Weight (steps 5,4,3)
% on Simplices
% [x,y,z,w]=SSQUADTRFW(k,d,t,wt,A1,v10,A2,v20,nr) realizes Steps 5,4 and 3 
% of the algorithm from the article "Quadrature Algorithms for High 
% Dimensional Singular Integrands on Simplices", NUMERICAL ALGORITHMS, 70 (2015), 
% pp. 847-874 when monomial weights in regular directions are incorporated 
% in quadrature weights (i.e. for Gauss-Jacobi quadrature in regular directions)
%
% Calls SUBQUAD realizing Steps 8,7,6 
% calls QUAD2RFS realizing Step 2, 
% calls QUAD2PHS realizing Step 1.
% Variables z are necessary to avoid subtractive cancellation
%
% Input parameters:
% k        dimension of the intersection of two simplices
% d        spatial dimension
% [t,wt]   node(s) and weight(s) of a quadrature rule on [0,1]^{2*d}
% A1,v10   parameters of the affine transformation of the first reference
%          simplex to the physical coordinates
% A2,v20   parameters of the affine transformation of the second reference
%          simplex to the physical coordinates
% nr       number of quadrature points in regular directions
%
% Output parameters:
% [x,y,z]  node(s) of the transformed quadrature rule
% w        weight(s) of the transformed quadrature rule

% Alexey Chernov, University of Bonn, Germany (corresponding author)
% Current affiliation: University of Oldenburg, Germany
% Email: alexey.chernov@uni-oldenburg.de
%
% Christoph Schwab, ETH Zurich, Switzerland
% Email: schwab@sam.math.ethz.ch
%
% Tobias von Petersdorff, University of Maryland, College Park, USA
% Email: tvp@math.umd.edu
%
% Last revised May 7, 2014

zh = [];
if(k<=0)
    if(k==-1)
        [q,wq] = gjquadrs(d,nr);
        [t,w] = tensquad(q,wq,q,wq);
    else
        [t,w] = subquadw(0,0,d,t,wt,nr);
    end
    u = t(:,1:d);   v = t(:,d+1:2*d);
else
    u = []; v = []; w = []; u_= []; v_= [];	w_= [];	zh_= [];
    for j=0:k-1
        % compute a basic subquadrature for a particular j
        [s,ws] = subquadw(j,k,d,t,wt,nr);
        P = 1:2*d;
    
        while(~isempty(P))
            ucheck = P(1:d-k);         ztilde = P(2*(d-k)+1:2*d-k);     
            vcheck = P(d-k+1:2*(d-k)); utilde = P(2*d-k+1:2*d);
            % permutations
            [u_,v_,zh_,w_] = ...
                quad2rfs(k,d,[s(:,utilde),            s(:,ucheck)],...
                             [s(:,utilde)+s(:,ztilde),s(:,vcheck)],...
                              s(:,ztilde),ws);
            u = [u; u_]; v = [v; v_]; zh = [zh; zh_]; w = [w; w_];
            % reflections
            [u_,v_,zh_,w_] = ...
                quad2rfs(k,d,[s(:,utilde)+s(:,ztilde),s(:,ucheck)],...
                             [s(:,utilde),            s(:,vcheck)],...
                             -s(:,ztilde),ws);
            u = [u; u_]; v = [v; v_]; zh = [zh; zh_]; w = [w; w_];
            % next permutation       
            P = nextperm(j,k,d,P);
        end
    end
end

[x,y,z,w] = quad2phs(k,d,u,v,zh,w,A1,v10,A2,v20);
