function [t1,w1,t2,w2] = quadrc(k,d,quadtype_id,nr,ns,GJweight)
% QUADRC setting up a quadrature rule on the reference (unit) cube
% [t1,w1,t2,w2] = quadrc(k,d,quadtype_id,nr,ns,GJweight) initializes 
% various types of quadrature rules on the reference unit cube [0,1]^{2*d}
%
% input parameters:
% k           dimension of the intersection of two simplices
% d           spatial dimension
% quadtype_id identification of the quadrature rule
% nr          number of quadrature points in regular directions
% ns          number of points in singular directions
% GJweight    the order of the singularity (needed if Gauss-Jacobi rule is
%             used in the singular direction)

% Alexey Chernov, University of Bonn, Germany (corresponding author)
% Current affiliation: University of Oldenburg, Germany
% Email: alexey.chernov@uni-oldenburg.de
%
% Christoph Schwab, ETH Zurich, Switzerland
% Email: schwab@sam.math.ethz.ch
%
% Tobias von Petersdorff, University of Maryland, College Park, USA
% Email: tvp@math.umd.edu
%
% Last revised May 7, 2014

sep = 3; % separation: 1:sep-1 and sep:2*d

% distant simplices, use pure GL
if(quadtype_id==1  && k==-1), quadtype_id=3;  end   
% distant simplices, use pure GJ
if(quadtype_id==31 && k==-1), quadtype_id=34;  end  

switch quadtype_id
    case 1  % CGL - TPGL    
        [r,wr]= glquad(nr); t2=r; w2 = wr;
        for j=sep+1:2*d,    [t2,w2] = tensquad(t2,w2,r,wr);   end
        [t1,w1] = cglquad(ns);
        for j=2:sep-1,      [t1,w1] = tensquad(t1,w1,r,wr);   end
    case 2  %  GJ - TPGL
        [r,wr]= glquad(nr); t2=r; w2 = wr;
        for j=sep+1:2*d,    [t2,w2] = tensquad(t2,w2,r,wr);   end
        [t1,w1] = gjquad(0,GJweight,nr);  w1 = w1./t1.^GJweight;
        for j=2:sep-1,      [t1,w1] = tensquad(t1,w1,r,wr);   end
    case 3  %  pure GL
        [r,wr]= glquad(nr); t2=r; w2 = wr;
        for j=sep+1:2*d,    [t2,w2] = tensquad(t2,w2,r,wr);   end
        t1=r; w1 = wr;
        for j=2:sep-1,      [t1,w1] = tensquad(t1,w1,r,wr);   end
    case 31 % CGL - GJ, sep = 2
        t2 = []; w2 = [];
        [t1,w1] = cglquad(ns);
    case 32 %  GJ - GJ, sep = 2
        t2 = []; w2 = [];
        [t1,w1] = gjquad(0,GJweight,nr);  w1 = w1./t1.^GJweight;
    case 33 %  GL - GJ   , sep = 2 
        t2 = []; w2 = [];
        [t1,w1] = glquad(nr); 
    case 34 %  pure GJ   , sep = 1; t1, w1 are dummy
        t2 = []; w2 = [];
        t1 = 1;  w1 = 1; 
end
