function [u,v,zh,w] = quad2rfs(k,d,u,v,zh,w)
% QUAD2RFS transformation of a quadrature rule to the reference simplex (step 2)
% [u,v,zh,w] = QUAD2RFS(k,d,u,v,zh,w) realizes Step 2 of the algorithm 
% from the article "Quadrature Algorithms for High Dimensional Singular 
% Integrands on Simplices", NUMERICAL ALGORITHMS, 70 (2015), pp. 847-874.
% Keeping zh is necessary to avoid subtractive cancellation
%
% Input parameters:
% k        dimension of the intersection of two simplices
% d        spatial dimension
% [u,v,zh] node(s) of the input quadrature rule
% w        weight(s) of the input quadrature rule
%
% Output parameters:
% [u,v,zh] node(s) of the transformed quadrature rule
% w        weight(s) of the transformed quadrature rule

% Alexey Chernov, University of Bonn, Germany (corresponding author)
% Current affiliation: University of Oldenburg, Germany
% Email: alexey.chernov@uni-oldenburg.de
%
% Christoph Schwab, ETH Zurich, Switzerland
% Email: schwab@sam.math.ethz.ch
%
% Tobias von Petersdorff, University of Maryland, College Park, USA
% Email: tvp@math.umd.edu
%
% Last revised May 7, 2014

tilde = 1:k;
check = max(k,0)+1:d;

udelta = sum(u(:,check),2);
vdelta = sum(v(:,check),2);
    
zh = bsxfun(@times,zh,1-vdelta) + bsxfun(@times,u(:,tilde),udelta-vdelta);
    
udelta = 1 - udelta;
vdelta = 1 - vdelta;

u(:,tilde) = bsxfun(@times,u(:,tilde),udelta);
v(:,tilde) = bsxfun(@times,v(:,tilde),vdelta);
w = w .* (udelta.*vdelta).^k;
