function [u,v,z,w] =quad2phs(k,d,u,v,zh,w,A1,v10,A2,v20)
% QUAD2PHS transformation of a quadrature rule to the physical simplex (step 1)
% [u,v,zh,w] = QUAD2PHS(k,d,u,v,zh,w) realizes Step 1 of the algorithm 
% from the article "Quadrature Algorithms for High Dimensional Singular 
% Integrands on Simplices", NUMERICAL ALGORITHMS, 70 (2015), pp. 847-874.
% Keeping z is necessary to avoid subtractive cancellation
%
% Input parameters:
% k        dimension of the intersection of two simplices
% d        spatial dimension
% [u,v,zh] node(s) of the input quadrature rule
% w        weight(s) of the input quadrature rule
% A1,v10   parameters of the affine transformation of the first reference
%          simplex to the physical coordinates
% A2,v20   parameters of the affine transformation of the second reference
%          simplex to the physical coordinates
%
% Output parameters:
% [u,v,z]  node(s) of the transformed quadrature rule
% w        weight(s) of the transformed quadrature rule

% Alexey Chernov, University of Bonn, Germany (corresponding author)
% Current affiliation: University of Oldenburg, Germany
% Email: alexey.chernov@uni-oldenburg.de
%
% Christoph Schwab, ETH Zurich, Switzerland
% Email: schwab@sam.math.ethz.ch
%
% Tobias von Petersdorff, University of Maryland, College Park, USA
% Email: tvp@math.umd.edu
%
% Last revised May 7, 2014

hat = 1:k;
check = max(k,0)+1:d;

if(k<=0)
    z =                 v(:,check)*A2(check,:) - u(:,check)*A1(check,:);
elseif(k<d)
    z = zh*A1(hat,:) + v(:,check)*A2(check,:) - u(:,check)*A1(check,:);
elseif(k==d)
    z = zh*A1(hat,:);
end
if(k==-1)
    z = bsxfun(@plus,z,v20-v10);
end
u = bsxfun(@plus,u*A1,v10);
v = bsxfun(@plus,v*A2,v20);
       
w = w*abs(det(A1)*det(A2));

 
