function U = nextperm(j,k,d,U)
% NEXTPERM get "Next Permutation" for iteration over integration subdomains 
% U = NEXTPERM(j,k,d,U) depending on the input parameters, the routine 
% generates next to U permutation in the lexicographic order. 
% See section 3.5 of the article "Quadrature Algorithms for High 
% Dimensional Singular Integrands on Simplices", NUMERICAL ALGORITHMS, 
% 70 (2015), pp. 847-874.
%
% Input parameters:
% j      index of the permutation
% k      dimension of the intersection of two simplices
% d      spatial dimension
% U      current permutation
%
% Output parameters:
% U      next permutation
%
% Example: sequence of all permutations k=3 and d=4 and j = 0,1,2:
%
% j=0 : U=[1 2 3 4 5 6 7 8]
% j=1 : U=[1 2 3 4 5 6 7 8] 
%       U=[1 2 4 3 5 7 6 8]
%       U=[1 2 4 5 3 7 8 6]
% j=2 : U=[1 2 3 4 5 6 7 8]
%       U=[1 2 3 5 4 6 8 7]
%       U=[1 2 5 3 4 8 6 7]

% Alexey Chernov, University of Bonn, Germany (corresponding author)
% Current affiliation: University of Oldenburg, Germany
% Email: alexey.chernov@uni-oldenburg.de
%
% Christoph Schwab, ETH Zurich, Switzerland
% Email: schwab@sam.math.ethz.ch
%
% Tobias von Petersdorff, University of Maryland, College Park, USA
% Email: tvp@math.umd.edu
%
% Last revised May 7, 2014

P = 1:2*d;
P(U) = 1:2*d;  % back permutation
    
Nj = 2*(d-k) + (1:j);
U = P(Nj) - 2*(d-k);
% compute the subpermutation U
flag = 0;
for i=j:-1:1
    if(U(i)<k-j+i)
        U(i) = U(i)+1;
        U(i+1:j) = U(i:j-1)+1;
        flag = 1;
        break;
    end
end
if flag == 0
    U = []; 
else % compute full permutation P
    P = 1:k;
    P(U(1:j)) = 0;
    [r,c,w] = find(P);
    P(1:2*(d-k)) = 1:2*(d-k);
    P(2*(d-k)+1:2*d-k) = 2*(d-k) + [U(1:j) w];
    P(2*d-k+1:2*d) = 2*d-k + [U(1:j) w];
    
    U = 1:2*d;
    U(P) = 1:2*d;  % back permutation
end
