function [x,w] = gjquad(alpha,beta,N)
% GJQUAD one-dimensional Gauss-Jacobi quadrature rule
% [x,w]=GJQUAD(alpha,beta,N) computes Gauss-Jacobi quadrature rule on [0,1] 
% by solving a symmetric tridiagonal eigenvalue problem 
% (Hesthaven/Warburton'2008, Appendix A)
%
% Input parameters:
% n     number of quadrature nodes and weights
% 
% Output parameters:
% [x,w] nodes and weights of the Gauss-Legendre quadrature rule

% Alexey Chernov, University of Bonn, Germany (corresponding author)
% Current affiliation: University of Oldenburg, Germany
% Email: alexey.chernov@uni-oldenburg.de
%
% Christoph Schwab, ETH Zurich, Switzerland
% Email: schwab@sam.math.ethz.ch
%
% Tobias von Petersdorff, University of Maryland, College Park, USA
% Email: tvp@math.umd.edu
%
% Last revised May 7, 2014

N = N-1;
if (N==0), x(1)=(alpha-beta)/(alpha+beta+2); w(1) = 2; return; end;

% Compute J
J = zeros(N+1);
h1 = 2*(0:N)+alpha+beta;
J = diag(-1/2*(alpha^2-beta^2)./(h1+2)./h1) + ...
    diag(2./(h1(1:N)+2).*sqrt((1:N).*((1:N)+alpha+beta).*...
    ((1:N)+alpha).*((1:N)+beta)./(h1(1:N)+1)./(h1(1:N)+3)),1);
if (abs(alpha+beta)<10*eps), J(1,1)=0.0;end;
J = J + J';

% Compute quadrature nodes and weights 
[V,D] = eig(J); x = diag(D);
w = (V(1,:)').^2/(alpha+beta+1)*gamma(alpha+1)*...
    gamma(beta+1)/gamma(alpha+beta+1);

x = (x+1)./2;
