function [Q,nfeval] = eval_ssquad(F,vertexlist,quadtype_id,nr,ns,SingOrder)
% EVAL_SSQUAD evaluation of the quadrature rule for singularities on simplices
% [Q,nfeval] = EVAL_SSQUAD(F,vertexlist,quadtype_id,nr,ns,SingOrder)
% manages computation and evaluation of the quadrature rule for
% singularities. 
%
% Input parameters:
% F           handle used in calling the singular integrand directly
% vertexlist  list of vertices of two simplices in physical coordinates
% quadtype_id identification of the quadrature rule (see QUADRC for the 
%             description)
% nr          number of quadrature points in regular directions
% ns          number of points in singular directions
% SingOrder   the order of the singularity after transformation 
%             (needed if Gauss-Jacobi rule is used in the singular direction)
%
% Output parameters:
% Q           value of the quadrature
% nfeval      total number of quadrature nodes used

% Alexey Chernov, University of Bonn, Germany (corresponding author)
% Current affiliation: University of Oldenburg, Germany
% Email: alexey.chernov@uni-oldenburg.de
%
% Christoph Schwab, ETH Zurich, Switzerland
% Email: schwab@sam.math.ethz.ch
%
% Tobias von Petersdorff, University of Maryland, College Park, USA
% Email: tvp@math.umd.edu
%
% Last revised February 2, 2015


% compute space dimension d and dimension of the intersection k
d =         min(size(vertexlist));
k = 2*d+1 - max(size(vertexlist));
% compute parameters of the affine transformation to the physical coordinates
idx2 = [1:k+1 d+2:2*d-k+1];
v10 = vertexlist(     1 ,:);
v20 = vertexlist(idx2(1),:);
A1(1:d,:) = bsxfun(@minus,vertexlist(     2:d+1 ,:),v10);
A2(1:d,:) = bsxfun(@minus,vertexlist(idx2(2:d+1),:),v20);

% compute quadrature rule on [0,1]^(2*d)
[t1,w1,t2,w2] = quadrc(k,d,quadtype_id,nr,ns,SingOrder+2*d-k-1);


Q = 0;  
nfeval = 0;
n = length(w1);
for i=1:n
    if(quadtype_id < 10) % Gauss-Legendre quadrature in regular coordinates
        [t,wt] = tensquad(t1(i,:),w1(i),t2,w2);
        [x,y,z,w]=ssquadtrf(k,d,t,wt,A1,v10,A2,v20);
    else                 % Gauss-Jacobi quadrature in regular coordinates
        [x,y,z,w]=ssquadtrfw(k,d,t1(i,:),w1(i),A1,v10,A2,v20,nr);        
    end
    Q = Q + dot(F(x,y,z),w);
    nfeval = nfeval + length(w);
end
    


