function [t,wt] = cpscube(active,t,wt)
% CPSCUBE collapsing coordinate transformation for the unit cube (step 8)
% [t,wt] = CPSCUBE(active,t,wt) realizes Step 8 of the algorithm
% from the article "Quadrature Algorithms for High Dimensional Singular 
% Integrands on Simplices", NUMERICAL ALGORITHMS, 70 (2015), pp. 847-874
% when monomial weights in regular directions are not incorporated in 
% the quadrature weights (as e.g. for Gauss-Legendre quadrature in regular 
% directions). 
%
% Input parameters:
% active contains indices of the variables participating in the collapsing
%        coordinate transformation (s1, s2, possibly s3) in the quadrature 
%        matrix t; active(1) = 1 is the index of the composite 
%        Gauss-Legendre quadrature
% [t,wt] node(s) and weight(s) of a quadrature rule in [0,1]^{2d}, where
%        the first component t(:,1) corresponds to the singular variable
%
% Output parameters:
% [t,wt] node(s) and weight(s) of the transformed quadrature

% Alexey Chernov, University of Bonn, Germany (corresponding author)
% Current affiliation: University of Oldenburg, Germany
% Email: alexey.chernov@uni-oldenburg.de
%
% Christoph Schwab, ETH Zurich, Switzerland
% Email: schwab@sam.math.ethz.ch
%
% Tobias von Petersdorff, University of Maryland, College Park, USA
% Email: tvp@math.umd.edu
%
% Last revised May 7, 2014

d = length(active);
t(:,active(2:d)) = bsxfun(@times,t(:,active(1)),t(:,active(2:d)));

wt = wt.*(t(:,active(1)).^(d-1));
lt = size(t,1);
for k=2:d
    u = 1:size(t,2); u([active(1) active(k)]) = u([active(k) active(1)]);
    t((k-1)*lt+1:k*lt,:) =  t(1:lt,u);
   wt((k-1)*lt+1:k*lt)   = wt(1:lt);
end
