%VARMFDM - Main function for the MFDM analysis of the b.v.problem in
%variational weak formulation
%
%   F = varMFDM(N,X,Y,BC,T,nt,G,teta) returns the vector F (N x 1), which is the 
%   numerical nodal (primary) solution of the variationally formulated b.v.p. 
%   (torsion of the prismatic bar). It is based upon the N nodes 
%   discretization, with coordinates X,Y, with boundary codes BC (0-internal 
%   node,1-boundary node). G is a shear (Kirchhoff) modulus wherea teta is a 
%   torsional angle (external load). Numerical integration is performed on
%   the triangular mesh with nt triangles, with vertices (nodes numbers)
%   stored in matrix T (nt x 3)
%
%   Example:
%       X = [0 0 0 0 1 1 1 1 2 2 2 2 3 3 3 3]';
%       Y = [0 1 2 3 0 1 2 3 0 1 2 3 0 1 2 3]';
%       BC = [1 1 1 1 1 0 0 1 1 0 0 1 1 1 1 1];
%       T = delaunay(X,Y,{'Qt','Qbb','Qc','Qz'});
%       nt = size(T,1);
%       N = 16;
%       G = 1;
%       teta = pi/6;
%       F = varMFDM(N,X,Y,BC,T,nt,G,teta)
%       
%   see also star, mwls, localMFDM, postprocessingMFDM

%Copyright by Sawomir Milewski, Cracow University of Technology,
%slawek@L5.pk.edu.pl

function F = varMFDM(N,X,Y,BC,T,nt,G,teta)
A = zeros(N,N); % global coefficient matrix
B = zeros(N,1); % global right hand side vector
m = 9; % number of nodes in a MFD star
if N<m
    m = N;
end
nG = 3; % number of Gauss points in triangle
Gw = [1/3 1/3 1/3]; % integration weights
for k=1:nt % loop over triangles
    w = T(k,:)'; % nodes numbers in triangle vertices
    Xt = X(w);
    Yt = Y(w);
    % location of Gauss points in triangles    
    Gpx = [0.5*(X(w(1))+X(w(2))) 0.5*(X(w(2))+X(w(3))) 0.5*(X(w(1))+X(w(3)))];
    Gpy = [0.5*(Y(w(1))+Y(w(2))) 0.5*(Y(w(2))+Y(w(3))) 0.5*(Y(w(1))+Y(w(3)))];
    wek = [Xt(1) - Xt(2), Yt(1) - Yt(2); Xt(2) - Xt(3), Yt(2) - Yt(3);...
        Xt(3)-Xt(1), Yt(3)-Yt(1)];
    dl = [sqrt(wek(1,1)^2 + wek(1,2)^2), sqrt(wek(2,1)^2 + wek(2,2)^2),...
        sqrt(wek(3,1)^2 + wek(3,2)^2)];
    p = 0.5*sum(dl);
    % Jacobian of transformation matrix - area of triangle
    J = sqrt(p*(p-dl(1))*(p-dl(2))*(p-dl(3))); 
    for p=1:nG % loop over Gauss points in triangle
        xg = Gpx(p); % x-coordinate of the Gauss point
        yg = Gpy(p); % y-coordinate of the Gauss point
        S = star(N,xg,yg,X,Y,m); % MFD star generation    
         % MFD formulas generation (MWLS) for trial function (m star nodes, p=1)
        MF = mwls(xg,yg,X,Y,S,m,1);
         % MFD formulas generation (MWLS) for test function (3 star nodes, p=1)
        Mv = mwls(xg,yg,X,Y,w,3,1);  
        % aggregation of the global coefficient matrix   
        A(w(1),S(1:m,1)) = A(w(1),S(1:m,1)) - ...
            J*Gw(p)*(MF(2,:)*Mv(2,1) + MF(3,:)*Mv(3,1));
        A(w(2),S(1:m,1)) = A(w(2),S(1:m,1)) - ...
            J*Gw(p)*(MF(2,:)*Mv(2,2) + MF(3,:)*Mv(3,2));
        A(w(3),S(1:m,1)) = A(w(3),S(1:m,1)) - ...
            J*Gw(p)*(MF(2,:)*Mv(2,3) + MF(3,:)*Mv(3,3));         
        % aggregation of the global right hand side vector     
        B(w) = B(w) - 2*G*teta*J*Gw(p)*Mv(1,:)';    
    end
end

% imposition of the Dirichlet boundary conditions
I = eye(N,N);
Id = diag(BC);
Ip = I - Id;
A = Ip*A*Ip + Id;
B = Ip*B;

F = A\B; % solution of the MFD equations