%postprocessingMFDM - Main function for postprocessing of the MFDM results 
%of the shear stresses analysis
%
%   [StrN,StrT,Res] = postprocessingMFDM(N,X,Y,T,nt,F,G,teta) returns three
%   matices, namely 
%       StrN (N x 3) - array of stresses at nodes: componet tzx, 
%           component tzy and total value tz
%       StrT (nt x 3) - array of mean stresses (forces) over triangles: 
%           componet tzx, component tzy and total value tz
%       Res (nt x 1) - vector of residual error
%   Function is based upon:
%      N - number of nodes
%      X (N x 1) - x-coordinates of nodes
%      Y (N x 1) - y-coordinates of nodes
%      T (nt x 3)- list of triangles
%      nt - number of triangles
%      F (N x 1) - nodal values of Prandtl function (primary MFD solution)
%      G - shear modulus
%      teta - torsional angle (load)
%
%   Example:
%       X = [0 0 0 0 1 1 1 1 2 2 2 2 3 3 3 3]';
%       Y = [0 1 2 3 0 1 2 3 0 1 2 3 0 1 2 3]';
%       BC = [1 1 1 1 1 0 0 1 1 0 0 1 1 1 1 1];
%       T = delaunay(X,Y,{'Qt','Qbb','Qc','Qz'});
%       nt = size(T,1);
%       N = 16;
%       G = 1;
%       teta = pi/6;
%       F = localMFDM(N,X,Y,BC,G,teta);
%       disp('Recovered nodal and integral results for the local b.v.p.');
%       [StrN,StrT,Res] = postprocessingMFDM(N,X,Y,T,nt,F,G,teta)
%       F = varMFDM(N,X,Y,BC,T,nt,G,teta);
%       disp('Recovered nodal and integral results for the variational
%       b.v.p.');
%       [StrN,StrT,Res] = postprocessingMFDM(N,X,Y,T,nt,F,G,teta)
%       
%   see also star, mwls, localMFDM, varMFDM

%Copyright by Sawomir Milewski, Cracow University of Technology,
%slawek@L5.pk.edu.pl

function [StrN,StrT,Res] = postprocessingMFDM(N,X,Y,T,nt,F,G,teta)
m = 16;
if m>N
    m = N;
end
StrN = zeros(N,3); % array of stresses at nodes: componet tzx, component tzy and total value tz
for i=1:N % loop over nodes
    S = star(N,X(i),Y(i),X,Y,m);
    M = mwls(X(i),Y(i),X,Y,S,m,1,0.1);      
    Fx = M(2,:)*F(S(1:m,1));
    Fy = M(3,:)*F(S(1:m,1));
    StrN(i,:) = [Fy; -Fx; sqrt(Fx^2 + Fy^2)];
end
nG = 3; % number of Gauss points in triangle
Gw = [1/3 1/3 1/3]; % integration weights
StrT = zeros(nt,3); % array of stresses at triangles: componet tzx, component tzy and total value tz
Res = zeros(nt,1); % vector of residual error
for k=1:nt % loop over triangles
    w = T(k,:)'; % nodes numbers in triangle vertices
    Xt = X(w);
    Yt = Y(w);   
    % location of Gauss points in triangles    
    Gpx = [0.5*(X(w(1)) + X(w(2))) 0.5*(X(w(2)) + X(w(3))) 0.5*(X(w(1)) + X(w(3)))];
    Gpy = [0.5*(Y(w(1)) + Y(w(2))) 0.5*(Y(w(2)) + Y(w(3))) 0.5*(Y(w(1)) + Y(w(3)))];
    wek = [Xt(1) - Xt(2), Yt(1) - Yt(2); Xt(2) - Xt(3), Yt(2) - Yt(3);...
        Xt(3)-Xt(1), Yt(3)-Yt(1)];
    dl = [sqrt(wek(1,1)^2 + wek(1,2)^2), sqrt(wek(2,1)^2 + wek(2,2)^2),...
        sqrt(wek(3,1)^2 + wek(3,2)^2)];
    p = 0.5*sum(dl);
    % Jacobian of transformation matrix - area of triangle
    J = sqrt(p*(p-dl(1))*(p-dl(2))*(p-dl(3))); 
    tzx = 0;   
    tzy = 0;   
    res = 0;
    for p=1:nG % loop over Gauss points in triangle
        xg = Gpx(p);      
        yg = Gpy(p); 
        S = star(N,xg,yg,X,Y,m); % MFD star generation 
         % MFD formulas generation (MWLS) for trial function (m star nodes, p=2)
        M = mwls(xg,yg,X,Y,S,m,2,0.1);
        F_der = M*F(S(1:m,1)); % set of derivatives values (F,Fx,Fy,Fxx,Fxy,Fyy)
        tzx = tzx + J*Gw(p)*F_der(3);       
        tzy = tzy - J*Gw(p)*F_der(2);
        res = res + abs(F_der(4) + F_der(6) + 2*G*teta);
    end
    StrT(k,1) = tzx;    
    StrT(k,2) = tzy;    
    StrT(k,3) = sqrt(tzx^2 + tzy^2);
    Res(k) = res;
end