%MWLS - MFD formulas generation by means of the local MWLS
%approximation 
%
%   [M] = mwls(x,y,X,Y,S,m) returns the MFD coefficients matrix 
%   M ((p+1)(p+2)/2 x m), based upon the local approximation of the second 
%   order spanned over the m nodes (X,Y) at point (x,y) with the singular 
%   weights.   
%   x,y - coordinates of the considered point
%   X (N x 1) - x-coordinates of nodes
%   Y (N x 1) - y-coordinates of nodes
%   S (m x 2) - set of m nodes numbers of the MFD star and their distances
%   from point (x,y)
%   m - number of nodes in the MFD star
%   Matrix M contains the MFD coefficients for the subsequent derivatives
%   from 0th (function value) up to and including full p-th order (p-th order
%   derivatives) evaluated at point (x,y), based upon the m nodal values
%
%   [M] = mwls(x,y,X,Y,S,m,p) allows for determination of the local
%   approximation rank (p=0,1,2), default value is p=2
%
%   [M] = mwls(x,y,X,Y,S,m,p,g) allows for determination of the local
%   smoothing parameter (g=0 - singular weights, without smoothing, g>0 -
%   nonsingular weight, with smoothing), default value is g=0
%
%   Example:
%       X = [0 0 0 0 1 1 1 1 2 2 2 2 3 3 3 3]';
%       Y = [0 1 2 3 0 1 2 3 0 1 2 3 0 1 2 3]';
%       F = X.^2 + Y.^2;
%       x = 1.5;
%       y = 1.5;
%       N = 16;
%       m = 12;
%       S = star(N,x,y,X,Y,m);
%       M = mwls(x,y,X,Y,S,m)
%       disp('numerical values: F,Fx,Fy,Fxx,Fxy,Fyy')
%       M*F(S(:,1))
%       disp('analytical values: F,Fx,Fy,Fxx,Fxy,Fyy')
%       [x^2+y^2; 2*x; 2*y; 2; 0; 2]
%
%   see also star, localMFDM, varMFDM, postprocessingMFDM

%Copyright by Sawomir Milewski, Cracow University of Technology,
%slawek@L5.pk.edu.pl

function [M] = mwls(x,y,X,Y,S,m,p,g) 
if nargin<8
    g = 0;
end
if nargin<7
    p = 2;
end
s = 6; 
if p==1
    s = 3;
end

h = X(S(1:m,1))'-x; % x-distance to the star central point
k = Y(S(1:m,1))'-y; % y-distance to the star central point
d = sqrt(h.^2 + k.^2); % total distance to the star central point
 % 2D interpolates up to and including second order
z = [ones(1,m); h; k; 0.5*h.^2; h.*k; 0.5*k.^2]';
P = z(:,1:s);
% diagonal weights matrix with singular (g=0) or non-singular (g>0) weights
W = diag(1./((d.^2 + (g^4)./(d.^2 + g^2 + 10^-15)).^(p+1) + 10^-15)); 
M = (P'*W*P)^-1*P'*W; % matrix of the MFD formulas