%LOCALMFDM - Main function for the MFDM analysis of the b.v.problem in local
%formulation
%
%   F = localMFDM(N,X,Y,BC,G,teta) returns the vector F (N x 1), which is the 
%   numerical nodal (primary) solution of the locally formulated b.v.p. 
%   (torsion of the prismatic bar). It is based upon the N nodes 
%   discretization, with coordinates X,Y, with boundary codes BC (0-internal 
%   node,1-boundary node). G is a shear (Kirchhoff) modulus wherea teta is a 
%   torsional angle (external load)
%
%   Example:
%       X = [0 0 0 0 1 1 1 1 2 2 2 2 3 3 3 3]';
%       Y = [0 1 2 3 0 1 2 3 0 1 2 3 0 1 2 3]';
%       BC = [1 1 1 1 1 0 0 1 1 0 0 1 1 1 1 1];
%       N = 16;
%       G = 1;
%       teta = pi/6;
%       F = localMFDM(N,X,Y,BC,G,teta)
%       
%   see also star, mwls, varMFDM, postprocessingMFDM

%Copyright by Sawomir Milewski, Cracow University of Technology,
%slawek@L5.pk.edu.pl

function F = localMFDM(N,X,Y,BC,G,teta)
A = zeros(N,N); % global coefficient matrix
B = zeros(N,1); % global right hand side vector
m = 9; % number of nodes in a MFD star
if N<m
    m = N;
end
for i=1:N % loop over nodes
    if BC(i)==0 % collocation for internal nodes
        S = star(N,X(i),Y(i),X,Y,m); % MFD star generation
        M = mwls(X(i),Y(i),X,Y,S,m); % MFD formulas generation (MWLS)
        % MFD equations generation - left hand side
        A(i,S(1:m,1)) = M(4,:) + M(6,:); 
        B(i) = -2*G*teta;  % MFD equations generation - right hand side
    else
        % collocation for boundary nodes (Dirichlet homogeneous condition)
        A(i,i) = 1; 
    end
end
F = A\B; %solution of the MFD equations

