
%TEST - Script created for testing purposes of the Matlab functions:
% star, mwls, localMFDM, varMFDM, postprocessingMFDM
%
%   Problem description: determination of the shear stresses in
%       - in a prismatic bar of rectangular cross section (task 1)
%       - in a railroad rail domain (task 2)
%
%   Program parameters:
%       task - task type (1 - rectangular bar with data generated here, 
%           2 - railroad rail with data written 
%           in the "railroadrailDATA.txt"), default is task=1
%       formulation - b.v.p. formulation (1 - local, 2 - variational 
%           Galerkin), default is formulation=1
%       teta - value of the torsional angle (load), default is teta=pi/6
%       G - value of the Kirchhoff modulus (material parameter), default is
%          G=1
%       a,b - bar dimensions, defaults are a=b=2
%       nx - number of nodes in the x-direction (3,4,....), default is
%          nx=15
%       ny - number of nodes in the y-direction (3,4,....), default is
%          ny=15
%
%   Example
%       Type TEST in Command Window to run the program with default 
%       parameters and to display the appropriate results (in both text 
%       and graphical forms)
%       Edit TEST in Matlab Editor, change variables
%       teta,G,task,formulation,a,b,nx,ny and run the program
%
%   see also star, mwls, localMFDM, varMFDM, postprocessingMFDM

%Copyright by Sawomir Milewski, Cracow University of Technology,
%slawek@L5.pk.edu.pl

teta = pi/6;
G = 1;

task = 1;
formulation = 1;

a = 2;
b = 2;
nx = 15;
ny = 15;

close all
clc

if task==1 % generation of the regular mesh of nodes in the rectangle
    hx = a/(nx-1); % mesh modulus in the x-direction
    hy = b/(ny-1); % mesh modulus in the y-direction
    N = nx*ny; % total number of nodes
    [X,Y] = meshgrid(0:hx:a,0:hy:b); % regular mesh generator
    X = X(:); % vector of x-coordinates of nodes
    Y = Y(:); % vector of y-coordinates of nodes
    BC = zeros(N,1); % codes of nodes (0 - internal node)
    BC([1:nx, N-nx:N, 1:nx:end, nx:nx:end]) = 1; % codes of nodes (1 - boundary node)
    T = delaunay(X,Y,{'Qt','Qbb','Qc','Qz'}); % mesh of the Delaunay triangles
    nt = size(T,1); % total number of triangles
end

if task==2 % here, data for the railroad rail domain (irregular cloud of nodes) 
    % are read from the disc file
    data_file = fopen('railroadrailDATA.txt','rt');
    N = fscanf(data_file,'%d',[1 1]);
    Z = fscanf(data_file,'%f',[3*N 1]);
    X = Z(1:3:end);
    Y = Z(2:3:end);
    BC = Z(3:3:end);
    nt = fscanf(data_file,'%d',[1 1]);
    Z = fscanf(data_file,'%f',[3*nt 1]);
    T = [Z(1:3:end)'; Z(2:3:end)'; Z(3:3:end)']';
    fclose(data_file);
end

if formulation == 1 
    %here, calculations for the local b.v.p. formulation are performed
    disp('PRIMARY NODAL SOLUTION OF THE LOCALLY FORMULATED B.V.P.');
    F = localMFDM(N,X,Y,BC,G,teta) % solution of the local b.v. problem
    disp('RECOVERED NODAL SOLUTION OF THE LOCALLY FORMULATED B.V.P.');
    [StrN,StrT,Res] = postprocessingMFDM(N,X,Y,T,nt,F,G,teta) 
    % postprocessing of the results obtained from the local b.v. problem
    form = 'LOCAL FORMULATION';
end
if formulation == 2 %here, calculations for the variational Galerkin 
    %b.v.p.formulation are performed
    disp('PRIMARY NODAL SOLUTION OF THE VARIATIONAL B.V.P.');
    F = varMFDM(N,X,Y,BC,T,nt,G,teta) % solution of the variational b.v. problem
    disp('RECOVERED NODAL SOLUTION OF THE VARIATIONAL B.V.P.');
    [StrN,StrT,Res] = postprocessingMFDM(N,X,Y,T,nt,F,G,teta) 
    % postprocessing of the results obtained from the variational b.v. problem
    form = 'VARIATIONAL FORMULATION';
end

%exemplary graphical presentation of the nodal results obtained from the
%variational b.v. problem

fig = figure(1);
set(fig,'Position',get(0,'ScreenSize'));

x = X(T(:,:))';
y = Y(T(:,:))';
z1 = StrN(:,1);
z2 = StrN(:,2);
z3 = StrN(:,3);
z = [F(T(:,:)) ; z1(T(:,:)) ; z2(T(:,:)) ; z3(T(:,:))]';
for k=1:4    % loop over figures
    subplot(2,2,k);
    hold on
    grid on
    
    p = patch(x,y,z(:,(k-1)*nt+1:k*nt),z(:,(k-1)*nt+1:k*nt));
    set(p,'Edgecolor','interp');
    
    view(3);
    if task==1
        axis([min(X) max(X) min(Y) max(Y) min(min(z(:,(k-1)*nt+1:k*nt))) ...
            max(max(z(:,(k-1)*nt+1:k*nt)))]);
    end
    if task==2
        axis equal;
    end
    s = 'PRANDTL FUNCTION';
    if k==2
        s = 'SHEAR STRESS COMPONENT \tau_z_x';
    end
    if k==3
        s = 'SHEAR STRESS COMPONENT \tau_z_y';
    end
    if k==4
        s = 'TOTAL SHEAR STRESS \tau_z';
    end
    t = title(s);
    set(t,'String',{'MFDM',form,['SOLUTION: ' s],['max = ' ...
        num2str(max(max(abs(z(:,(k-1)*nt+1:k*nt)))))]});
    set(t,'FontSize',12,'FontWeight','bold');
end
