function NPoutput(rec,histout,data)
%NPOUTPUT   Print numerical results and visualize images.
%
%   NPOUTPUT(REC,HISTOUT,DATA) prints the numerical results of the method 
%   and visualize the restored image REC, the exact image OBJ, the data 
%   image GN and the error image -abs(REC-OBJ). 
%   HISTOUT is an array storing numerical performance of the method used
%   for computing REC. For the NCGP method:
%       HISTOUT = [ERR F_OBJ G_PROJ TOT_FFT CG_IT]
%   where ERR is a column vector with the relative error values at the
%   NCGP iterates; F_OBJ is a column vector with the objective 
%   function values; G_PROJ is a column vector with the projected gradient 
%   norm values; TOT_FFT is a column vector with the number of performed 
%   ffts; CG_IT is the number of performed CG iterations.
%   For the NP and QNP methods:
%       HISTOUT = [ERR F_OBJ G_PROJ TOT_FFT].
%   DATA is a structure array with the problem data OBJ and GN.

obj = data.obj; % exact image
gn = data.gn; % blurred and noisy image

[k,m] = size(histout);
rel_err = histout(:,1);
f_obj = histout(:,2);
g_proj = histout(:,3);
Tot_FFT = (histout(:,4));

if m>4
    cg_it = (histout(:,5));
end

if m == 4
   fprintf('\n\n');
   fprintf('rel. error  \t  objective \t proj. grad. \t   fft \t    k \n');
   for i=1:k
        fprintf('%5.4e \t %5.4e \t %5.4e \t %5d \t  %3d \n',...
            rel_err(i),f_obj(i),g_proj(i),Tot_FFT(i),i);
   end
else
   fprintf('\n\n');
   fprintf('rel. error  \t  objective \t proj. grad. \t');
   fprintf('fft \t    CG \t   k \n');
   for i=1:k
        fprintf('%5.4e \t %5.4e \t %5.4e \t %5d \t %5d \t %3d \n',...
            rel_err(i),f_obj(i),g_proj(i),Tot_FFT(i),cg_it(i),i);
   end
end

% measures of the reconstructin quality
MSE = mserror(obj,rec); % Mean Squared Error
PSNR = psnratio(obj,rec,max(max(obj))); % Peak Signal-to-Noise Ratio 
MAE = maerror(obj,rec); % Mean Average Error
if exist('ssim','file');
   [mssim] = ssim(obj,rec); % Structural Similarity Index
   fprintf('\n');
   fprintf('SSIM     \t     MSE     \t     PSNR     \t     MAE        \n');
   fprintf('%e \t %e \t %e \t %e  \n',mssim,MSE,PSNR,MAE);
else
   fprintf('\n');
   fprintf('MSE     \t     PSNR     \t     MAE        \n');
   fprintf('%e \t %e \t %e  \n',MSE,PSNR,MAE);
end

% Plots
figure; plot(rel_err); title('Relative error');
figure; plot(f_obj); title('Objective function');

% Dispaly images
figure; imagesc(rec); colormap(gray); 
set(gca,'XTick',[],'YTick',[]); title('restored image');
figure; imagesc(gn); colormap(gray); 
set(gca,'XTick',[],'YTick',[]); title('data image');
figure; imagesc(obj); colormap(gray); 
set(gca,'XTick',[],'YTick',[]); title('exact image');
figure; imagesc(-abs(rec-obj)); colormap(gray); 
set(gca,'XTick',[],'YTick',[]); colorbar; title('error image');





 
 


