function [A,b,x] = deriv2_alt(n,example)
%DERIV2 Test problem: computation of the second derivative.
%
% [A,b,x] = deriv2_alt(n,example)
%
% This is a mildly ill-posed Fredholm integral equation of the
% first kind. It is discretized by a Nystrom method based on 
% the trapezoidal rule. The kernel K is the Green's function for
% the second derivative:
%
%    K(s,t) =  s(t-1),  s <  t,
%    K(s,t) =  t(s-1),  s >= t.
%
% Both s and t live in the interval [0,1]. The right-hand side 
% function g and the corresponding solution f can be choose to be:
%
%    example = 1: g(s) = (s^3 - s)/6,         f(t) = t.
%
%    example = 2: g(s) = exp(s) + (1-e)s - 1, f(t) = exp(t).
%
%    example = 3: g(s) = (4s^3 - 3s)/24,              s <  0.5,
%                 g(s) = (-4s^3 + 12s^2 - 9s + 1)/24, s >= 0.5,
%
%                 f(t) = t,    t <  0.5,
%                 f(t) = 1-t,  t >= 0.5.
 
% Reference:  L. M. Delves and  J. L. Mohamed, "Computational Methods
% for Integral Equations", Cambridge % University Press, 1985; p. 310. 

% Initialization.
if (nargin==1), example = 1; end
h = 1/(n-1); A = zeros(n,n); h2 = h^2;

% Compute the matrix A.
for i = 1:n
  A(i,i) = h2*(i^2 - (n+1)*i + n);
  for j = (i+1):n
    A(i,j) = h2*(i-1)*(j-n);
    A(j,i) = A(i,j);
  end
end
for i = 1:n
    A(i,1) = A(i,1)/2;
    A(i,n) = A(i,n)/2;
end

% Compute the right-hand side vector b.
if (nargout>1)
    b = zeros(n,1);
    if (example==1)
        for i=1:n
            s_i = (i-1)*h;
            b(i) = (s_i^3 - s_i)/6;
        end
    elseif (example==2)
        for i=1:n
            s_i = (i-1)*h;
            b(i) = exp(s_i) + (1-exp(1))*s_i - 1;
        end
    else
        error('Illegal value of example')
    end
end

% Compute the solution vector x.
if (nargout==3)
    x = zeros(n,1);
    if (example==1)
        for i=1:n
            s_i = (i-1)*h;
            x(i) = s_i;
        end
    elseif (example==2)
        for i=1:n
            s_i = (i-1)*h;
            x(i) = exp(s_i);
        end
    else
        error('Illegal value of example')
    end
end

