function irlbablkdemo(Action)
%  Interactive command window for the irlbablk routine. 
%  This command window can be used to run all of the examples
%  in the paper "Restarted Block Lanczos Bidiagonalization
%  Methods" by J. Baglama and L. Reichel
%  Numerical Algorithms 43 (2006) 251-272
%  The program irlbablk can be run by itself from the 
%  MATLAB command line.
%
% See also IRLBABLK.

%  AUTHORS:
%  James Baglama     
%  Mathematics Department 
%  University of Rhode Island
%  Kingston, RI 02881
%  E-mail: jbaglama@math.uri.edu
%  
%  Lothar Reichel    
%  Mathematics Department
%  Kent State University
%  Kent, OH 44242 
%  E-mail: reichel@mcs.kent.edu

% REFERENCES:
% 1.) J. Baglama and L. Reichel, "Augmented Implicitly Restarted Lanczos 
%     Bidiagonalization Methods", SIAM J. Sci. Comput., 27 (2005), pp. 19-42. 
% 2.) J. Baglama and L. Reichel, "Restarted Block Lanczos Bidiagonalization
%     Methods", Numerical Algorithms, 43 (2006) pp. 251-272.

% Date: 10/27/2006

% Global variables.
global methodtouse noget guiresidual FLAG A OPTS matrixprod waithan dnbls;

% Demo is being initialized
if nargin==0

    % Check for existence of programs.    
    if exist('irlba','file') ~= 2 || exist('irlba','file') ~= 2 
       error('ERROR: The functions IRLBA and IRLBABLK are required for the DEMO to run.'); 
    end
    
    Action='initialize'; 
end

% Determine which action to take.
switch Action
   
  case 'initialize' % Initialize values.
       OPTS.sigma='LS'; OPTS.k=6; OPTS.blsz=3; OPTS.m=10; 
       OPTS.maxit=1000; OPTS.adjust=3; OPTS.tol = 1d-6; 
       OPTS.aug=[]; OPTS.reorth='ONE';
       matrixprod=[]; dnbls = OPTS.m; noget=1;
       methodtouse = 'Yblock';
       
       % Set default matrix to be the matrix in EXAMPLE 1.
       n = 200; A = sparse(diag(1:n)); A(2,2) = 1+1/n^4;
   
       % Call function to set up user interface. 
       graphwin
       
  case 'method_blk'  % Set the block method to be used.
       set(findobj(gcf,'Tag','method_blk'),'Value',1);
       set(findobj(gcf,'Tag','method_nonblk'),'Value',0);
       set(findobj(gcf,'Tag','BLSZ'),'Value',3);
       set(findobj(gcf,'Tag','BLSZ'),'String',' 1 | 2 | 3 | 4 | 5 | 6 ');
       set(findobj(gcf,'Tag','BLSZ'),'UserData',[1 2 3 4 5 6]);
       OPTS.blsz = 3; methodtouse='Yblock';
       
  case 'method_nonblk'  % Set the block method to be used. 
       set(findobj(gcf,'Tag','method_blk'),'Value',0);
       set(findobj(gcf,'Tag','method_nonblk'),'Value',1);
       set(findobj(gcf,'Tag','BLSZ'),'Value',1);
       set(findobj(gcf,'Tag','BLSZ'),'String',' 1 ');
       set(findobj(gcf,'Tag','BLSZ'),'UserData',1);
       OPTS.blsz = 1; methodtouse='Nblock';
       
  case 'examples' % Get the example to be used.
       exA = get(findobj(gcf,'Tag','Aexamples'),'Value');
       noget = 1;
       
       switch exA
           case 1
               n = 200; A = sparse(diag(1:n)); A(2,2) = 1+1/n^4;
           case 2
               A=delsq(numgrid('S',20));
           case 3
               x=zeros(130,1); w=4.5; x(1) = 1;
               for i=2:8
                 x(i) = (w/(i-1))^2*(sin((i-1)/w))^2;
               end
               A=sparse(toeplitz(x,x));
           case 4  
               mmfile = fopen('hypatia','r');
               if ( mmfile == -1 )
                  noget = 0;
                  % Clear plot and set axis.
                  plot(0.5,0.5,'Color',[0.8 0.8 0.8]);
                  set(gca,'XTick',[],'YTick',[],'Visible','off');
                  axis normal;
                  % Output Error message.
                  Str1 = 'Error! File hypatia not found!';
                  Str2 = '  ';
                  Str3 = ' File can be found on Netlib or';
                  Str4 = ' http://www.math.uri.edu/~jbaglama';
                  D=str2mat(Str1,Str2,Str3,Str4);
                  text(0,0.5,D,'Fontsize',8,'Color','r'); 
               else

                   % Output a wait bar. 
                   waithan = waitbar(0,'Please wait loading matrix HYPATIA...');
  
                   % Read comment line then foget it.
                   commentline = fgets(mmfile);
                   while length(commentline) > 0 && commentline(1) == '%',
                         commentline = fgets(mmfile);
                   end
                   
                   % Read size information
                   [sizeinfo,count] = sscanf(commentline,'%d%d%d');
                   while ( count == 0 )
                        commentline =  fgets(mmfile);
                        [sizeinfo,count] = sscanf(commentline,'%d%d%d');
                   end
                   rows = sizeinfo(1); cols = sizeinfo(2); 
                   entries = sizeinfo(3);
                   
                   % Read in matrix hypatia.
                   [T,count] = fscanf(mmfile,'%f',3);
                   T = [T; fscanf(mmfile,'%f')];
                   T = reshape(T,3,entries)';
                   A = sparse(T(:,1), T(:,2), T(:,3), rows , cols);
                   
                   % Close wait bar.
                   close(waithan);
               end
               
              
       end      
           
  case 'examplematrix' % Radio button for examples.
       noget = 1;
       set(findobj(gcf,'Tag','examplematrix'),'Value',1);
       set(findobj(gcf,'Tag','matrix'),'Value',0);
       set(findobj(gcf,'Tag','matrixUser'),'String',' ');
       set(findobj(gcf,'Tag','Aexamples'),'String',str2mat(... 
       ' A = diag[1, 1+1/200^4, 3, ..., 200] ',...
       ' A = delsq(numgrid(''S'',20)) ',...
       ' A = 130 x 130 Toeplitz matrix     ',...
       ' A = 11,390 x 1,265 term-by-document'));
      set(findobj(gcf,'Tag','userA'),'Visible','off');
      n = 200; A = sparse(diag(1:n)); A(2,2) = 1+1/n^4;

  case 'matrix' % Radio button for user example.
       noget = 1;
       set(findobj(gcf,'Tag','examplematrix'),'Value',0);
       set(findobj(gcf,'Tag','matrix'),'Value',1);
       set(findobj(gcf,'Tag','matrixUser'),'String','A=sparse(diag(1:100));');
       set(findobj(gcf,'Tag','Aexamples'),'String',' ');
       set(findobj(gcf,'Tag','Aexamples'),'Value',1);
       set(findobj(gcf,'Tag','userA'),'Visible','on');
        A = sparse(diag(1:100));

       
  case 'matrixUser' % Get user input matrix A. No testing of input.
       umat = get(findobj(gcf,'Tag','matrixUser'),'String');
       if ~isempty(umat),eval(umat); end 

  case 'ADJUST' % Set option ADJUST to be values ranging from 1 to 10.
       data = get(findobj(gcf,'Tag','ADJUST'),'UserData');
       OPTS.adjust = data(get(findobj(gcf,'Tag','ADJUST'),'Value'));

  case 'AUG' % Set option AUG to be either 'HARM' or 'RITZ'.
       data = get(findobj(gcf,'Tag','AUG'),'UserData');
       OPTS.aug = data(get(findobj(gcf,'Tag','AUG'),'Value'),:);
 
  case 'BLSZ' % Set option BLSZ to be a value between 1 and 6.
       data = get(findobj(gcf,'Tag','BLSZ'),'UserData');
       OPTS.blsz = data(get(findobj(gcf,'Tag','BLSZ'),'Value'));
    
  case 'K' % Set option K to be a value between 1 and 10.
       data = get(findobj(gcf,'Tag','K'),'UserData');
       OPTS.k = data(get(findobj(gcf,'Tag','K'),'Value'));
         
  case 'MAXIT' % Set option MAXIT to be values ranging from 100 to 6000.
       data = get(findobj(gcf,'Tag','MAXIT'),'UserData');
       OPTS.maxit = data(get(findobj(gcf,'Tag','MAXIT'),'Value'));
     
  case 'M' % Set option M to be a value between 3 and 20.
       data = get(findobj(gcf,'Tag','M'),'UserData');
       OPTS.m = data(get(findobj(gcf,'Tag','M'),'Value'));
   
  case 'REORTH' % Set option REORTH to be 1 or 2.
       data = get(findobj(gcf,'Tag','REORTH'),'UserData');
       OPTS.reorth = data(get(findobj(gcf,'Tag','REORTH'),'Value'),:); 
 
  case 'SIGMA' % Set option SIGMA to be 'SS' or 'LS'.   
       data = get(findobj(gcf,'Tag','SIGMA'),'UserData');
       OPTS.sigma = data(get(findobj(gcf,'Tag','SIGMA'),'Value'),:);
       if strcmp(OPTS.sigma,'SS')
          set(findobj(0,'Tag','AUG'),'Value',1);
          OPTS.aug ='HARM'; 
       end   
       
  case 'TOL' % Set option TOL to be values ranging from 1d-3 to 1d-16.
       data = get(findobj(gcf,'Tag','TOL'),'UserData');
       OPTS.tol = data(get(findobj(gcf,'Tag','TOL'),'Value'));
       
  case 'GetSVDs' % Call IRLBABLK to compute the  singular values.
      
       if noget
           % Clear plot and set axis.
          plot(0.5,0.5,'Color',[0.8 0.8 0.8]);
          set(gca,'XTick',[],'YTick',[],'Visible','off');
          axis normal;
        
          % Output a wait bar. 
           waithan = waitbar(0,'Please wait...');

          % Save the orginal number of blocks M.
          dnbls = OPTS.m;
      
          % Call IRLBABLK and compute the cputime.
          if strcmp(methodtouse,'Yblock')
             tic; D=irlbablk(A,OPTS); tim = toc;
          else
             OPTS.m_b = OPTS.m; 
             tic; D=irlba(A,OPTS); tim = toc;
          end   
             
          % Reset the number of blocks if M changed.
          if OPTS.m ~= dnbls
             if dnbls >= 3
                OPTS.m = dnbls;
                set(findobj(0,'Tag','M'),'Value',dnbls);
                if dnbls > 20
                   S = ' 3 | 4 | 5 | 6 | 7 | 8 | 9 | 10 | 11 | 12 | 13 | 14 | 15 | 16 | 17 | 18 | 19 | 20 |  ';
                   DS = num2str(dnbls);
                   S = strcat(S,DS);
                   set(findobj(0,'Tag','M'),'String',S);
                   set(findobj(0,'Tag','M'),'Userdata',...
                   [3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 dnbls]);
                end
             else    
                set(findobj(0,'Tag','M'),'String','ERROR!');
             end
          end   
      
          % Close wait bar.
          close(waithan); 
       
          if FLAG(1)==0
              
             % Set numeric values to strings for output.
             D=num2str(D,10); tim = num2str(tim);

             semilogy(guiresidual);
             xs = size(guiresidual,1);
             xs_3 = (xs-1)/3;
             xlim([1 xs]); 
             xaxis=str2mat('Sum of matrix-vector products with A and A^T',...
               ['       (CPU time: ',tim,' secs.)']);
             xlabel(xaxis,'fontsize',8);
             pos = get(gca,'position');
             set(gca,'position',[0.1 0.4 pos(3) pos(4)]);
             set(gca,'XTick',[1 1+xs_3 1+2*xs_3 1+3*xs_3]);
             matrixprod = matrixprod + 3- rem(matrixprod,3); 
             prod_3 = matrixprod/3;
             set(gca,'XTickLabel',num2str([0, prod_3, 2*prod_3, 3*prod_3]'));    
             h_legend=legend(D);
             set(h_legend,'FontSize',6);
             if strcmp(OPTS.sigma,'SS')
                titleA = str2mat([num2str(OPTS.k),' Smallest singular values of A']);
             else
                 titleA = str2mat([num2str(OPTS.k),' Largest singular values of A']);
             end
             title(titleA,'FontSize',8);
       else
          axis([0 1 0 1]);
          Str1 = '               NO CONVERGENCE!';
          Str2 = ' Try the following suggestions.';
          Str3 = ' 1.) Increase block size BLSZ. ';
          Str4 = ' 2.) Increase the number of vectors M.';
          Str5 = ' 3.) Decrease the tolerance, TOL.';
          Str6 = ' 4.) More iterations may be required. Increase MAXIT.';
          D=str2mat(Str1,Str2,Str3,Str4,Str5,Str6);
          text(0,0.5,D,'Fontsize',8,'Color','r');
          end
       
       end
           
      
  case 'Setup' % Print the initial message at start up.
       % Clear plot.
       plot(0.5,0.5,'Color',[0.8 0.8 0.8]);
       set(gca,'XTick',[],'YTick',[],'Visible','off');
       axis([0 1 0 1]);
       Str1aa=' ';
       Str1a = ' ';
       Str1 = 'Interactive command window for the routines';
       Str1b = 'IRLBABLK and IRLBA.';
       Str2 = ' ';
       Str3 = 'This is a demo to illustrate the examples in the paper';
       Str5 = '"Restarted Block Lanczos Bidiagonalization Methods"';
       Str6 = 'by J. Baglama and L. Reichel.';
       Str7 = 'Select an example or input a matrix, then set the ';
       Str8 = 'options by using the pull down menus on the right.';
       Str9 = 'To get singular values, select the ''Get singular values'' button.';
       Str10 = ' ';
       Str11 = 'For more information on the options move the';
       Str12 = 'mouse over the option and press the right mouse'; 
       Str13= 'button or press the Help button.';
       Str14=' ';
       Str15='Authors: J. Baglama and L. Reichel';
       Str16='Date: 10 - 27 - 2006';
       D=str2mat(Str1aa,Str1a,Str1,Str1b,Str2,Str3,Str5,Str6,Str7,Str8,Str9,Str10,...
                 Str11,Str12,Str13,Str14,Str15,Str16);
       text(0,0.5,D,'Fontsize',8,'Color','r');

      
  case 'HelpIRLBABLK' % Output help window.
       Helpdemo;
      
  case 'ADJUSTinfo' % Output information about ADJUST. 
       % Clear plot.
       plot(0.5,0.5,'Color',[0.8 0.8 0.8]);
       set(gca,'XTick',[],'YTick',[],'Visible','off');
       axis([0 1 0 1]);
       Str1=' ADJUST - Initial number of vectors added';
       Str2='          to the K restart vectors.';
       D=str2mat(Str1,Str2);
       text(0,0.5,D,'Fontsize',10,'Color','r');
  
  case 'AUGinfo' % Output information about AUG. 
       % Clear plot.
       plot(0.5,0.5,'Color',[0.8 0.8 0.8]);
       set(gca,'XTick',[],'YTick',[],'Visible','off');
       axis([0 1 0 1]);
       Str1=' AUG - Vectors used to augment the Krylov space';
       Str2='  ';
       Str3=' HARM - Harmonic Ritz vectors.';
       Str4=' RITZ - Ritz vectors.';
       D=str2mat(Str1,Str2,Str3,Str4);
       text(0,0.5,D,'Fontsize',10,'Color','r');
      
  case 'BLSZinfo' % Output information about BLSZ.
       % Clear plot.
       plot(0.5,0.5,'Color',[0.8 0.8 0.8]);
       set(gca,'XTick',[],'YTick',[],'Visible','off');
       axis([0 1 0 1]);
       Str1=' BLSZ - Block size of the Lanczos ';
       Str2 = '      bidiagonal matrix.';
       D=str2mat(Str1,Str2);
       text(0,0.5,D,'Fontsize',10,'Color','r');

  case 'Kinfo' % Output information about K.
       % Clear plot.
       plot(0.5,0.5,'Color',[0.8 0.8 0.8]);
       set(gca,'XTick',[],'YTick',[],'Visible','off');
       axis([0 1 0 1]);
       Str1=' K - Number of desired singular values.';
       D=str2mat(Str1);
       text(0,0.5,D,'Fontsize',10,'Color','r');

  case 'MAXITinfo' % Output information about MAXIT.
       % Clear plot.
       plot(0.5,0.5,'Color',[0.8 0.8 0.8]);
       set(gca,'XTick',[],'YTick',[],'Visible','off');
       axis([0 1 0 1]);
       Str1=' MAXIT  - Maximum number of iterations/restarts.';
       D=str2mat(Str1);
       text(0,0.5,D,'Fontsize',10,'Color','r');       
       
  case 'NBLSinfo' % Output information about M.
       % Clear plot.
       plot(0.5,0.5,'Color',[0.8 0.8 0.8]);
       set(gca,'XTick',[],'YTick',[],'Visible','off');
       axis([0 1 0 1]);
       Str1=' M - number steps in the block  ';
       Str2 ='        Lanczos bidiagonal method.';
       D=str2mat(Str1,Str2);
       text(0,0.5,D,'Fontsize',10,'Color','r');      
 
  case 'REORTHinfo' % Output information about REORTH.
       % Clear plot.
       plot(0.5,0.5,'Color',[0.8 0.8 0.8]);
       set(gca,'XTick',[],'YTick',[],'Visible','off');
       axis([0 1 0 1]);
       Str1=' REORTH - specifying whether to use one-sided full';
       Str2 ='         reorthogonalization or two-sided';
       Str3 = ' ';
       Str4 = 'ONE - One-sided full reorthogonalization.';
       Str5 = 'TWO - Two-sided full reorthogonalization.';
       D=str2mat(Str1,Str2,Str3,Str4,Str5);
       text(0,0.5,D,'Fontsize',10,'Color','r');   
       
  case 'SIGMAinfo' % Output information about SIGMA. 
       % Clear plot.
       plot(0.5,0.5,'Color',[0.8 0.8 0.8]);
       set(gca,'XTick',[],'YTick',[],'Visible','off');
       axis([0 1 0 1]);
       Str1=' SIGMA - Which singular values.';
       Str2=' ';
       Str3=     '    SS - Smallest singular values.';
       Str4=     '    LS - Largest singular values. ';
       D=str2mat(Str1,Str2,Str3,Str4);
       text(0,0.5,D,'Fontsize',10,'Color','r');
      
  case 'TOLinfo' % Output information about TOL.
       % Clear plot.
       plot(0.5,0.5,'Color',[0.8 0.8 0.8]);
       set(gca,'XTick',[],'YTick',[],'Visible','off');
       axis([0 1 0 1]);
       Str1 =' TOL - Tolerance used for convergence.';
       Str2 ='       Convergence is determined when ';            
       Str3 ='       || A*V - U*S ||_2 <= TOL*||A||_2';
       Str4 ='       ||A||_2 is approximated by largest';
       Str5 ='       singular value.  ';
       D=str2mat(Str1,Str2,Str3,Str4,Str5);
       text(0,0.5,D,'Fontsize',10,'Color','r');
       
end       

%===================================================================
function graphwin

% This function sets up the graphical user interface.

% Set values and figures.
StdUnit = 'point';
StdColor = get(0,'DefaultUIcontrolBackgroundColor');
PointsPerPixel = 72/get(0,'ScreenPixelsPerInch');

% Open a new figure
 a = figure('Color',[0.8 0.8 0.8], ...
   'Position',[25 45 560 420], ...
   'Visible','off', ...
   'NumberTitle','off', ...
   'IntegerHandle','off', ...
   'Number','off',...  
   'Name','"Restarted Block Lanczos Bidiagonalization Methods" by J. Baglama and L. Reichel');

% Add the axes to the new figure a.
b = axes('Parent',a, ...
   'Units',StdUnit, ...
   'Color',[0.8 0.8 0.8], ...
   'Position',PointsPerPixel*[18 179 289 227], ...
   'Units','Normalized',...
   'XColor',[0.8 0.8 0.8], ...
   'XGrid','off', ...
   'XTick',[],...
   'YColor',[0.8 0.8 0.8], ...
   'YGrid','off', ...
   'YTick',[]);

% Set no labels.
set(get(b,'XLabel'),'visible','off');
set(get(b,'YLabel'),'visible','off');

% Add various frames to the bottom and right-hand side of the figure.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'Position',PointsPerPixel*[380 100 170 300], ...
   'Units','Normalized',...
   'Style','frame');

b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'Position',PointsPerPixel*[14 10 292 90], ...
   'Units','Normalized',...
   'Style','frame');


% Add radio buttons and text for specifying a type of matrix.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'Position',PointsPerPixel*[32 80 249 13], ...
   'Units','Normalized',...
   'Style','text',...
   'Tag','SelectA',...
   'String','Select matrix A');
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'Position',PointsPerPixel*[32 15 249 13], ...
   'Units','Normalized',...
   'Style','text',...
   'Tag','userA',...
   'Visible','off', ...
   'FontSize',6,...
   'String','Must press enter after user input matrix A.');
b = uicontrol('Parent',a, ...
   'Style','radiobutton', ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'Position',PointsPerPixel*[32 60 22 20], ...
   'Units','Normalized',...
   'Value',1,...
   'Tag','examplematrix',...
   'CallBack','irlbablkdemo(''examplematrix'')');
b = uicontrol('Parent',a, ...
   'Style','radiobutton', ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'Position',PointsPerPixel*[32 30 22 20], ...
   'Units','Normalized',...
   'Value',0,...
   'Tag','matrix',...
   'CallBack','irlbablkdemo(''matrix'')');
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',[1 1 1], ...
   'Position',PointsPerPixel*[52 30 240 20], ...
   'Units','Normalized',...
   'Style','edit', ...
   'Horiz','left',...
   'Tag','matrixUser',...
   'CallBack','irlbablkdemo(''matrixUser'')'); 
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',[1 1 1], ...
   'Style','popup', ...
   'Visible','on',...
   'Position',PointsPerPixel*[52 60 240 17], ...
   'Units','Normalized',...
   'Tag','Aexamples',...
   'String',str2mat(...
   ' A = diag[1, 1+1/200^4, 3, ..., 200] ',...
   ' A = delsq(numgrid(''S'',20)) ',...
   ' A = 130 x 130 Toeplitz matrix     ',...
   ' A = 11,390 x 1,265 term-by-document'),...
   'CallBack','irlbablkdemo(''examples'')');

% Add radio buttons and text for which method to use.
b = uicontrol('Parent',a, ...
   'Style','radiobutton', ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'Position',PointsPerPixel*[390 105 22 20], ...
   'Units','Normalized',...
   'Value',1,...
   'Tag','method_blk',...
   'CallBack','irlbablkdemo(''method_blk'')');
b = uicontrol('Parent',a, ...
   'Style','radiobutton', ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'Position',PointsPerPixel*[470 105 22 20], ...
   'Units','Normalized',...
   'Value',0,...
   'Tag','method_nonblk',...
   'CallBack','irlbablkdemo(''method_nonblk'')');
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'Position',PointsPerPixel*[405 110 50 13], ...
   'Units','Normalized',...
   'Style','text',...
   'Tag','SelectA',...
   'String','irlbablk');
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'Position',PointsPerPixel*[485 110 30 13], ...
   'Units','Normalized',...
   'Style','text',...
   'Tag','SelectA',...
   'String','irlba');


% Text next to the options field
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'HorizontalAlignment','left', ...
   'Position',PointsPerPixel*[360 400 200 17], ...
   'Units','Normalized',...
   'FontSize',6,...
   'String','Use right mouse button to get info', ...
   'Style','text');
% SIGMA text.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'HorizontalAlignment','left', ...
   'Position',PointsPerPixel*[390 375 127 17], ...
   'Units','Normalized',...
   'String','SIGMA', ...
   'ButtonDownFcn','irlbablkdemo(''SIGMAinfo'')',...
   'Style','text');
% K text.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'HorizontalAlignment','left', ...
   'Position',PointsPerPixel*[390 345 127 17], ...
   'Units','Normalized',...
   'String','K', ...
   'ButtonDownFcn','irlbablkdemo(''Kinfo'')',...
   'Style','text');
% BLSZ text.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'HorizontalAlignment','left', ...
   'Position',PointsPerPixel*[390 315 127 17], ...
   'Units','Normalized',...
   'String','BLSZ', ...
   'ButtonDownFcn','irlbablkdemo(''BLSZinfo'')',...
   'Style','text');
% M text
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'HorizontalAlignment','left', ...
   'Position',PointsPerPixel*[390 285 127 17], ...
   'Units','Normalized',...
   'String','M', ...
   'ButtonDownFcn','irlbablkdemo(''NBLSinfo'')',...
   'Style','text');
% ADJUST text.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'HorizontalAlignment','left', ...
   'Position',PointsPerPixel*[390 255 108 17], ...
   'Units','Normalized',...
   'String','ADJUST', ...
   'ButtonDownFcn','irlbablkdemo(''ADJUSTinfo'')',...
   'Style','text');
% AUG text.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'HorizontalAlignment','left', ...
   'Position',PointsPerPixel*[390 225 109 17], ...
   'Units','Normalized',...
   'String','AUG', ...
   'ButtonDownFcn','irlbablkdemo(''AUGinfo'')',...
   'Style','text');
% REORTH text.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'HorizontalAlignment','left', ...
   'Position',PointsPerPixel*[390 195 109 17], ...
   'Units','Normalized',...
   'String','REORTH', ...
   'ButtonDownFcn','irlbablkdemo(''REORTHinfo'')',...
   'Style','text');
% MAXIT text.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'HorizontalAlignment','left', ...
   'Position',PointsPerPixel*[390 165 109 17], ...
   'Units','Normalized',...
   'String','MAXIT', ...
   'ButtonDownFcn','irlbablkdemo(''MAXITinfo'')',...
   'Style','text');
% TOL text.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',StdColor, ...
   'HorizontalAlignment','left', ...
   'Position',PointsPerPixel*[390 135 109 17], ...
   'Units','Normalized',...
   'String','TOL', ...
   'ButtonDownFcn','irlbablkdemo(''TOLinfo'')',...
   'Style','text');


% Add controls for specifying various options.

% SIGMA controls.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',[1 1 1], ...
   'Style','popup', ...
   'Position',PointsPerPixel*[452 375 81 20], ...
   'Units','Normalized',...
   'Tag','SIGMA',...
   'Value',2,...
   'String',' SS | LS', ...
   'UserData',str2mat('SS', 'LS'),...
   'CallBack','irlbablkdemo(''SIGMA'')');
% K controls.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',[1 1 1], ...
   'Style','popup', ...
   'Position',PointsPerPixel*[452 345 81 20], ...
   'Units','Normalized',...
   'Tag','K',...
   'Value',6,...
   'String',' 1 | 2 | 3 | 4 | 5 | 6 ', ...
   'UserData',[1 2 3 4 5 6], ...
   'CallBack','irlbablkdemo(''K'')');
% BLSZ controls.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',[1 1 1], ...
   'Style','popup', ...
   'Position',PointsPerPixel*[452 315 81 20], ...
   'Units','Normalized',...
   'Tag','BLSZ',...
   'Value',3,...
   'String',' 1 | 2 | 3 | 4 | 5 | 6 ', ...
   'UserData',[1 2 3 4 5 6], ...
   'CallBack','irlbablkdemo(''BLSZ'')');
% M control.   
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'Style','popup', ...
   'BackgroundColor',[1 1 1], ...
   'Position',PointsPerPixel*[452 285 81 20], ...
   'Units','Normalized',...
   'Tag','M',...
   'Value',10,...
   'String',' 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9 | 10 | 11 | 12 | 13 | 14 | 15 | 16 | 17 | 18 | 19 | 20 ', ...
   'UserData',[1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20], ...
   'CallBack','irlbablkdemo(''M'')');   
% ADJUST control.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',[1 1 1], ...
   'Style','popup', ...
   'Position',PointsPerPixel*[452 255 81 20], ...
   'Units','Normalized',...
   'Tag','ADJUST',...
   'Value',3,...
   'String',' 1 | 2 | 3 | 4 | 5 | 6 ', ...
   'UserData',[1 2 3 4 5 6], ...
   'CallBack','irlbablkdemo(''ADJUST'')');
% AUG control.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',[1 1 1], ...
   'Style','popup', ...
   'Position',PointsPerPixel*[452 225 81 20], ...
   'Units','Normalized',...
   'Tag','AUG',...
   'Value',2,...
   'String',' HARM | RITZ ', ...
   'UserData',str2mat('HARM', 'RITZ'),...
   'CallBack','irlbablkdemo(''AUG'')');
% REORTH control
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',[1 1 1], ...
   'Style','popup', ...
   'Position',PointsPerPixel*[452 195 81 20], ...
   'Units','Normalized',...
   'Tag','REORTH',...
   'Value',1,...
   'String',' ONE| TWO ', ...
   'UserData',str2mat('ONE', 'TWO'),...
   'CallBack','irlbablkdemo(''REORTH'')');
% MAXIT control.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',[1 1 1], ...
   'Style','popup', ...
   'Position',PointsPerPixel*[452 165 81 20], ...
   'Units','Normalized',...
   'Tag','MAXIT',...
   'Value',6,...
   'String',' 100 | 200 | 400 | 600 | 800 | 1000 | 2000 | 4000 | 6000 ', ...
   'UserData',[100  200  400  600  800  1000  2000  4000  6000], ...
   'CallBack','irlbablkdemo(''MAXIT'')');
% TOL control.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'BackgroundColor',[1 1 1], ...
   'Style','popup', ...
   'Position',PointsPerPixel*[452 135 81 20], ...
   'Units','Normalized',...
   'Tag','TOL',...
   'Value',4,...
   'String',' 1d-3 | 1d-4 | 1d-5 | 1d-6 | 1d-7 | 1d-8 | 1d-9 | 1d-10 | 1d-12 | 1d-16  ', ...
   'UserData',[1d-3  1d-4  1d-5  1d-6  1d-7  1d-8  1d-9  1d-10  1d-12 1d-16], ...
   'CallBack','irlbablkdemo(''TOL'')');
 
% Add push buttons for Get singular values, Close and Help.
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'Position',PointsPerPixel*[380 65 170 26], ...
   'Units','Normalized',...
   'Tag','GetEigs',...
   'String','Get singular values -->', ...
   'CallBack','irlbablkdemo(''GetSVDs'')');
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'CallBack','irlbablkdemo(''HelpIRLBABLK'')', ...
   'Position',PointsPerPixel*[380 30 80 26], ...
   'Units','Normalized',...
   'String','Help');
b = uicontrol('Parent',a, ...
   'Unit',StdUnit, ...
   'CallBack','close(gcf);', ...
   'Position',PointsPerPixel*[470 30 80 26], ...
   'Units','Normalized',...
   'String','Close');

% Set the figure for initial output of message.
set(a,'visible','on');

% Call irlbablkdemo to print the initial message.
irlbablkdemo('Setup');

% Re-set the figure for callback.
set(a,'visible','on','HandleVisibility','CallBack');

%======================================================       

function Helpdemo

HelpStr={'Restarted Block Lanczos Bidiagonalization Methods Demonstration';
' ';
' IRLBABLK will find a few singular values and singular vectors of A'; 
' such that A*V = U*S and A''*U = V*S, where V is an orthonormal n x k matrix'; 
' of "right" singular vectors, U is an orthonormal p x k matrix of "left" singular'; 
' vectors, and S is a k x k diagonal matrix of singular values. This program'; 
' utilizes the a Augmented Block Lanczos bidiagonalization method to find the';
' singular triples. The program uses only matrix vector products to find the';
' singular triples at either end of the spectrum.'; 
' '
' This demo is to help illustrate the different options that are available ';
' in the IRLBABLK routine. The user must select a matrix either by choosing ';
' a radio button next to a preset matrix or entering a matrix in the text field ';
' provided. The user can then change some or all of the options on the right hand ';
' side of the figure by using the pull down menus. After selecting a matrix and ';
' setting the options use the ''Get singular values'' button to find the singular values.';
' ';
' Pressing the right mouse button when the mouse is over the option will give';
' a short explanation of the option.';
' ';
'  ADJUST - Initial number of vectors added to the K restart vectors. After';
'           vectors start to converge more vectors are added to help'; 
'           increase convergence.';
' ';
'  AUG    - Vectors used to augment the Krylov space. Choices are';
'           either Ritz (RITZ) vectors or harmonic (HARM) Ritz vectors.';
' ';
'  BLSZ   - Block size of the Lanczos bidiagonal matrix.';        
' ';
'  K      - Number of desired singular values.';             
' ';
'  MAXIT  - Maximum number of iterations, i.e. maximum number of restarts.';                           
' ';
'  M   - Number of steps in the block Lanczos bidiagonalization method. The program may increase';
'           M to ensure certain requirements are satisfied. A warning message';
'           will be displayed if M increases.';                          
' ';
'  REORTH - Three letter string specifying whether to use one-sided full reorthogonalization';
'           or two-sided. One-sided is performed only on the "short" vectors.';
'           Two-sided orthogonality is used when cond(A) estimated by cond(B) > 1/sqrt(eps).';
' ';
'  SIGMA  - Two letter string specifying the location of the desired singular values.';            
'           ''LS''  Largest singular values.';
'           ''SS''  Smallest singular values.';                                                                        
'  ';                               
'  TOL    - Tolerance used for convergence. Convergence is determined when';             
'           || A*V - U*S ||_2 <= TOL*||A||_2. Where ||A||_2 is approximated by';                                   
'           the largest singular value of all projection matrices.';  
' ';
'  AUTHORS:';
'  James Baglama     University of Rhode Island  E-mail: jbaglama@math.uri.edu';
'  Lothar Reichel    Kent State University       E-mail: reichel@math.kent.edu';
' ';
' REFERENCES: ';
' 1.) J. Baglama and L. Reichel, "Augmented Implicitly Restarted Lanczos ';
'     Bidiagonalization Methods", SIAM J. Sci. Comput., 27 (2005), pp. 19-42.'; 
' 2.) J. Baglama and L. Reichel, "Restarted Block Lanczos Bidiagonalization';
'     Methods", submitted for publication.';
' ';};
      
helpwin(HelpStr,'IRLBABLKDEMO Help');
         
