function examples(Action)
% This function is used to display the four examples in the paper
% "Restarted Block Lanczos Bidiagonalization Methods", by 
% James Baglama and Lothar Reichel in 
% Numerical Algorithms 43 (2006) 251-272

% Clear guiresidual.
clear guiresidual;

% Global variables.
global guiresidual example3; example3 = 0;
global S_exact exact_error;
global A_web MV_tim count;

% Determine which example to run.
switch Action
    
    % Example 1. A = diag[1, 1+1/200^4, 3, ..., 200].
    % Computing the two smallest singular values.
    case 'example1'

        % Set common parameters for IRLBA and IRLBABLK.
        OPTS.ADJUST = 3; OPTS.K = 2; OPTS.SIGMA ='SS'; OPTS.TOL = 1d-10;
        OPTS.M = 20; OPTS.AUG = 'RITZ'; OPTS.REORTH = 'ONE';

        % Set up the matrix.
        n = 200; A = sparse(diag(1:n)); A(2,2) = 1+1/n^4;

        % Set up rand vectors 
        V = randn(n,2);
     
        % Call IRLBA.
        OPTS.V = V(:,1); S = irlba(A,OPTS); 
        nonblksing = guiresidual; 
        guiresidual=[];
     
        % Call IRLBABLK for m = 10.
        OPTS.BLSZ=2; OPTS.M=10; OPTS.V = V;
        S = irlbablk(A,OPTS);
        blksing10 = guiresidual; guiresidual=[];
     
        % Call IRLBABLK for m = 20.
        OPTS.M=20; S = irlbablk(A,OPTS); 
        blksing20 = guiresidual; guiresidual=[];
        
        % Plot the solutions on a subgraph.
        figure; 
        x1 = size(nonblksing,1); x2 = size(blksing10,1);
        xmax = max(x1,x2); xlim([1 xmax]);
        subplot(2,1,1); 
        semilogy(nonblksing(:,1),'b:'); hold on;
        semilogy(nonblksing(:,2),'r:');
        semilogy(blksing10(:,1),'k'); 
        semilogy(blksing10(:,2),'g');
        h_legend = legend('irlba (Ritz(1))','irlba (Ritz(2))','irlbablk (Ritz(1))','irlbablk (Ritz(2))');
         set(h_legend,'FontSize',6);
         title('A = sparse(diag(1:200)); A(2,2) = 1+1/(200)^4  (irlbablk r=2  m=10) (irlba m=20)','Fontsize',8);
        semilogy(1:xmax,1d-6*ones(xmax),'k');
        xlabel('Number of restarts','Fontsize',8); text(-20,10^(-6),'10^{-6}');
        subplot(2,1,2);
        x3 = size(blksing20,1); xmax = max(x1,x3); xlim([1 xmax]);
        semilogy(nonblksing(:,1),'b:'); hold on; 
        semilogy(nonblksing(:,2),'r:');
        semilogy(blksing20(:,1),'k'); 
        semilogy(blksing20(:,2),'g');
        h_legend = legend('irlba (Ritz(1))','irlba (Ritz(2))','irlbablk (Ritz(1))','irlbablk (Ritz(2))');
        set(h_legend,'FontSize',6);
        title('A = sparse(diag(1:200)); A(2,2) = 1+1/(200)^4  (irlbablk r=2  m=20) (irlba m=20)','Fontsize',8);
        semilogy(1:xmax,1d-6*ones(xmax),'k'); text(-10,10^(-6),'10^{-6}');
        xlabel('Number of restarts','Fontsize',8);
     
    % Example 2.  A = delsq(numgrid(''S'',20)).
    % Computing the 6 largest singular values.
    case 'example2'
        
        % Set up the matrix.
        A=delsq(numgrid('S',20)); n = 324;
        
        % Set up parameters for IRLBABLK.
        OPTS.MAXIT = 5000; OPTS.ADJUST = 3; OPTS.K = 6; 
        OPTS.SIGMA ='LS'; OPTS.TOL = eps; OPTS.M = 20; 
        OPTS.AUG = 'RITZ'; OPTS.REORTH = 'TWO';
        
        % Set up rand vectors 
        V = randn(n,6);
         
        for i=1:5
          OPTS.BLSZ=i; OPTS.M = floor(20/i); 
          OPTS.V = V(:,1:i);
          tic; [U,S,V,FLAG] = irlbablk(A,OPTS); tim(i) = toc;
          iter(i) = size(guiresidual,1); % length is the number of iterations.
          mvp(i) = FLAG(2);
        end
        
        % Plot the CPU times on a subgraph.
        figure; 
        subplot(2,1,1); 
        bar(tim./iter,1,'w');
        set(gca,'XTickLabel',{'r=1, m=20', 'r=2, m= 10', 'r=3, m=6', 'r=4, m=5',...
                'r=5, m=4'},'Visible','on','FontSize',8);
        Str1 = 'Avg. CPU  '; Str2='time per  '; Str3='iteration  ';
        Str4='(secs.)  '; D=str2mat(Str1,Str2,Str3,Str4);
        ylabel(D,'Rotation',0,'fontsize',6); 
        title('A=delsq(numgrid(''S'',20));  r=block size    m=# of blocks');
        subplot(2,1,2)
        bar(tim,1,'w');
        Str1 = 'Total  '; Str2='CPU time  '; Str3='(secs.)  '; 
        D=str2mat(Str1,Str2,Str3);
        ylabel(D,'Rotation',0,'fontsize',6);
        text(0.75,tim(1)+tim(1)/4,strcat('mvp=',num2str(mvp(1))),'Fontsize',8);
        text(1.75,tim(2)+tim(1)/4,strcat('mvp=',num2str(mvp(2))),'Fontsize',8);
        text(2.75,tim(3)+tim(1)/4,strcat('mvp=',num2str(mvp(3))),'Fontsize',8);
        text(3.75,tim(4)+tim(1)/4,strcat('mvp=',num2str(mvp(4))),'Fontsize',8);
        text(4.75,tim(5)+tim(1)/4,strcat('mvp=',num2str(mvp(5))),'Fontsize',8);
        set(gca,'XTickLabel',{'r=1, m=20', 'r=2, m= 10', 'r=3, m=6', 'r=4, m=5',...
                'r=5, m=4'},'Visible','on','FontSize',8);

    % Example 3.  A = 130 x 130 Toeplitz matrix.
    % Computing the 4 smallest singular values.
    case 'example3'
        
        %Set to true for computing exact residuals.
        example3 = 1;
        
        % Reset exact_error;
        exact_error=[];
        
        % Set up the matrix. 
        x=zeros(130,1);  w=4.5; x(1) = 1;
        for m=2:8
          x(m) = 1/((m-1)/w)^2*(sin((m-1)/w))^2;
        end
        A=sparse(toeplitz(x,x));  
        S_exact =[0.000002316854850 0.000008358301508 0.000772216008660 ...
                  0.008653782928600];
              
        % Set common parameters for IRLBABLK.
        OPTS.MAXIT = 2000; OPTS.ADJUST = 3; OPTS.K = 4; 
        OPTS.SIGMA ='SS'; OPTS.TOL = 1d-6; OPTS.M = 10; 
        OPTS.AUG = 'HARM'; OPTS.BLSZ = 4; OPTS.V = randn(130,4); 

        S = irlbablk(A,OPTS);
        Hval = exact_error;  exact_error=[];

        OPTS.AUG = 'RITZ';

        S=irlbablk(A,OPTS);
        Rval = exact_error;  exact_error=[];

        % Plot the solutions on a subgraph.
        figure; 
        subplot(2,2,1); semilogy(Rval(:,1),'-'); hold on; 
        semilogy(Hval(:,1),':'); xlabel('Number of restarts','Fontsize',8); 
        title('\sigma=2.32 10^{-6}','Fontsize',8);
        subplot(2,2,2); semilogy(Rval(:,2),'-'); hold on;
        semilogy(Hval(:,2),':'); legend('Ritz','Harmonic Ritz');
        title('\sigma=8.36 10^{-6}','Fontsize',8);
        xlabel('Number of restarts','Fontsize',8); 
        subplot(2,2,3); semilogy(Rval(:,3),'-'); hold on; 
        semilogy(Hval(:,3),':'); xlabel('Number of restarts','Fontsize',8); 
        title('\sigma = 7.72 10^{-4}','Fontsize',8);
        subplot(2,2,4); semilogy(Rval(:,4),'-'); hold on; 
        semilogy(Hval(:,4),':'); xlabel('Number of restarts','Fontsize',8); 
        title('\sigma = 8.65 10^{-3}','Fontsize',8);
   
    % Example 4.  A = 11,390 x 1,265 term-by-document.
    % Computing the 10 largest singular values.
    case 'example4'
        
        % Set up the matrix.
        mmfile = fopen('hypatia','r');
        if ( mmfile == -1 )
            
            % Output Error message.
            Str1 = 'Error! File hypatia not found!';
            Str2 = '  ';
            Str3 = ' File can be found on Netlib or';
            Str4 = ' http://www.math.uri.edu/~jbaglama';
            error(str2mat(Str1,Str2,Str3,Str4));
     
        else

            disp('Please wait loading matrix HYPATIA...');

            % Read comment line then foget it.
            commentline = fgets(mmfile);
            while length(commentline) > 0 && commentline(1) == '%',
                  commentline = fgets(mmfile);
            end
                   
            % Read size information
            [sizeinfo,count] = sscanf(commentline,'%d%d%d');
            while ( count == 0 )
                   commentline =  fgets(mmfile);
                   [sizeinfo,count] = sscanf(commentline,'%d%d%d');
            end
            rows = sizeinfo(1); cols = sizeinfo(2); 
            entries = sizeinfo(3);
                   
            % Read in matrix hypatia.
            [T,count] = fscanf(mmfile,'%f',3);
            T = [T; fscanf(mmfile,'%f')];
            T = reshape(T,3,entries)';
            A_web = sparse(T(:,1), T(:,2), T(:,3), rows , cols);
                   
                   
        end
        [p,n] = size(A_web);

        % Number of runs.
        num_iter = 4;
  
        % Set up rand vectors so calls with different parameters
        % use the same rand vector.
        V = randn(n,num_iter);

       % % Set common parameters for IRLBABLK.
       OPTS.AUG = 'RITZ'; OPTS.REORTH = 'ONE'; OPTS.TOl = 1d-6; 
       OPTS.K = 10; OPTS.sigma = 'LS';
       
       % Computer the singular values for the different 
       % block sizes
       for iter = 1:num_iter
         OPTS.V = V(:,1:iter); OPTS.BLSZ = iter; 
         OPTS.M = floor(20/iter);
         if iter == 3, OPTS.M = 7; end
         
         % Set mvprod timings to zero.  
         MV_tim=0; count = 0;
         tt=cputime; [U,S,V,FLAG] = irlbablk(@Aprod,p,n,OPTS);  
         total_timing = cputime-tt; mprod = FLAG(2);
         disp(' ');
         disp('--------------------------------------------------------------------');
         disp(sprintf('IRLBABLK'));
         disp(sprintf('Block size = %d',OPTS.BLSZ));
         disp(sprintf('Number of blocks = %d',OPTS.M));
         disp(sprintf('Matrix-vector Products = %d' ,mprod));
         disp(sprintf('Number of Matrix access = %d' ,count));
         disp(sprintf('Matrix-vector Products timing = %d secs.',MV_tim));
         disp(sprintf('Non-Matrix-vector Products timing = %d secs.',total_timing-MV_tim));
         disp(sprintf('TOTAL time = %d secs.',total_timing));
         disp('--------------------------------------------------------------------'); 
         disp(' '); 
       end     
        
    otherwise
         Str1='Please input:';
         Str2='examples(''example1'')';
         Str3='examples(''example2'')';
         Str4='examples(''example3'')';
         Str5='examples(''example4'')';
         D=str2mat(Str1,Str2,Str3,Str4,Str5);
         disp(D);
end        

% Function used to compute matrix-vector products and
% record timings for example 4.
function Y = Aprod(varargin)
global A_web MV_tim count; 
 x = varargin{1};
 tt=cputime;
 if strcmp(varargin{5},'F')
    Y = A_web*x;
 else
    Y = (x'*A_web)';
 end  
 MV_tim = MV_tim + cputime-tt; 
 count = count + 1;