// Author:   Jeff Dicker
// For:      Dr. Yves Lucet
// Whatsit:  Unit tests for fusion, fusionsci
// 	     Comparison of fusionsci (sort using buip-in functions) with fusion (merge sort)
//
// Test-script for the Linear-time Legendre Transform (Yves Lucet - 09/05/97)
//// ported to Octave (Yves Lucet - 11/18/04)
//// ported to Scilab w/ Unit Tests (Jeff Dicker - 05/09/05)
///////////////////////////////////////////////////////////////////////////

// Beware that taking random numbers affects the computation time.
// Indeed if s_1<...<s_m<c_1<...<c_p, both functions are very fast
// (since they do not look at all slopes S). Unfortunately, this case gives
// poor numerical accuracy.
///////////////////////////////////////////////////////////////////////////

//fusion and fusionsci are internal functions that should not be directly manipulated
//the code below is for unit testing purposes only
mode(-1);
__test_set = "Fusion";

//unit tests
function b = testFusionAllTouchFirst()
	C = [8 9 10 12]';
	S = [1 2 3 4 5]';
	result = fusion(C, S);
	b = isequal(result, [1 1 1 1 1]');
endfunction

function b = testFusionNoneTouch()
	C = [-4 -2 0 1 4]';
	S = [6 8 9 10 12]';
	result = fusion(C, S);
	// it should return the index 6 size(S, 2) times, because
	// none of the points will touch
	b = isequal(result, [6 6 6 6 6]');
endfunction

function b = testFusionMiddleOne()
	C = [-4 -2 0 2 4]';
	S = [-3 -1 1 3]';
	result = fusion(C, S);
	// S(1) = -3 <= C(2) = -2     [2]
	// S(2) = -1 <= C(3) = 0      [3]
	// S(3) = 1  <= C(4) = 2      [4]
	// S(4) = 3  <= C(5) = 4      [5]
	b = isequal(result, [2 3 4 5]');
endfunction

function b = testFusionMiddleTwo()
	C = [-8 4 7 9 36]';
	S = [5 6 13]';
	result = fusion(C, S);
	// S(1) = 5  <= C(3) = 7      [3]
	// S(2) = 6  <= C(3) = 7      [3]
	// S(3) = 13 <= C(5) = 36     [5]
	b = isequal(result, [3 3 5]');
endfunction

function b = testFusionAllTouchFirstsci()
	C = [8 9 10 12]';
	S = [1 2 3 4 5]';
	result = fusionsci(C, S);
	b = isequal(result, [1 1 1 1 1]');
endfunction

function b = testFusionNoneTouchsci()
	C = [-4 -2 0 1 4]';
	S = [6 8 9 10 12]';
	result = fusionsci(C, S);
	// it should return the index 6 size(S, 2) times, because
	// none of the points will touch
	b = isequal(result, [6 6 6 6 6]');
endfunction

function b = testFusionMiddleOnesci()
	C = [-4 -2 0 2 4]';
	S = [-3 -1 1 3]';
	result = fusionsci(C, S);
	// S(1) = -3 <= C(2) = -2     [2]
	// S(2) = -1 <= C(3) = 0      [3]
	// S(3) = 1  <= C(4) = 2      [4]
	// S(4) = 3  <= C(5) = 4      [5]
	b = isequal(result, [2 3 4 5]');
endfunction

function b = testFusionMiddleTwosci()
	C = [-8 4 7 9 36]';
	S = [5 6 13]';
	result = fusionsci(C, S);
	// S(1) = 5  <= C(3) = 7      [3]
	// S(2) = 6  <= C(3) = 7      [3]
	// S(3) = 13 <= C(5) = 36     [5]
	b = isequal(result, [3 3 5]');
endfunction

function b = testAllFusionsAreTheSame()
	X=[-5:0.5:5]';
	Y=X.^2;
	C=(Y(2:size(Y,1))-Y(1:size(Y,1)-1))./(X(2:size(X,1))-X(1:size(X,1)-1));
	S=C-0.25*ones(size(C, 1));
	fSsci = fusionsci(C, S);
	fSfus = fusion(C, S);
	b = isequal(fSsci, fSfus);
endfunction


function b = runTestFile
	b = %t;	
	r = testWrapper(testAllFusionsAreTheSame, "testAllFusionsAreTheSame");
	b = checkForFail(r, b);
	r = testWrapper(testFusionAllTouchFirst, "testFusionAllTouchFirst");
	b = checkForFail(r, b);
	r = testWrapper(testFusionNoneTouch, "testFusionNoneTouch");
	b = checkForFail(r, b);
	r = testWrapper(testFusionMiddleOne, "testFusionMiddleOne");
	b = checkForFail(r, b);
	r = testWrapper(testFusionMiddleTwo, "testFusionMiddleTwo");
	b = checkForFail(r, b);
	r = testWrapper(testFusionAllTouchFirstsci, "testFusionAllTouchFirstsci");
	b = checkForFail(r, b);
	r = testWrapper(testFusionNoneTouchsci, "testFusionNoneTouchsci");
	b = checkForFail(r, b);
	r = testWrapper(testFusionMiddleOnesci, "testFusionMiddleOnesci");
	b = checkForFail(r, b);
	r = testWrapper(testFusionMiddleTwosci, "testFusionMiddleTwosci");
	b = checkForFail(r, b);
endfunction
