/*
** SUPERF.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
** REVISION DATE: May, 1999
**
**
** MODULES CALLED: None 
**
** -------------------------------------------------------------------------
** SUBROUTINES AND FUNCTIONS DESCRIPTION:
**
** >>>>>>>>>>>>>>>
**
** void genfact(int n,int V[])
**
** This subroutine computes the list of binomials 
**    / n \
**   |     |   for i=0,...,n.
**    \ i /     
** The result is stored in the array V 
** which must have at least n+1 columns.
**
** >>>>>>>>>>>>>>>
**
** void gengeneralfact(int n,int V[],struct indextype lista_ass[])
**
** This subroutine computes the set of generalized binomials 
**   /   n   \
**   |       |
**   \ i,j,k /      
** by the formula n!/(i!j!k!) where i+j+k=n. 
**
** The result is stored in the array V, which must have at least (n+1)*(n+2)/2 
** elements, following the lexicographic order of the triples (i,j,k).
**
** In the body of the subroutine, it is used the local variable 
** 'h' that varies from 0 to (n+1)*(n+2)/2-1 while  
** triples '(lista_ass[h].i, lista_ass[h].j, lista_ass[h].k)'
** list all triples (i,j,k) in lexicographic order.
** 
**
**  >>>>>>>>>>>>>>>
**
** void ass(int h,int n,struct indextype* result)
**
** This subroutine completes the association between the sequential index
** of the control point in a list and the corresponding tridimensional index 
** (i,j,k) for a polynomial triangular Bezier patch of degree 'n' in 
** barycentric coordinates. 
** It is assumed that the list of control points is ordered first
** with respect to 'i', and then with respect to 'j' (both from 0 to n). 
**
** The resulting triple of indexes is stored in the structure 'result'.
**
** >>>>>>>>>>>>>>>
**
** int listindex(int n,int i,int j)
**
** This subroutine computes the sequential position of the control point
** described by the triple (i,j,k) where i+j+k=n, where 'n' is the degree
** of the triangular polynomial Bezier patch. 
** It is assumed that the list of control points is ordered first
** first with respect to 'i', and then with respect to 'j' (both from 0 to n). 
**
** >>>>>>>>>>>>>>>
**
** int triitpbez(int n,double[][MAXDIM] itpt,double[][MAXDIM] cpt,
**               double[] u,double[] v,int dim,
**               struct indextype[] lista_ass,int[] genfactlist)
**
** This function interpolates the (n+1)*(n+2)/2 x 'dim'-dimensional
** array of points 'itpt' (where dim<=MAXDIM) according to the
** the array parameters u and v. 
** These latter must agree with the following requirements. 
** Let a=(1,0), b=(0,1) and c=(0,0) be the
** vertexes of the fundamental triangle in (R^2), and
** (1,0,0), (0,1,0) and (0,0,1) the corresponding barycentric 
** coordinates. With reference to this triangle, let us consider a
** new point whose cartesian coordinates are (u,v). Then, the
** corresponding barycentric coordinates are
** simply 
**        ubar=u, vbar=v and wbar=1-u-v.
**
** The program calling this function must supply the pointer to
** the list of all possible associations, i.e. the variable
** 'lista_ass', in order to get all tridimensional barycentric 
** indexes (i,j,k) ordered first 
** with respect to 'i', and then with respect to j. 
**
** The program calling this function must supply the pointer to
** the list of generalized binomials, n!/(i!k!k!),
** ordered first with respect to 'i' from 0 to n, and then with respect  
** to 'j' from 0 to n-i.
**
** Function result:
**       'triitpbez==0': all is OK
**       'triitpbez==1': n<1 or (n+1)*(n+2)/2 > MAXORD
**       'triitpbez==3': the interpolation matrix, built by this routine,
**                       is a singular or it is ill-conditioned.
**
** >>>>>>>>>>>>>>>
**
** int trihornbez(int n,int dim,double[][MAXDIM] cpt,int[] genfactlist,
**                double[] ucalcpar,double[] vcalcpar,
**                int numpt,double[][MAXDIM] calcpt)
**
** This function evaluates the Bezier surface described
** by the triangular net of control points listed in the array 'cpt'.
** The control points should be at least (n+1)*(n+2)/2 
** (the number of control points for a n-degree triangular polynomial Bezier patch).
**
** The function computes 'numpt' points (1<=numpt<=MAXCALC) corresponding
** to the values given in the vectors 'ucalcpar' and 'vcalcpar'
** that must have, at least 'numpt' elements. 
** The resulting values are stored in the first 'numpt' entries of the
** output array 'calcpt'.
** 
** The computed points are always MAXDIM-dimensional, like the
** input control points in the array 'cpt', but the first 'MAXDIM-dim'
** columns of the output matrix 'calcpt' are computed in a faster
** way using the linear precision formula. 
**
** The function needs the list of the generalized binomials
** n!/(i!j!k!) i=0,...,n  j=0,...,n-i. 
**
** It uses a Horner-like method to compute the points. 
**
** Function result:
**          'trihornbez==0':  all is OK;
**          'trihornbez==1': numpt<1 or numpt>MAXCALC
**                           or (n+1)*(n+2)/2 > MAXORD. 
**
** 
** >>>>>>>>>>>>>>>
**
** int minmaxbox(int n,int dim,double[][MAXDIM] cpt,
**               double[] min,double[] max,double minamp)
**
** This function takes as input the 'dim'-dimensional array 'cpt', 
** with 'MAXDIM' real columns, 
** and at least 'n+1' 'dim'-dimensional control points (its rows)
** in order to compute their minimum and maximum values
**
** The results are stored in the 'dim'-dimensional vectors 'min' and
** 'max' (with MAXDIM real columns).
** If the difference between maximum and minimum values is greater
** than the minimum value allowed, that is 'minamp', the maximum and minimum
** values are stretched by 'minamp/2' so that their difference
** becomes greater than minamp.
**
** Function result:
**              'minampbox==0': all is OK;
**              'minampbox==0': dim<0 or dim>MAXDIM.
**
** -------------------------------------------------------------------------- 
*/

void genfact(n,V)
int n,V[];
{
int i;
/* */
V[0]=1;
for (i=1;i<=n;++i) V[i]=V[i-1]*(n-i+1)/i;
}

void gengeneralfact(n,V,lista_ass)
int n,V[];
struct indextype lista_ass[];
{
 int l=(n+1)*(n+2)/2;
 int h;
 /* */
 V[0]=1;
 for (h=1;h<l;++h)
  {
   if (lista_ass[h].j > 0)
     V[h]=(V[h-1]*(n-lista_ass[h].i-lista_ass[h].j+1))/lista_ass[h].j;
   else if (lista_ass[h].j == 0)
	  V[h]=( V[ h-(n-lista_ass[h].i+2) ] * (n-lista_ass[h].i+1) ) /
	       lista_ass[h].i;
  }
}

void ass(h,n,result)
int h,n;
struct indextype *result;
{
 int ii;
 /* */
 ii=0; 
 while (h+1 > ((ii+1)*(2*n-ii+2))/2) ii=ii+1;
 /* The second member of the inequality is the number of control points
    having their index i in the range from 0 to ii. */
 result->i=ii;
 result->j=h-(ii*(2*n-ii+3))/2; /* now ii is 1 unity greater than it was */
 result->k=n-ii -result->j;
}

int listindex(n,i,j)
int n,i,j;
{
 return ( ((2*n-i+3)*i)/2+j );
}

int triitpbez(n,itpt,cpt,u,v,dim,lista_ass,genfactlist)
int n,dim;
double itpt[][MAXDIM],cpt[][MAXDIM],u[],v[];
int genfactlist[];
struct indextype lista_ass[];
{
int l=(n+1)*(n+2)/2-1;
int s,h,err;
double A[MAXORD][MAXORD];
double temp;
int tempintarray[MAXORD];  /* for Gauss */
int gsspvt();
void gengeneralfact();

if (n<1 || l+1>MAXORD)
 return 1;
for (h=0;h<=l;++h) A[0][h]=(double)genfactlist[h];
for (s=1;s<=l;++s) for (h=0;h<=l;++h) A[s][h]=A[0][h];
for (s=0;s<=l;++s) for (h=0;h<=l;++h)
{

 if (lista_ass[h].i==0) temp=1.;
   else temp=pow(u[s],(double)lista_ass[h].i);
 A[s][h]=A[s][h]*temp;
 if (lista_ass[h].j==0) temp=1.;
   else temp=pow(v[s],(double)lista_ass[h].j);
 A[s][h]=A[s][h]*temp;
 if (lista_ass[h].k==0) temp=1.;
   else temp=pow((1.-u[s]-v[s]),(double)lista_ass[h].k);
 A[s][h]=A[s][h]*temp;
}
err=gsspvt(A,cpt,itpt,MAXORD,MAXDIM,l+1,dim,tempintarray);
if (err==1) return 2;
if (err==2) return 1;
return 0;
}

int trihornbez(n,dim,cpt,genfactlist,ucalcpar,vcalcpar,numpt,calcpt)
int n,dim,numpt;
double cpt[][MAXDIM],ucalcpar[],vcalcpar[],calcpt[][MAXDIM];
int genfactlist[];
{
 int l=(n+1)*(n+2)/2;
 double temp;
 int i,j,h,h1,h2,h3,ptcont,dimcont;

 if (numpt<1 || numpt>MAXCALC || l>MAXORD)
  return 1;

 h1=listindex(n,n,0);
 h2=listindex(n,0,n);
 h3=listindex(n,0,0);
 h=h1;
 for (ptcont=0;ptcont<numpt;++ptcont)
 { 
  for (dimcont=MAXDIM-dim;dimcont<MAXDIM;++dimcont)
   calcpt[ptcont][dimcont]=cpt[h][dimcont];
 }
 for (i=n-1;i>=0;--i)
  for (ptcont=0;ptcont<numpt;++ptcont)
  {
   for (dimcont=0;dimcont<MAXDIM-dim;++dimcont)
   {
    calcpt[ptcont][dimcont]=
      cpt[h1][dimcont]*ucalcpar[ptcont]+cpt[h2][dimcont]*vcalcpar[ptcont]+
      cpt[h3][dimcont]*(1-ucalcpar[ptcont]-vcalcpar[ptcont]);
   } /* fine for su dimcont */
   for (dimcont=MAXDIM-dim;dimcont<MAXDIM;++dimcont)
   {
    /* computing temp as a function of i,ptcont,dimcont */ 
    h=listindex(n,i,0);
    temp=cpt[h][dimcont]*genfactlist[h];
    for (j=1;j<=n-i;++j)
    {
     temp=temp*(1.-ucalcpar[ptcont]-vcalcpar[ptcont]) +
          genfactlist[h+j]*pow(vcalcpar[ptcont],j)*cpt[h+j][dimcont];
    }
    /* ending computation of temp */
    calcpt[ptcont][dimcont]=calcpt[ptcont][dimcont]*ucalcpar[ptcont]+temp;
   } /* end for dimcont */
  } /* end for ptcont */
 return 0;
}

int minmaxbox(n,dim,cpt,min,max,minamp)
int n,dim;
double cpt[][MAXDIM];
double min[],max[],minamp;
{
int i,j;

if (dim < 0 || dim > MAXDIM)
 return 1;
/* initialization to the first point */
for (j=0;j<dim;++j)
{
min[j]=cpt[0][j];
max[j]=cpt[0][j];
}
/* check over other points */
for (i=1;i<=n;++i)
 for (j=0;j<dim;++j)
 {
 if (cpt[i][j] < min[j]) min[j]=cpt[i][j];
 else if (cpt[i][j] > max[j]) max[j]=cpt[i][j];
 }
for (j=0;j<dim;++j)
 if ((max[j]-min[j]) < minamp)
  {
   min[j]=min[j]-minamp/2;
   max[j]=max[j]+minamp/2;
  }
return 0;
}



