/*
** PWS_D_B.C 
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
**
** REVISION DATE: May, 1999  
**
** MODULES CALLED: None 
**
** --------------------------------------------------------------------------
**
** SUBROUTINES AND FUNCTIONS DESCRIPTION:
**
** >>>>>>>>>>>>>>>>>>>>
**
** double derivecptpwsb_v(int i,int j)
**
** GLOBAL VARIABLES USED: u1,v1,u2,v2,u3,v3,t1,t2,t3
** t1,t2,t3 must be computed externally as:
**	   t1=-((u3-u2)*(u2-u4)+(v3-v2)*(v2-v4))/
**	       (pow(u3-u2,2.)+pow(v3-v2,2.));
**	   t2=-((u1-u3)*(u3-u4)+(v1-v3)*(v3-v4))/
**	       (pow(u1-u3,2.)+pow(v1-v3,2.));
**	   t3=-((u2-u1)*(u1-u4)+(v2-v1)*(v1-v4))/
**	       (pow(u2-u1,2.)+pow(v2-v1,2.));
**
** This function computes the derivative of the control point d_i 
** ('i' is in the range [0,30])
** of the Powell-Sabin element of B-Type as a function of the partial
** derivatives zu1,zv1,zu2,zv2,zu3,zv3 with respect to the 'j'-th partial
** derivative in the specified order from 0 to 5.
** The triangle over which the function operates is described by the
** vertices (u1,v1), (u2,v2), (u3,v3), while (u4,v4) is the barycenter. 
**
** Inputs
**        i: the point index (between 0 and 30);
**        j: the derivative index (between 0 and 5);
**
** Output
**        derivecptpwsb_v: d d_i/ d^j .
**
**
** >>>>>>>>>>>>>>>>>>>>
**
** double compute_d_j_pwsb(int j,int i)
**
** GLOBAL VARIABLES USED: u1,v1,u2,v2,u3,v3,z1,z2,z3,t1,t2,t3
** t1,t2,t3 must be computed externally as:
**	   t1=-((u3-u2)*(u2-u4)+(v3-v2)*(v2-v4))/
**	       (pow(u3-u2,2.)+pow(v3-v2,2.));
**	   t2=-((u1-u3)*(u3-u4)+(v1-v3)*(v3-v4))/
**	       (pow(u1-u3,2.)+pow(v1-v3,2.));
**	   t3=-((u2-u1)*(u1-u4)+(v2-v1)*(v1-v4))/
**	       (pow(u2-u1,2.)+pow(v2-v1,2.));
**
** This function computes the control point d_j ('j' is in the range [0,30]) 
** of thePowell-Sabin element of B-Type assigning to the unknown parameters
** zu1,zv1,zu2,zv2,zu3,zv3 the value given by the 'i'-th vector of the
** canonical basis of R^6.
** (In the body of the function the vectors of the canonical basis are
** described by e0, e1, ..., e5). 
** The triangle over which the function operates is described by the
** vertices (u1,v1), (u2,v2), (u3,v3), while (u4,v4) is the barycenter. 
** The parameters z1,z2,z3 are the values of the triangular patch 
** relative to the vertices V1,V2,V3 that are, by end-point interpolation,
** the values of the control points d0,d4,d8.
**
** Inputs
**        j: the point index (between 0 and 30);
**        i: the index of the element of the canonical basis 
**          (between 0 and 5);
**
** Output
**        compute_d_j_pwsb: the value of d_j.
**
**
** >>>>>>>>>>>>>>>>>>>>
**
** double compute_d_j_pwsb_null(int j)
**
** GLOBAL VARIABLES USED: z1,z2,z3,t1,t2,t3
** t1,t2,t3 must be computed externally as:
**	   t1=-((u3-u2)*(u2-u4)+(v3-v2)*(v2-v4))/
**	       (pow(u3-u2,2.)+pow(v3-v2,2.));
**	   t2=-((u1-u3)*(u3-u4)+(v1-v3)*(v3-v4))/
**	       (pow(u1-u3,2.)+pow(v1-v3,2.));
**	   t3=-((u2-u1)*(u1-u4)+(v2-v1)*(v1-v4))/
**	       (pow(u2-u1,2.)+pow(v2-v1,2.));
**
** This function computes the control point d_j ('j' is in the range [0,30]) 
** of the Powell-Sabin element of B-Type assigning to the unknown parameters
** zu1,zv1,zu2,zv2,zu3,zv3 the value 0.
** The parameters z1,z2,z3 are the values of the triangular patch 
** relative to the vertices V1,V2,V3 that are, by end-point interpolation,
** the values of the control points d0,d4,d8.
**
** Input
**        j: the point index (between 0 and 30);
** Output
**        compute_d_j_pwsb_null: the value of d_j at 0.
**
**
** -------------------------------------------------------------------------
*/

extern double u1,v1,u2,v2,u3,v3,z1,z2,z3,t1,t2,t3;

double derivecptpwsb_v(i,j)
int i,j;
{
 /* Beginning of derivation by lookup table */
 if (i==0)     /* d0 */
 {
  return(0.);
 }
 else if (i==1)     /* d1 */
 {
  if (j==0) return((u2-u1)/4);
  if (j==1) return((v2-v1)/4);
 }
 else if (i==2)     /* d2 */
 {
  if (j==0) return((u2-u1)/8);
  if (j==1) return((v2-v1)/8);
  if (j==2) return((u1-u2)/8);
  if (j==3) return((v1-v2)/8);
 }
 else if (i==3)     /* d3 */
 {
  if (j==2) return((u1-u2)/4);
  if (j==3) return((v1-v2)/4);
 }
 else if (i==4)     /* d4 */
 {
  return(0.);
 }
 else if (i==5)     /* d5 */
 {
  if (j==2) return((u3-u2)/4);
  if (j==3) return((v3-v2)/4);
 }
 else if (i==6)     /* d6 */
 {
  if (j==2) return((u3-u2)/8);
  if (j==3) return((v3-v2)/8);
  if (j==4) return((u2-u3)/8);
  if (j==5) return((v2-v3)/8);
 }
 else if (i==7)     /* d7 */
 {
  if (j==4) return((u2-u3)/4);
  if (j==5) return((v2-v3)/4);
 }
 else if (i==8)     /* d8 */
 {
  return(0.);
 }
 else if (i==9)     /* d9 */
 {
  if (j==4) return((u1-u3)/4);
  if (j==5) return((v1-v3)/4);
 }
 else if (i==10)     /* d10 */
 {
  if (j==0) return((u3-u1)/8);
  if (j==1) return((v3-v1)/8);
  if (j==4) return((u1-u3)/8);
  if (j==5) return((v1-v3)/8);
 }
 else if (i==11)     /* d11 */
 {
  if (j==0) return((u3-u1)/4);
  if (j==1) return((v3-v1)/4);
 }
 else if (i==12)     /* d12 */
 {
  if (j==0) return(-(2*u1-u2-u3)/8);
  if (j==1) return(-(2*v1-v2-v3)/8);
 }
 else if (i==13)     /* d13 */
 {
  if (j==0) return((6*t3*(u1-u2)-6*u1+5*u2+u3)/16);
  if (j==1) return((6*t3*(v1-v2)-6*v1+5*v2+v3)/16);
  if (j==2) return((6*t3*(u1-u2)-2*u1+u2+u3)/16);
  if (j==3) return((6*t3*(v1-v2)-2*v1+v2+v3)/16);
 }
 else if (i==14)     /* d14 */
 {
  if (j==0) return((6*t3*(u1-u2)-5*u1+4*u2+u3)/12);
  if (j==1) return((6*t3*(v1-v2)-5*v1+4*v2+v3)/12);
  if (j==2) return((6*t3*(u1-u2)-2*u1+u2+u3)/12);
  if (j==3) return((6*t3*(v1-v2)-2*v1+v2+v3)/12);
 }
 else if (i==15)     /* d15 */
 {
  if (j==0) return((6*t3*(u1-u2)-5*u1+4*u2+u3)/16);
  if (j==1) return((6*t3*(v1-v2)-5*v1+4*v2+v3)/16);
  if (j==2) return((6*t3*(u1-u2)-u1+u3)/16);
  if (j==3) return((6*t3*(v1-v2)-v1+v3)/16);
 }
 else if (i==16)     /* d16 */
 {
  if (j==2) return((u1-2*u2+u3)/8);
  if (j==3) return((v1-2*v2+v3)/8);
 }
 else if (i==17)     /* d17 */
 {
  if (j==2) return((6*t1*(u2-u3)+u1-6*u2+5*u3)/16);
  if (j==3) return((6*t1*(v2-v3)+v1-6*v2+5*v3)/16);
  if (j==4) return((6*t1*(u2-u3)+u1-2*u2+u3)/16);
  if (j==5) return((6*t1*(v2-v3)+v1-2*v2+v3)/16);
 }
 else if (i==18)     /* d18 */
 {
  if (j==2) return((6*t1*(u2-u3)+u1-5*u2+4*u3)/12);
  if (j==3) return((6*t1*(v2-v3)+v1-5*v2+4*v3)/12);
  if (j==4) return((6*t1*(u2-u3)+u1-2*u2+u3)/12);
  if (j==5) return((6*t1*(v2-v3)+v1-2*v2+v3)/12);
 }
 else if (i==19)     /* d19 */
 {
  if (j==2) return((6*t1*(u2-u3)+u1-5*u2+4*u3)/16);
  if (j==3) return((6*t1*(v2-v3)+v1-5*v2+4*v3)/16);
  if (j==4) return((6*t1*(u2-u3)+u1-u2)/16);
  if (j==5) return((6*t1*(v2-v3)+v1-v2)/16);
 }
 else if (i==20)     /* d20 */
 {
  if (j==4) return((u1+u2-2*u3)/8);
  if (j==5) return((v1+v2-2*v3)/8);
 }
 else if (i==21)     /* d21 */
 {
  if (j==0) return(-(6*t2*(u1-u3)-u1-u2+2*u3)/16);
  if (j==1) return(-(6*t2*(v1-v3)-v1-v2+2*v3)/16);
  if (j==4) return(-(6*t2*(u1-u3)-5*u1-u2+6*u3)/16);
  if (j==5) return(-(6*t2*(v1-v3)-5*v1-v2+6*v3)/16);
 }
 else if (i==22)     /* d22 */
 {
  if (j==0) return(-(6*t2*(u1-u3)-u1-u2+2*u3)/12);
  if (j==1) return(-(6*t2*(v1-v3)-v1-v2+2*v3)/12);
  if (j==4) return(-(6*t2*(u1-u3)-4*u1-u2+5*u3)/12);
  if (j==5) return(-(6*t2*(v1-v3)-4*v1-v2+5*v3)/12);
 }
 else if (i==23)     /* d23 */
 {
  if (j==0) return(-(6*t2*(u1-u3)-u2+u3)/16);
  if (j==1) return(-(6*t2*(v1-v3)-v2+v3)/16);
  if (j==4) return(-(6*t2*(u1-u3)-4*u1-u2+5*u3)/16);
  if (j==5) return(-(6*t2*(v1-v3)-4*v1-v2+5*v3)/16);
 }
 else if (i==24)     /* d24 */
 {
  if (j==0) return(-3*(t2*(u1-u3)+t3*(u2-u1)+u1-u2)/16);
  if (j==1) return(-3*(t2*(v1-v3)+t3*(v2-v1)+v1-v2)/16);
  if (j==2) return((6*t3*(u1-u2)-2*u1+u2+u3)/32);
  if (j==3) return((6*t3*(v1-v2)-2*v1+v2+v3)/32);
  if (j==4) return(-(6*t2*(u1-u3)-4*u1-u2+5*u3)/32);
  if (j==5) return(-(6*t2*(v1-v3)-4*v1-v2+5*v3)/32);
 }
 else if (i==25)     /* d25 */
 {
  if (j==0) return((6*t3*(u1-u2)-5*u1+4*u2+u3)/32);
  if (j==1) return((6*t3*(v1-v2)-5*v1+4*v2+v3)/32);
  if (j==2) return(3*(t1*(u2-u3)+t3*(u1-u2)-u2+u3)/16);
  if (j==3) return(3*(t1*(v2-v3)+t3*(v1-v2)-v2+v3)/16);
  if (j==4) return((6*t1*(u2-u3)+u1-2*u2+u3)/32);
  if (j==5) return((6*t1*(v2-v3)+v1-2*v2+v3)/32);
 }
 else if (i==26)     /* d26 */
 {
  if (j==0) return(-(6*t2*(u1-u3)-u1-u2+2*u3)/32);
  if (j==1) return(-(6*t2*(v1-v3)-v1-v2+2*v3)/32);
  if (j==2) return((6*t1*(u2-u3)+u1-5*u2+4*u3)/32);
  if (j==3) return((6*t1*(v2-v3)+v1-5*v2+4*v3)/32);
  if (j==4) return(3*(t1*(u2-u3)+t2*(u3-u1)+u1-u3)/16);
  if (j==5) return(3*(t1*(v2-v3)+t2*(v3-v1)+v1-v3)/16);
 }
 else if (i==27)     /* d27 */
 {
  if (j==0) return(-(6*t2*(u1-u3)+6*t3*(u2-u1)+4*u1-5*u2+u3)/24);
  if (j==1) return(-(6*t2*(v1-v3)+6*t3*(v2-v1)+4*v1-5*v2+v3)/24);
  if (j==2) return((6*t3*(u1-u2)-2*u1+u2+u3)/24);
  if (j==3) return((6*t3*(v1-v2)-2*v1+v2+v3)/24);
  if (j==4) return(-(6*t2*(u1-u3)-4*u1-u2+5*u3)/24);
  if (j==5) return(-(6*t2*(v1-v3)-4*v1-v2+5*v3)/24);
 }
 else if (i==28)     /* d28 */
 {
  if (j==0) return((6*t3*(u1-u2)-5*u1+4*u2+u3)/24);
  if (j==1) return((6*t3*(v1-v2)-5*v1+4*v2+v3)/24);
  if (j==2) return((6*t1*(u2-u3)+6*t3*(u1-u2)-u1-4*u2+5*u3)/24);
  if (j==3) return((6*t1*(v2-v3)+6*t3*(v1-v2)-v1-4*v2+5*v3)/24);
  if (j==4) return((6*t1*(u2-u3)+u1-2*u2+u3)/24);
  if (j==5) return((6*t1*(v2-v3)+v1-2*v2+v3)/24);
 }
 else if (i==29)     /* d29 */
 {
  if (j==0) return(-(6*t2*(u1-u3)-u1-u2+2*u3)/24);
  if (j==1) return(-(6*t2*(v1-v3)-v1-v2+2*v3)/24);
  if (j==2) return((6*t1*(u2-u3)+u1-5*u2+4*u3)/24);
  if (j==3) return((6*t1*(v2-v3)+v1-5*v2+4*v3)/24);
  if (j==4) return((6*t1*(u2-u3)+6*t2*(u3-u1)+5*u1-u2-4*u3)/24);
  if (j==5) return((6*t1*(v2-v3)+6*t2*(v3-v1)+5*v1-v2-4*v3)/24);
 }
 else if (i==30)     /* d30 */
 {
  if (j==0) return(-(6*t2*(u1-u3)+6*t3*(u2-u1)+4*u1-5*u2+u3)/36);
  if (j==1) return(-(6*t2*(v1-v3)+6*t3*(v2-v1)+4*v1-5*v2+v3)/36);
  if (j==2) return((6*t1*(u2-u3)+6*t3*(u1-u2)-u1-4*u2+5*u3)/36);
  if (j==3) return((6*t1*(v2-v3)+6*t3*(v1-v2)-v1-4*v2+5*v3)/36);
  if (j==4) return((6*t1*(u2-u3)+6*t2*(u3-u1)+5*u1-u2-4*u3)/36);
  if (j==5) return((6*t1*(v2-v3)+6*t2*(v3-v1)+5*v1-v2-4*v3)/36);
 }
 /* security */
 return(0.);
}

double compute_d_j_pwsb(j,i)
int j,i;
{
 /* Inizio risposte */
 if (i==0) /* Computation of dj(e0) */
 {
  if (j==0) return(z1);
  if (j==1) return(-(u1-u2-4*z1)/4);
  if (j==2) return(-(u1-u2-4*(z1+z2))/8);
  if (j==3) return(z2);
  if (j==4) return(z2);
  if (j==5) return(z2);
  if (j==6) return(z2/2+z3/2);
  if (j==7) return(z3);
  if (j==8) return(z3);
  if (j==9) return(z3);
  if (j==10) return(-(u1-u3-4*(z1+z3))/8);
  if (j==11) return(-(u1-u3-4*z1)/4);
  if (j==12) return(-(2*u1-u2-u3-8*z1)/8);
  if (j==13) return((6*t3*(u1-u2-2*z1+2*z2)-6*u1+5*u2+u3+16*z1)/16);
  if (j==14) return((6*t3*(u1-u2-2*z1+2*z2)-5*u1+4*u2+u3+12*z1)/12);
  if (j==15) return((6*t3*(u1-u2-2*z1+2*z2)-5*u1+4*u2+u3+12*z1+4*z2)/16);
  if (j==16) return(z2);
  if (j==17) return((4*z2-3*t1*(z2-z3))/4);
  if (j==18) return(t1*(z3-z2)+z2);
  if (j==19) return(-(3*t1*(z2-z3)-3*z2-z3)/4);
  if (j==20) return(z3);
  if (j==21) return(-(6*t2*(u1-u3+2*(z3-z1))-u1-u2+2*(u3-8*z3))/16);
  if (j==22) return(-(6*t2*(u1-u3+2*(z3-z1))-u1-u2+2*(u3-6*z3))/12);
  if (j==23) return(-(6*t2*(u1-u3+2*(z3-z1))-u2+u3-4*(z1+3*z3))/16);
  if (j==24) return(-(3*t2*(u1-u3+2*(z3-z1))-3*t3*(u1-u2-2*z1+2*z2)+3*u1-3*u2-2*(5*z1+3*z3))/16);
  if (j==25) return(-(12*t1*(z2-z3)-6*t3*(u1-u2-2*z1+2*z2)+5*u1-4*u2-u3-4*(3*z1+5*z2))/32);
  if (j==26) return(-(12*t1*(z2-z3)+6*t2*(u1-u3+2*(z3-z1))-u1-u2+2*(u3-2*(3*z2+5*z3)))/32);
  if (j==27) return(-(6*t2*(u1-u3+2*(z3-z1))-6*t3*(u1-u2-2*z1+2*z2)+4*u1-5*u2+u3-12*(z1+z3))/24);
  if (j==28) return(-(12*t1*(z2-z3)-6*t3*(u1-u2-2*z1+2*z2)+5*u1-4*u2-u3-12*(z1+z2))/24);
  if (j==29) return(-(12*t1*(z2-z3)+6*t2*(u1-u3+2*(z3-z1))-u1-u2+2*(u3-6*(z2+z3)))/24);
  if (j==30) return(-(12*t1*(z2-z3)+6*t2*(u1-u3+2*(z3-z1))-6*t3*(u1-u2-2*z1+2*z2)+4*u1-5*u2+u3-12*(z1+z2+z3))/36);
 }
 else if (i==1) /* Computation of dj(e1) */
 {
  if (j==0) return(z1);
  if (j==1) return(-(v1-v2-4*z1)/4);
  if (j==2) return(-(v1-v2-4*(z1+z2))/8);
  if (j==3) return(z2);
  if (j==4) return(z2);
  if (j==5) return(z2);
  if (j==6) return(z2/2+z3/2);
  if (j==7) return(z3);
  if (j==8) return(z3);
  if (j==9) return(z3);
  if (j==10) return(-(v1-v3-4*(z1+z3))/8);
  if (j==11) return(-(v1-v3-4*z1)/4);
  if (j==12) return(-(2*v1-v2-v3-8*z1)/8);
  if (j==13) return((6*t3*(v1-v2-2*z1+2*z2)-6*v1+5*v2+v3+16*z1)/16);
  if (j==14) return((6*t3*(v1-v2-2*z1+2*z2)-5*v1+4*v2+v3+12*z1)/12);
  if (j==15) return((6*t3*(v1-v2-2*z1+2*z2)-5*v1+4*v2+v3+12*z1+4*z2)/16);
  if (j==16) return(z2);
  if (j==17) return((4*z2-3*t1*(z2-z3))/4);
  if (j==18) return(t1*(z3-z2)+z2);
  if (j==19) return(-(3*t1*(z2-z3)-3*z2-z3)/4);
  if (j==20) return(z3);
  if (j==21) return(-(6*t2*(v1-v3+2*(z3-z1))-v1-v2+2*(v3-8*z3))/16);
  if (j==22) return(-(6*t2*(v1-v3+2*(z3-z1))-v1-v2+2*(v3-6*z3))/12);
  if (j==23) return(-(6*t2*(v1-v3+2*(z3-z1))-v2+v3-4*(z1+3*z3))/16);
  if (j==24) return(-(3*t2*(v1-v3+2*(z3-z1))-3*t3*(v1-v2-2*z1+2*z2)+3*v1-3*v2-2*(5*z1+3*z3))/16);
  if (j==25) return(-(12*t1*(z2-z3)-6*t3*(v1-v2-2*z1+2*z2)+5*v1-4*v2-v3-4*(3*z1+5*z2))/32);
  if (j==26) return(-(12*t1*(z2-z3)+6*t2*(v1-v3+2*(z3-z1))-v1-v2+2*(v3-2*(3*z2+5*z3)))/32);
  if (j==27) return(-(6*t2*(v1-v3+2*(z3-z1))-6*t3*(v1-v2-2*z1+2*z2)+4*v1-5*v2+v3-12*(z1+z3))/24);
  if (j==28) return(-(12*t1*(z2-z3)-6*t3*(v1-v2-2*z1+2*z2)+5*v1-4*v2-v3-12*(z1+z2))/24);
  if (j==29) return(-(12*t1*(z2-z3)+6*t2*(v1-v3+2*(z3-z1))-v1-v2+2*(v3-6*(z2+z3)))/24);
  if (j==30) return(-(12*t1*(z2-z3)+6*t2*(v1-v3+2*(z3-z1))-6*t3*(v1-v2-2*z1+2*z2)+4*v1-5*v2+v3-12*(z1+z2+z3))/36);
 }
 else if (i==2) /* Computation of dj(e2) */
 {
  if (j==0) return(z1);
  if (j==1) return(z1);
  if (j==2) return((u1-u2+4*z1+4*z2)/8);
  if (j==3) return((u1-u2+4*z2)/4);
  if (j==4) return(z2);
  if (j==5) return(-(u2-u3-4*z2)/4);
  if (j==6) return(-(u2-u3-4*(z2+z3))/8);
  if (j==7) return(z3);
  if (j==8) return(z3);
  if (j==9) return(z3);
  if (j==10) return(z1/2+z3/2);
  if (j==11) return(z1);
  if (j==12) return(z1);
  if (j==13) return((6*t3*(u1-u2-2*z1+2*z2)-2*u1+u2+u3+16*z1)/16);
  if (j==14) return((6*t3*(u1-u2-2*z1+2*z2)-2*u1+u2+u3+12*z1)/12);
  if (j==15) return((6*t3*(u1-u2-2*z1+2*z2)-u1+u3+12*z1+4*z2)/16);
  if (j==16) return((u1-2*u2+u3+8*z2)/8);
  if (j==17) return((6*t1*(u2-u3-2*z2+2*z3)+u1-6*u2+5*u3+16*z2)/16);
  if (j==18) return((6*t1*(u2-u3-2*z2+2*z3)+u1-5*u2+4*u3+12*z2)/12);
  if (j==19) return((6*t1*(u2-u3-2*z2+2*z3)+u1-5*u2+4*u3+12*z2+4*z3)/16);
  if (j==20) return(z3);
  if (j==21) return((3*t2*(z1-z3)+4*z3)/4);
  if (j==22) return(t2*(z1-z3)+z3);
  if (j==23) return((3*t2*(z1-z3)+z1+3*z3)/4);
  if (j==24) return((12*t2*(z1-z3)+6*t3*(u1-u2-2*z1+2*z2)-2*u1+u2+u3+4*(5*z1+3*z3))/32);
  if (j==25) return((3*t1*(u2-u3-2*z2+2*z3)+3*t3*(u1-u2-2*z1+2*z2)-3*u2+3*u3+6*z1+10*z2)/16);
  if (j==26) return((6*t1*(u2-u3-2*z2+2*z3)+12*t2*(z1-z3)+u1-5*u2+4*u3+12*z2+20*z3)/32);
  if (j==27) return((12*t2*(z1-z3)+6*t3*(u1-u2-2*z1+2*z2)-2*u1+u2+u3+12*z1+12*z3)/24);
  if (j==28) return((6*t1*(u2-u3-2*z2+2*z3)+6*t3*(u1-u2-2*z1+2*z2)-u1-4*u2+5*u3+12*z1+12*z2)/24);
  if (j==29) return((6*t1*(u2-u3-2*z2+2*z3)+12*t2*(z1-z3)+u1-5*u2+4*u3+12*z2+12*z3)/24);
  if (j==30) return((6*t1*(u2-u3-2*z2+2*z3)+12*t2*(z1-z3)+6*t3*(u1-u2-2*z1+2*z2)-u1-4*u2+5*u3+12*(z1+z2+z3))/36);
 }
 else if (i==3) /* Computation of dj(e3) */
 {
  if (j==0) return(z1);
  if (j==1) return(z1);
  if (j==2) return((v1-v2+4*z1+4*z2)/8);
  if (j==3) return((v1-v2+4*z2)/4);
  if (j==4) return(z2);
  if (j==5) return(-(v2-v3-4*z2)/4);
  if (j==6) return(-(v2-v3-4*(z2+z3))/8);
  if (j==7) return(z3);
  if (j==8) return(z3);
  if (j==9) return(z3);
  if (j==10) return(z1/2+z3/2);
  if (j==11) return(z1);
  if (j==12) return(z1);
  if (j==13) return((6*t3*(v1-v2-2*z1+2*z2)-2*v1+v2+v3+16*z1)/16);
  if (j==14) return((6*t3*(v1-v2-2*z1+2*z2)-2*v1+v2+v3+12*z1)/12);
  if (j==15) return((6*t3*(v1-v2-2*z1+2*z2)-v1+v3+12*z1+4*z2)/16);
  if (j==16) return((v1-2*v2+v3+8*z2)/8);
  if (j==17) return((6*t1*(v2-v3-2*z2+2*z3)+v1-6*v2+5*v3+16*z2)/16);
  if (j==18) return((6*t1*(v2-v3-2*z2+2*z3)+v1-5*v2+4*v3+12*z2)/12);
  if (j==19) return((6*t1*(v2-v3-2*z2+2*z3)+v1-5*v2+4*v3+12*z2+4*z3)/16);
  if (j==20) return(z3);
  if (j==21) return((3*t2*(z1-z3)+4*z3)/4);
  if (j==22) return(t2*(z1-z3)+z3);
  if (j==23) return((3*t2*(z1-z3)+z1+3*z3)/4);
  if (j==24) return((12*t2*(z1-z3)+6*t3*(v1-v2-2*z1+2*z2)-2*v1+v2+v3+4*(5*z1+3*z3))/32);
  if (j==25) return((3*t1*(v2-v3-2*z2+2*z3)+3*t3*(v1-v2-2*z1+2*z2)-3*v2+3*v3+6*z1+10*z2)/16);
  if (j==26) return((6*t1*(v2-v3-2*z2+2*z3)+12*t2*(z1-z3)+v1-5*v2+4*v3+12*z2+20*z3)/32);
  if (j==27) return((12*t2*(z1-z3)+6*t3*(v1-v2-2*z1+2*z2)-2*v1+v2+v3+12*z1+12*z3)/24);
  if (j==28) return((6*t1*(v2-v3-2*z2+2*z3)+6*t3*(v1-v2-2*z1+2*z2)-v1-4*v2+5*v3+12*z1+12*z2)/24);
  if (j==29) return((6*t1*(v2-v3-2*z2+2*z3)+12*t2*(z1-z3)+v1-5*v2+4*v3+12*z2+12*z3)/24);
  if (j==30) return((6*t1*(v2-v3-2*z2+2*z3)+12*t2*(z1-z3)+6*t3*(v1-v2-2*z1+2*z2)-v1-4*v2+5*v3+12*(z1+z2+z3))/36);
 }
 else if (i==4) /* Computation of dj(e4) */
 {
  if (j==0) return(z1);
  if (j==1) return(z1);
  if (j==2) return(z1/2+z2/2);
  if (j==3) return(z2);
  if (j==4) return(z2);
  if (j==5) return(z2);
  if (j==6) return((u2-u3+4*z2+4*z3)/8);
  if (j==7) return((u2-u3+4*z3)/4);
  if (j==8) return(z3);
  if (j==9) return((u1-u3+4*z3)/4);
  if (j==10) return((u1-u3+4*z1+4*z3)/8);
  if (j==11) return(z1);
  if (j==12) return(z1);
  if (j==13) return((4*z1-3*t3*(z1-z2))/4);
  if (j==14) return(t3*(z2-z1)+z1);
  if (j==15) return(-(3*t3*(z1-z2)-3*z1-z2)/4);
  if (j==16) return(z2);
  if (j==17) return((6*t1*(u2-u3-2*z2+2*z3)+u1-2*u2+u3+16*z2)/16);
  if (j==18) return((6*t1*(u2-u3-2*z2+2*z3)+u1-2*u2+u3+12*z2)/12);
  if (j==19) return((6*t1*(u2-u3-2*z2+2*z3)+u1-u2+12*z2+4*z3)/16);
  if (j==20) return((u1+u2-2*u3+8*z3)/8);
  if (j==21) return(-(6*t2*(u1-u3+2*(z3-z1))-5*u1-u2+2*(3*u3-8*z3))/16);
  if (j==22) return(-(6*t2*(u1-u3+2*(z3-z1))-4*u1-u2+5*u3-12*z3)/12);
  if (j==23) return(-(6*t2*(u1-u3+2*(z3-z1))-4*u1-u2+5*u3-4*(z1+3*z3))/16);
  if (j==24) return(-(6*t2*(u1-u3+2*(z3-z1))+12*t3*(z1-z2)-4*u1-u2+5*u3-4*(5*z1+3*z3))/32);
  if (j==25) return((6*t1*(u2-u3-2*z2+2*z3)+12*t3*(z2-z1)+u1-2*u2+u3+4*(3*z1+5*z2))/32);
  if (j==26) return((3*t1*(u2-u3-2*z2+2*z3)-3*t2*(u1-u3+2*(z3-z1))+3*u1-3*u3+6*z2+10*z3)/16);
  if (j==27) return(-(6*t2*(u1-u3+2*(z3-z1))+12*t3*(z1-z2)-4*u1-u2+5*u3-12*(z1+z3))/24);
  if (j==28) return((6*t1*(u2-u3-2*z2+2*z3)+12*t3*(z2-z1)+u1-2*u2+u3+12*z1+12*z2)/24);
  if (j==29) return((6*t1*(u2-u3-2*z2+2*z3)-6*t2*(u1-u3+2*(z3-z1))+5*u1-u2-4*u3+12*z2+12*z3)/24);
  if (j==30) return((6*t1*(u2-u3-2*z2+2*z3)-6*t2*(u1-u3+2*(z3-z1))+12*t3*(z2-z1)+5*u1-u2-4*u3+12*(z1+z2+z3))/36);
 }
 else if (i==5) /* Computation of dj(e5) */
 {
  if (j==0) return(z1);
  if (j==1) return(z1);
  if (j==2) return(z1/2+z2/2);
  if (j==3) return(z2);
  if (j==4) return(z2);
  if (j==5) return(z2);
  if (j==6) return((v2-v3+4*z2+4*z3)/8);
  if (j==7) return((v2-v3+4*z3)/4);
  if (j==8) return(z3);
  if (j==9) return((v1-v3+4*z3)/4);
  if (j==10) return((v1-v3+4*z1+4*z3)/8);
  if (j==11) return(z1);
  if (j==12) return(z1);
  if (j==13) return((4*z1-3*t3*(z1-z2))/4);
  if (j==14) return(t3*(z2-z1)+z1);
  if (j==15) return(-(3*t3*(z1-z2)-3*z1-z2)/4);
  if (j==16) return(z2);
  if (j==17) return((6*t1*(v2-v3-2*z2+2*z3)+v1-2*v2+v3+16*z2)/16);
  if (j==18) return((6*t1*(v2-v3-2*z2+2*z3)+v1-2*v2+v3+12*z2)/12);
  if (j==19) return((6*t1*(v2-v3-2*z2+2*z3)+v1-v2+12*z2+4*z3)/16);
  if (j==20) return((v1+v2-2*v3+8*z3)/8);
  if (j==21) return(-(6*t2*(v1-v3+2*(z3-z1))-5*v1-v2+2*(3*v3-8*z3))/16);
  if (j==22) return(-(6*t2*(v1-v3+2*(z3-z1))-4*v1-v2+5*v3-12*z3)/12);
  if (j==23) return(-(6*t2*(v1-v3+2*(z3-z1))-4*v1-v2+5*v3-4*(z1+3*z3))/16);
  if (j==24) return(-(6*t2*(v1-v3+2*(z3-z1))+12*t3*(z1-z2)-4*v1-v2+5*v3-4*(5*z1+3*z3))/32);
  if (j==25) return((6*t1*(v2-v3-2*z2+2*z3)+12*t3*(z2-z1)+v1-2*v2+v3+4*(3*z1+5*z2))/32);
  if (j==26) return((3*t1*(v2-v3-2*z2+2*z3)-3*t2*(v1-v3+2*(z3-z1))+3*v1-3*v3+6*z2+10*z3)/16);
  if (j==27) return(-(6*t2*(v1-v3+2*(z3-z1))+12*t3*(z1-z2)-4*v1-v2+5*v3-12*(z1+z3))/24);
  if (j==28) return((6*t1*(v2-v3-2*z2+2*z3)+12*t3*(z2-z1)+v1-2*v2+v3+12*z1+12*z2)/24);
  if (j==29) return((6*t1*(v2-v3-2*z2+2*z3)-6*t2*(v1-v3+2*(z3-z1))+5*v1-v2-4*v3+12*z2+12*z3)/24);
  if (j==30) return((6*t1*(v2-v3-2*z2+2*z3)-6*t2*(v1-v3+2*(z3-z1))+12*t3*(z2-z1)+5*v1-v2-4*v3+12*(z1+z2+z3))/36);
 }
  /* security */
 return(0.);
}

double compute_d_j_pwsb_null(j)
int j;
{
 /* Computation of dj(0) */
 {
  if (j==0) return(z1);
  if (j==1) return(z1);
  if (j==2) return(z1/2+z2/2);
  if (j==3) return(z2);
  if (j==4) return(z2);
  if (j==5) return(z2);
  if (j==6) return(z2/2+z3/2);
  if (j==7) return(z3);
  if (j==8) return(z3);
  if (j==9) return(z3);
  if (j==10) return(z1/2+z3/2);
  if (j==11) return(z1);
  if (j==12) return(z1);
  if (j==13) return((4*z1-3*t3*(z1-z2))/4);
  if (j==14) return(t3*(z2-z1)+z1);
  if (j==15) return(-(3*t3*(z1-z2)-3*z1-z2)/4);
  if (j==16) return(z2);
  if (j==17) return((4*z2-3*t1*(z2-z3))/4);
  if (j==18) return(t1*(z3-z2)+z2);
  if (j==19) return(-(3*t1*(z2-z3)-3*z2-z3)/4);
  if (j==20) return(z3);
  if (j==21) return((3*t2*(z1-z3)+4*z3)/4);
  if (j==22) return(t2*(z1-z3)+z3);
  if (j==23) return((3*t2*(z1-z3)+z1+3*z3)/4);
  if (j==24) return((3*t2*(z1-z3)+3*t3*(z2-z1)+5*z1+3*z3)/8);
  if (j==25) return(-(3*t1*(z2-z3)+3*t3*(z1-z2)-3*z1-5*z2)/8);
  if (j==26) return(-(3*t1*(z2-z3)+3*t2*(z3-z1)-3*z2-5*z3)/8);
  if (j==27) return((t2*(z1-z3)+t3*(z2-z1)+z1+z3)/2);
  if (j==28) return(-(t1*(z2-z3)+t3*(z1-z2)-z1-z2)/2);
  if (j==29) return(-(t1*(z2-z3)+t2*(z3-z1)-z2-z3)/2);
  if (j==30) return(-(t1*(z2-z3)+t2*(z3-z1)+t3*(z1-z2)-z1-z2-z3)/3);
 }
  /* security */
  return(0.);
}











