/*
** FUNZ_Q18.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
**
** REVISION DATE: May, 1999  
**
** MODULES CALLED: NONE
**
** -----------------------------------------------------------------------
** BACKGROUND
**
** Let (xi,yi,qi), i=0,...,N-1 be N points in the space R^3. 
** Let us consider their projections (xi,yi) onto the plane z=0 and 
** let T be a triangulation of these points. 
** A triangulation of them is the set of the triangles having
** these points and only them as vertices, so that two distinct triangles 
** intersect at most in a common edge or in a common vertex. 
** 
** The aim of this program is the computation of the value of the functional
**
** phi(v)=   sum  [ Integral[ (Qxx^2 + 2*Qxy^2 + Qyy^2  ) dx dy ] ]     
**          t in T      t      
** 
** where Q is a Q18 polynomial patch, restricted to the triangle t.
**
**
** REFERENCES
**
**  M. Morandi Cecchi, S. De Marchi, D. Fasoli,
**     "A Package for Representing C^1 Interpolating Surfaces: 
**      Application to the Lagoon of Venice's Bed."
**      Numer. Algorithms, 1999.
**
**  D. Fasoli,
**      Laurea's Thesis, University of Padua, 1995 (in Italian). 
**
**  R.E. Barnhill and G. Farin:
**    "C^1 quintic interpolation over triangles: 
**     Two explicit representations"
**     Internat. J. Numer. Methods Engrg., 17(12):1763-1778, 1981.
**
** ------------------------------------------------------------------
**
** SYNOPSIS: funz_q18 file1 file2 file3
**
** 'file1' is the data file made up of 3-dimensional points. 
** The coordinates of the points may be separated by any 
** combination of spaces, tabs and carriage returns, 
** even though we recommend to use a row per point for clearness. 
** 
** 'file2' is the triangulation data file in the format generated by
** the program 'do_trian'. The file has three sections. 
** The first, starting with a single capital letter `C' in the first row, 
** lists in subsequent blocks separated by -1 the index of
** a node and those of its neighbours in a counterclockwise
** order. 
** Note: this program uses only this first section and the heading of the
** second section, that is a single capital letter `B' in the row.
**
** 'file3' contains the derivatives data in the following format.  
** The values in a same row refer to the same vertex of the triangulation.
** The first column is the x-derivative, the second one is the y-derivative,
** the third one is the xx-derivative, the fourth one is the xy-derivative
** and the last one is the yy-derivative.
**
** The output is the value of the functional of the Q18 interpolation function
** for the points specified in file1, the triangulation specified in file2
** and the derivatives values specified in file3.
** The format of the output is handled by the two constants WIDTH and PREC
** that are the width of the output field, and the number of decimal digits.
**        
** The program supports at most MAX_NUM_NODI (#DEFINE constant) input points.
** If more points are given, it exits by printing a warning message.
** 

** -------------------------------------------------------------------------------
** SUBROUTINES AND FUNCTIONS:
**
** double sum_cpt3_prod(double b[],double c[])
**
** This function computes the sum of the control points of the product of
** two cubic triangular polynomial patches. Their control points are stored
** in the input arrays 'b' and 'c'.
** 
** The indexes 0 up to 9 are used instead of the corresponding 
** tridimensional indexes 'rst' (used by b_{rst} and c_{rst} ordered 
** first with respect to r and then with respect to s, 
** that is 003,012,021,030,102,111,120,201,210,300).
**
** In the body of the function, the symbol 'a' represent the product patch.
**
**
** --------------------------------------------------------------------------
*/

#include<stdio.h>
#include<stdlib.h>
#include<math.h>
#define MAX_NUM_NODI 100000
#define WIDTH 20   /* Format width of functional value */
#define PREC 10    /* Decimal precision width of functional value */

double sum_cpt3_prod();

/* Declarations for global variables */
double q1,q2,q3;
double e1x,e1y,e2x,e2y,e3x,e3y,t1,t2,t3;
double q1x,q2x,q3x,q1y,q2y,q3y,q1xx,q2xx,q3xx,q1xy,q2xy,q3xy,q1yy,q2yy,q3yy;

void main(argc,argv)
int argc;
char *argv[];
{
  FILE *infile, *trifile, *derfile;
  char base_buffer[22];
  /* Control points of Q18 polynomial patch labeled from 0 to 20
     (value 21 is used only to initialize IndT) */
  int IndT[6][6]={{0,1,2,3,4,5},       {6,7,8,9,10,21},
		  {11,12,13,14,21,21}, {15,16,17,21,21,21},
		  {18,19,21,21,21,21}, {20,21,21,21,21,21}};
  long int num_nodo,iv1,iv2,iv3,nt;
  double x[MAX_NUM_NODI],y[MAX_NUM_NODI],z[MAX_NUM_NODI],
	 zx[MAX_NUM_NODI],zy[MAX_NUM_NODI],
	 zxx[MAX_NUM_NODI],zxy[MAX_NUM_NODI],zyy[MAX_NUM_NODI];
  double d[21];
  double x1,y1,x2,y2,x3,y3,At;
  double c01,c11,c31,c41,c02,c12,c32,c42,c03,c13,c33,c43;
  int i,r,s;
  double cpt3[3][10],temp_calc,phi;

 infile=NULL; trifile=NULL; derfile=NULL;
 if (argc==1)
   {
    printf("\nStart the program with three parameters:\n");
    printf("First parameter: data file name.\n");
    printf("Second parameter: triangulation file name.\n");
    printf("Third parameter: derivative file name.\n");
    exit(1);
   }
 /* Reading */
 infile=fopen(argv[1],"r");
 if (infile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[1]);
    exit(1);
   }
 if (argc==2)
   {
    printf("Missing triangulation file.\n");
    exit(1);
   }
 if (argc==3)
   {
    printf("Missing derivative file.\n");
    exit(1);
   }
 /* opening derivative file */
 derfile=fopen(argv[3],"r");
 if (derfile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[3]);
    exit(1);
   }
 /* data reading */
 printf("Wait for data reading.\n");
 num_nodo=0;
 while (!feof(infile))
 {
  if (num_nodo<MAX_NUM_NODI)
   {
    fscanf(infile," %lf %lf %lf ",&x[num_nodo],&y[num_nodo],&z[num_nodo]);
      /* spaces are essential part of the input format string */
    if (feof(derfile))
    {
     printf("Derivative file %s in a no correct format.\n",argv[3]);
     exit(1);
    }
    fscanf(derfile," %lf %lf %lf %lf %lf ",
	   &zx[num_nodo],&zy[num_nodo],
	   &zxx[num_nodo],&zxy[num_nodo],&zyy[num_nodo]);
      /* spaces are essential part of the input format string */
    num_nodo=num_nodo+1;
   }
   else
   {
    printf("Too many input data.\n");
    exit(1);
   }
 }
 if (num_nodo<3)
    {
     printf("I need at least three input points.\n");
     exit(1);
    }
 fclose(infile);
 fclose(derfile);
 /* end of data reading */

 /* opening triangulation file */
 trifile=fopen(argv[2],"r");
 if (trifile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[2]);
    exit(1);
   }
 if (fgetc(trifile)!='C')
   {
    printf("Triangulation file %s in a no correct format.\n",argv[2]);
    exit(1);
   }
 else fgetc(trifile); /* carriage return */

 /* MAIN LOOP OF FUNCTIONAL COMPUTATION OVER TRIANGLES */
 printf("Starting functional computation for each triangle.\n");
 phi=0.;
 fscanf(trifile,"%ld ",&iv1);
 nt=1;
 /* Triangle counter */
 do
   {
    fscanf(trifile,"%ld %ld ",&iv2,&iv3);
    while(iv3!=-1)
     {
      if ((iv1<iv2)&&(iv1<iv3))
	{/* New triangle: beginning computations */
	 x1=x[iv1]; x2=x[iv2]; x3=x[iv3];
	 y1=y[iv1]; y2=y[iv2]; y3=y[iv3];
	 q1=z[iv1]; q2=z[iv2]; q3=z[iv3];
	 q1x=zx[iv1]; q2x=zx[iv2]; q3x=zx[iv3];
	 q1y=zy[iv1]; q2y=zy[iv2]; q3y=zy[iv3];
	 q1xx=zxx[iv1]; q2xx=zxx[iv2]; q3xx=zxx[iv3];
	 q1xy=zxy[iv1]; q2xy=zxy[iv2]; q3xy=zxy[iv3];
	 q1yy=zyy[iv1]; q2yy=zyy[iv2]; q3yy=zyy[iv3];
	 e1x=x3-x2; e1y=y3-y2; e2x=x1-x3; e2y=y1-y3; e3x=x2-x1; e3y=y2-y1;
	 At=fabs((x2*y3-x3*y2+x3*y1-x1*y3+x1*y2-x2*y1)/2.);
	 t1=-(e1x*e3x+e1y*e3y)/(pow(e1x,2.)+pow(e1y,2.));
	 t2=-(e2x*e1x+e2y*e1y)/(pow(e2x,2.)+pow(e2y,2.));
	 t3=-(e3x*e2x+e3y*e2y)/(pow(e3x,2.)+pow(e3y,2.));
	 /* Computing control points for the interpolation patch */

	 d[20]=q1;
	 d[5]=q2;
	 d[0]=q3;
	 d[19]= (e3x*q1x+e3y*q1y)/5.+q1;
	 d[18]=-(e2x*q1x+e2y*q1y)/5.+q1;
	 d[10]=-(e3x*q2x+e3y*q2y)/5.+q2;
	 d[4]= +(e1x*q2x+e1y*q2y)/5.+q2;
	 d[1]= -(e1x*q3x+e1y*q3y)/5.+q3;
	 d[6]= +(e2x*q3x+e2y*q3y)/5.+q3;
	 d[17]=(pow(e3x,2)*q1xx+2*e3x*e3y*q1xy+pow(e3y,2)*q1yy)/20.
	       +2*(d[19])-q1;
	 d[14]=(pow(e3x,2)*q2xx+2*e3x*e3y*q2xy+pow(e3y,2)*q2yy)/20.
	       +2*(d[10])-q2;
	 d[3]=(pow(e1x,2)*q2xx+2*e1x*e1y*q2xy+pow(e1y,2)*q2yy)/20.
	      +2*(d[4])-q2;
	 d[2]=(pow(e1x,2)*q3xx+2*e1x*e1y*q3xy+pow(e1y,2)*q3yy)/20.
	      +2*(d[1])-q3;
	 d[11]=(pow(e2x,2)*q3xx+2*e2x*e2y*q3xy+pow(e2y,2)*q3yy)/20.
	       +2*(d[6])-q3;
	 d[15]=(pow(e2x,2)*q1xx+2*e2x*e2y*q1xy+pow(e2y,2)*q1yy)/20.
	       +2*(d[18])-q1;
	 d[16]=-(e3x*e2x*q1xx+(e3x*e2y+e3y*e2x)*q1xy+e3y*e2y*q1yy)/20.
	       +(d[19])+(d[18])-q1;
	 d[9]=-(e3x*e1x*q2xx+(e3x*e1y+e3y*e1x)*q2xy+e3y*e1y*q2yy)/20.
	      +(d[10])+(d[4])-q2;
	 d[7]=-(e1x*e2x*q3xx+(e1x*e2y+e1y*e2x)*q3xy+e1y*e2y*q3yy)/20.
	      +(d[1])+(d[6])-q3;
	 c01=d[10]-(1-t1)*q2-t1*d[4];
	 c11=d[9]-(1-t1)*d[4]-t1*d[3];
	 c31=d[7]-(1-t1)*d[2]-t1*d[1];
	 c41=d[6]-(1-t1)*d[1]-t1*q3;
	 c02=-(1-t2)*q3-t2*d[6]+d[1];
	 c12=-(1-t2)*d[6]-t2*d[11]+d[7];
	 c32=-(1-t2)*d[15]-t2*d[18]+d[16];
	 c42=-(1-t2)*d[18]-t2*q1+d[19];
	 c03=-t3*d[19]+d[18]-(1-t3)*q1;
	 c13=-t3*d[17]+d[16]-(1-t3)*d[19];
	 c33=-t3*d[10]+d[9]-(1-t3)*d[14];
	 c43=-t3*q2+d[4]-(1-t3)*d[10];
	 d[8]=(-c01+4*c11+4*c31-c41)/6.+(1-t1)*d[3]+t1*d[2];
	 d[12]=(-c02+4*c12+4*c32-c42)/6.+(1-t2)*d[11]+t2*d[15];
	 d[13]=(-c03+4*c13+4*c33-c43)/6.+(1-t3)*d[17]+t3*d[14];
	 /* Computing second derivatives for the interpolation function
	    (their control points) */
	 i=0; for (r=0;r<=3;++r) for (s=0;s<=3-r;++s)
	 {
	  temp_calc=d[IndT[r+2][s]]*e1y*e1y;
	  temp_calc+=d[IndT[r][s+2]]*e2y*e2y;
	  temp_calc+=d[IndT[r][s]]*e3y*e3y;
	  temp_calc+=d[IndT[r+1][s+1]]*2*e1y*e2y;
	  temp_calc+=d[IndT[r+1][s]]*2*e1y*e3y;
	  temp_calc+=d[IndT[r][s+1]]*2*e2y*e3y;
	  cpt3[0][i]=temp_calc;

	  temp_calc=-d[IndT[r+2][s]]*e1y*e1x;
	  temp_calc-=d[IndT[r][s+2]]*e2y*e2x;
	  temp_calc-=d[IndT[r][s]]*e3y*e3x;
	  temp_calc-=d[IndT[r+1][s+1]]*(e1y*e2x+e2y*e1x);
	  temp_calc-=d[IndT[r+1][s]]*(e1y*e3x+e3y*e1x);
	  temp_calc-=d[IndT[r][s+1]]*(e2y*e3x+e3y*e2x);
	  cpt3[1][i]=temp_calc;

	  temp_calc=d[IndT[r+2][s]]*e1x*e1x;
	  temp_calc+=d[IndT[r][s+2]]*e2x*e2x;
	  temp_calc+=d[IndT[r][s]]*e3x*e3x;
	  temp_calc+=d[IndT[r+1][s+1]]*2*e1x*e2x;
	  temp_calc+=d[IndT[r+1][s]]*2*e1x*e3x;
	  temp_calc+=d[IndT[r][s+1]]*2*e2x*e3x;
	  cpt3[2][i]=temp_calc;
	  i++;
	 }  /* end for over s (and over r) */
	 /* Computation of the functional over the triangle and updating */
	 temp_calc=sum_cpt3_prod(cpt3[0],cpt3[0])+
		   2.*sum_cpt3_prod(cpt3[1],cpt3[1])+
		   sum_cpt3_prod(cpt3[2],cpt3[2]);
	 /* 
             The constant 25./(28.*pow(At,3)) comes from
            (5./At^2)^2 * (2*At/((6+1)(6+2))) */

	 temp_calc=temp_calc*25./(28.*pow(At,3));
	 phi=phi+temp_calc;

	 if (nt%100==0)
	   printf("Done %ld triangles.\n",nt); nt++;
	} /* End of current triangle */
      iv2=iv3;
      fscanf(trifile,"%ld ",&iv3);
     }  /* End of the reading of the row in triangulation file */
    fscanf(trifile,"%s ",base_buffer);
    iv1=atol(base_buffer);
   }  /* End of reading triangulation file at the loop exit */
 while (base_buffer[0]!='B');
 fclose(trifile);
 printf("Functional value: %*.*f\n",WIDTH,PREC,phi);
}

double sum_cpt3_prod(b,c)
double b[],c[];
{
 double temp=0.;

 /* adding a_006 */
 temp=temp+b[0]*c[0];
 /* adding a_015 */
 temp=temp+(b[0]*c[1]+b[1]*c[0])/2;
 /* adding a_024 */
 temp=temp+(b[0]*c[2]+3*b[1]*c[1]+b[2]*c[0])/5;
 /* adding a_033 */
 temp=temp+(b[0]*c[3]+9*b[1]*c[2]+9*b[2]*c[1]+b[3]*c[0])/20;
 /* adding a_042 */
 temp=temp+(b[1]*c[3]+3*b[2]*c[2]+b[3]*c[1])/5;
 /* adding a_051 */
 temp=temp+(b[2]*c[3]+b[3]*c[2])/2;
 /* adding a_060 */
 temp=temp+b[3]*c[3];
 /* adding a_105 */
 temp=temp+(b[0]*c[4]+b[4]*c[0])/2;
 /* adding a_114 */
 temp=temp+(2*b[0]*c[5]+3*b[1]*c[4]+3*b[4]*c[1]+2*b[5]*c[0])/10;
 /* adding a_123 */
 temp=temp+(b[0]*c[6]+6*b[1]*c[5]+3*b[2]*c[4]
	    +3*b[4]*c[2]+6*b[5]*c[1]+b[6]*c[0])/20;
 /* adding a_132 */
 temp=temp+(3*b[1]*c[6]+6*b[2]*c[5]+b[3]*c[4]
	    +b[4]*c[3]+3*(2*b[5]*c[2]+b[6]*c[1]))/20;
 /* adding a_141 */
 temp=temp+(3*b[2]*c[6]+2*b[3]*c[5]+2*b[5]*c[3]+3*b[6]*c[2])/10;
 /* adding a_150 */
 temp=temp+(b[3]*c[6]+b[6]*c[3])/2;
 /* adding a_204 */
 temp=temp+(b[0]*c[7]+3*b[4]*c[4]+b[7]*c[0])/5;
 /* adding a_213 */
 temp=temp+(b[0]*c[8]+3*b[1]*c[7]+6*b[4]*c[5]
	    +6*b[5]*c[4]+3*b[7]*c[1]+b[8]*c[0])/20;
 /* adding a_222 */
 temp=temp+(b[1]*c[8]+b[2]*c[7]+b[4]*c[6]+4*b[5]*c[5]
	    +b[6]*c[4]+b[7]*c[2]+b[8]*c[1])/10;
 /* adding a_231 */
 temp=temp+(3*b[2]*c[8]+b[3]*c[7]+6*b[5]*c[6]
	    +6*b[6]*c[5]+b[7]*c[3]+3*b[8]*c[2])/20;
 /* adding a_240 */
 temp=temp+(b[3]*c[8]+3*b[6]*c[6]+b[8]*c[3])/5;
 /* adding a_303 */
 temp=temp+(b[0]*c[9]+b[9]*c[0]+9*(b[4]*c[7]+b[7]*c[4]))/20;
 /* adding a_312 */
 temp=temp+(b[9]*c[1]+b[1]*c[9]+3*(b[4]*c[8]
	    +2*b[5]*c[7]+2*b[7]*c[5]+b[8]*c[4]))/20;
 /* adding a_321 */
 temp=temp+(b[9]*c[2]+b[2]*c[9]+3*(2*b[5]*c[8]
	    +b[6]*c[7]+b[7]*c[6]+2*b[8]*c[5]))/20;
 /* adding a_330 */
 temp=temp+(b[9]*c[3]+b[3]*c[9]+9*(b[6]*c[8]+b[8]*c[6]))/20;
 /* adding a_402 */
 temp=temp+(b[9]*c[4]+b[4]*c[9]+3*b[7]*c[7])/5;
 /* adding a_411 */
 temp=temp+(2*b[9]*c[5]+2*b[5]*c[9]+3*(b[7]*c[8]+b[8]*c[7]))/10;
 /* adding a_420 */
 temp=temp+(b[9]*c[6]+b[6]*c[9]+3*b[8]*c[8])/5;
 /* adding a_501 */
 temp=temp+(b[9]*c[7]+b[7]*c[9])/2;
 /* adding a_510 */
 temp=temp+(b[9]*c[8]+b[8]*c[9])/2;
 /* adding a_600 */
 temp=temp+b[9]*c[9];
 return(temp);
}



