/*
** FUNZ_CLT.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
**
** REVISION DATE: May, 1999  
**
** MODULES CALLED: NONE
**
** ------------------------------------------------------------------------
** BACKGROUND
**
** Let (xi,yi,qi), i=0,...,N-1 be N points in the space R^3. 
** Let us consider their projections (xi,yi) onto the plane z=0 and 
** let T be a triangulation of these points. 
** A triangulation of them is the set of the triangles 
** having these points and only them as vertices, so that two distinct 
** triangles intersect at most in a common edge or in a common vertex. 
** 
** The aim of this program is the computation of the value of the functional
**
** phi(v)=   sum  [ Integral[ (Qxx^2 + 2*Qxy^2 + Qyy^2  ) dx dy ] ]     
**          t in T      t      
** 
** where Q is the Clough-Tocher polynomial patch, restricted to the 
** triangle t.
**
**
** REFERENCES
**
**  M. Morandi Cecchi, S. De Marchi, D. Fasoli,
**      "A Package for Representing C^1 Interpolating Surfaces: 
**       Application to the Lagoon of Venice's Bed."
**       Numer. Algorithms, 1999.
**
**  D. Fasoli,
**      Laurea's Thesis, University of Padua, 1995 (in Italian).
**
**  G. Farin:
**     "Triangular Bernstein-Bezier patches",
**      Comput. Aided Geom. Design,3:83-127, 1986.
** 
**
** --------------------------------------------------------------------------
**
** SYNOPSIS: funz_clt file1 file2 file3
**
** 'file1' is the data file made up of 3-dimensional points. 
** The coordinates of the points may be separated by any 
** combination of spaces, tabs and carriage returns, 
** even though we recommend to use a row per point for clearness. 
** 
** 'file2' is the triangulation data file in the format generated by
** the program 'do_trian'. The file has three sections. 
** The first, starting with a single capital letter `C' in the first row, 
** lists in subsequent blocks separated by -1 the index of
** a node and those of its neighbours in a counterclockwise
** order. 
** Note: this program uses only this first section and the heading of the
** second section, that is a single capital letter `B' in the row.
**
** 'file3' contains the derivatives data in the following format. 
** The values in a same row refer to the same vertex of the triangulation.
** The first column is the x-derivative, the second one is the y-derivative,
** the third, the fourth and the fifth ones contain the value 0.
**
** The output is the value of the functional of the Clough-Tocher 
** interpolation function at the points specified in file1, 
** the triangulation specified in file2 and the derivatives values 
** specified in file3.
** The format of the output is handled by the two constants WIDTH and PREC
** that are the width of the output field, and the number of decimal digits.
**        
** The program supports at most MAX_NUM_NODI (#DEFINE constant) input points.
** If more points are given, it exits by printing a warning message.
** 
** ------------------------------------------------------------------
**
** SUBROUTINES AND FUNCTIONS
**
** double sum_cpt1_prod(double b[], double c[])
**
** This function computes the sum of the control points of the product
** of two linear triangular polynomial patches. Their control points
** are stored in the input arrays 'b' and 'c'. 
**
** The indexes 0,1,2 are used instead of the corresponding tridimensional 
** indexes 'rst' (used as b_{rst} and c_{rst}, ordered first with respect 
** to r and then with respect to s, that is 001,010,100). 
**
** In the body of the function, the symbol 'a' represent the product patch.
**
**
** -------------------------------------------------------------------------------
*/

#include<stdio.h>
#include<stdlib.h>
#include<math.h>
#define MAX_NUM_NODI 100000
#define WIDTH 20   /* Format output width of functional value */
#define PREC 10    /* Decimal precision with of functional value */

double sum_cpt1_prod();

/* Declarations for global variables */
double q1,q2,q3;
double u1,v1,u2,v2,u3,v3,t1,t2,t3;
double zu1,zu2,zu3,zv1,zv2,zv3;

void main(argc,argv)
int argc;
char *argv[];
{
  FILE *infile, *trifile, *derfile;
  char base_buffer[20];
  long int num_nodo,iv1,iv2,iv3,nt;
  double x[MAX_NUM_NODI],y[MAX_NUM_NODI],z[MAX_NUM_NODI],
	 zx[MAX_NUM_NODI],zy[MAX_NUM_NODI];
  double d[20];
  /* Control points of Clough-Tocher element
     labeled from 1 to 19 (0 index is not used) */
  double u4,v4,At,trash;
  int i,r,s;
  double cpt1[9][3],temp_calc,phi;
  int IndT1[4][4]={{2,9,4,1},{8,13,5,0},{17,16,0,0},{19,0,0,0}};
  int IndT2[4][4]={{3,12,7,2},{11,14,8,0},{18,17,0,0},{19,0,0,0}};
  int IndT3[4][4]={{1,6,10,3},{5,15,11,0},{16,18,0,0},{19,0,0,0}};

 infile=NULL; trifile=NULL; derfile=NULL;
 if (argc==1)
   {
    printf("\nStart the program with three parameters:\n");
    printf("First parameter: data file name.\n");
    printf("Second parameter: triangulation file name.\n");
    printf("Third parameter: derivative file name.\n");
    exit(1);
   }
 /* Reading */
 infile=fopen(argv[1],"r");
 if (infile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[1]);
    exit(1);
   }
 if (argc==2)
   {
    printf("Missing triangulation file.\n");
    exit(1);
   }
 if (argc==3)
   {
    printf("Missing derivative file.\n");
    exit(1);
   }
 /* opening derivative */
 derfile=fopen(argv[3],"r");
 if (derfile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[3]);
    exit(1);
   }
 /* data reading */
 printf("Wait for data reading.\n");
 num_nodo=0;
 while (!feof(infile))
 {
  if (num_nodo<MAX_NUM_NODI)
   {
    fscanf(infile," %lf %lf %lf ",&x[num_nodo],&y[num_nodo],&z[num_nodo]);
      /* spaces are essential part of the input format string */
    if (feof(derfile))
    {
     printf("Derivative file %s in a no correct format.\n",argv[3]);
     exit(1);
    }
    fscanf(derfile," %lf %lf %lf %lf %lf ",
	   &zx[num_nodo],&zy[num_nodo],&trash,&trash,&trash);
      /* spaces are essential part of the input format string */
    num_nodo=num_nodo+1;
   }
   else
   {
    printf("Too many input data.\n");
    exit(1);
   }
 }
 if (num_nodo<3)
    {
     printf("I need at least three input points.\n");
     exit(1);
    }
 fclose(infile);
 fclose(derfile);
 /* end of data reading */

 /* opening triangulation file */
 trifile=fopen(argv[2],"r");
 if (trifile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[2]);
    exit(1);
   }
 if (fgetc(trifile)!='C')
   {
    printf("Triangulation file %s in a no correct format.\n",argv[2]);
    exit(1);
   }
 else fgetc(trifile); /* carriage return */

 /* MAIN LOOP OF FUNCTIONAL COMPUTATION OVER TRIANGLES */
 printf("Starting functional computation for each triangle.\n");
 phi=0.;
 fscanf(trifile,"%ld ",&iv1);
 nt=1;
 /* Counter of current triangle */
 do
   {
    fscanf(trifile,"%ld %ld ",&iv2,&iv3);
    while(iv3!=-1)
     {
      if ((iv1<iv2)&&(iv1<iv3))
	{/* New triangle: beginning computations */
	 u1=x[iv1]; u2=x[iv2]; u3=x[iv3];
	 v1=y[iv1]; v2=y[iv2]; v3=y[iv3];
	 q1=z[iv1]; q2=z[iv2]; q3=z[iv3];
	 zu1=zx[iv1]; zu2=zx[iv2]; zu3=zx[iv3];
	 zv1=zy[iv1]; zv2=zy[iv2]; zv3=zy[iv3];
	 u4=(u1+u2+u3)/3.; v4=(v1+v2+v3)/3.;
	 At=fabs((u2*v3-u3*v2+u3*v1-u1*v3+u1*v2-u2*v1)/2.);
	 t1= ((u4-u1)*(u2-u1)+(v4-v1)*(v2-v1))/
	     (pow((u2-u1),2.)+pow((v2-v1),2.));
	 t2= ((u4-u2)*(u3-u2)+(v4-v2)*(v3-v2))/
	     (pow((u3-u2),2.)+pow((v3-v2),2.));
	 t3= ((u4-u3)*(u1-u3)+(v4-v3)*(v1-v3))/
	     (pow((u1-u3),2.)+pow((v1-v3),2.));
	 /* Computation of control points for the interpolation function */
	 d[1]=q1;
	 d[2]=q2;
	 d[3]=q3;
	 d[4]=((u2-u1)*zu1+(v2-v1)*zv1)/3.+q1;
	 d[5]=((u4-u1)*zu1+(v4-v1)*zv1)/3.+q1;
	 d[6]=((u3-u1)*zu1+(v3-v1)*zv1)/3.+q1;
	 d[7]=((u3-u2)*zu2+(v3-v2)*zv2)/3.+q2;
	 d[8]=((u4-u2)*zu2+(v4-v2)*zv2)/3.+q2;
	 d[9]=((u1-u2)*zu2+(v1-v2)*zv2)/3.+q2;
	 d[10]=((u1-u3)*zu3+(v1-v3)*zv3)/3.+q3;
	 d[11]=((u4-u3)*zu3+(v4-v3)*zv3)/3.+q3;
	 d[12]=((u2-u3)*zu3+(v2-v3)*zv3)/3.+q3;
	 d[13]=(d[5]+d[8]+(t1-1)*d[1]+(2-3*t1)*d[4]+(3*t1-1)*d[9]
		-t1*d[2])/2.;
	 d[14]=(d[8]+d[11]+(t2-1)*d[2]+(2-3*t2)*d[7]+(3*t2-1)*d[12]
		-t2*d[3])/2.;
	 d[15]=(d[11]+d[5]+(t3-1)*d[3]+(2-3*t3)*d[10]+(3*t3-1)*d[6]
		-t3*d[1])/2.;
	 d[16]=(d[15]+d[5]+d[13])/3.;
	 d[17]=(d[13]+d[8]+d[14])/3.;
	 d[18]=(d[14]+d[11]+d[15])/3.;
	 d[19]=(d[18]+d[16]+d[17])/3.;
	 /* Computation of second derivatives for the interpolation function
	    (its control points) */
	 i=0; for (r=0;r<=1;++r) for (s=0;s<=1-r;++s)
	 {
	  /* First triangle V4 V1 V2 */
	  temp_calc= d[IndT1[r+2][s]]*(v2-v1)*(v2-v1);
	  temp_calc+=d[IndT1[r][s+2]]*(v4-v2)*(v4-v2);
	  temp_calc+=d[IndT1[r][s]]*(v1-v4)*(v1-v4);
	  temp_calc+=d[IndT1[r+1][s+1]]*2*(v2-v1)*(v4-v2);
	  temp_calc+=d[IndT1[r+1][s]]*2*(v2-v1)*(v1-v4);
	  temp_calc+=d[IndT1[r][s+1]]*2*(v4-v2)*(v1-v4);
	  cpt1[0][i]=temp_calc;

	  temp_calc=-d[IndT1[r+2][s]]*(v2-v1)*(u2-u1);
	  temp_calc-=d[IndT1[r][s+2]]*(v4-v2)*(u4-u2);
	  temp_calc-=d[IndT1[r][s]]*(v1-v4)*(u1-u4);
	  temp_calc-=d[IndT1[r+1][s+1]]*
		     ((v2-v1)*(u4-u2)+(v4-v2)*(u2-u1));
	  temp_calc-=d[IndT1[r+1][s]]*
		     ((v2-v1)*(u1-u4)+(v1-v4)*(u2-u1));
	  temp_calc-=d[IndT1[r][s+1]]*
		     ((v4-v2)*(u1-u4)+(v1-v4)*(u4-u2));
	  cpt1[1][i]=temp_calc;

	  temp_calc=d[IndT1[r+2][s]]*(u2-u1)*(u2-u1);
	  temp_calc+=d[IndT1[r][s+2]]*(u4-u2)*(u4-u2);
	  temp_calc+=d[IndT1[r][s]]*(u1-u4)*(u1-u4);
	  temp_calc+=d[IndT1[r+1][s+1]]*2*(u2-u1)*(u4-u2);
	  temp_calc+=d[IndT1[r+1][s]]*2*(u2-u1)*(u1-u4);
	  temp_calc+=d[IndT1[r][s+1]]*2*(u4-u2)*(u1-u4);
	  cpt1[2][i]=temp_calc;

	  /* Second triangle V4 V2 V3 */
	  temp_calc=d[IndT2[r+2][s]]*(v3-v2)*(v3-v2);
	  temp_calc+=d[IndT2[r][s+2]]*(v4-v3)*(v4-v3);
	  temp_calc+=d[IndT2[r][s]]*(v2-v4)*(v2-v4);
	  temp_calc+=d[IndT2[r+1][s+1]]*2*(v3-v2)*(v4-v3);
	  temp_calc+=d[IndT2[r+1][s]]*2*(v3-v2)*(v2-v4);
	  temp_calc+=d[IndT2[r][s+1]]*2*(v4-v3)*(v2-v4);
	  cpt1[3][i]=temp_calc;

	  temp_calc=-d[IndT2[r+2][s]]*(v3-v2)*(u3-u2);
	  temp_calc-=d[IndT2[r][s+2]]*(v4-v3)*(u4-u3);
	  temp_calc-=d[IndT2[r][s]]*(v2-v4)*(u2-u4);
	  temp_calc-=d[IndT2[r+1][s+1]]*
		     ((v3-v2)*(u4-u3)+(v4-v3)*(u3-u2));
	  temp_calc-=d[IndT2[r+1][s]]*
		     ((v3-v2)*(u2-u4)+(v2-v4)*(u3-u2));
	  temp_calc-=d[IndT2[r][s+1]]*
		     ((v4-v3)*(u2-u4)+(v2-v4)*(u4-u3));
	  cpt1[4][i]=temp_calc;

	  temp_calc=d[IndT2[r+2][s]]*(u3-u2)*(u3-u2);
	  temp_calc+=d[IndT2[r][s+2]]*(u4-u3)*(u4-u3);
	  temp_calc+=d[IndT2[r][s]]*(u2-u4)*(u2-u4);
	  temp_calc+=d[IndT2[r+1][s+1]]*2*(u3-u2)*(u4-u3);
	  temp_calc+=d[IndT2[r+1][s]]*2*(u3-u2)*(u2-u4);
	  temp_calc+=d[IndT2[r][s+1]]*2*(u4-u3)*(u2-u4);
	  cpt1[5][i]=temp_calc;

	  /* Third triangle V4 V3 V1 */
	  temp_calc=d[IndT3[r+2][s]]*(v1-v3)*(v1-v3);
	  temp_calc+=d[IndT3[r][s+2]]*(v4-v1)*(v4-v1);
	  temp_calc+=d[IndT3[r][s]]*(v3-v4)*(v3-v4);
	  temp_calc+=d[IndT3[r+1][s+1]]*2*(v1-v3)*(v4-v1);
	  temp_calc+=d[IndT3[r+1][s]]*2*(v1-v3)*(v3-v4);
	  temp_calc+=d[IndT3[r][s+1]]*2*(v4-v1)*(v3-v4);
	  cpt1[6][i]=temp_calc;

	  temp_calc=-d[IndT3[r+2][s]]*(v1-v3)*(u1-u3);
	  temp_calc-=d[IndT3[r][s+2]]*(v4-v1)*(u4-u1);
	  temp_calc-=d[IndT3[r][s]]*(v3-v4)*(u3-u4);
	  temp_calc-=d[IndT3[r+1][s+1]]*
		     ((v1-v3)*(u4-u1)+(v4-v1)*(u1-u3));
	  temp_calc-=d[IndT3[r+1][s]]*
		     ((v1-v3)*(u3-u4)+(v3-v4)*(u1-u3));
	  temp_calc-=d[IndT3[r][s+1]]*
		     ((v4-v1)*(u3-u4)+(v3-v4)*(u4-u1));
	  cpt1[7][i]=temp_calc;

	  temp_calc=d[IndT3[r+2][s]]*(u1-u3)*(u1-u3);
	  temp_calc+=d[IndT3[r][s+2]]*(u4-u1)*(u4-u1);
	  temp_calc+=d[IndT3[r][s]]*(u3-u4)*(u3-u4);
	  temp_calc+=d[IndT3[r+1][s+1]]*2*(u1-u3)*(u4-u1);
	  temp_calc+=d[IndT3[r+1][s]]*2*(u1-u3)*(u3-u4);
	  temp_calc+=d[IndT3[r][s+1]]*2*(u4-u1)*(u3-u4);
	  cpt1[8][i]=temp_calc;

	  i++;
	 }  /* end for over s (and over r) */
	 /* Functional computation over the triangle and updating */
	 temp_calc=(sum_cpt1_prod(cpt1[0],cpt1[0])+
		    2.*sum_cpt1_prod(cpt1[1],cpt1[1])+
		    sum_cpt1_prod(cpt1[2],cpt1[2]));
	 temp_calc+=(sum_cpt1_prod(cpt1[3],cpt1[3])+
		     2.*sum_cpt1_prod(cpt1[4],cpt1[4])+
		     sum_cpt1_prod(cpt1[5],cpt1[5]));
	 temp_calc+=(sum_cpt1_prod(cpt1[6],cpt1[6])+
		     2.*sum_cpt1_prod(cpt1[7],cpt1[7])+
		     sum_cpt1_prod(cpt1[8],cpt1[8]));
	 /* The constant 3./(8. *pow(At/3.,3.)
            comes from (6/4 a^2)^2 * (2a / (2+1)(2+2) ) 
            where a=At/3 */
         temp_calc=temp_calc*3./(8. *pow(At/3.,3.));
	 phi=phi+temp_calc;

	 if (nt%100==0)
	   printf("Done %ld triangles.\n",nt); nt++;
	} /* End of current triangle */
      iv2=iv3;
      fscanf(trifile,"%ld ",&iv3);
     }  /* End of the reading of the row in the triangulation file */
    fscanf(trifile,"%s ",base_buffer);
    iv1=atol(base_buffer);
   }  /* End of reading triangulation file at the loop exit */
 while (base_buffer[0]!='B');
 fclose(trifile);
 printf("Functional value: %*.*f\n",WIDTH,PREC,phi);
}

double sum_cpt1_prod(b,c)
double b[],c[];
{
 double temp=0.;

 /* adding a_002 */
 temp=temp+b[0]*c[0];
 /* adding a_011 */
 temp=temp+(b[1]*c[0]+b[0]*c[1])/2;
 /* adding a_020 */
 temp=temp+b[1]*c[1];
 /* adding a_101 */
 temp=temp+(b[2]*c[0]+b[0]*c[2])/2;
 /* adding a_110 */
 temp=temp+(b[2]*c[1]+b[1]*c[2])/2;
 /* addubg a_200 */
 temp=temp+b[2]*c[2];
 return(temp);
}


