/*
** DO_TRIAN.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
**
** REVISION DATE: May, 1999  
**
** MODULES CALLED: NONE 
**
** --------------------------------------------------------------------------
** BACKGROUND
**  
** Let (xi,yi,zi) i=0,...,N-1 be N points in the space R^3. Let us 
** consider their projections (xi,yi) onto the plane z=0.
** T is a triangulation of these points, if it is a set of triangles having
** all these points, and only them, as vertices so that two
** distinct triangles intersect at most in a common edge or in a common
** vertex.
** The domain of the triangulation, that is the area of the plane covered by
** its triangles, is usually the convex hull of the points, also called nodes
** of the triangulation.
** Among all possible triangulations of these points, the Delaunay triangulation
** T is characterized by the following property: 
** all circumcircles of the triangles of T do not contain any node inside.
** This triangulation, that is not uniquely defined if there are
** at least two triangles with the same circumcircle, 
** maximizes the smallest angle among all the angles
** of all triangles of the triangulations of the same points (over the same 
** domain).   
** 
**
** METHODOLOGY
**
** The Delaunay triangulation is built by induction over the number of nodes.
**
** A) The triangle having P0,P1,P2 for vertices is a Delaunay triangulation
**    of these points (if they are not collinear ones).
**
** B) Let D_(n-1) be a Delaunay triangulation of the points P0,P1, ..., P(n-1).
**
**    There may be three different cases.
**
**      1) P(n) is in the convex hull of P0,P1, ..., P(n-1).
**         Let Sigma be the set of all triangles of D_(n-1) whose 
**         circumscribed circle contains the point P(n) inside.
**         Let {b}_j (j=1,...,k) the set of edges of triangles of Sigma, not 
**         common to two triangles of Sigma.
**         Then
** 
**             D_(n) = ( D_(n-1) \ Sigma )   U    U  ( P(n)b(j) ) 
**                                            j=1,...,k 
**
**      2) P(n) is not in any circumscribed circle of a triangle of D_(n-1).
**         Let {e}_j (j=1,...,h) the set of edges of the boundary of the convex
**         hull of the points P0,P1, ..., P(n-1) whose lines passing through 
**         separe P(n) from the set of other points.
**         Then
**
**             D_(n) = D_(n-1)   U      U     ( P(n)e(j) )
**                                 j=1,...,h
**
**      3) P(n) is not in the convex hull of P0,P1, ..., P(n-1), but in
**         at least one circumscribed circle to a triangle of D_(n-1).
**         Let Sigma be the set of all triangles of D_(n-1) whose 
**         circumscribed circle contains the point P(n) inside.
**         Let {b}_j (j=1,...,k) the set of edges of triangles of Sigma, not 
**         common to two triangles of Sigma, that don't separe P(n) from
**         the set of other vertices of triangles of Sigma.
**         Let {e}_j (j=1,...,h) the set of edges of the boundary of the convex
**         hull of the points P0,P1, ..., P(n-1) whose lines passing through 
**         separe P(n) from the set of other points, that don't belong
**         to triangles of Sigma.
**         Then
**
**        D_(n) = (D_(n-1)\ Sigma) U   U  (P(n)b(j)) U    U  (P(n)e(i))  
**                                  j=1,...,k          i=1,...,h
**
** 
** REFERENCES
** 
**  J.-J. Risler, 
**     "Mathematical Methods for C.A.D."
**     Cambridge University Press, 1992.
** 
** --------------------------------------------------------------------------
** SYNOPSIS:  do_trian file1 input_flag [file2] [-change]
**
** 'file1' is the data file made up of 2-dimensional points
** or 3-dimensional points. 
** The coordinates of the points may be separated by any 
** combination of spaces, tabs and carriage returns, 
** even though it is recommended to use a row for each point. 
** The program works properly only if there are no points repeated 
** twice or more.
** 
** 'input_flag' may be '-i2' or '-i3': use '-i2' for x-y coordinates in 
** file1 or '-i3' for a x-y-z coordinates in file1 (in this case the 
** z-coordinate is not considered by the construction algorithm).
**
** 'file2' is the Delaunay triangulation data file generated by this program
** over the convex hull of the input points, also known as nodes
** (omit it to use standard output). This file has three sections.
** -- 
** The first, starting with a single capital letter C in the first row, 
** lists in subsequent blocks separated by -1 the index of
** a node and the indices of its neighbours in a counterclockwise
** order. If the node does not belong to the boundary of the convex hull,
** the first node in the list of its neighbours is repeated again before 
** the separating value -1.
** --
** The second section, starting with a single capital letter B alone in a row,
** lists the edges belonging to the boundary of the convex hull of all nodes
** in a counterclockwise direction using the following 5-columns matrix format:
** <column 1>: index of starting node of the edge
** <column 2>: index of ending node of the edge
** <column 3>: index of the first vertex of the triangle ownering the edge
** <column 4>: index of the second vertex of the triangle ownering the edge
** <column 5>: index of the third vertex of the triangle ownering the edge.
** --
** The third section, starting with a single capital letter T alone in a row,
** list all the triangles of the triangulation built using the following
** 12-columns matrix format:
** <column  1>: index of the first vertex of the triangle, V1
** <column  2>: index of the second vertex of the triangle, V2
** <column  3>: index of the third vertex of the triangle, V3
** <column  4>: index of the first vertex of the neighbouring triangle along
**              the edge V1V2, or -1 if this triangle does not exist.
** <column  5>: index of the second vertex of the neighbouring triangle along
**              the edge V1V2, or -1 if this triangle does not exist.
** <column  6>: index of the third vertex of the neighbouring triangle along
**              the edge V1V2, or -1 if this triangle does not exist.
** <column  7>: index of the first vertex of the neighbouring triangle along
**              the edge V2V3, or -1 if this triangle does not exist.
** <column  8>: index of the second vertex of the neighbouring triangle along
**              the edge V2V3, or -1 if this triangle does not exist.
** <column  9>: index of the third vertex of the neighbouring triangle along
**              the edge V2V3, or -1 if this triangle does not exist.
** <column 10>: index of the first vertex of the neighbouring triangle along
**              the edge V3V1, or -1 if this triangle does not exist.
** <column 11>: index of the second vertex of the neighbouring triangle along
**              the edge V3V1, or -1 if this triangle does not exist.
** <column 12>: index of the third vertex of the neighbouring triangle along
**              the edge V3V1, or -1 if this triangle does not exist.        
**
** If the fourth parameter '-change' is given the program does not build the
** triangulation writing its representation in the file 'file2', but instead
** it reads the triangulation stored in the file 'file2' and provides the user
** with an interactive shell supporting the following commands:
** <add>    : Add a new node
** <ptlist> : View the current nodes and their indices.
** <exit>   : Save in the specified files the new data and the
**            new triangulation and exits from the interactive shell.
**
**
** ----------------------------------------------------------------------------
** FUNCTIONS AND PROCEDURES DESCRIPTION:
**
** int MakeDelaunay(int* num_nodi, double[] x,double[] y,double[] z,
**                  struct triang_list** Delaunay,struct edge_list** convex_hull,
**                  int change_mode,FILE* readtrifile,int input_mode)
**
** The function, if change_mode==0, 
** computes the Delaunay triangulation
** for a set of *num_nodi DISTINCT 2D points (when input_mode==3 
** for 3D points the z-coordinate is ignored) whoose x,y coordinates are
** stored in double precision in the x,y arrays large enough to store
** them. 
** THE BEHAVIOUR OF THE FUNCTION IS NOT FORSEEN IF THERE ARE REPEATED POINTS.
** If change_mode==1 instead, 
** the function requires the reading of all the
** triangles listed in the triangulation file together with the informations
** about their neighbouring ones and the reading of the convex hull from 
** the same triangulation file. Then, it supplies the user with an
** interactive shell for the insertion of new nodes without rebuild 
** all the triangulation.
** 
** In any case the function returns *num_nodi, the updated number of nodes.
**
** The triangulation is returned in the list pointed by *Delaunay
** whose elements are triang_list structures. 
** If P is a pointer to such a structure:
** P->triangolo.v1, P->triangolo.v2, P->triangolo.v3 
** are the indexes of the points being the vertices of the triangle.
** P->confine_v1v2, P->confine_v2v3, P->confine_v3v1 
** are the pointers to the neighbouring triangles (null pointers if
** there isn't the neighbouring triangle along the triangle edge considered)
** P->next is the pointer to the next triangle in the list or 
** the null pointer if there are no more triangles.
** The set of all triangle edges belonging to the convex hull of the
** triangulation is returned in the list pointed by *convex_hull    
** whose elements are triang_list structures. 
** If P is a pointer to such a structure:
** P->bordo.v1, P->bordo.v2 are the indexes of the vertices of a triangle
** edge in the convex hull, sorted as the edge is seen 
** visiting the convex hull in a counterclockwise order.
** P->source_triang is the pointer to the unique triangle to which
** the triangle edge in the convex hull belongs.
** P->next is the pointer to the next triangle edge in the convex hull
** or the null pointer if there are no more such triangle edges.
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
**
** void strong_reading(struct triang_list** TriangEndPt,
**                     struct edge_list** ConvexHEndPt, FILE* trifile)
** 
** This function creates the list of triangles described by the file pointed
** by trifile with the value 'Delaunay_Triang' for the 'tipo' field and
** sets the fields 'confine_v1v2', 'confine_v2v3', 'confine_v3v1' 
** for each triangle of the list.
** The function creates the list of edges belonging to the convex hull with
** the correct value for the field 'source_triang'.
**
** Before performing these operations *TriangEndPt points to a dummy
** triangle (not really used) such that (*TriangEndPt)->next==NULL and
** *ConvexHEndPt points to a dummy edge (not really used) such that
** (*ConvexHEndPt)->next==NULL.
**
** When these operations have been performed the first triangle in the list
** of triangles is pointed by the local variable
** TriangBeginPt and the last one by *TriangEndPt
** (returned to the caller), while the last edge in the list of edges is
** pointed by *ConvexHEndPt (returned to the caller too).
** The ends of the lists are visible from the properties:
** (*TriangEndPt)->next == NULL 
** (*ConvexHEndPt)->next== NULL.
**
** The triangulation file opened for reading is closed. 
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>
** 
** void AddMutualNeighbour(struct triang_list* Pt1,int Sr1,
**                         struct triang_list* Pt2,int Sr2)
**
** Let Sr1,Sr2 numbers among 0,1,2 which state the rotations in the
** counterclockwise direction needed to transform the triangles pointed by
** Pt1 and Pt2 into their canonical form.
** Let us assume for a triangle having for vertices V1,V2,V3 
** that V1+1=V2, V2+1=V3 and V3+1=V1.
** Then, this function mutually updates the pointers to the neighbouring
** triangles along the edge V3+2Sr1,V1+2Sr1 of the first triangle and the
** edge V1+2Sr2,V2+2Sr2 of the second triangle. 
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
**
** int circoscrivi(double x1,double y1,double x2,double y2,
**                 double x3,double y3,double* xc,double* yc,double* r)
**
** It computes the circle passing through the points (x1,y1), (x2,y2), (x3,y3).
** 
** The function returns
**     i)  the center (xc,yc) and  the radius 'r' of the circle  
**         if the function result is 'circoscrivi==0';
**    ii)  if 'r==NULL' the computation of the radius is not performed;
**   iii)  if 'circoscrivi==1' the three points are aligned or nearby aligned 
**            ones (there will be a denominator smaller than EPS_ALLIN) 
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>><<
**
** geometric_side which_side(double xp,double yp,double xi,double yi,
**                           double xj,double yj)
**
** This function considers the vector (xj,yj)-(xi,yi) and the line passing
** through and returns 'dentro_weak' if the point (xp,yp) lies in the closed
** half-plane on the left of the line, while it returns 'fuori_strong' if
** the point (xp,yp) lies in the opened half-plane on the right of the line.
**
** The function uses the sign of the third component of the cross vector 
** assuming it is zero if its magnitude is smaller then EPS_ALLIN.  
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
**
** geometric_side inside_convex_hull(double xtest,double ytest,
**                                   struct edge_list* convex_hull_LocPt,
**                                   double[] x,double[] y)
**
** This function considers the test point (xtest,ytest), the arrays 'x' and
** 'y' containing the coordinates of nodes, and the pointer 'convex_hull_LocPt'
** to the list of edges making up the convex hull of the nodes.
**
** It returns 'dentro_weak' if the test point is inside the convex hull and
** 'fuori_strong' if it is outside.
** The convex hull is considered to be closed.
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>><<
**
** geometric_side outside_hole(double xtest,double ytest,
**                             double xc,double yc,double r)
**
** This function considers the test point (xtest,ytest) and the circle
** having the point (xc,yc) for center and 'r' for radius.
**
** The function returns 'dentro_weak' if the test point lies outside the
** open circle (hole) and fuori_strong if it lies inside.
** (The computation is done with reference to the region outside the hole
**  and therefore the values returned seem to be inverted even though they
**  aren't).
** The boundary of the circle of radius 'r' 
** is determined with a tolerance
** of EPS_ALLIN over the circle having for radius 1. 
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
**
** void Delaunay_to_sigma(long int n,double[] x,double[] y,
**                       struct triang_list* DelaunayLoc,long int* numero_azioni)
**
** This function signs with the 'sigma_type' value the 'tipo' field of all
** the triangles where the node (x[n],y[n]) lies inside the circumcircle,
** starting from the first cell of the list of triangles pointed by
** DelaunayLoc.
**
** The function returns (by reference)
** the 'numero_azioni' parameter which is the
** number of signed triangles (0 if there aren't).
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
**
** void evaluate_edges(struct triang_list* DelaunayLoc,long int n,
**                     double[] x,double[] y, int mode,
**                     struct edge_list* b_e_edges_Loc,long int* critical_arrival)
**
** NOTE for the caller:
** Output is returned in the list pointed by b_e_edges_Loc->next according
** to the following convention: 
** the first cell of the lists is a dummy one (it is not really used). 
**
** CASE MODE=1
** Applied if the point (x[n],y[n]) lies inside the closed convex hull 
** of the points (x[0],y[0]), ... , (x[n-1],y[n-1]).
** --
** The function visits the list pointed by DelaunayLoc and selects the
** triangles where the 'tipo' field has the 'sigma_triang' value, that are
** those where the opened circumcircle has got the point (x[n],y[n]) inside
** (test over the 'tipo' field).
** Then, for each selected triangle, the function detects 'b_edges' sides,
** that are triangle edges not shared by two 'sigma' triangles, testing 
** if the neighbouring triangle does not exist or if it exists and it is
** a 'sigma' triangle. Then the function adds these 'b_edges' sides to
** the list pointed by 'b_e_edges_Loc->next' so that the point (x[n],y[n])
** lies on the 'dentro_weak' region (see 'which-side' function) with 
** respect to the oriented 'b_edge' side ('b_edges' sides are the edges
** of a closed region having the point (x[n],y[n]) inside). 
** The adding of the edges is performed at the tail of the list, it is not 
** performed, or it is performed at the head of the list as described
** in the called function 'update_b_edges'.
** Here, we say only that that '*critical_arrival' is, if different from -1,
** the index of the ending vertex of a 'b_edge' side belonging to a
** 'sigma' triangle (only to one), such that the point (x[n],y[n]) lies 
** over the edge itself. This parameter is returned back so that the
** caller function may consider not to add the triangle spanning
** from the vertex (x[n],y[n]) to this edge, and may be able to
** deal with the fields relative to neighbouring triangles agreeing. 
** The head of the list shall be really used 
** by the function 'add_new_triangle'.
** The function updates the field 'source_triang' of added 'b_edges' with
** the pointer to the neighbouring triangles (they are 'Delaunay_triang' ones
** instead of 'sigma_triang' ones) if they exist, or with NULL otherwise.
**
** CASE MODE=2
** Applied if the point (x[n],y[n]) is outside the union 
** of all the circumcircles 
** of the triangles belonging to the Delaunay triangulation 
** of the points (x[0],y[0]), ..., (x[n-1],y[n-1]).
** --
** The function uses the listed pointed by Delaunay_Loc to detect the edges
** belonging to the boundary of the convex hull.
** Such an edge is tested to see if it divides the point (x[n],y[n]) from
** the points (x[0],y[0]),...,(x[n-1],y[n-1]) so that the point (x[n],y[n])
** lies outside the closed half-plane starting from the edge and containing
** the other points. 
** If so, the edge (a 'e_edge' side) is added to the list pointed by
** b_e_edges_Loc->next together with the pointer to the ownering triangle.
** The adding is done at the tail of the list, apart from the case where
** the point (x[n],y[n]) lies in the 'dentro_weak' region with respect to
** the vector (x[o_v_old],y[o_v_old])-(x[n],y[n]), if o_v_old is the index of
** the starting vertex of the edge currently at the head of the list.
** In this latter case, the adding is done at the head of the list, that is
** in a new cell pointed by b_e_edges_Loc_Saved->next.
** For the dividing test, the function uses the 'which_side' function accepting 
** the test if the point (x[n],y[n]) lies in the 'fuori_strong' region with
** respect to the edge currently being examined. 
** Once the edge has passed the test, it is added to the list (at the tail or at
** the head of the same) with the orientation suggested by the test itself.
**
** CASE MODE=3
** Applied if the point (x[n],y[n]) is outside the closed convex hull of
** the points (x[0],y[0]), ... , (x[n-1],y[n-1]), but it lies inside at least
** one circumcircle among those of the triangles belonging to the
** Delaunay triangulation of the points (x[0],y[0]), ... , (x[n-1],y[n-1]).
** This case mixes the previous ones.
** --
** Here 'b_edges' sides must satisfy the further condition not to divide the
** point (x[n],y[n]) from the third vertex of the unique 'sigma' triangle
** to which the candidate 'b_edge' side belongs (when MODE=1 this condition is
** always verified).
** This condition means that this third vertex lies on the same open
** half-plane of the point (x[n],y[n]) starting from the line passing
** trough the candidate 'b_edge' side. 
** The condition is tested by the 
** called function 'update_b_edges' (which calls in turn the function 
** 'which_side') verifying if the point (x[n],y[n]) is in the 'fuori_strong'
** region with respect to the candidate 'b_edge' side.  
** Once the edge has passed the test, it is added to the list (always at the
** tail of the same) with the orientation suggested by the test itself, which
** is opposite to the one of the case labeled by MODE=1, since here the
** point (x[n],y[n]) is outside the convex hull of the other points. 
** --
** Here 'e_edges' sides must satisfy the further condition not to be edges
** of 'sigma' triangles (when MODE=2 this condition is always verified).
** --
** The function mixes the procedures described by the cases labeled by
** MODE=1 and MODE=2 updated with the new requirements, detects 'b_edges' and
** 'e_edges' sides and adds them to the list pointed by b_e_edges_Loc,
** at the tail or at the head of the same as specified in the cases
** labeled by MODE=2, apart from the storing of 'b_edges' sides shown above.
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>
**
** void update_b_edges(struct triang_list* neighbourPt, int mode,
**                     struct edge_list* b_e_edges_Orig,
**                     struct edge_list** b_edges_Loc,
**	    	       long int n, double[] x,double[] y,
**                     long int o_v,long int a_v,long int oa_v,
**                     long int* critical_arrival) 
**
** This function works properly only if MODE=1 or MODE=3.
** --
** CASE MODE=1
** Let us consider a 'sigma' triangle having for vertices the points
** labeled by the input parameters 'o_v','a_v' and 'oa_v'.
** Then 'o_v' and 'a_v' indices are eventually swapped to guarantee that
** the third vertex of the 'sigma' triangle, whose index is supplied by
** the parameter 'oa_v', is on the 'dentro_weak' region with respect to the
** vector (x[a_v],y[a_v])-(x[o_v],y[o_v]). The test is done using this
** vertex instead of the point (x[n],y[n]) directly, since this latter one
** may belong to the line passing through the vector. In the following we
** shall refer to this situation as to the case where *critical_arrival==a_v.
** The function updates the list pointed by *b_e_edges_Loc adding the
** edge starting from (x[o_v],y[o_v]) and ending to (x[a_v],y[a_v]) if the
** triangle pointed by 'neighbourPt' has not got the value 'sigma_triang' for
** the 'tipo' field (or if it doesn't exist) together with the setting
** for the 'source_triang' field supplied by 'neighbourPt' too.
** Once the edge has passed the test, it may be added to the list at the tail 
** of the same, it may not to be added to the list, or it may be added to the
** list at the head of the same. 
** If *critical_arrival=-1 the edge is added at the tail of the list. 
** If the point (x[n],y[n]) lies on the convex hull boundary (but not over
** another point: they are distinct), that is if (x[a_v],y[a_v]) is in the
** 'dentro_weak' region with respect to the vector (x[n],y[n])-(x[o_v],y[o_v])
**  (the 'dentro_weak' region is the same of (x[oa_v],y[oa_v]) by means of
**  the eventual initial swapping), the edge is not added to the list.
** In this latter case *critical_arrival is set to 'a_v' and, if 'a_v' is 
** already in the list pointed by b_e_edges_Orig->next as first vertex of an
** edge, this edge is swapped with the one currently at the head of the list.
** If (*critical_arrival)>-1, supposing 'a_v_old' its value, the edge is
** added at the head of the list (in the cell pointed by 'b_e_edges_Orig->next')
** if a_v_old==o_v.  
** All this procedure has the aim to avoid adding a wrong triangle whose
** vertices (x[n],y[n]), (x[o_v],y[o_v]) and (x[a_v],y[a_v]) are (nearby)
** collinear ones.
** --
** CASE MODE=3
** Parameters 'o_v' and 'a_v' are eventually swapped to guarantee that the
** point (x[n],y[n]) is on the 'fuori_strong' region with respect to the 
** vector (x[a_v],y[a_v])-(x[o_v],y[o_v]). (This time this requirement may
** be achieved testing directly the point (x[n],y[n]) since it is outside
** the convex hull).
** Then, besides all the requirements described in the case labeled by MODE=1,
** in order to update the list (with the edge and with its ownering triangle)
** we require that the point (x[oa_v],y[oa_v]) is in the 'fuori_strong' region
** with respect to the vector (x[a_v],y[a_v])-(x[o_v],y[o_v]).
** If these requirements succeed, the adding of the 'b_edge' side is done
** at the tail of the list, apart from the case 
** where the point (x[o_v],y[o_v]) is in the 'dentro_weak' region 
** with respect to the vector (x[o_v_old],y[o_v_old])-(x[n],y[n]), if we label
** with 'o_v_old' the index of the first vertex of the edge currently being
** at the head of the list. In this latter case the adding is done at the
** head of the list (in the cell pointed by 'b_e_edges_Orig->next') 
** --
** IN BOTH CASES
** the pointer '*b_edges_Loc' is returned back to the caller function such that
** (*b_edges_Loc)->next==NULL.
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>><<
**
** void update_e_edges(struct triang_list* triangPt,
**                     struct edge_list* b_e_edges_Orig,
**                     struct edge_list** e_edges_Loc,
**                     long int n,double[] x,double[] y,
**                     long int o_v,long int a_v)
**
** This function works properly only
** if the point (x[n],y[n]) is outside the union 
** of all the circumcircles 
** of the triangles belonging to the Delaunay triangulation 
** of the points (x[0],y[0]), ..., (x[n-1],y[n-1]).
**
** Let us consider a triangle having for vertices the points
** labeled by the input parameters 'o_v','a_v' and the point (x[n],y[n]).
** Then 'o_v' and 'a_v' indices are swapped if (x[n],y[n]) is
** in the 'dentro_weak' region with respect to the vector
** (x[a_v],y[a_v])-(x[o_v],y[o_v]), so that the point (x[n],y[n]) is in
** the 'fuori_strong' region; if this does not happen the point (x[n],y[n]) is
** collinear with the vector (x[a_v],y[a_v])-(x[o_v],y[o_v]), but it is
** outside the vector itself (and outside the convex hull): in this case 
** the preliminary test is failed.
** Otherwise, the function tests the edge 
** (x[a_v],y[a_v])-(x[o_v],y[o_v]) so see if all points 
** (x[0],y[0]), ..., (x[n-1],y[n-1]) belong to the 'dentro_weak' region
** with respect to the oriented edge.
** Once the test has been passed, the function adds the edge to the list
** pointed by (*e_edges_Loc) with the 'source_triang' field set to 'triangPt'.
** The adding of the 'e_edge' side is done at the tail of the list, apart from
** the case where the point (x[o_v],y[o_v]) is in the 'dentro_weak' region
** with respect to the vector (x[o_v_old],y[o_v_old])-(x[n],y[n]), if 
** 'o_v_old' is the index of the first vertex of the edge currently being
** at the head of the list: in this latter case the adding is done at the 
** head of the list (in the cell pointed by 'b_e_edges_Orig->next').
** --
** the pointer '*e_edges_Loc' is returned back to the caller function such that
** (*e_edges_Loc)->next==NULL.
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
**
** void remove_sigma_triang(struct triang_list* DelaunayLoc,
**                          struct triang_list** DelaunayPt)
**
** The function visits the list pointed by 'DelaunayLoc' and removes all
** the triangles having the 'sigma_triang' value for the 'tipo' field
** skipping the first cell of the list (it is not a triangle by convention)
** in order to avoid the removal at the head of the list.
** The function returns back the pointer '*DelaunayPt' to the last triangle,
** such that (*DelaunayPt)->next==NULL.
** The function does not work if DelaunayLoc==NULL (convention broken) or if
** DelaunayLoc-next==NULL (there are no triangles).
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
**
** void erase_neighbour(struct triang_list* triangPt,
**                      struct triang_list* neighbourPt)
**
** The function removes the 'neighbourPt' boundary (if it is not already NULL)
** of the triangle pointed by 'triangPt' making the assumption that anyway
** there cannot exist two equal boundaries.
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
** void add_new_triangle(struct triang_list** DelaunayPt,
**                       struct edge_list** b_e_edges_Loc,
**                       int mode,long intn,long int* critical_arrival)
**
** The function visits the list pointed by 'b_e_edges_Loc->next' (skipping
** the first cell), considers the corresponding edges 
** (x[a_v],y[a_v])-(x[o_v],y[o_v]) and creates a list of triangles having
** for vertices v1=(x[n],y[n]), v2=(x[o_v],y[o_v]),v3=(x[a_v],y[a_v]) and
** 'Delaunay_triang' for the value of the 'tipo' field.
** However, the order of such triangles is not the same of that of the edges
** visited, but the triangle having for vertices v1new,v2new,v3new follows
** the triangle having for vertices v1old,v2old,v3old if and only if
** v1new=v2old=(xnew,ynew) and v3old=v2new; anyway, the first of these triangles
** is the one corresponding to the first 'b_e_edge' side of the list.
** Then the function mutually updates the boundaries of new triangles with
** the ones of old triangles getting the boundary informations from the
** 'source_triang' fields of the edges of 'b_e_edges_Loc' list.
** Note that the removal of 'sigma' triangles, applied before the call to
** this function, does not mean that some of the triangles pointed by
** the 'source_triang' fields do not exist, since both 'b_edges' sides
** and 'e_edges' sides have their own 'source_triang' fields (when they are
** not NULL) pointing to triangles having 'Delaunay_triang' 
** for the value of the 'tipo' field.
** Then (in all cases labeled by mode==1,2,3) the function mutually updates 
** the boundaries 'v3old_v1old' and 'v1new_v2new' for each couple (old,new)
** of consecutive triangles added.
** Then, if mode==1, the boundary 'v1_v2' of the first triangle added is
** mutually updated with the boundary 'v3_v1' of the last triangle added if
** (*critical_arrival=-1); these boundaries are instead set to NULL if
** mode==1 and (*critical_arrival)>-1 and anyway if mode=2 or mode=3.
** The function returns back the pointer 'DelaunayPt' to the last triangle
** such that DelaunayPt->next==NULL.
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
**
** void remake_convex_hull(struct triang_list* DelaunayLoc,
**                         struct edge_list* convex_hull_Loc,
**                         double[] x,double[] y)
**
** The function visits the list pointed by DelaunayLoc (from the first cell),
** considers the edges of the boundary of the convex hull together with 
** their ownering triangles, and adds them at the tail of the list pointed by
** convex_hull_Loc->next (skipping the first cell of the list).
** The storing of the indices of the edges vertices is done such that the 
** third point of the triangle (the one different from the edge vertices) is
** in the 'dentro_weak' region with respect to the oriented edge.
**  
** x,y : the arrays containing points coordinates.
**
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
**
** void update_convex_hull (struct triang_list* triangPt,
**                          struct edge_list** convex_hull_Loc,
**                          double[] x,double[] y,
**                          long int o_v,long int a_v,long int oa_v)
**
** The function creates an edge starting from (x[o_v],y[o_v]) and ending
** to (x[a_v],y[a_v]), eventually swaps 'o_v' and 'a_v' if the point
** (x[oa_v],y[oa_v]) is in the 'fuori_strong' region with respect to the
** edge created, and adds it to the list pointed by '(*convex_hull_Loc)->next' 
** with the 'source_triang' field set to 'triangPt'.
** The function returns back 'convex_hull_Loc' such that 
** (*convex_hull_Loc)->next==NULL.
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
**
** void write_node_neighbours(struct triang_list* triangulation,
**                            struct edge_list* convex_hull, long int node,
**                            FILE* id_file,double[] x,double[] y)
**
** The function visits the triangulation pointed by 'triangulation'
** (the first cell is a dummy one: it is not really used) in order to search
** for a triangle among whose vertices there is 'node' as a point. 
** Then it writes to the
** file labeled by 'id_file' the 'node' point followed 
** by all its neighbouring points
** (the ones connected to it with an edge of a triangle belonging to 
** 'triangulation') with the same order as we visit in the counterclockwise
** direction the polygon (even if not closed) made up by all the edges
** of the triangles having the 'node' points opposite to the edges theirselves.
** If the 'node' point belongs to the boundary of the convex hull, the first
** triangle considedered is the one ownering the edge for which the 'node'
** point is the first vertex, instead of the first triangle having 
** the 'node' point as a vertex in the sequence given by the triangulation.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#define EPS_ALLIN 1.e-13
#define MAX_NUM_NODI 100000

#define WIDTH 20   /* Format width of real numbers written to files */
#define PREC 10    /* Decimal precision width of real numbers
                      written to files */

typedef enum {dentro_weak, fuori_strong} geometric_side;
typedef enum {Delaunay_triang,sigma_triang} triang_type;
typedef struct {long int v1; long int v2;} edge;
typedef struct {long int v1; long int v2; long int v3;
		triang_type tipo;} triang;
struct edge_list {edge bordo;
		  struct triang_list *source_triang;
		  struct edge_list *next;};
struct triang_list {triang triangolo;
		    struct triang_list *confine_v1v2;
		    struct triang_list *confine_v2v3;
		    struct triang_list *confine_v3v1;
		    struct triang_list *next;};

/* base function */
int MakeDelaunay();

/* other functions */
int circoscrivi();
void Delaunay_to_sigma(),
     evaluate_edges(),update_e_edges(),update_b_edges(),
     remove_sigma_triang(),erase_neighbour(),add_new_triangle(),
     remake_convex_hull(),update_convex_hull(),
     write_node_neigbours(),strong_reading(),AddMutualNeighbour();
geometric_side which_side(),inside_convex_hull(),outside_hole();

int MakeDelaunay(num_nodi,x,y,z,Delaunay,convex_hull,change_mode,readtrifile,
		 input_mode)
long int *num_nodi;
int change_mode; /* Direct call (0) or interactive environment(1) */
int input_mode;  /* 2D points (2) or 3D points (3) */
double x[],y[],z[];
struct triang_list *(*Delaunay);
struct edge_list *(*convex_hull);
FILE *readtrifile; /* Used only if change_mode==1 */
{
 long int n,i,j,k,num_new_nodi,num_azioni,critical_arrival;
 int insert_mode,convex_hull_verso,change_exit_flag;
 double xc,yc,r;
 struct triang_list *DelaunayPt,*DelaunayFirstTriPt;
 struct edge_list *b_e_edges,*convex_hull_Pt,*removing_edge_Pt;
 char change_command[10];

 if (circoscrivi(x[0],y[0],x[1],y[1],x[2],y[2],&xc,&yc,&r)==1)
 {
  return(1); 
  /* The first three nodes are collinear ones or nearby collinear ones */
 };
 /*
 ** INVARIANT PROPERTY: Every allocation of dynamical memory (malloc),
 ** apart from the first element in the list (it will never be used), is done
 ** using the 'next' field of the relative pointer that, after the allocation,
 ** goes forward; when the allocation is not done this field becomes the
 ** null pointer. 
 */

 /* The first element in the list pointed by (*Delaunay) is never used 
    to make easier the insertion and the removal of nodes */
  (*Delaunay)=malloc(sizeof(struct triang_list));
 if (*Delaunay==NULL) {printf("Not enough memory"); exit(1); }
 (*Delaunay)->triangolo.v1=-1;
 (*Delaunay)->triangolo.v2=-1;                    /* security */
 (*Delaunay)->triangolo.v3=-1;
 (*Delaunay)->triangolo.tipo=Delaunay_triang;
 (*Delaunay)->confine_v1v2=NULL;
 (*Delaunay)->confine_v2v3=NULL;                  /* security */
 (*Delaunay)->confine_v3v1=NULL;
 (*Delaunay)->next=NULL;

 /* The first element in the list pointed by (*convex_hull) is never used
    to make easier the insertion and the removal of triangle edges */
 (*convex_hull)=malloc(sizeof(struct edge_list));
 if (*convex_hull==NULL) {printf("Not enough memory"); exit(1); }
 (*convex_hull)->bordo.v1=-1;
 (*convex_hull)->bordo.v2=-1;  /* security */
 (*convex_hull)->source_triang=NULL;
 (*convex_hull)->next=NULL;

 /* The first element in the list pointed by b_e_edges is never used */
 b_e_edges=malloc(sizeof(struct edge_list));
 if (b_e_edges==NULL) {printf("Not enough memory"); exit(1); }
 b_e_edges->bordo.v1=-1;
 b_e_edges->bordo.v2=-1;  /* security */
 b_e_edges->source_triang=NULL;
 b_e_edges->next=NULL;

 if (change_mode==0)
 {
  /* BEGINNING TRIANGOLATION BUILDING */
  DelaunayPt=(*Delaunay);
  DelaunayPt->next=malloc(sizeof(struct triang_list));
  if (DelaunayPt->next==NULL) {printf("Not enough memory"); exit(1); }
  DelaunayPt=DelaunayPt->next;
  DelaunayPt->triangolo.v1=0;
  DelaunayPt->triangolo.v2=1;
  DelaunayPt->triangolo.v3=2;
  DelaunayPt->triangolo.tipo=Delaunay_triang;
  DelaunayPt->confine_v1v2=NULL;
  DelaunayPt->confine_v2v3=NULL;
  DelaunayPt->confine_v3v1=NULL;
  DelaunayPt->next=NULL;
  if (which_side(x[2],y[2],x[0],y[0],x[1],y[1])==dentro_weak)
    convex_hull_verso=0; /* senso antiorario */
   else convex_hull_verso=1; /* senso orario */
  convex_hull_Pt=(*convex_hull);
  convex_hull_Pt->next=malloc(sizeof(struct edge_list));
  if (convex_hull_Pt->next==NULL) {printf("Not enough memory"); exit(1);}
  convex_hull_Pt=convex_hull_Pt->next;
  if (convex_hull_verso==0)
    {
     convex_hull_Pt->bordo.v1=0; convex_hull_Pt->bordo.v2=1;
    }
   else
    {
     convex_hull_Pt->bordo.v1=1; convex_hull_Pt->bordo.v2=0;
    }
  convex_hull_Pt->source_triang=DelaunayPt; /* punto all'unico triangolo */
  convex_hull_Pt->next=malloc(sizeof(struct edge_list));
  if (convex_hull_Pt->next==NULL) {printf("Not enough memory"); exit(1);}
  convex_hull_Pt=convex_hull_Pt->next;
  if (convex_hull_verso==0)
    {
     convex_hull_Pt->bordo.v1=1; convex_hull_Pt->bordo.v2=2;
    }
   else
    {
     convex_hull_Pt->bordo.v1=2; convex_hull_Pt->bordo.v2=1;
    }
  convex_hull_Pt->source_triang=DelaunayPt;
  convex_hull_Pt->next=malloc(sizeof(struct edge_list));
  if (convex_hull_Pt->next==NULL) {printf("Not enough memory"); exit(1);}
  convex_hull_Pt=convex_hull_Pt->next;
  if (convex_hull_verso==0)
    {
     convex_hull_Pt->bordo.v1=2; convex_hull_Pt->bordo.v2=0;
    }
   else
    {
     convex_hull_Pt->bordo.v1=0; convex_hull_Pt->bordo.v2=2;
    }
  convex_hull_Pt->source_triang=DelaunayPt;
  convex_hull_Pt->next=NULL;
 } /* End of first triangle */
 /* INTERACTIVE SHELL */
 if (change_mode==1)
 {
  printf("\nWait for the complete reading of triangulation.\n");
  /* The initialization is done considering that there isn't yet any triangle
  ** and any triangle edge on the border */
  DelaunayPt=(*Delaunay);
  convex_hull_Pt=(*convex_hull);
  strong_reading(&DelaunayPt,&convex_hull_Pt,readtrifile);
  /* When the function exits DelaunayPt points to the latest triangle readed */ 
  /* and convex_hull_PT to the latest triangle edge in the convex hull readed */
  printf("\n\nUse the following commands:\n");
  printf("add:     Add a new node.\n");
  printf("ptlist:  View the current nodes and their indices.\n");
  printf("exit:    Save in the specified files the new data and the\n");
  printf("         new triangulation and exits from the interactive shell.\n\n");
 }
 change_exit_flag=0;
 do
 {
 num_new_nodi=0;
 /* This position is usefel for commands like 'exit' too, and not only for 
 ** the case: change_mode==0 */ 
  if (change_mode==1)
  {
   printf(">> ");
   scanf(" %s",change_command);
   if (strcmp(change_command,"add")==0)
   {
    if (*num_nodi==MAX_NUM_NODI) /* nodes labeled from 0 */
    {
     printf("Maximum number of nodes reached.\n");
     printf("I cannot add new nodes.\n\n");
     /* ritorno al menu */
    }
    else  /* *num_nodi < MAX_NUM_NODI */
    {
     switch (input_mode)
     {
      case 2: printf("Write the new node x-y.\n");
	      scanf("%lf %lf",&x[*num_nodi],&y[*num_nodi]);
	      break;
      case 3: printf("Write the new node x-y-z.\n");
	      scanf("%lf %lf %lf",&x[*num_nodi],&y[*num_nodi],&z[*num_nodi]);
	      break;
      }
     num_new_nodi=1;
    }
   }
   if (strcmp(change_command,"ptlist")==0)
   {
    printf("\n");
    for (n=0;n<*num_nodi;++n)
    {
     printf("%5ld %*.*f %*.*f\n",n,WIDTH,PREC,x[n],WIDTH,PREC,y[n]);
    }
    printf("\n");
   }
   if (strcmp(change_command,"exit")==0)
     change_exit_flag=1;
  }
  if (change_mode==0) n=3; else n=*num_nodi;
  if (  change_mode==0 ||
       ((change_mode==1) && strcmp(change_command,"add")==0) ||
       ((change_mode==1) && strcmp(change_command,"addfile")==0) )
      while (n<(*num_nodi)+num_new_nodi)
	/* 
           nodes labeled from 0 to *num_nodi-1 if change_mode==0,
           nodes labeled from *num_nodi to *num_nodi+num_new_nodi-1
           if change_mode==1 and change_command = "add" or "addfile"
	*/
  {
   insert_mode=3; /* Eventually, it shall be updated later */
   if (inside_convex_hull(x[n],y[n],(*convex_hull)->next,x,y)==dentro_weak)
     insert_mode=1;
   Delaunay_to_sigma(n,x,y,(*Delaunay)->next,&num_azioni);
   if (num_azioni==0) insert_mode=2;
   critical_arrival=-1;
   /* meaning: now the new point does not belong to the convex hull */
   evaluate_edges((*Delaunay)->next,n,x,y,insert_mode,b_e_edges,
		  &critical_arrival);
   remove_sigma_triang(*Delaunay,&DelaunayPt); /*Now DelaunayPt->next=NULL*/
   /* The edges of neighbours of the triangles removed have been updated*/
   add_new_triangle(&DelaunayPt,b_e_edges,insert_mode,n,&critical_arrival);
   /* Again DelaunayPt is such that DelunayPt->next=NULL */ 
   /*
   ** After this routine the 'tipo' field of new triangles is Delaunay_triang,
   ** they have been sorted using the key for the list and their fields
   ** relative to neighbouring ones have been mutually updated using
   ** 'b_e_edges' list data. The inner boundaries among new triangles have
   ** been updated too, together with the boundaries of the first and last
   ** triangle added. Now b_e_edges->next=NULL 
   */
   /* Now we compute again the convex hull (if insert_mode==1 this 
   ** operation is performed only to change the owner triangle of 
   ** triangle edges in the convex hull).
   */
   /* beginning removal of old convex hull */
   convex_hull_Pt=(*convex_hull)->next; /* Real convex hull starting entry */
   while (convex_hull_Pt!=NULL)
    {
     removing_edge_Pt=convex_hull_Pt;
     convex_hull_Pt=convex_hull_Pt->next;
     free(removing_edge_Pt);
    }
   (*convex_hull)->next=NULL;
   /* end of the removal for the old convex hull */
   /* The following statement does not use the 
      first cell in the triangulation and updates ownering triangles too */
   remake_convex_hull((*Delaunay)->next,*convex_hull,x,y);
   if ((n-change_mode*(*num_nodi))%50==49)
     printf("Completed %ld nodes\n",(n-change_mode*(*num_nodi))+1);
   n++;
  } /* end while loop oever n */
  if (change_mode==0) change_exit_flag=1;
    else *num_nodi=*num_nodi+num_new_nodi; /* New number of nodes */
 } while (change_exit_flag==0);
 /* Making invisible the convention used for the first unused cells */
 (*Delaunay)=(*Delaunay)->next;
 (*convex_hull)=(*convex_hull)->next;
 return 0; /* OK */
}

void strong_reading(TriangEndPt,ConvexHEndPt,trifile)
struct triang_list *(*TriangEndPt);
struct edge_list *(*ConvexHEndPt);
FILE *trifile;
{
 long int cont_nodi,iv1,iv2,iv3,temp_index,iv2_first;
 struct triang_list *CellFirstPt,*CellCurrentPt,*CellPrecPt,*TriangBeginPt;
 int CellFirstShiftRight, CellCurrentShiftRight, CellPrecShiftRight;
 int open_cella,flag,stop,first_triangle_flag=1;
 char base_buffer[20];

 /* LOOP OVER CELLS */
  fscanf(trifile,"%ld ",&iv1);
  do
   {
    fscanf(trifile,"%ld %ld ",&iv2,&iv3);
    /* Remembering 'v2' vertex if, later, the cell will appear to be closed */
    iv2_first=iv2;
    if ((iv1<iv2)&&(iv1<iv3))
    { /* Canonical form - First triangle */
     (*TriangEndPt)->next=malloc(sizeof(struct triang_list));
     if ((*TriangEndPt)->next==NULL)
       {printf("Not enough memory.\n"); exit(1);}
     (*TriangEndPt)=(*TriangEndPt)->next;
     (*TriangEndPt)->triangolo.v1=iv1;
     (*TriangEndPt)->triangolo.v2=iv2;
     (*TriangEndPt)->triangolo.v3=iv3;
     (*TriangEndPt)->triangolo.tipo=Delaunay_triang;
     (*TriangEndPt)->next=NULL;
     if (first_triangle_flag==1)
      {
       TriangBeginPt=*TriangEndPt;
       first_triangle_flag=0;
      }
     CellFirstPt=(*TriangEndPt);
     CellFirstShiftRight=0;
    }
    else
    {
     /* Conversion of the triangle into its canonical form 
        where iv1<iv2, iv2<iv3 */
     CellFirstShiftRight=0;
     if (iv1>iv2 || iv1>iv3)
     {
      temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
      CellFirstShiftRight++;
     }
     /* Completing the test */
     if (iv1>iv2 || iv1>iv3)
     {
      temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
      CellFirstShiftRight++;
     }
     CellFirstPt=TriangBeginPt;
     stop=0;
     do
     {
      if (CellFirstPt->triangolo.v1==iv1)
	if (CellFirstPt->triangolo.v2==iv2)
	  if (CellFirstPt->triangolo.v3==iv3) stop=1;
      if (stop==0) CellFirstPt=CellFirstPt->next;
     }
     while (stop==0 && CellFirstPt!=NULL);
     if (CellFirstPt==NULL)
     {
      printf("I cannot find a triangle.\n");
      exit(1);
     }
     /* Converting the triangle into its original form */
     if (CellFirstShiftRight==1) /* making shift_left */
     {
      temp_index=iv1; iv1=iv2; iv2=iv3; iv3=temp_index;
     }
     else
     if (CellFirstShiftRight==2) /*making a new shift_right (-2=1 mod 3)*/
     {
      temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
     }
    } /* end of first triangle not in canonical form */
    switch(CellFirstShiftRight)
    {
     case 0: CellFirstPt->confine_v1v2=NULL; break;
     case 1: CellFirstPt->confine_v2v3=NULL; break;
     case 2: CellFirstPt->confine_v3v1=NULL; break;
    }
    CellCurrentPt=CellFirstPt; CellCurrentShiftRight=CellFirstShiftRight;
    while(iv3!=-1) /* loop over the remaining nodes of the cell */
     {
      iv2=iv3;
      fscanf(trifile,"%ld ",&iv3);
      if (iv3!=-1)
      {
       CellPrecPt=CellCurrentPt;
       CellPrecShiftRight=CellCurrentShiftRight;
       if ((iv1<iv2)&&(iv1<iv3))
       { /* Canonical form */
	(*TriangEndPt)->next=malloc(sizeof(struct triang_list));
	if ((*TriangEndPt)->next==NULL)
	  {printf("Not enough memory.\n"); exit(1);}
	(*TriangEndPt)=(*TriangEndPt)->next;
	(*TriangEndPt)->triangolo.v1=iv1;
	(*TriangEndPt)->triangolo.v2=iv2;
	(*TriangEndPt)->triangolo.v3=iv3;
	(*TriangEndPt)->triangolo.tipo=Delaunay_triang;
	(*TriangEndPt)->next=NULL;
	CellCurrentPt=(*TriangEndPt);
	CellCurrentShiftRight=0;
       }
       else
       {
	/* Conversion of the triangle into its cononical form 
           where iv1<iv2, iv2<iv3 */
	CellCurrentShiftRight=0;
	if (iv1>iv2 || iv1>iv3)
	{
	 temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
	 CellCurrentShiftRight++;
	}
	/* Completing the test */
	if (iv1>iv2 || iv1>iv3)
	{
	 temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
	 CellCurrentShiftRight++;
	}
	CellCurrentPt=TriangBeginPt;
	stop=0;
	do
	{
	 if (CellCurrentPt->triangolo.v1==iv1)
	   if (CellCurrentPt->triangolo.v2==iv2)
	     if (CellCurrentPt->triangolo.v3==iv3) stop=1;
	 if (stop==0) CellCurrentPt=CellCurrentPt->next;
	}
	while (stop==0 && CellCurrentPt!=NULL);
	if (CellCurrentPt==NULL)
	{
	 printf("I cannot find a triangle.\n");
	 exit(1);
	}
	/* Converting the triangle into its original form */
	if (CellCurrentShiftRight==1) /* making shift_left */
	{
	 temp_index=iv1; iv1=iv2; iv2=iv3; iv3=temp_index;
	}
	else
	if (CellCurrentShiftRight==2)/*making a new shift_right(-2=1 mod 3)*/
	{
	 temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
	}
       } /* end of current triangle not in canonical form */
       AddMutualNeighbour(CellPrecPt,CellPrecShiftRight,
			  CellCurrentPt,CellCurrentShiftRight);
      } /* end of test over iv3 */
      else /* iv3==-1 and therefore iv2 is the last index of the cell */
      {
       if (iv2_first==iv2) /* closed cell */
	 AddMutualNeighbour(CellCurrentPt,CellCurrentShiftRight,
			    CellFirstPt,CellFirstShiftRight);
       else /* opened cell */
       {
	(*ConvexHEndPt)->next=malloc(sizeof(struct edge_list));
	if ((*ConvexHEndPt)->next==NULL)
	{
	 printf("Not enough memory.\n"); exit(1);
	}
	(*ConvexHEndPt)=(*ConvexHEndPt)->next;
	switch(CellCurrentShiftRight)
	{
	 case 0: CellCurrentPt->confine_v3v1=NULL;
		 (*ConvexHEndPt)->bordo.v1=CellCurrentPt->triangolo.v3;
		 (*ConvexHEndPt)->bordo.v2=CellCurrentPt->triangolo.v1;
		 (*ConvexHEndPt)->source_triang=CellCurrentPt;
		 break;
	 case 1: CellCurrentPt->confine_v1v2=NULL;
		 (*ConvexHEndPt)->bordo.v1=CellCurrentPt->triangolo.v1;
		 (*ConvexHEndPt)->bordo.v2=CellCurrentPt->triangolo.v2;
		 (*ConvexHEndPt)->source_triang=CellCurrentPt;
		 break;
	 case 2: CellCurrentPt->confine_v2v3=NULL;
		 (*ConvexHEndPt)->bordo.v1=CellCurrentPt->triangolo.v2;
		 (*ConvexHEndPt)->bordo.v2=CellCurrentPt->triangolo.v3;
		 (*ConvexHEndPt)->source_triang=CellCurrentPt;
		 break;
	}
	(*ConvexHEndPt)->next=NULL;
       }
      } /* end of the case where iv2 is the last index of the cell */
     } /* End of reading the row in the triangulation */
    fscanf(trifile,"%s ",base_buffer);
    iv1=atol(base_buffer);
   }  /* End of reading triangulation file at the loop exit */
  while (base_buffer[0]!='B');
 fclose(trifile);
}

void AddMutualNeighbour(Pt1,Sr1,Pt2,Sr2)
int Sr1,Sr2;
struct triang_list *Pt1, *Pt2;
{
 switch(Sr1)
 {
  case 0: Pt1->confine_v3v1=Pt2; break;
  case 1: Pt1->confine_v1v2=Pt2; break;
  case 2: Pt1->confine_v2v3=Pt2; break;
 }
 switch(Sr2)
 {
  case 0: Pt2->confine_v1v2=Pt1; break;
  case 1: Pt2->confine_v2v3=Pt1; break;
  case 2: Pt2->confine_v3v1=Pt1; break;
 }
}


int circoscrivi(x1,y1,x2,y2,x3,y3,xc,yc,r)
double x1,y1,x2,y2,x3,y3,*xc,*yc,*r;
{
double Denom;

Denom=(2*(x1*(y2-y3)+x2*(y3-y1)+x3*(y1-y2)));
if (fabs(Denom)<EPS_ALLIN) return(1); 
*xc=(pow(x1,2.)*(y2-y3)
     +pow(x2,2.)*(y3-y1)
     +(y1-y2)*(pow(x3,2.)
	       +(y2-y3)*(y1-y3)) )
    /Denom;
*yc=-(pow(x1,2.)*(x2-x3)
      -x1*(pow(x2,2.)
      -pow(x3,2.)
      +(y2-y3)*(y2+y3))
      +pow(x2,2.)*x3
      -x2*(pow(x3,2.)
	   +(y3-y1)*(y1+y3))
      +x3*(y2-y1)*(y1+y2) )
     /Denom;
*r=sqrt(pow((*xc-x1),2.)+pow((*yc-y1),2.));
return(0);
}

geometric_side which_side(xp,yp,xi,yi,xj,yj)
double xp,yp,xi,yi,xj,yj;
{
 if ( (xj-xi)*(yp-yi)-(xp-xi)*(yj-yi) >= -EPS_ALLIN) return (dentro_weak);
						else return (fuori_strong);
}

geometric_side inside_convex_hull(xtest,ytest,convex_hull_LocPt,x,y)
double xtest,ytest,x[],y[];
struct edge_list *convex_hull_LocPt;
{
 geometric_side answer=dentro_weak;
 long int i,j;

 while ((convex_hull_LocPt != NULL) && (answer==dentro_weak))
 {
  i=convex_hull_LocPt->bordo.v1;
  j=convex_hull_LocPt->bordo.v2;
  answer=which_side(xtest,ytest,x[i],y[i],x[j],y[j]);
  convex_hull_LocPt=convex_hull_LocPt->next;
 }
 return (answer);
}

geometric_side outside_hole(xtest,ytest,xc,yc,r)
double xtest,ytest,xc,yc,r;
{
 double rsquare=pow(r,2.);

 if (1./rsquare*pow((xtest-xc),2.)+
     1./rsquare*pow((ytest-yc),2.) >=1.-EPS_ALLIN )
	 return dentro_weak;
    else return fuori_strong;
}

void Delaunay_to_sigma(n,x,y,DelaunayLoc,numero_azioni)
long int n,*numero_azioni;
double x[],y[];
struct triang_list *DelaunayLoc;
{
 long int i,j,k;
 double xc,yc,r;

 *numero_azioni=0;
 while (DelaunayLoc!=NULL)
 {
  i=DelaunayLoc->triangolo.v1;
  j=DelaunayLoc->triangolo.v2;
  k=DelaunayLoc->triangolo.v3;
  if(circoscrivi(x[i],y[i],x[j],y[j],x[k],y[k],&xc,&yc,&r)==1)
  {
   printf("Find nearly aligned points.\n");
   printf("I cannot compute the circumcenter.\n");
   exit(1);
  };
  if (outside_hole(x[n],y[n],xc,yc,r)==fuori_strong)
  {
   DelaunayLoc->triangolo.tipo=sigma_triang;
   *numero_azioni=*numero_azioni+1;
  }
  DelaunayLoc=DelaunayLoc->next;
 }
}

void evaluate_edges(DelaunayLoc,n,x,y,mode,b_e_edges_Loc,critical_arrival)
struct triang_list *DelaunayLoc;
struct edge_list *b_e_edges_Loc;
long int n, *critical_arrival;
int mode;
double x[],y[];
{
 long int i,j,k; 
 /* indices of the vertices of the triangle pointed by DelaunayLoc */
 struct edge_list *b_e_edges_Loc_Saved;

 b_e_edges_Loc_Saved=b_e_edges_Loc;
 while (DelaunayLoc!=NULL)
 {
  i=DelaunayLoc->triangolo.v1;
  j=DelaunayLoc->triangolo.v2;
  k=DelaunayLoc->triangolo.v3;
  if ((mode==1) || (mode==3))
  {
   if (DelaunayLoc->triangolo.tipo==sigma_triang)
   {
    update_b_edges(DelaunayLoc->confine_v1v2,mode,
		   b_e_edges_Loc_Saved,&b_e_edges_Loc,
		   n,x,y,i,j,k,critical_arrival);
    update_b_edges(DelaunayLoc->confine_v2v3,mode,
		   b_e_edges_Loc_Saved,&b_e_edges_Loc,
		   n,x,y,j,k,i,critical_arrival);
    update_b_edges(DelaunayLoc->confine_v3v1,mode,
		   b_e_edges_Loc_Saved,&b_e_edges_Loc,
		   n,x,y,k,i,j,critical_arrival);
   }
   else /* The triangle is not a 'sigma' one and mode=1 or mode=3 */
    if (mode==3)
    {
     if (DelaunayLoc->confine_v1v2==NULL)
       update_e_edges(DelaunayLoc,b_e_edges_Loc_Saved,&b_e_edges_Loc,
		      n,x,y,i,j);
     if (DelaunayLoc->confine_v2v3==NULL)
       update_e_edges(DelaunayLoc,b_e_edges_Loc_Saved,&b_e_edges_Loc,
		      n,x,y,j,k);
     if (DelaunayLoc->confine_v3v1==NULL)
       update_e_edges(DelaunayLoc,b_e_edges_Loc_Saved,&b_e_edges_Loc,
		      n,x,y,k,i);
    }
  }  /* end mode=1 or mode=3 */
  if (mode==2)       /* in this case 'sigma' triangles connot exist */
    {
     if (DelaunayLoc->confine_v1v2==NULL)
       update_e_edges(DelaunayLoc,b_e_edges_Loc_Saved,&b_e_edges_Loc,
		      n,x,y,i,j);
     if (DelaunayLoc->confine_v2v3==NULL)
       update_e_edges(DelaunayLoc,b_e_edges_Loc_Saved,&b_e_edges_Loc,
		      n,x,y,j,k);
     if (DelaunayLoc->confine_v3v1==NULL)
       update_e_edges(DelaunayLoc,b_e_edges_Loc_Saved,&b_e_edges_Loc,
		      n,x,y,k,i);
    }
  DelaunayLoc=DelaunayLoc->next;
 }  /* fine ciclo while */
}

void update_b_edges(neighbourPt,mode,b_e_edges_Orig,b_edges_Loc,
		    n,x,y,o_v,a_v,oa_v,critical_arrival)
struct triang_list *neighbourPt;
struct edge_list *(*b_edges_Loc), *b_e_edges_Orig;
long int n,o_v,a_v,oa_v,*critical_arrival;
int mode;
double x[],y[];
{
int add_flag=0,head_insertion_flag=0;
long int tempint,test_index,i;
geometric_side tempside;
struct edge_list *temp_edge_Pt, *scroll_edge_Pt;
    /* scambio se la parte e' sbagliata */
    if (mode==1) {test_index=oa_v; tempside=fuori_strong;};
    if (mode==3) {test_index=n; tempside=dentro_weak;};
    if (which_side(x[test_index],y[test_index],x[o_v],y[o_v],x[a_v],y[a_v])
	==tempside)
	   {
	    tempint=o_v;
	    o_v=a_v;
	    a_v=tempint;
	   }
    /* base conditions for the insertion */
    if (neighbourPt==NULL) add_flag=1;
      else
      if (neighbourPt->triangolo.tipo!=sigma_triang)
	add_flag=1;
    /* further conditions for the insertion  */
    if (add_flag==1)
	switch(mode)
	{
	 case 1: add_flag=
		    (which_side(x[a_v],y[a_v],x[o_v],y[o_v],x[n],y[n])
		     ==fuori_strong);
		 if ((add_flag==0)&&( (*critical_arrival) ==-1))
		      {
		       (*critical_arrival)=a_v;
                       /* visiting the list to search for an edge
                          having a_v as starting vertex */
		       scroll_edge_Pt=b_e_edges_Orig;
		       while (scroll_edge_Pt->next!=NULL)
		       {
			if (scroll_edge_Pt->next->bordo.v1==a_v) /* swap */
			  {
			   /* be careful in the swap steps */
			   if (b_e_edges_Orig!=scroll_edge_Pt)
			    {
			     temp_edge_Pt=scroll_edge_Pt->next;
			     scroll_edge_Pt->next=scroll_edge_Pt->next->next;
			     temp_edge_Pt->next=b_e_edges_Orig->next;
			     b_e_edges_Orig->next=temp_edge_Pt;
                             /* If the head of the list has been swapped with
			        the last element of the same, *b_edges_Loc must
                                be updated so that b_edges_Loc->next==NULL
                                as agreed */
			     if (scroll_edge_Pt->next==NULL)
			       (*b_edges_Loc)=scroll_edge_Pt;
			     break; /* from while loop */
			    }
			  }
			scroll_edge_Pt=scroll_edge_Pt->next;
		       }
		      }
		 break;  /* switch */
	 case 3: add_flag=
		    (which_side(x[oa_v],y[oa_v],x[o_v],y[o_v],x[a_v],y[a_v])
		     ==fuori_strong);
		 break;
	}
    if (add_flag==1) /* success */
    {
     switch (mode)  /* testing for the need of head insertion */
     {
      case 1: if ((*critical_arrival)==o_v) head_insertion_flag=1;
	      break;
      case 3: if (b_e_edges_Orig->next!=NULL) /* otherwise insert at the tail */
	      {
	       i=b_e_edges_Orig->next->bordo.v1;
	       if (which_side(x[o_v],y[o_v],x[n],y[n],x[i],y[i])
		   ==dentro_weak)
		 head_insertion_flag=1;
	      }
	      break;
     }
     /* insertion */
     temp_edge_Pt=malloc(sizeof(struct edge_list));
     if (temp_edge_Pt==NULL) {printf("Not enough memory"); exit(1); }
     temp_edge_Pt->bordo.v1=o_v;
     temp_edge_Pt->bordo.v2=a_v;
     temp_edge_Pt->source_triang=neighbourPt;
     switch(head_insertion_flag)
     {
      case 0: (*b_edges_Loc)->next=temp_edge_Pt;
	      (*b_edges_Loc)=(*b_edges_Loc)->next;
	      (*b_edges_Loc)->next=NULL;
	      break;
      case 1: temp_edge_Pt->next=b_e_edges_Orig->next;
	      b_e_edges_Orig->next=temp_edge_Pt;
	      /* generally, that is if I have already inserted any node,
	         (*b_edges_Loc) is not increased since the insertion has
                 been done at the head of the list.
                 Otherwise, if this is the first insertion, (*b_edges_Loc)
                 must increase (point to the next cell) anyway.
                 The first insertion is detected by the following test */
	      if ((*b_edges_Loc)==(b_e_edges_Orig))
		(*b_edges_Loc)=(*b_edges_Loc)->next;
	      break;
     } /* switch end */
    }   /* insertion end */
}

void update_e_edges(triangPt,b_e_edges_Orig,e_edges_Loc,n,x,y,o_v,a_v)
struct triang_list *triangPt;
struct edge_list *(*e_edges_Loc), *b_e_edges_Orig;
long int n,o_v,a_v;
double x[],y[];
{
int add_flag=0, head_insertion_flag=0;
long int cont,tempint,i;
struct edge_list *temp_edge_Pt;

      /* Swapping the ends of the vector (x[a_v],y[a_v])-(x[o_v],y[o_v])
      ** if (x[n],y[n]) is in the wrong region (the 'dentro_weak' one), 
      ** so it may be in the right region (the 'fuori_strong' one). 
      ** If the swapping is not powerful enough to achieve its aim,
      ** this means that the point (x[n],y[n]) is in the 'dentro_weak' 
      ** region again, or in other words, that it is collinear with the
      ** vector (x[a_v],y[a_v])-(x[o_v],y[o_v]) and, at the same time,
      ** outside the vector itself. In this latter case the candidate edge
      ** cannot be a 'e_edge' side. 
      */
      if (which_side(x[n],y[n],x[o_v],y[o_v],x[a_v],y[a_v])==dentro_weak)
      {
       tempint=o_v; o_v=a_v; a_v=tempint;
      }

      if (which_side(x[n],y[n],x[o_v],y[o_v],x[a_v],y[a_v])==dentro_weak)
	add_flag=0;
      else
      {
       cont=0;
       add_flag=1;      /* eventually it shall change later */
       while ((cont<n)&&(add_flag==1))
       {
	if ((cont!=o_v) && (cont!=a_v))
	  if (which_side(x[cont],y[cont],x[o_v],y[o_v],x[a_v],y[a_v])
	      ==fuori_strong) add_flag=0;
	cont++;
       }
      }
      if (add_flag==1)        /* success */
      {
       /* testing for the need of head insertion */
       if (b_e_edges_Orig->next!=NULL)  /*otherwise tail insertion */
	 {
	  i=b_e_edges_Orig->next->bordo.v1;
	  if (which_side(x[o_v],y[o_v],x[n],y[n],x[i],y[i])==dentro_weak)
	    head_insertion_flag=1;
	 }
       temp_edge_Pt=malloc(sizeof(struct edge_list));
       if (temp_edge_Pt==NULL) {printf("Not enough memory"); exit(1); }
       temp_edge_Pt->bordo.v1=o_v;
       temp_edge_Pt->bordo.v2=a_v;
       temp_edge_Pt->source_triang=triangPt;
       switch(head_insertion_flag)
       {
	case 0: (*e_edges_Loc)->next=temp_edge_Pt;
		(*e_edges_Loc)=(*e_edges_Loc)->next;
		(*e_edges_Loc)->next=NULL;
		break;
	case 1: temp_edge_Pt->next=b_e_edges_Orig->next;
		b_e_edges_Orig->next=temp_edge_Pt;
		/* (*e_edges_Loc) is not increased since the adding has been done
                   at the head of the list */
		break;
       } /* switch end*/
      } /* insertion end */
}

void remove_sigma_triang(DelaunayLoc,DelaunayPt)
struct triang_list *DelaunayLoc,*(*DelaunayPt);
{
 struct triang_list *sigma_triang_Pt;

 if (DelaunayLoc!=NULL)
 {
  while (DelaunayLoc->next!=NULL)
  {
   if (DelaunayLoc->next->triangolo.tipo==sigma_triang)
   {
    /* beginning of removal */
    sigma_triang_Pt=DelaunayLoc->next;
    /* emptying the pointers to the boundaries for the neighbouring triangles
      of our 'sigma' triangle */
    if (sigma_triang_Pt->confine_v1v2!=NULL)
      erase_neighbour(sigma_triang_Pt->confine_v1v2,sigma_triang_Pt);
    if (sigma_triang_Pt->confine_v2v3!=NULL)
      erase_neighbour(sigma_triang_Pt->confine_v2v3,sigma_triang_Pt);
    if (sigma_triang_Pt->confine_v3v1!=NULL)
      erase_neighbour(sigma_triang_Pt->confine_v3v1,sigma_triang_Pt);
    DelaunayLoc->next=DelaunayLoc->next->next; /* removal */
    free(sigma_triang_Pt);
    /* removal end */
   }
   else DelaunayLoc=DelaunayLoc->next;
   /*
   ** NOTE: DelaunayLoc is increased (in order to update DelaunayLoc->next)
   **       only if the triangle has not been removed, since otherwise
   **       the removal step itself has increased DelaunayLoc->next.
   */
  } /* while loop end*/
  (*DelaunayPt)=DelaunayLoc;  /* Here DelaunayLoc->next==NULL */
 } /* control test end */
}

void erase_neighbour(triangPt,neighbourPt)
struct triang_list *triangPt,*neighbourPt;
{
 if (triangPt->confine_v1v2==neighbourPt) triangPt->confine_v1v2=NULL;
 else
 if (triangPt->confine_v2v3==neighbourPt) triangPt->confine_v2v3=NULL;
 else
 if (triangPt->confine_v3v1==neighbourPt) triangPt->confine_v3v1=NULL;
}

void add_new_triangle(DelaunayPt,b_e_edges_Loc,mode,n,critical_arrival)
struct triang_list *(*DelaunayPt);
struct edge_list *b_e_edges_Loc;
long int n,*critical_arrival;
int mode;
{
 struct edge_list *edge_temp_Pt;
 struct edge_list *removing_edge_Pt;
 struct triang_list *triang_temp_Pt, *DelaunayPtSaved, *DelaunayPrecTriPt;
 long int i,j,k,o_v,a_v;
 int continua_loop;

 edge_temp_Pt=b_e_edges_Loc;
 DelaunayPtSaved=(*DelaunayPt);
 DelaunayPrecTriPt=NULL;
 /* inizio ciclo while */
 while (edge_temp_Pt->next!=NULL)
 {
  o_v=edge_temp_Pt->next->bordo.v1;
  a_v=edge_temp_Pt->next->bordo.v2;
  triang_temp_Pt=edge_temp_Pt->next->source_triang;
  /* inizio rimozione */
  removing_edge_Pt=edge_temp_Pt->next;
  edge_temp_Pt->next=removing_edge_Pt->next;
  free(removing_edge_Pt);
  /* fine rimozione */
  (*DelaunayPt)->next=malloc(sizeof(struct triang_list));
  if ((*DelaunayPt)->next==NULL) {printf("Not enough memory"); exit(1);}
  (*DelaunayPt)=(*DelaunayPt)->next;
  (*DelaunayPt)->triangolo.v1=n;
  (*DelaunayPt)->triangolo.v2=o_v;
  (*DelaunayPt)->triangolo.v3=a_v;
  (*DelaunayPt)->triangolo.tipo=Delaunay_triang;
  (*DelaunayPt)->confine_v2v3=triang_temp_Pt;
  /* mutually updating the boundaries with old triangles */
  if (triang_temp_Pt!=NULL)
  {
   i=triang_temp_Pt->triangolo.v1;
   j=triang_temp_Pt->triangolo.v2;
   k=triang_temp_Pt->triangolo.v3;
   if ( ((i==o_v)&&(j==a_v)) || ((j==o_v)&&(i==a_v)) )
     triang_temp_Pt->confine_v1v2=(*DelaunayPt);
   if ( ((j==o_v)&&(k==a_v)) || ((k==o_v)&&(j==a_v)) )
     triang_temp_Pt->confine_v2v3=(*DelaunayPt);
   if ( ((k==o_v)&&(i==a_v)) || ((i==o_v)&&(k==a_v)) )
     triang_temp_Pt->confine_v3v1=(*DelaunayPt);
  }
  /* mutuallu updating the boundary with the previous triangle */
  (*DelaunayPt)->confine_v1v2=DelaunayPrecTriPt;
  if (DelaunayPrecTriPt!=NULL)
    DelaunayPrecTriPt->confine_v3v1=(*DelaunayPt);
  DelaunayPrecTriPt=(*DelaunayPt);
  /* risetto edge_temp_Pt */
  edge_temp_Pt=b_e_edges_Loc;
  continua_loop=(edge_temp_Pt->next!=NULL);
  while (continua_loop)
  {
   if (edge_temp_Pt->next==NULL)
     {
      continua_loop=0;
      printf("Error: probably the nodes aren't distinct.\n");
      printf("Otherwise recompile the program with a smaller EPS_ALLIN\n");
      exit(1);
     }
   else
     {
      if (edge_temp_Pt->next->bordo.v1==a_v) continua_loop=0;
	else edge_temp_Pt=edge_temp_Pt->next;
     }
  } /* end 'while structure' to set edge_temp_Pt */
 } /* fine while */
 switch(mode)     /* updating ending boundaries */
 {
  case 1: if ((*critical_arrival)==-1)
	  {
	   (*DelaunayPt)->confine_v3v1=DelaunayPtSaved->next;
	   DelaunayPtSaved->next->confine_v1v2=(*DelaunayPt);
	  }
	  else
	   (*DelaunayPt)->confine_v3v1=NULL;  
           /* the other boundary is already NULL */
	  break;
  case 2:
  case 3: (*DelaunayPt)->confine_v3v1=NULL;  
          /* the ohter boundary is already NULL */
	  break;
 }
 (*DelaunayPt)->next=NULL;
}

void remake_convex_hull(DelaunayLoc,convex_hull_Loc,x,y)
struct triang_list *DelaunayLoc;
struct edge_list *convex_hull_Loc;
double x[],y[];
{
 long int i,j,k;

 while (DelaunayLoc!=NULL)
 {
  i=DelaunayLoc->triangolo.v1;
  j=DelaunayLoc->triangolo.v2;
  k=DelaunayLoc->triangolo.v3;
  /* do not alter i,j,k since a triangle may have more than one boundary */
  if (DelaunayLoc->confine_v1v2==NULL)
    update_convex_hull(DelaunayLoc,&convex_hull_Loc,x,y,i,j,k);
  if (DelaunayLoc->confine_v2v3==NULL)
    update_convex_hull(DelaunayLoc,&convex_hull_Loc,x,y,j,k,i);
  if (DelaunayLoc->confine_v3v1==NULL)
    update_convex_hull(DelaunayLoc,&convex_hull_Loc,x,y,k,i,j);
  DelaunayLoc=DelaunayLoc->next;
 }
}

void update_convex_hull (triangPt,convex_hull_Loc,x,y,o_v,a_v,oa_v)
long int o_v,a_v,oa_v;
struct triang_list *triangPt;
struct edge_list *(*convex_hull_Loc);
double x[],y[];
{
 long int tempint;

     (*convex_hull_Loc)->next=malloc(sizeof(struct edge_list));
     if ((*convex_hull_Loc)->next==NULL)
	{printf("Not enough memory"); exit(1); }
     (*convex_hull_Loc)=(*convex_hull_Loc)->next;
     if (which_side(x[oa_v],y[oa_v],x[o_v],y[o_v],x[a_v],y[a_v])
	 ==fuori_strong) {tempint=a_v; a_v=o_v; o_v=tempint;}
     (*convex_hull_Loc)->bordo.v1=o_v;
     (*convex_hull_Loc)->bordo.v2=a_v;
     (*convex_hull_Loc)->source_triang=triangPt;
     (*convex_hull_Loc)->next=NULL;
}

void write_node_neighbours(triangulation,convex_hull,node,id_file,x,y)
struct triang_list *triangulation;
struct edge_list *convex_hull;
long int node;
FILE* id_file;
double x[],y[];
{
 struct triang_list  *first_triangle_Pt, *current_triangle_Pt;
 struct edge_list *current_edge_Pt;
 long int current_neighbour,i,j,k,tempint;

 current_edge_Pt=convex_hull;
 while (current_edge_Pt!=NULL)
 {
  if (current_edge_Pt->bordo.v1==node)
    {
     first_triangle_Pt=current_edge_Pt->source_triang;
     current_neighbour=current_edge_Pt->bordo.v2;
     break;
    }
  current_edge_Pt=current_edge_Pt->next;
 }
 if (current_edge_Pt==NULL) /* normal exit from while loop */
 {
  first_triangle_Pt=triangulation;
  while (first_triangle_Pt!=NULL)
  {
   if (first_triangle_Pt->triangolo.v1==node)
     {
      current_neighbour=first_triangle_Pt->triangolo.v2; /* A choice */
      tempint=first_triangle_Pt->triangolo.v3;
      if(which_side(x[tempint],y[tempint],x[node],y[node],
		    x[current_neighbour],y[current_neighbour])==fuori_strong)
	current_neighbour=tempint;
      break;
     }
   else
   if (first_triangle_Pt->triangolo.v2==node)
     {
      current_neighbour=first_triangle_Pt->triangolo.v3; /* A choice */
      tempint=first_triangle_Pt->triangolo.v1;
      if(which_side(x[tempint],y[tempint],x[node],y[node],
		    x[current_neighbour],y[current_neighbour])==fuori_strong)
	current_neighbour=tempint;
      break;
     }
   else
   if (first_triangle_Pt->triangolo.v3==node)
     {
      current_neighbour=first_triangle_Pt->triangolo.v1; /* A choice */
      tempint=first_triangle_Pt->triangolo.v2;
      if(which_side(x[tempint],y[tempint],x[node],y[node],
		    x[current_neighbour],y[current_neighbour])==fuori_strong)
	current_neighbour=tempint;
      break;
     }
   else
   first_triangle_Pt=first_triangle_Pt->next;
  } /* fine while */
 } /* fine if*/
 /*
 ** 'first_triangle_Pt' points to the first triangle having the 'node' point 
 ** as a vertex and 'current_neighbour' to the first neighbouring point of
 ** the 'node' point.
 */
 fprintf(id_file,"%5ld  %5ld ",node,current_neighbour);
 current_triangle_Pt=first_triangle_Pt;
 do
 {
  i=current_triangle_Pt->triangolo.v1;
  j=current_triangle_Pt->triangolo.v2;
  k=current_triangle_Pt->triangolo.v3;
  if ((i==node)&&(j!=current_neighbour))
  {
   current_neighbour=j;
   current_triangle_Pt=current_triangle_Pt->confine_v1v2;
  }
  else
  if ((i==node)&&(k!=current_neighbour))
  {
   current_neighbour=k;
   current_triangle_Pt=current_triangle_Pt->confine_v3v1;
  }
  else
  if ((j==node)&&(i!=current_neighbour))
  {
   current_neighbour=i;
   current_triangle_Pt=current_triangle_Pt->confine_v1v2;
  }
  else
  if ((j==node)&&(k!=current_neighbour))
  {
   current_neighbour=k;
   current_triangle_Pt=current_triangle_Pt->confine_v2v3;
  }
  else
  if ((k==node)&&(i!=current_neighbour))
  {
   current_neighbour=i;
   current_triangle_Pt=current_triangle_Pt->confine_v3v1;
  }
  else
  if ((k==node)&&(j!=current_neighbour))
  {
   current_neighbour=j;
   current_triangle_Pt=current_triangle_Pt->confine_v2v3;
  }
  fprintf(id_file,"%5ld ",current_neighbour);
  /* For a point on the boundary this instruction is performed last time
  ** with the same value than the one called before the while loop: this is
  ** useful. 
  */
 }
 while ((current_triangle_Pt!=NULL) &&
	(current_triangle_Pt!=first_triangle_Pt) );
 /* printing a separator and carriage return */
 fprintf(id_file,"-1\n");
}

int main(argc,argv)
int argc;
char *argv[];
{
 long int num_nodo,current_nodo;
 int err,read_mode;
 /*
 ** read_mode==2  for x y   points as input
 ** read_mode==3  for x y z points as input using only x and y.
 */
 int change_mode=0;
 /*
 ** change_mode==0 to build a new triangulation
 ** change_mode==1 to alter an existing triangulation
 */
 long int c12i,c12j,c12k,c23i,c23j,c23k,c31i,c31j,c31k,ti,tj,tk;
 double x[MAX_NUM_NODI],y[MAX_NUM_NODI],z[MAX_NUM_NODI];
 /* z coordinate is used when we alter an existing triangulation */
 struct triang_list *Triangolazione, *TriPt;
 struct edge_list *Inviluppo_convesso, *IcPt;
 FILE *readdatfile=NULL, *readtrifile=NULL,
      *writedatfile=NULL, *writetrifile=NULL;
 char file_name_buffer[100];

 if (argc==1)
   {
    printf("\nStart the program with three or four parameters:\n");
    printf("First parameter: data file name.\n");
    printf("Second parameter: -i2 x-y data points. \n");
    printf("                  -i3 x-y-z data points.\n");
    printf("Third parameter: triangulation file name.\n");
    printf("                 (Omit it for screen output.)\n");
    printf("Fourth parameter: -change for the interactive shell\n");
    exit(1);
   }
 readdatfile=fopen(argv[1],"r");
 if (readdatfile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[1]);
    exit(1);
   }
 if (argc==2)
 {
  printf("Missing input format parameter.\n");
  exit(1);
 }
 if (strcmp(argv[2],"-i2")==0) read_mode=2;
   else if (strcmp(argv[2],"-i3")==0) read_mode=3;
    else
     {
      printf("Missing input points dimension.\n");
      exit(1);
     }
 if (argc>=5)
   if (strcmp(argv[4],"-change")==0) change_mode=1; /* For the shell */
 switch(change_mode)
 {
  case 0:
    if (argc==3) writetrifile=stdout; else writetrifile=fopen(argv[3],"w");
    break;
  case 1:
    readtrifile=fopen(argv[3],"r");
    if (fgetc(readtrifile)!='C')
    {
     printf("File %s in a no correct format.\n",argv[3]);
     exit(1);
    }
    break;
 } /* switch end */
 if ( (change_mode==0 && writetrifile==NULL)||
      (change_mode==1 && readtrifile==NULL) )
   {
    printf("Cannot open file: %s.\n",argv[3]);
    exit(1);
   }
 /* data reading */
 num_nodo=0;
 while (!feof(readdatfile))
 {
  if (num_nodo<MAX_NUM_NODI)
   {
    if (read_mode==2)
      fscanf(readdatfile," %lf %lf ",&x[num_nodo],&y[num_nodo]);
     else   /* read_mode==3 */
      fscanf(readdatfile," %lf %lf %lf ",&x[num_nodo],&y[num_nodo],
					 &z[num_nodo]);
    /* spaces are essential part of the input format string */
    num_nodo=num_nodo+1;
   }
  else
   {
    printf("Too many input data.\n");
    exit(1);
   }
 }
 if (num_nodo<3)
   {
    printf("I need at least three input points.\n");
    exit(1);
   }
 fclose(readdatfile);
 /* Calling triangulation */
 err=MakeDelaunay(&num_nodo,x,y,z,&Triangolazione,&Inviluppo_convesso,
		  change_mode,readtrifile,read_mode);
 if (err==1)
 {
  printf("\nThe first three points of file %s are nearby aligned.\n",argv[1]);
  exit(1);
 }
 if (change_mode==1)
 {
  printf("\nNew data file: ");
  scanf(" %s",file_name_buffer);
  writedatfile=fopen(file_name_buffer,"w");
  if (writedatfile==NULL)
  {
   printf("Cannot open file %s.\n",file_name_buffer);
   exit(1);
  }
  printf("New triangulation file: ");
  scanf(" %s",file_name_buffer);
  writetrifile=fopen(file_name_buffer,"w");
  if (writetrifile==NULL)
  {
   printf("Cannot open file %s.\n",file_name_buffer);
   exit(1);
  }
  /* output to 'writedatfile' file */
  for (current_nodo=0;current_nodo<num_nodo;current_nodo++)
  {
   switch(read_mode)
   {
    case 2: fprintf(writedatfile,"%*.*f %*.*f\n",WIDTH,PREC,x[current_nodo],
		    WIDTH,PREC,y[current_nodo]);
	    break;
    case 3: fprintf(writedatfile,"%*.*f %*.*f %*.*f\n",
		    WIDTH,PREC,x[current_nodo],
		    WIDTH,PREC,y[current_nodo],WIDTH,PREC,z[current_nodo]);
	    break;
   }
  }
  fclose(writedatfile);
 }
 /* output to 'writetrifile' file */
 fprintf(writetrifile,"C\n");
 for (current_nodo=0;current_nodo<num_nodo;current_nodo++)
  write_node_neighbours(Triangolazione,Inviluppo_convesso,
			current_nodo,writetrifile,x,y);
 IcPt=Inviluppo_convesso;
 fprintf(writetrifile,"B\n");
 while (IcPt!=NULL)
 {
  ti=IcPt->source_triang->triangolo.v1;
  tj=IcPt->source_triang->triangolo.v2;
  tk=IcPt->source_triang->triangolo.v3;
  fprintf(writetrifile,"%5ld %5ld  %5ld %5ld %5ld\n",
	  IcPt->bordo.v1,IcPt->bordo.v2,ti,tj,tk);
  IcPt=IcPt->next;
 }
 TriPt=Triangolazione;
 fprintf(writetrifile,"T\n");
 while (TriPt!=NULL)
 {
  c12i=-1; c12j=-1; c12k=-1;
  if (TriPt->confine_v1v2!=NULL)
  {
   c12i=TriPt->confine_v1v2->triangolo.v1;
   c12j=TriPt->confine_v1v2->triangolo.v2;
   c12k=TriPt->confine_v1v2->triangolo.v3;
  }
  c23i=-1; c23j=-1; c23k=-1;
  if (TriPt->confine_v2v3!=NULL)
  {
   c23i=TriPt->confine_v2v3->triangolo.v1;
   c23j=TriPt->confine_v2v3->triangolo.v2;
   c23k=TriPt->confine_v2v3->triangolo.v3;
  }
  c31i=-1; c31j=-1; c31k=-1;
  if (TriPt->confine_v3v1!=NULL)
  {
   c31i=TriPt->confine_v3v1->triangolo.v1;
   c31j=TriPt->confine_v3v1->triangolo.v2;
   c31k=TriPt->confine_v3v1->triangolo.v3;
  }
  fprintf(writetrifile,
    "%5ld %5ld %5ld  %5ld %5ld %5ld  %5ld %5ld %5ld  %5ld %5ld %5ld\n",
    TriPt->triangolo.v1,TriPt->triangolo.v2,TriPt->triangolo.v3,
    c12i,c12j,c12k,c23i,c23j,c23k,c31i,c31j,c31k);
  TriPt=TriPt->next;
 }
 fclose(writetrifile);
 return 0;
}
