/*
** DO_DER.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
** REVISION DATE: May, 1999
**
** MODULES CALLED: DO_DER_A.C
**                 LINSIST.C 
**
** ------------------------------------------------------------------------
** BACKGROUND
**
** Let (xi,yi,qi) i=0,...,N-1 be N points in the space R^3. 
** Let us consider their projections (xi,yi) onto the plane z=0 and 
** let T be a triangulation of them. 
** A triangulation T of a set of points is the set of triangles whose vertices 
** are the points and such that two distinct triangles intersect at 
** most in a common edge or in a common vertex. 
**
** The aim is the interpolation of the points (xi,yi,qi) with a bivariate
** C^1 function.
**
** Q18 interpolation method builds up such a function using a Q18 polynomial
** patch on each triangle, provided that we know the partial derivatives
** up to the order two at each vertex of the triangulation T.
** 
** The program computes the partial derivatives up to the order
** two at each point (xi,yi,qi) i=0,...,N-1. In what follows 
** we refer to them as to qix,qiy,qixx,qixy,qiyy i=0,...,N-1, or more often
** as to the (5N)-dimensional vector 'v'.
**
** For every fixed i, i=0,...,N-1, the program computes the vector 'v' in the
** "smoothest" possible way for a Q18 interpolating function over the
** triangulation TLoc(i) made up by all triangles having the 
** i-th interpolation point, Vi, for vertex (that is the cell centered in Vi).
** Then it considers only the values qix,qiy,qixx,qixy,qiy where i is now
** fixed.    
** The criteria considered to evaluate the "smoothness" of
** a Q18 interpolating function is the minimization of the functional
**
** phi(vLoc(i))=   sum  [ Integral[ (Qxx^2 + 2*Qxy^2 + Qyy^2  ) dx dy ] ]     
**             t in TLoc(i)  t      
** 
** with respect to vLoc(i), the same of v when T becomes TLoc(i).
**
** 
** METHODOLOGY 
**
** For every fixed i, i=0,...,N-1,
**
**     phi(vLoc(i))= 1/2 * vLoc(i)' A vLoc(i) + g' vLoc(i) + c 
**
** is minimum when vLoc(i) solves the system
**
**      A vLoc(i) + g = 0.
**
** Let M be the number of nodes in the cell TLoc(i).
** The program computes 
**  
**     c=phi(0), A, gj=phi(ej)-1/2*ajj-c, 
**
** where ej is the j-th vector of the canonical basis of R^(5M),
** and solves the resulting linear system with the Cholesky factorisation. 
**
** Then, it considers only the parameters relative to Vi, the center of TLoc(i).
**
** 
** REFERENCES
**
**  R.E. Barnhill and G. Farin:
**    "C^1 quintic interpolation over triangles: 
**     Two explicit representations"
**     Internat. J. Numer. Methods Engrg., 17(12):1763-1778, 1981.
**
**   R. J. Renka and A. K. Cline:
**     "A triangle-based C^1 interpolation method"
**      Rocky Mountain J. Math, 14(1):223-237, 1984.
**
**  P. Alfeld:
**    "Derivative Generation for multivariate 
**     scattered data functional minimization", 
**     Comp. Aided Graph. Design,2: 281-296, 1985.
**
**  D. Fasoli,
**      Laurea's Thesis, University of Padua, 1995 (in Italian). 
**
** 
** ------------------------------------------------------------------------
**
** SYNOPSIS:  do_der file1 file2 [file3]
**
** 'file1' is the data file made up of the 3-dimensional points. 
** The coordinates of the points may be separated by any 
** combination of spaces, tabs and carriage returns. 
** We recommend to use a row for each point, for clearness. 
** 
** 'file2' is the triangulation data file in the format generated by
** the program 'do_trian'. The file has three sections. 
** The first, starting with a single capital letter `C' in the first row, 
** lists in subsequent blocks separated by -1 the index of
** a node and those of its neighbours in a counterclockwise
** order. 
** Note: this program uses only this first section and the heading of the
** second section, that is a single capital letter `B' in the row.
**
** 'file3' is the output file, the file of the derivatives. 
** Omit it to use standard output. 
** The values in a same row refer to the same vertex of the triangulation.
** The first column is the x-derivative, the second the y-derivative,
** the third is the xx-derivative, the fourth is the xy-derivative
** and the last one is the yy-derivative.
** Output format is handled by the two constants WIDTH and PREC
** that are the width of the output field, and the number of decimal digits.
**        
** The program supports at most MAX_NUM_NODI (#DEFINE constant) input points.
** If more points are given, it exits by printing a warning.
**
** The program uses a temporary swap file called 'blocks.tmp' to store
** data during computations. You may safely remove it after the use. 
** 

**
** -----------------------------------------------------------------------
**
** SUBROUTINES AND FUNCTIONS:
**
** int global_index(int N, int i, int j)
**
** Inputs: 
**     N, the number of nodes of an open cell or of a closed one; 
**     j, the index of the triangle in the cell; 
**     i, the local index of the unknown parameter among 
**        q1x,q1y,q1xx,q1xy,q1yy,q2x,q2y,q2xx,q2xy,q2yy,q3x,q3y,q3xx,q3xy,q3yy
**        relative to the triangle (a value between 0 and 14).
** 
** Output: 
**     The global index of the unknown parameter
**     (a value between 0 and 5N-1) relative to the cell.
**
**     Locally, for each triangle, the parameters are sorted following
**     the vertices of the triangle and the sequence 
**     d/dx,d/dy,d2/dxdx,d2/dxdy,d2/dydy.
**
**     Globally, for the cell, 
**     the parameters are sorted following the nodes of the cell,
**     according to their index relative to the cell itself,
**     and the above sequence.
**
** -------------------------------------------------------------------------
*/

#include<stdio.h>
#include<stdlib.h>
#include<math.h>
#define MAX_NUM_NODI 100000
#define WIDTH 20   /* format output width for real numbers */
#define PREC 10    /* number of decimal digits of real numbers output */

double derivecptq18_v(),compute_d_rst_Q18(),compute_d_rst_Q18_null(),
       sum_cpt3_prod();
void free_ind_node_list();
int global_index();

/* Declarations for global variables */
double q1,q2,q3;
double e1x,e1y,e2x,e2y,e3x,e3y,t1,t2,t3;

struct ind_node_list {long int glob_index; struct ind_node_list *next;};

void free_ind_node_list(Ind_Node_List_Pt)
struct ind_node_list *Ind_Node_List_Pt;
{
 struct ind_node_list *tempPt;

 if (Ind_Node_List_Pt==NULL) return;
 while (Ind_Node_List_Pt->next!=NULL)
   {
    tempPt=Ind_Node_List_Pt->next;
    free(Ind_Node_List_Pt);
    Ind_Node_List_Pt=tempPt;
   }
 free(Ind_Node_List_Pt);
}

void main(argc,argv)
int argc;
char *argv[];
{
  struct triangle {long int v1; long int v2; long int v3;};

  size_t size_double;
  FILE *infile, *trifile, *outfile, *bfile;
  long int num_nodo,cont_nodi,temp_index,iv1,iv2,iv3,
	   triangle_number,triangle_counter,nt;
  double x[MAX_NUM_NODI],y[MAX_NUM_NODI],z[MAX_NUM_NODI];

  /* In a triangulation of MAX_NUM_NODI the number of triangles
     is given by the Eulero formula: t=2*(MAX_NUM_NODI-1)-eb where
     eb is the number of edges building the border of the triangulation
     Since eb>=3 we get an upper limit for the number of triangles */

  struct triangle triangle_order[2*MAX_NUM_NODI-5];
  double x1,y1,x2,y2,x3,y3,At;
  int i,j,ip,jp,N_C; /* N_C is the counter for the number of nodes in a cell */
  int ip_global,jp_global;
  int r,s,err,open_cella_flag; /* flag open-closed cell */
  struct ind_node_list *lista_vicini, *vicino_pt, *vicino_pt_temp;
  double cpt3[45][10],temp_calc;
  char base_buffer[20];
  int stop; /* For cycles */
  int shift_right;
  /* 
  ** Number of counterclockwise rotations needed to transform a triangle of
  ** a cell in the "canonical form" defined by iv1<iv2,iv1<iv3 where 
  ** iv1,iv2,iv3 are the indexes of the vertices of the triangle.
  */
  double *A, *v, *gg, *DiagL, c;

 infile=NULL; trifile=NULL; outfile=NULL; bfile=NULL;
 A=NULL; v=NULL; gg=NULL; DiagL=NULL;
 lista_vicini=NULL;
 size_double=sizeof(double);
 if (argc==1)
   {
    printf("\nStart program with three parameters:\n");
    printf("First parameter: data file name.\n");
    printf("Second parameter: triangulation file name.\n");
    printf("Third parameter: derivative file name.\n");
    printf("                 (Omit it for standard output).\n");
    exit(1);
   }
 /* Reading */
 infile=fopen(argv[1],"r");
 if (infile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[1]);
    exit(1);
   }
 if (argc==2)
   {
    printf("Missing triangulation file.\n");
    exit(1);
   }
 /* data reading */
 printf("Wait for data reading.\n"); 
 num_nodo=0;
 while (!feof(infile))
 {
  if (num_nodo<MAX_NUM_NODI)
   {
    fscanf(infile," %lf %lf %lf ",&x[num_nodo],&y[num_nodo],&z[num_nodo]);
      /* spaces are essential part of the input format string */
    num_nodo=num_nodo+1;
   }
   else
   {
    printf("Too many input data.\n");
    exit(1);
   }
 }
 if (num_nodo<3)
    {
     printf("I need at least three input points.\n");
     exit(1);
    }
 fclose(infile);
 /* end of data reading */
 /* opening triangulation file */
 trifile=fopen(argv[2],"r");
 if (trifile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[2]);
    exit(1);
   }
 if (fgetc(trifile)!='C')
   {
    printf("File %s in a no correct format.\n",argv[2]);
    exit(1);
   }
 else fgetc(trifile); /* reading carriage return */
 /* apro file output */
 if (argc==3) outfile=stdout; else outfile=fopen(argv[3],"w");
 if (outfile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[3]);
    exit(1);
   }
 /* HANDLING THE PARTICULAR CASE OF ONLY ONE TRIANGLE */
 if (num_nodo==3)
 {
  fclose(trifile);
  temp_calc=x[0]*(y[1]-y[2])+x[1]*(y[2]-y[0])+x[2]*(y[0]-y[1]);
  x1=-(y[0]*(z[1]-z[2])+y[1]*(z[2]-z[0])+y[2]*(z[0]-z[1]))/temp_calc;
  y1=(x[0]*(z[1]-z[2])+x[1]*(z[2]-z[0])+x[2]*(z[0]-z[1]))/temp_calc;
  /* Derivatives of the plane */
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,x1,WIDTH,PREC,y1,
	  WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,x1,WIDTH,PREC,y1,
	  WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,x1,WIDTH,PREC,y1,
	  WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  fclose(outfile);
  return; /* regular exit from main function */
 }
 /* OPENING TEMPORARY FILE */
 bfile=fopen("blocks.tmp","wb");
 if (bfile==NULL) 
   {
    printf("Cannot open temporary file for writing.\n");
    exit(1);
   }
 /* MAIN CYCLE, COMPUTATIONS OVER TRIANGLES, AND THEIR STORAGE */
 printf("Starting computations for the data of each triangle.\n");
 fscanf(trifile,"%ld ",&iv1);
 triangle_number=0;
 /* Current triangle number. At the end it is the number of triangles */
 do
   {
    fscanf(trifile,"%ld %ld ",&iv2,&iv3);
    while(iv3!=-1)
     {
      if ((iv1<iv2)&&(iv1<iv3))
	{/* New triangle: beginning computations and assembling phase */
	 x1=x[iv1]; x2=x[iv2]; x3=x[iv3];
	 y1=y[iv1]; y2=y[iv2]; y3=y[iv3];
	 q1=z[iv1]; q2=z[iv2]; q3=z[iv3];
	 e1x=x3-x2; e1y=y3-y2; e2x=x1-x3; e2y=y1-y3; e3x=x2-x1; e3y=y2-y1;
	 At=fabs((x2*y3-x3*y2+x3*y1-x1*y3+x1*y2-x2*y1)/2.);
	 t1=-(e1x*e3x+e1y*e3y)/(pow(e1x,2.)+pow(e1y,2.));
	 t2=-(e2x*e1x+e2y*e1y)/(pow(e2x,2.)+pow(e2y,2.));
	 t3=-(e3x*e2x+e3y*e2y)/(pow(e3x,2.)+pow(e3y,2.));
	 /* Computations and updating of the storing */
	 /* A matrix
         ** The computation is splitted in two parts in order to perform
         ** the first one only 15 times instead of 120 times as we should do
         ** using only the second part.
         */
	 for (ip=0;ip<=14;ip++)
	 {
	  i=0; for (r=0;r<=3;++r) for (s=0;s<=3-r;++s)
	  {
	   temp_calc=derivecptq18_v(r+2,s,ip)*e1y*e1y;
	   temp_calc+=derivecptq18_v(r,s+2,ip)*e2y*e2y;
	   temp_calc+=derivecptq18_v(r,s,ip)*e3y*e3y;
	   temp_calc+=derivecptq18_v(r+1,s+1,ip)*2*e1y*e2y;
	   temp_calc+=derivecptq18_v(r+1,s,ip)*2*e1y*e3y;
	   temp_calc+=derivecptq18_v(r,s+1,ip)*2*e2y*e3y;
	   cpt3[ip][i]=temp_calc;

	   temp_calc=-derivecptq18_v(r+2,s,ip)*e1y*e1x;
	   temp_calc-=derivecptq18_v(r,s+2,ip)*e2y*e2x;
	   temp_calc-=derivecptq18_v(r,s,ip)*e3y*e3x;
	   temp_calc-=derivecptq18_v(r+1,s+1,ip)*(e1y*e2x+e2y*e1x);
	   temp_calc-=derivecptq18_v(r+1,s,ip)*(e1y*e3x+e3y*e1x);
	   temp_calc-=derivecptq18_v(r,s+1,ip)*(e2y*e3x+e3y*e2x);
	   cpt3[ip+15][i]=temp_calc;

	   temp_calc=derivecptq18_v(r+2,s,ip)*e1x*e1x;
	   temp_calc+=derivecptq18_v(r,s+2,ip)*e2x*e2x;
	   temp_calc+=derivecptq18_v(r,s,ip)*e3x*e3x;
	   temp_calc+=derivecptq18_v(r+1,s+1,ip)*2*e1x*e2x;
	   temp_calc+=derivecptq18_v(r+1,s,ip)*2*e1x*e3x;
	   temp_calc+=derivecptq18_v(r,s+1,ip)*2*e2x*e3x;
	   cpt3[ip+30][i]=temp_calc;
	   i++;
	  } /* end of cycle over s (and over r) */
	 } /* end for ip */
	 for (ip=0;ip<=14;++ip)
	    for (jp=0;jp<=ip;++jp)
	     {
	      /* The constant 25./(28.*pow(At,3)) comes from
                (5./At^2)^2 * (2*At/((6+1)(6+2))) */
	      temp_calc=50./(28. *pow(At,3.))*
	      (sum_cpt3_prod(cpt3[ip],cpt3[jp])+
	       2.*sum_cpt3_prod(cpt3[ip+15],cpt3[jp+15])+
	       sum_cpt3_prod(cpt3[ip+30],cpt3[jp+30]));
	      fwrite((char *)&temp_calc,size_double,1,bfile);
	     }
	 /* gg vector */
	 for (ip=0;ip<=14;++ip)
	 {
	  i=0; for (r=0;r<=3;++r) for (s=0;s<=3-r;++s)
	  {
	   temp_calc=compute_d_rst_Q18(r+2,s,ip)*e1y*e1y;
	   temp_calc+=compute_d_rst_Q18(r,s+2,ip)*e2y*e2y;
	   temp_calc+=compute_d_rst_Q18(r,s,ip)*e3y*e3y;
	   temp_calc+=compute_d_rst_Q18(r+1,s+1,ip)*2*e1y*e2y;
	   temp_calc+=compute_d_rst_Q18(r+1,s,ip)*2*e1y*e3y;
	   temp_calc+=compute_d_rst_Q18(r,s+1,ip)*2*e2y*e3y;
	   cpt3[0][i]=temp_calc;

	   temp_calc=-compute_d_rst_Q18(r+2,s,ip)*e1y*e1x;
	   temp_calc-=compute_d_rst_Q18(r,s+2,ip)*e2y*e2x;
	   temp_calc-=compute_d_rst_Q18(r,s,ip)*e3y*e3x;
	   temp_calc-=compute_d_rst_Q18(r+1,s+1,ip)*(e1y*e2x+e2y*e1x);
	   temp_calc-=compute_d_rst_Q18(r+1,s,ip)*(e1y*e3x+e3y*e1x);
	   temp_calc-=compute_d_rst_Q18(r,s+1,ip)*(e2y*e3x+e3y*e2x);
	   cpt3[1][i]=temp_calc;

	   temp_calc=compute_d_rst_Q18(r+2,s,ip)*e1x*e1x;
	   temp_calc+=compute_d_rst_Q18(r,s+2,ip)*e2x*e2x;
	   temp_calc+=compute_d_rst_Q18(r,s,ip)*e3x*e3x;
	   temp_calc+=compute_d_rst_Q18(r+1,s+1,ip)*2*e1x*e2x;
	   temp_calc+=compute_d_rst_Q18(r+1,s,ip)*2*e1x*e3x;
	   temp_calc+=compute_d_rst_Q18(r,s+1,ip)*2*e2x*e3x;
	   cpt3[2][i]=temp_calc;
	   i++;
	  } /* end cycle over s (and over r) */
	  /* The constant 25./(28.*pow(At,3)) comes from
            (5./At^2)^2 * (2*At/((6+1)(6+2))) */
	  temp_calc=25./(28. *pow(At,3.))*
		    (sum_cpt3_prod(cpt3[0],cpt3[0])+
		     2.*sum_cpt3_prod(cpt3[1],cpt3[1])+
		     sum_cpt3_prod(cpt3[2],cpt3[2]));
	  fwrite((char *)&temp_calc,size_double,1,bfile);
	 } /* end for ip */
	 /* c constant */
	 i=0; for (r=0;r<=3;++r) for (s=0;s<=3-r;++s)
	 {
	  temp_calc=compute_d_rst_Q18_null(r+2,s)*e1y*e1y;
	  temp_calc+=compute_d_rst_Q18_null(r,s+2)*e2y*e2y;
	  temp_calc+=compute_d_rst_Q18_null(r,s)*e3y*e3y;
	  temp_calc+=compute_d_rst_Q18_null(r+1,s+1)*2*e1y*e2y;
	  temp_calc+=compute_d_rst_Q18_null(r+1,s)*2*e1y*e3y;
	  temp_calc+=compute_d_rst_Q18_null(r,s+1)*2*e2y*e3y;
	  cpt3[0][i]=temp_calc;

	  temp_calc=-compute_d_rst_Q18_null(r+2,s)*e1y*e1x;
	  temp_calc-=compute_d_rst_Q18_null(r,s+2)*e2y*e2x;
	  temp_calc-=compute_d_rst_Q18_null(r,s)*e3y*e3x;
	  temp_calc-=compute_d_rst_Q18_null(r+1,s+1)*(e1y*e2x+e2y*e1x);
	  temp_calc-=compute_d_rst_Q18_null(r+1,s)*(e1y*e3x+e3y*e1x);
	  temp_calc-=compute_d_rst_Q18_null(r,s+1)*(e2y*e3x+e3y*e2x);
	  cpt3[1][i]=temp_calc;

	  temp_calc=compute_d_rst_Q18_null(r+2,s)*e1x*e1x;
	  temp_calc+=compute_d_rst_Q18_null(r,s+2)*e2x*e2x;
	  temp_calc+=compute_d_rst_Q18_null(r,s)*e3x*e3x;
	  temp_calc+=compute_d_rst_Q18_null(r+1,s+1)*2*e1x*e2x;
	  temp_calc+=compute_d_rst_Q18_null(r+1,s)*2*e1x*e3x;
	  temp_calc+=compute_d_rst_Q18_null(r,s+1)*2*e2x*e3x;
	  cpt3[2][i]=temp_calc;
	  i++;
	 }
	 /* The constant 25./(28.*pow(At,3)) comes from
           (5./At^2)^2 * (2*At/((6+1)(6+2))) */
	 temp_calc=25./(28. *pow(At,3.))*
		   (sum_cpt3_prod(cpt3[0],cpt3[0])+
		    2.*sum_cpt3_prod(cpt3[1],cpt3[1])+
		    sum_cpt3_prod(cpt3[2],cpt3[2]));
	 fwrite((char *)&temp_calc,size_double,1,bfile);
	 triangle_order[triangle_number].v1=iv1;
	 triangle_order[triangle_number].v2=iv2;
	 triangle_order[triangle_number].v3=iv3;
	 if (triangle_number%25==24)
	   printf("Computed the data for %ld triangles.\n",
		  triangle_number+1); triangle_number++;
	} /* End of data computations for the current triangle */
      iv2=iv3;
      fscanf(trifile,"%ld ",&iv3);
     }  /* End reading the row of the triangulation file */
    fscanf(trifile,"%s ",base_buffer);
    iv1=atol(base_buffer);
   }  /* End reading of the triangulation file at the end of the cycle */
 while (base_buffer[0]!='B');
 /* There are triangle_number triangles labeled from 0 to triangle_number-1 */
 fclose(bfile);
 fseek(trifile,2L,SEEK_SET);   
 /* trifile points now to the first node of the first cell */
 printf("Starting computations for each node.\n");
 /* MAIN CYCLE OVER CELLS */
 bfile=fopen("blocks.tmp","rb");
 if (bfile==NULL)
 {
  printf("Cannot open temporary file for reading.\n");
  exit(1);
 }
 for (cont_nodi=0;cont_nodi<num_nodo;cont_nodi++)
 {
  N_C=0; iv1=cont_nodi;
  open_cella_flag=1;  /* I suppose the cell to be an open one */
  /* Salto il nodo base */
  fscanf(trifile," %ld ",&temp_index); N_C++;
  free_ind_node_list(lista_vicini);
  /* Creation of the first node */
  fscanf(trifile," %ld ",&temp_index); N_C++; 
  /* there exists at least a neighbouring node */
  lista_vicini=malloc(sizeof(struct ind_node_list));
  lista_vicini->glob_index=temp_index;
  vicino_pt=lista_vicini;
  /* I shall create the new memory locations to point with vicino_pt->next */
  /* I prepare the cycle for the reading of the other neighbouring nodes */
  fscanf(trifile," %ld ",&temp_index); /* N_C is increased later */
  while (temp_index!=-1)
  {
   N_C++;
   vicino_pt->next=malloc(sizeof(struct ind_node_list));
   vicino_pt->next->glob_index=temp_index;
   vicino_pt->next->next=NULL;
   vicino_pt=vicino_pt->next;
   fscanf(trifile," %ld ",&temp_index);
  }  /* Fine while lettura vicini */
  if (lista_vicini->glob_index==vicino_pt->glob_index)
    {
     open_cella_flag=0; /* The cell is a close one */
     N_C--; 
     /* N_C does not consider the second occurence 
        of the first neighbouring node */
    }
  /* In the case of a closed cell we have already read the second occurence
  ** of the first neighbouring node in order to optimize the computation
  ** of global indexes later.
  */
  /* Initialization of dynamical memory */
  A=malloc(size_double*25*N_C*N_C);
  if (A==NULL) {printf("Not enough memory\n"); exit(1); };
  gg=malloc(size_double*5*N_C);
  if (gg==NULL) {printf("Not enough memory\n"); exit(1); };
  v=malloc(size_double*5*N_C);
  if (v==NULL) {printf("Not enough memory\n"); exit(1); };
  DiagL=malloc(size_double*5*N_C);
  if (DiagL==NULL) {printf("Not enough memory\n"); exit(1); };
  /* Initialization of data */
  c=0.;
  for (i=0;i<5*N_C;i++)
   {
    for (j=0;j<=i;++j) A[i*5*N_C+j]=0.;
    gg[i]=0.;
   }
  /* Beginning computations for each triangle of every cell */
  vicino_pt=lista_vicini;
  /* initializing iv2 to optimize the for cycle with a false iv3 */
  iv3=vicino_pt->glob_index;
  vicino_pt=vicino_pt->next;
  /* N_C is the number of elements of the cell, nevertheless it is a closed
  ** one or an open one */
  for (nt=1;nt<=N_C-1-open_cella_flag;nt++)
  {
   iv2=iv3;
   iv3=vicino_pt->glob_index;
   /* Now if nt=N_C-1 in the case of a closed cell, iv3 is correct by
   ** means of the repetition done when reading the cell indexes */ 
   /* Conversion of the triangle into the canonical form where
      iv1<iv2, iv2<iv3 */
   shift_right=0;
   if (iv1>iv2 || iv1>iv3)
   {
    temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
    shift_right++;
   }
   /* completing the test */
   if (iv1>iv2 || iv1>iv3)
   {
    temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
    shift_right++;
   }
   /* Searching for the index of the current triangle */
   triangle_counter=0;
   stop=0;
   do
   {
   if (triangle_order[triangle_counter].v1!=iv1) triangle_counter++;
    else
     if (triangle_order[triangle_counter].v2!=iv2) triangle_counter++;
      else
       if (triangle_order[triangle_counter].v3!=iv3) triangle_counter++;
	else stop=1;
   }
   while (stop==0 && triangle_counter<triangle_number);
   if (triangle_counter==triangle_number)
    {
     printf("Cannot find the triangle in the temporary file.\n");
     exit(1);
    }
   /* triangle_counter is the index of the current triangle 
      in the triangulation */
   /* Transforming the triangle into the original form without changing
      the value of shift_right */
   if (shift_right==1) /* faccio uno shift_left */
   {
    temp_index=iv1; iv1=iv2; iv2=iv3; iv3=temp_index;
   }
   else
   if (shift_right==2) /* faccio un nuovo shift_right (-2=1 mod 3) */
   {
    temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
   }
   /* seeking the pointer in the temporary file */
   fseek(bfile,size_double*triangle_counter*136L,SEEK_SET);
   /* 
   ** There are 136 stored values per triangle, 120 ones
   ** for A matrix (15*16/2), 15 ones for vector gg and only one
   ** for the constant c.
   */
   /* Building linear system */
   /* Computation of the matrix A */
   for (ip=0;ip<=14;++ip)
   {
    ip_global=global_index(N_C,(ip+10*shift_right)%15,nt);
    for (jp=0;jp<=ip;++jp)
    {
     jp_global=global_index(N_C,(jp+10*shift_right)%15,nt);
     fread((char *)&temp_calc,size_double,1,bfile);
     if (jp_global<=ip_global)
       A[ip_global*5*N_C+jp_global]+=temp_calc;
      else
       A[jp_global*5*N_C+ip_global]+=temp_calc;
    }
   }
   /* First phase for gg */
   for (ip=0;ip<=14;++ip)
   {
    ip_global=global_index(N_C,(ip+10*shift_right)%15,nt);
    fread((char *)&temp_calc,size_double,1,bfile);
    gg[ip_global]+=temp_calc;
   }
   /* Computation of c */
   fread((char *)&temp_calc,size_double,1,bfile);
   c+=temp_calc;
   /* Second phase for gg */
   vicino_pt_temp=lista_vicini;
   for (i=1;i<N_C;++i)
   {
    if(vicino_pt_temp->glob_index!=iv2 && vicino_pt_temp->glob_index!=iv3)
    /* avoid test over iv1 when this one has the same value of cont_nodi */
     {
      gg[5*i]+=temp_calc;
      gg[5*i+1]+=temp_calc;
      gg[5*i+2]+=temp_calc;
      gg[5*i+3]+=temp_calc;
      gg[5*i+4]+=temp_calc;
     }
     vicino_pt_temp=vicino_pt_temp->next;
     /* The structure agrees with the loop of the previous reading */
   }
   vicino_pt=vicino_pt->next;
  }
  /* transforming phi (computed in gg) into -g (computed in gg too) */
  for (i=0;i<5*N_C;++i) gg[i]=-gg[i]+A[i*5*N_C+i]/2.+c;
  /* getting v as the solution of the linear system Av=gg 
    (remember that -g=gg) */
  err=cholesky(A,v,gg,5*N_C,1,5*N_C,1,DiagL);
  if (err==1) 
  {
   printf("Singular or bad matrix for the node %ld.\n",
	   cont_nodi);
   printf("Given false solution with all zeros. \n");
   for (i=0;i<5*N_C;++i) v[i]=0.;
  }
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,v[0],WIDTH,PREC,v[1],
	  WIDTH,PREC,v[2],WIDTH,PREC,v[3],WIDTH,PREC,v[4]);
  free(A); free(gg); free(v); free(DiagL);
  if (cont_nodi%25==24)
    printf("Completed %ld nodes\n",(cont_nodi+1));
 } /* end of main loop */
 fclose(bfile);
 fclose(trifile);
 fclose(outfile);
}

int global_index(N,i,j)
int N,i,j;
{
 switch(i/5)
 {
  /* return statement allows me to avoid break statements 
     in the various cases */
  case 0: /* Relative to the first vertex of the triangle */
	  return (i%5);
  case 1: /* Relative to the second vertex of the triangle */
	  return(5*j+i%5);
  case 2: /* Relative to the third vertex of the triangle */
	  return(5*(j%(N-1)+1)+i%5);
 }
}














