/*
** CLTGLDER.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
** REVISION DATE: May, 1999
**
** MODULES CALLED: CLT_D_A.C
**                 LINSIST.C 
**
** --------------------------------------------------------------------
**
** BACKGROUND
**
** Let (xi,yi,qi) i=0,N-1 be N points in R^3. Let us consider their
** projections (xi,yi) onto the plane z=0 and let T be a triangulation
** of these points. A triangulation of them is a set of triangles having
** for vertices all these points (and only these ones) so that two
** distinct triangles intersect at most in a common edge or in a common
** vertex. 
**
** The aim is to build the bivariate C1-function that interpolates
** at the points (xi,yi,qi). 
**
** The Clough-Tocher's interpolation method builds up such a function 
** using a polynomial patch on each triangle, provided that 
** one knows the first derivatives at each vertex of the 
** triangulation T. 
** The Clough-Tocher's polynomial element,
** splits a triangle in three sub-triangles (using the barycenter)
** over which it builds three cubic polynomial patches.
** 
** The program computes these partial derivatives (in the following
** we shall refer to them as to qix,qiy i=0,...,N-1, or more often
** as to the 2N-dimensional vector v) in the
** "smoothest" possible way for a Clough-Tocher interpolating function 
** over the whole triangulation T. 
** To evaluate the "smoothness" of a Clough-Tocher 
** interpolating function the program minimizes the functional
**
**     phi(v)=   sum  [ Integral[ (Qxx^2 + 2*Qxy^2 + Qyy^2  ) dx dy ] ]     
**               t in T      t      
** 
** with respect to v.
** 
** METHODOLOGY 
**
**     phi(v)= 1/2 * v' A v + g' v + c 
**
** is minimum when v solves 
**
**     A v + g = 0.
** 
** The program computes 
**
**     c=phi(0), A, gi=phi(ei)-1/2*aii-c, 
**
** where ei is the i-th vector of the canonical basis of R^(2N), 
** and solves the resulting linear system with 
** the Cholesky factorisation.
**
** 
** REFERENCES
**
**  G. Farin:
**     "Triangular Bernstein-Bezier patches",
**      Comput. Aided Geom. Design,3:83-127, 1986.
**
**  P. Alfeld:
**    "Derivative Generation for multivariate 
**     scattered data functional minimization", 
**     Comp. Aided Graph. Design,2: 281-296, 1985.
**
**  D. Fasoli,
**      Laurea's Thesis, University of Padua, 1995 (in Italian). 
**
**
** --------------------------------------------------------------------
** SYNOPSIS:  cltglder file1 file2 [file3]
**
** file1 is the data file of the 3-dimensional points. 
** The coordinates of the points may be separated by any 
** combination of spaces, tabs and carriage returns, 
** It is recommend to use a row per point, for clearness. 
** 
** file2 is the triangulation data file in the format generated by
** the program 'do_trian'. 
** This file has three sections:
** the first, starting with a single capital letter C in the first row, 
** lists in subsequent blocks separated by -1, the index of
** a node and those of its neighbours in a counterclockwise order. 
** The program uses only this first section and the heading of the
** second section, that is a capital letter B alone in a row.
**
** file3 is the output derivative data file. 
** If file3 is omitted, the standard output is used instead.
** The values in a same row refer to the same vertex of the triangulation.
** The first column is the x-derivative, the second one is the y-derivative,
** the third one is 0, the fourth one is 0 and the last one is 0 too.
** The format of the output file is handled by the constants WIDTH and PREC
** (#DEFINE constants) that are the width of the output field, 
** and the digits of precision.
**        
** The program reads at most MAX_NUM_NODI (#DEFINE constant) input points.
** If more points are given, a warning message will be printed.
** 
* -------------------------------------------------------------------------------
** SUBROUTINES AND FUNCTIONS
**
** int global_index(int i,int iv1,int iv2,int iv3) 
**
** This function transforms i, a local index in the range 0..5 
** corresponding to the sequence of parameters 
** qx(V1), qy(V1),
** qx(V2), qy(V2), 
** qx(V3), qy(V3) 
** for the triangle V1,V2,V3 into a global index according to
** the vertex among V1,V2,V3 to which it refers to. 
** Global indexes are sorted by the index of the node 
** in the triangulation
** and by the derivation sequence qx,qy.  
**
** Output: the global index of the parameter.
**
** ------------------------------------------------------------------------------
*/

#include<stdio.h>
#include<stdlib.h>
#include<math.h>
#define MAX_NUM_NODI 100000
#define WIDTH 20   /* Format width of real numbers written to files */
#define PREC 10    /* Decimal precision width of real numbers written to files */

double derivecptclt_v(),compute_d_j_clt(),compute_d_j_clt_null(),
       sum_cpt1_prod();
int global_index();

/* Declarations for global variables */
double q1,q2,q3;
double u1,v1,u2,v2,u3,v3,t1,t2,t3;

void main(argc,argv)
int argc;
char *argv[];
{
  FILE *infile, *trifile, *outfile;
  char base_buffer[20];
  int num_nodo,iv1,iv2,iv3;	   
  double x[MAX_NUM_NODI],y[MAX_NUM_NODI],z[MAX_NUM_NODI];
  double u4,v4,At;
  int i,j,ip_global,jp_global,nt;
  int ip,jp,r,s,err;
  double cpt1[54][3],temp_calc;
  double *A, *v, *gg, c;
  double *DiagL;
  int IndT1[4][4]={{2,9,4,1},{8,13,5,0},{17,16,0,0},{19,0,0,0}};
  int IndT2[4][4]={{3,12,7,2},{11,14,8,0},{18,17,0,0},{19,0,0,0}};
  int IndT3[4][4]={{1,6,10,3},{5,15,11,0},{16,18,0,0},{19,0,0,0}};

 infile=NULL; trifile=NULL; outfile=NULL; 
 A=NULL; v=NULL; gg=NULL; DiagL=NULL;
 if (argc==1)
   {
    printf("\nStart the program with three parameters:\n");
    printf("First parameter: data file name.\n");
    printf("Second parameter: triangulation file name.\n");
    printf("Third parameter: derivative file name.\n");
    printf("                 (Omit it for screen output.)\n");
    exit(1);
   }
 /* Reading */
 infile=fopen(argv[1],"r");
 if (infile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[1]);
    exit(1);
   }
 if (argc==2)
   {
    printf("Missing triangulation file.\n");
    exit(1);
   }
 /* data reading */
 printf("Wait for data reading.\n"); 
 num_nodo=0;
 while (!feof(infile))
 {
  if (num_nodo<MAX_NUM_NODI)
   {
    fscanf(infile," %lf %lf %lf ",&x[num_nodo],&y[num_nodo],&z[num_nodo]);
      /* the spaces are essential part of the input form */
    num_nodo=num_nodo+1;
   }
   else
   {
    printf("Too many input data.\n");
    exit(1);
   }
 }
 if (num_nodo<3)
    {
     printf("I need at least three input points.\n");
     exit(1);
    }
 fclose(infile);
 /* end of data reading */
 /* opening triangulation file */
 trifile=fopen(argv[2],"r");
 if (trifile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[2]);
    exit(1);
   }
 if (fgetc(trifile)!='C')
   {
    printf("File %s in a no correct format.\n",argv[2]);
    exit(1);
   }
 else fgetc(trifile); /* reading carriage return */
 /* apro file output */
 if (argc==3) outfile=stdout; else outfile=fopen(argv[3],"w");
 if (outfile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[3]);
    exit(1);
   }
 /* Creation of the dynamical structures needed by the program */
 A=malloc(sizeof(double)*4*num_nodo*num_nodo);
 if (A==NULL) {printf("Not enough memory\n"); exit(1); };
 gg=malloc(sizeof(double)*2*num_nodo);
 if (gg==NULL) {printf("Not enough memory\n"); exit(1); };
 v=malloc(sizeof(double)*2*num_nodo);
 if (v==NULL) {printf("Not enough memory\n"); exit(1); };
 DiagL=malloc(sizeof(double)*2*num_nodo);
 if (DiagL==NULL) {printf("Not enough memory\n"); exit(1); };
 /* Data initialization */
  c=0.;
  for (i=0;i<2*num_nodo;i++)
   {
    for (j=0;j<=i;++j) A[i*2*num_nodo+j]=0.;
    gg[i]=0.;
   } 

 /* MAIN CYCLE OF COMPUTATIONS OVER TRIANGLES */
 printf("Starting computations for the data of each triangle.\n");
 fscanf(trifile,"%d ",&iv1);
 nt=1;
 /* Counter for the triangle being assembled */
 do
   {
    fscanf(trifile,"%d %d ",&iv2,&iv3);
    while(iv3!=-1)
     {
      if ((iv1<iv2)&&(iv1<iv3))
	{/* New triangle: beginning computations and assembling phase */
	 u1=x[iv1]; u2=x[iv2]; u3=x[iv3];
	 v1=y[iv1]; v2=y[iv2]; v3=y[iv3];
	 q1=z[iv1]; q2=z[iv2]; q3=z[iv3];
	 u4=(u1+u2+u3)/3.; v4=(v1+v2+v3)/3.;
	 At=fabs((u2*v3-u3*v2+u3*v1-u1*v3+u1*v2-u2*v1)/2.);
	 t1= ((u4-u1)*(u2-u1)+(v4-v1)*(v2-v1))/
	     (pow((u2-u1),2.)+pow((v2-v1),2.));
	 t2= ((u4-u2)*(u3-u2)+(v4-v2)*(v3-v2))/
	     (pow((u3-u2),2.)+pow((v3-v2),2.));
	 t3= ((u4-u3)*(u1-u3)+(v4-v3)*(v1-v3))/
	     (pow((u1-u3),2.)+pow((v1-v3),2.));
	 /* Computations and updating assembling phase */
  	 /* c constant*/
         i=0; for (r=0;r<=1;++r) for (s=0;s<=1-r;++s)
	 {
	  /* First triangle V4 V1 V2 */
	  temp_calc=compute_d_j_clt_null(IndT1[r+2][s])*(v2-v1)*(v2-v1);
	  temp_calc+=compute_d_j_clt_null(IndT1[r][s+2])*(v4-v2)*(v4-v2);
	  temp_calc+=compute_d_j_clt_null(IndT1[r][s])*(v1-v4)*(v1-v4);
	  temp_calc+=compute_d_j_clt_null(IndT1[r+1][s+1])*2*(v2-v1)*(v4-v2);
	  temp_calc+=compute_d_j_clt_null(IndT1[r+1][s])*2*(v2-v1)*(v1-v4);
	  temp_calc+=compute_d_j_clt_null(IndT1[r][s+1])*2*(v4-v2)*(v1-v4);
	  cpt1[0][i]=temp_calc;

	  temp_calc=-compute_d_j_clt_null(IndT1[r+2][s])*(v2-v1)*(u2-u1);
	  temp_calc-=compute_d_j_clt_null(IndT1[r][s+2])*(v4-v2)*(u4-u2);
	  temp_calc-=compute_d_j_clt_null(IndT1[r][s])*(v1-v4)*(u1-u4);
	  temp_calc-=compute_d_j_clt_null(IndT1[r+1][s+1])*
		     ((v2-v1)*(u4-u2)+(v4-v2)*(u2-u1));
	  temp_calc-=compute_d_j_clt_null(IndT1[r+1][s])*
		     ((v2-v1)*(u1-u4)+(v1-v4)*(u2-u1));
	  temp_calc-=compute_d_j_clt_null(IndT1[r][s+1])*
		     ((v4-v2)*(u1-u4)+(v1-v4)*(u4-u2));
	  cpt1[1][i]=temp_calc;

	  temp_calc=compute_d_j_clt_null(IndT1[r+2][s])*(u2-u1)*(u2-u1);
	  temp_calc+=compute_d_j_clt_null(IndT1[r][s+2])*(u4-u2)*(u4-u2);
	  temp_calc+=compute_d_j_clt_null(IndT1[r][s])*(u1-u4)*(u1-u4);
	  temp_calc+=compute_d_j_clt_null(IndT1[r+1][s+1])*2*(u2-u1)*(u4-u2);
	  temp_calc+=compute_d_j_clt_null(IndT1[r+1][s])*2*(u2-u1)*(u1-u4);
	  temp_calc+=compute_d_j_clt_null(IndT1[r][s+1])*2*(u4-u2)*(u1-u4);
	  cpt1[2][i]=temp_calc;

	  /* Second triangle V4 V2 V3 */
	  temp_calc=compute_d_j_clt_null(IndT2[r+2][s])*(v3-v2)*(v3-v2);
	  temp_calc+=compute_d_j_clt_null(IndT2[r][s+2])*(v4-v3)*(v4-v3);
	  temp_calc+=compute_d_j_clt_null(IndT2[r][s])*(v2-v4)*(v2-v4);
	  temp_calc+=compute_d_j_clt_null(IndT2[r+1][s+1])*2*(v3-v2)*(v4-v3);
	  temp_calc+=compute_d_j_clt_null(IndT2[r+1][s])*2*(v3-v2)*(v2-v4);
	  temp_calc+=compute_d_j_clt_null(IndT2[r][s+1])*2*(v4-v3)*(v2-v4);
	  cpt1[3][i]=temp_calc;

	  temp_calc=-compute_d_j_clt_null(IndT2[r+2][s])*(v3-v2)*(u3-u2);
	  temp_calc-=compute_d_j_clt_null(IndT2[r][s+2])*(v4-v3)*(u4-u3);
	  temp_calc-=compute_d_j_clt_null(IndT2[r][s])*(v2-v4)*(u2-u4);
	  temp_calc-=compute_d_j_clt_null(IndT2[r+1][s+1])*
		     ((v3-v2)*(u4-u3)+(v4-v3)*(u3-u2));
	  temp_calc-=compute_d_j_clt_null(IndT2[r+1][s])*
		     ((v3-v2)*(u2-u4)+(v2-v4)*(u3-u2));
	  temp_calc-=compute_d_j_clt_null(IndT2[r][s+1])*
		     ((v4-v3)*(u2-u4)+(v2-v4)*(u4-u3));
	  cpt1[4][i]=temp_calc;

	  temp_calc=compute_d_j_clt_null(IndT2[r+2][s])*(u3-u2)*(u3-u2);
	  temp_calc+=compute_d_j_clt_null(IndT2[r][s+2])*(u4-u3)*(u4-u3);
	  temp_calc+=compute_d_j_clt_null(IndT2[r][s])*(u2-u4)*(u2-u4);
	  temp_calc+=compute_d_j_clt_null(IndT2[r+1][s+1])*2*(u3-u2)*(u4-u3);
	  temp_calc+=compute_d_j_clt_null(IndT2[r+1][s])*2*(u3-u2)*(u2-u4);
	  temp_calc+=compute_d_j_clt_null(IndT2[r][s+1])*2*(u4-u3)*(u2-u4);
	  cpt1[5][i]=temp_calc;

	  /* Third triangle V4 V3 V1 */
	  temp_calc=compute_d_j_clt_null(IndT3[r+2][s])*(v1-v3)*(v1-v3);
	  temp_calc+=compute_d_j_clt_null(IndT3[r][s+2])*(v4-v1)*(v4-v1);
	  temp_calc+=compute_d_j_clt_null(IndT3[r][s])*(v3-v4)*(v3-v4);
	  temp_calc+=compute_d_j_clt_null(IndT3[r+1][s+1])*2*(v1-v3)*(v4-v1);
	  temp_calc+=compute_d_j_clt_null(IndT3[r+1][s])*2*(v1-v3)*(v3-v4);
	  temp_calc+=compute_d_j_clt_null(IndT3[r][s+1])*2*(v4-v1)*(v3-v4);
	  cpt1[6][i]=temp_calc;

	  temp_calc=-compute_d_j_clt_null(IndT3[r+2][s])*(v1-v3)*(u1-u3);
	  temp_calc-=compute_d_j_clt_null(IndT3[r][s+2])*(v4-v1)*(u4-u1);
	  temp_calc-=compute_d_j_clt_null(IndT3[r][s])*(v3-v4)*(u3-u4);
	  temp_calc-=compute_d_j_clt_null(IndT3[r+1][s+1])*
		     ((v1-v3)*(u4-u1)+(v4-v1)*(u1-u3));
	  temp_calc-=compute_d_j_clt_null(IndT3[r+1][s])*
		     ((v1-v3)*(u3-u4)+(v3-v4)*(u1-u3));
	  temp_calc-=compute_d_j_clt_null(IndT3[r][s+1])*
		     ((v4-v1)*(u3-u4)+(v3-v4)*(u4-u1));
	  cpt1[7][i]=temp_calc;

	  temp_calc=compute_d_j_clt_null(IndT3[r+2][s])*(u1-u3)*(u1-u3);
	  temp_calc+=compute_d_j_clt_null(IndT3[r][s+2])*(u4-u1)*(u4-u1);
	  temp_calc+=compute_d_j_clt_null(IndT3[r][s])*(u3-u4)*(u3-u4);
	  temp_calc+=compute_d_j_clt_null(IndT3[r+1][s+1])*2*(u1-u3)*(u4-u1);
	  temp_calc+=compute_d_j_clt_null(IndT3[r+1][s])*2*(u1-u3)*(u3-u4);
	  temp_calc+=compute_d_j_clt_null(IndT3[r][s+1])*2*(u4-u1)*(u3-u4);
	  cpt1[8][i]=temp_calc;
	  i++;
	 }  /* End for over s (and over r) */
	 temp_calc=(sum_cpt1_prod(cpt1[0],cpt1[0])+
		    2.*sum_cpt1_prod(cpt1[1],cpt1[1])+
		    sum_cpt1_prod(cpt1[2],cpt1[2]));
	 temp_calc+=(sum_cpt1_prod(cpt1[3],cpt1[3])+
		     2.*sum_cpt1_prod(cpt1[4],cpt1[4])+
		     sum_cpt1_prod(cpt1[5],cpt1[5]));
	 temp_calc+=(sum_cpt1_prod(cpt1[6],cpt1[6])+
		     2.*sum_cpt1_prod(cpt1[7],cpt1[7])+
		     sum_cpt1_prod(cpt1[8],cpt1[8]));
         /* The constant 3./(8. *pow(At/3.,3.)
          comes from (6/4 a^2)^2 * (2a / (2+1)(2+2) )  
          where a=At/3 */
	 temp_calc=temp_calc*3./(8. *pow(At/3.,3.));
	 c=c+temp_calc;
	 /* gg vector */
         /* First phase */
	 for (i=0;i<2*num_nodo;++i)
	 {
	  if(iv1!=i/2 && iv2!=i/2 && iv3!=i/2)
	  {
	   gg[i]+=temp_calc;
	  }
	 }
	 /* Second phase */
	 for (ip=0;ip<=5;++ip)
	 {
	  i=0; for (r=0;r<=1;++r) for (s=0;s<=1-r;++s)
	  {
	   /* First triangle V4 V1 V2 */
	   temp_calc=compute_d_j_clt(IndT1[r+2][s],ip)*(v2-v1)*(v2-v1);
	   temp_calc+=compute_d_j_clt(IndT1[r][s+2],ip)*(v4-v2)*(v4-v2);
	   temp_calc+=compute_d_j_clt(IndT1[r][s],ip)*(v1-v4)*(v1-v4);
	   temp_calc+=compute_d_j_clt(IndT1[r+1][s+1],ip)*2*(v2-v1)*(v4-v2);
	   temp_calc+=compute_d_j_clt(IndT1[r+1][s],ip)*2*(v2-v1)*(v1-v4);
	   temp_calc+=compute_d_j_clt(IndT1[r][s+1],ip)*2*(v4-v2)*(v1-v4);
	   cpt1[0][i]=temp_calc;

	   temp_calc=-compute_d_j_clt(IndT1[r+2][s],ip)*(v2-v1)*(u2-u1);
	   temp_calc-=compute_d_j_clt(IndT1[r][s+2],ip)*(v4-v2)*(u4-u2);
	   temp_calc-=compute_d_j_clt(IndT1[r][s],ip)*(v1-v4)*(u1-u4);
	   temp_calc-=compute_d_j_clt(IndT1[r+1][s+1],ip)*
		      ((v2-v1)*(u4-u2)+(v4-v2)*(u2-u1));
	   temp_calc-=compute_d_j_clt(IndT1[r+1][s],ip)*
		      ((v2-v1)*(u1-u4)+(v1-v4)*(u2-u1));
	   temp_calc-=compute_d_j_clt(IndT1[r][s+1],ip)*
		      ((v4-v2)*(u1-u4)+(v1-v4)*(u4-u2));
	   cpt1[1][i]=temp_calc;

	   temp_calc=compute_d_j_clt(IndT1[r+2][s],ip)*(u2-u1)*(u2-u1);
	   temp_calc+=compute_d_j_clt(IndT1[r][s+2],ip)*(u4-u2)*(u4-u2);
	   temp_calc+=compute_d_j_clt(IndT1[r][s],ip)*(u1-u4)*(u1-u4);
	   temp_calc+=compute_d_j_clt(IndT1[r+1][s+1],ip)*2*(u2-u1)*(u4-u2);
	   temp_calc+=compute_d_j_clt(IndT1[r+1][s],ip)*2*(u2-u1)*(u1-u4);
	   temp_calc+=compute_d_j_clt(IndT1[r][s+1],ip)*2*(u4-u2)*(u1-u4);
	   cpt1[2][i]=temp_calc;

	   /* Second triangle V4 V2 V3 */
	   temp_calc=compute_d_j_clt(IndT2[r+2][s],ip)*(v3-v2)*(v3-v2);
	   temp_calc+=compute_d_j_clt(IndT2[r][s+2],ip)*(v4-v3)*(v4-v3);
	   temp_calc+=compute_d_j_clt(IndT2[r][s],ip)*(v2-v4)*(v2-v4);
	   temp_calc+=compute_d_j_clt(IndT2[r+1][s+1],ip)*2*(v3-v2)*(v4-v3);
	   temp_calc+=compute_d_j_clt(IndT2[r+1][s],ip)*2*(v3-v2)*(v2-v4);
	   temp_calc+=compute_d_j_clt(IndT2[r][s+1],ip)*2*(v4-v3)*(v2-v4);
	   cpt1[3][i]=temp_calc;

	   temp_calc=-compute_d_j_clt(IndT2[r+2][s],ip)*(v3-v2)*(u3-u2);
	   temp_calc-=compute_d_j_clt(IndT2[r][s+2],ip)*(v4-v3)*(u4-u3);
	   temp_calc-=compute_d_j_clt(IndT2[r][s],ip)*(v2-v4)*(u2-u4);
	   temp_calc-=compute_d_j_clt(IndT2[r+1][s+1],ip)*
		      ((v3-v2)*(u4-u3)+(v4-v3)*(u3-u2));
	   temp_calc-=compute_d_j_clt(IndT2[r+1][s],ip)*
		      ((v3-v2)*(u2-u4)+(v2-v4)*(u3-u2));
	   temp_calc-=compute_d_j_clt(IndT2[r][s+1],ip)*
		      ((v4-v3)*(u2-u4)+(v2-v4)*(u4-u3));
	   cpt1[4][i]=temp_calc;

	   temp_calc=compute_d_j_clt(IndT2[r+2][s],ip)*(u3-u2)*(u3-u2);
	   temp_calc+=compute_d_j_clt(IndT2[r][s+2],ip)*(u4-u3)*(u4-u3);
	   temp_calc+=compute_d_j_clt(IndT2[r][s],ip)*(u2-u4)*(u2-u4);
	   temp_calc+=compute_d_j_clt(IndT2[r+1][s+1],ip)*2*(u3-u2)*(u4-u3);
	   temp_calc+=compute_d_j_clt(IndT2[r+1][s],ip)*2*(u3-u2)*(u2-u4);
	   temp_calc+=compute_d_j_clt(IndT2[r][s+1],ip)*2*(u4-u3)*(u2-u4);
	   cpt1[5][i]=temp_calc;

	   /* Third triangle V4 V3 V1 */
	   temp_calc=compute_d_j_clt(IndT3[r+2][s],ip)*(v1-v3)*(v1-v3);
	   temp_calc+=compute_d_j_clt(IndT3[r][s+2],ip)*(v4-v1)*(v4-v1);
	   temp_calc+=compute_d_j_clt(IndT3[r][s],ip)*(v3-v4)*(v3-v4);
	   temp_calc+=compute_d_j_clt(IndT3[r+1][s+1],ip)*2*(v1-v3)*(v4-v1);
	   temp_calc+=compute_d_j_clt(IndT3[r+1][s],ip)*2*(v1-v3)*(v3-v4);
	   temp_calc+=compute_d_j_clt(IndT3[r][s+1],ip)*2*(v4-v1)*(v3-v4);
	   cpt1[6][i]=temp_calc;

	   temp_calc=-compute_d_j_clt(IndT3[r+2][s],ip)*(v1-v3)*(u1-u3);
	   temp_calc-=compute_d_j_clt(IndT3[r][s+2],ip)*(v4-v1)*(u4-u1);
	   temp_calc-=compute_d_j_clt(IndT3[r][s],ip)*(v3-v4)*(u3-u4);
	   temp_calc-=compute_d_j_clt(IndT3[r+1][s+1],ip)*
		      ((v1-v3)*(u4-u1)+(v4-v1)*(u1-u3));
	   temp_calc-=compute_d_j_clt(IndT3[r+1][s],ip)*
		      ((v1-v3)*(u3-u4)+(v3-v4)*(u1-u3));
	   temp_calc-=compute_d_j_clt(IndT3[r][s+1],ip)*
		      ((v4-v1)*(u3-u4)+(v3-v4)*(u4-u1));
	   cpt1[7][i]=temp_calc;

	   temp_calc=compute_d_j_clt(IndT3[r+2][s],ip)*(u1-u3)*(u1-u3);
	   temp_calc+=compute_d_j_clt(IndT3[r][s+2],ip)*(u4-u1)*(u4-u1);
	   temp_calc+=compute_d_j_clt(IndT3[r][s],ip)*(u3-u4)*(u3-u4);
	   temp_calc+=compute_d_j_clt(IndT3[r+1][s+1],ip)*2*(u1-u3)*(u4-u1);
	   temp_calc+=compute_d_j_clt(IndT3[r+1][s],ip)*2*(u1-u3)*(u3-u4);
	   temp_calc+=compute_d_j_clt(IndT3[r][s+1],ip)*2*(u4-u1)*(u3-u4);
	   cpt1[8][i]=temp_calc;
	   i++;
	  } /* End for over s (and over r) */
	  temp_calc=(sum_cpt1_prod(cpt1[0],cpt1[0])+
		     2.*sum_cpt1_prod(cpt1[1],cpt1[1])+
		     sum_cpt1_prod(cpt1[2],cpt1[2]));
	  temp_calc+=(sum_cpt1_prod(cpt1[3],cpt1[3])+
		      2.*sum_cpt1_prod(cpt1[4],cpt1[4])+
		      sum_cpt1_prod(cpt1[5],cpt1[5]));
	  temp_calc+=(sum_cpt1_prod(cpt1[6],cpt1[6])+
		      2.*sum_cpt1_prod(cpt1[7],cpt1[7])+
		      sum_cpt1_prod(cpt1[8],cpt1[8]));
         /* The constant 3./(8. *pow(At/3.,3.)
            comes from (6/4 a^2)^2 * (2a / (2+1)(2+2) ) 
            where a=At/3 */
	  temp_calc=temp_calc*3./(8. *pow(At/3.,3.));
          gg[global_index(ip,iv1,iv2,iv3)]+=temp_calc;  
	 } /* End ip for*/
	 /* A matrix
         ** The computation is splitted in two parts in order to perform
         ** the first one only 6 times instead of 21 times as we should do
         ** using only the second part.
         */
	 for (ip=0;ip<=5;ip++)
	 {
	  i=0; for (r=0;r<=1;++r) for (s=0;s<=1-r;++s)
	  {
	   /* First triangle V4 V1 V2 */
	   temp_calc=derivecptclt_v(IndT1[r+2][s],ip)*(v2-v1)*(v2-v1);
	   temp_calc+=derivecptclt_v(IndT1[r][s+2],ip)*(v4-v2)*(v4-v2);
	   temp_calc+=derivecptclt_v(IndT1[r][s],ip)*(v1-v4)*(v1-v4);
	   temp_calc+=derivecptclt_v(IndT1[r+1][s+1],ip)*2*(v2-v1)*(v4-v2);
	   temp_calc+=derivecptclt_v(IndT1[r+1][s],ip)*2*(v2-v1)*(v1-v4);
	   temp_calc+=derivecptclt_v(IndT1[r][s+1],ip)*2*(v4-v2)*(v1-v4);
	   cpt1[ip][i]=temp_calc;

	   temp_calc=-derivecptclt_v(IndT1[r+2][s],ip)*(v2-v1)*(u2-u1);
	   temp_calc-=derivecptclt_v(IndT1[r][s+2],ip)*(v4-v2)*(u4-u2);
	   temp_calc-=derivecptclt_v(IndT1[r][s],ip)*(v1-v4)*(u1-u4);
	   temp_calc-=derivecptclt_v(IndT1[r+1][s+1],ip)*
		      ((v2-v1)*(u4-u2)+(v4-v2)*(u2-u1));
	   temp_calc-=derivecptclt_v(IndT1[r+1][s],ip)*
		      ((v2-v1)*(u1-u4)+(v1-v4)*(u2-u1));
	   temp_calc-=derivecptclt_v(IndT1[r][s+1],ip)*
		      ((v4-v2)*(u1-u4)+(v1-v4)*(u4-u2));
	   cpt1[ip+6][i]=temp_calc;

	   temp_calc=derivecptclt_v(IndT1[r+2][s],ip)*(u2-u1)*(u2-u1);
	   temp_calc+=derivecptclt_v(IndT1[r][s+2],ip)*(u4-u2)*(u4-u2);
	   temp_calc+=derivecptclt_v(IndT1[r][s],ip)*(u1-u4)*(u1-u4);
	   temp_calc+=derivecptclt_v(IndT1[r+1][s+1],ip)*2*(u2-u1)*(u4-u2);
	   temp_calc+=derivecptclt_v(IndT1[r+1][s],ip)*2*(u2-u1)*(u1-u4);
	   temp_calc+=derivecptclt_v(IndT1[r][s+1],ip)*2*(u4-u2)*(u1-u4);
	   cpt1[ip+12][i]=temp_calc;

	   /* Second triangle V4 V2 V3 */
	   temp_calc=derivecptclt_v(IndT2[r+2][s],ip)*(v3-v2)*(v3-v2);
	   temp_calc+=derivecptclt_v(IndT2[r][s+2],ip)*(v4-v3)*(v4-v3);
	   temp_calc+=derivecptclt_v(IndT2[r][s],ip)*(v2-v4)*(v2-v4);
	   temp_calc+=derivecptclt_v(IndT2[r+1][s+1],ip)*2*(v3-v2)*(v4-v3);
	   temp_calc+=derivecptclt_v(IndT2[r+1][s],ip)*2*(v3-v2)*(v2-v4);
	   temp_calc+=derivecptclt_v(IndT2[r][s+1],ip)*2*(v4-v3)*(v2-v4);
	   cpt1[ip+18][i]=temp_calc;

	   temp_calc=-derivecptclt_v(IndT2[r+2][s],ip)*(v3-v2)*(u3-u2);
	   temp_calc-=derivecptclt_v(IndT2[r][s+2],ip)*(v4-v3)*(u4-u3);
	   temp_calc-=derivecptclt_v(IndT2[r][s],ip)*(v2-v4)*(u2-u4);
	   temp_calc-=derivecptclt_v(IndT2[r+1][s+1],ip)*
		      ((v3-v2)*(u4-u3)+(v4-v3)*(u3-u2));
	   temp_calc-=derivecptclt_v(IndT2[r+1][s],ip)*
		      ((v3-v2)*(u2-u4)+(v2-v4)*(u3-u2));
	   temp_calc-=derivecptclt_v(IndT2[r][s+1],ip)*
		      ((v4-v3)*(u2-u4)+(v2-v4)*(u4-u3));
	   cpt1[ip+24][i]=temp_calc;

	   temp_calc=derivecptclt_v(IndT2[r+2][s],ip)*(u3-u2)*(u3-u2);
	   temp_calc+=derivecptclt_v(IndT2[r][s+2],ip)*(u4-u3)*(u4-u3);
	   temp_calc+=derivecptclt_v(IndT2[r][s],ip)*(u2-u4)*(u2-u4);
	   temp_calc+=derivecptclt_v(IndT2[r+1][s+1],ip)*2*(u3-u2)*(u4-u3);
	   temp_calc+=derivecptclt_v(IndT2[r+1][s],ip)*2*(u3-u2)*(u2-u4);
	   temp_calc+=derivecptclt_v(IndT2[r][s+1],ip)*2*(u4-u3)*(u2-u4);
	   cpt1[ip+30][i]=temp_calc;

	   /* Third triangle V4 V3 V1 */
	   temp_calc=derivecptclt_v(IndT3[r+2][s],ip)*(v1-v3)*(v1-v3);
	   temp_calc+=derivecptclt_v(IndT3[r][s+2],ip)*(v4-v1)*(v4-v1);
	   temp_calc+=derivecptclt_v(IndT3[r][s],ip)*(v3-v4)*(v3-v4);
	   temp_calc+=derivecptclt_v(IndT3[r+1][s+1],ip)*2*(v1-v3)*(v4-v1);
	   temp_calc+=derivecptclt_v(IndT3[r+1][s],ip)*2*(v1-v3)*(v3-v4);
	   temp_calc+=derivecptclt_v(IndT3[r][s+1],ip)*2*(v4-v1)*(v3-v4);
	   cpt1[ip+36][i]=temp_calc;

	   temp_calc=-derivecptclt_v(IndT3[r+2][s],ip)*(v1-v3)*(u1-u3);
	   temp_calc-=derivecptclt_v(IndT3[r][s+2],ip)*(v4-v1)*(u4-u1);
	   temp_calc-=derivecptclt_v(IndT3[r][s],ip)*(v3-v4)*(u3-u4);
	   temp_calc-=derivecptclt_v(IndT3[r+1][s+1],ip)*
		      ((v1-v3)*(u4-u1)+(v4-v1)*(u1-u3));
	   temp_calc-=derivecptclt_v(IndT3[r+1][s],ip)*
		      ((v1-v3)*(u3-u4)+(v3-v4)*(u1-u3));
	   temp_calc-=derivecptclt_v(IndT3[r][s+1],ip)*
		      ((v4-v1)*(u3-u4)+(v3-v4)*(u4-u1));
	   cpt1[ip+42][i]=temp_calc;

	   temp_calc=derivecptclt_v(IndT3[r+2][s],ip)*(u1-u3)*(u1-u3);
	   temp_calc+=derivecptclt_v(IndT3[r][s+2],ip)*(u4-u1)*(u4-u1);
	   temp_calc+=derivecptclt_v(IndT3[r][s],ip)*(u3-u4)*(u3-u4);
	   temp_calc+=derivecptclt_v(IndT3[r+1][s+1],ip)*2*(u1-u3)*(u4-u1);
	   temp_calc+=derivecptclt_v(IndT3[r+1][s],ip)*2*(u1-u3)*(u3-u4);
	   temp_calc+=derivecptclt_v(IndT3[r][s+1],ip)*2*(u4-u1)*(u3-u4);
	   cpt1[ip+48][i]=temp_calc;
	   i++;
	  } /* End for about s (and about r) */
	 } /* end ip for */
	 for (ip=0;ip<=5;++ip)
           {
            ip_global=global_index(ip,iv1,iv2,iv3);
	    for (jp=0;jp<=ip;++jp)
	     {
              jp_global=global_index(jp,iv1,iv2,iv3);
	      temp_calc=(sum_cpt1_prod(cpt1[ip],cpt1[jp])+
			 2.*sum_cpt1_prod(cpt1[ip+6],cpt1[jp+6])+
			 sum_cpt1_prod(cpt1[ip+12],cpt1[jp+12]));
	      temp_calc+=(sum_cpt1_prod(cpt1[ip+18],cpt1[jp+18])+
			  2.*sum_cpt1_prod(cpt1[ip+24],cpt1[jp+24])+
			  sum_cpt1_prod(cpt1[ip+30],cpt1[jp+30]));
	      temp_calc+=(sum_cpt1_prod(cpt1[ip+36],cpt1[jp+36])+
			  2.*sum_cpt1_prod(cpt1[ip+42],cpt1[jp+42])+
			  sum_cpt1_prod(cpt1[ip+48],cpt1[jp+48]));
	      /* The constant 3./(8. *pow(At/3.,3.)
                 comes from (6/4 a^2)^2 * (2a / (2+1)(2+2) ) 
                 where a=At/3 */
              temp_calc=temp_calc*6./(8. *pow(At/3.,3.));
	      if(ip_global>=jp_global)
	        A[ip_global*2*num_nodo+jp_global]+=temp_calc;
	       else
	        A[jp_global*2*num_nodo+ip_global]+=temp_calc;
	     } /* end jp for */
            } /* end ip for */

	 if (nt%25==0)
	   printf("Assembled %d triangles.\n",nt); nt++;
	} /* End of the computations and of the assembling phase for the
	  **   current triangle */
      iv2=iv3;
      fscanf(trifile,"%d ",&iv3);
     }  /* End of the reading of the row in the triangulation file */
    fscanf(trifile,"%s ",base_buffer);
    iv1=atol(base_buffer);
   }  /* End of reading triangulation file at the exit of the cycle */
 while (base_buffer[0]!='B');
 fclose(trifile);
  /* transforming phi (computed in gg) to -g (computed in gg too) */
  for (i=0;i<2*num_nodo;++i) gg[i]=-gg[i]+A[i*2*num_nodo+i]/2.+c;
  printf("Solving system.\n");
  /* getting v as the solution of the linear system Av=gg 
  ** (remember that -g=gg) */
  err=cholesky(A,v,gg,2*num_nodo,1,2*num_nodo,1,DiagL);
  if (err==1) 
  {
   printf("Singular or bad matrix.\n");
   exit(1);
  }
  printf("Output copying.\n");
  for (i=0;i<num_nodo;++i)
  {
   fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	   WIDTH,PREC,v[2*i],WIDTH,PREC,v[2*i+1],
	   WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  }
  fclose(outfile);
}

int global_index(i,iv1,iv2,iv3)
int i,iv1,iv2,iv3;
{
 switch(i/2)
 {
  /* return statement allows me to avoid break statements 
     in the various cases */
  case 0: /* Relative to the first vertex of the triangle */
	  return (2*iv1+i%2);
  case 1: /* Relative to the second vertex of the triangle */
	  return (2*iv2+i%2);
  case 2: /* Relative to the third vertex of the triangle */
	  return (2*iv3+i%2);
  default: printf("The conversion of indexes is not possible.\n"); exit(1);
 }
}

