(* :Title: datachv52.m   *)

(* :Context: chebyshev`datachv52` *)

(* :Author:  Zlia da ROCHA *)

(* :Address: 
            Departamento de Matemtica Aplicada da
            Faculdade de Cincias do Porto
            Rua das Taipas, 135
            4050 Porto
            Portugal
  
            Tel: ++351 02 2080313
            Fax: ++351 02 2004109
            Email: mrdioh@fc.up.pt
*)
(* :Summary:
    The datachv52 package contains the definition form,
    gammaexact and betaexact that the package chv needs,
    corresponding to the example 5.2 of the first paper
    cited bellow.    
*)
(* :Date: July 1999. *)

(* :Available at http://netlib.bell-labs.com/ in the 
    numeralgo library in netlib:       *)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 3.0 *)  
 
(* :Keywords: chebyshev's method, recurrence relations, 
              orthogonal polynomials of dimension d,
              d-dimensional forms.
*)
(* :Sources:

 - Z. da Rocha, Shohat-Favard and Chebyshev's
   methods in d-orthogonality, Numerical Algorithms,
   20 (1999) 139-164.
   
 - About the example treated here see  p.286 of:
 
   K. Douak, The relation of the d-orthogonal
   polynomials to the Appell polynomials,  
   J. Comput. Appl. Math., 70 (1996) 279-295.
*)

(* :Warnings: How to load this package.

This package must be putted in a directory named
chebyshev. 
Note that the directory containing the chebyshev
directory must be on the file search path $Path.

This package contains the definitions needed in the
package chv.m. 
The context corresponding to this package
(chebyshev`datachv52`) must be written in the 
BeginPackage command of the package chv.m.
Then, when you load the package chv.m doing 
<<chebyshev`chv` or Needs["chebyshev`chv`"] in any
notebook, this package is automatically loaded. 
*)


BeginPackage["chebyshev`datachv52`"]

datachv52::usage="is a package that contains
the definitions of:

- the moments of the d forms with respect to which
  the polynomials are orthogonal of dimension d
  (form),
  
- the exact expression of the d gamma coefficients
  and the beta coefficient that appear in the 
  (d+1)-order recurrence relation satisfied by 
  orthogonal polynomials of dimension d, in the case
  they are known (gammaexact, betaexact);
  
corresponding to the example 5.2 of the paper

Z. da Rocha, Shohat-Favard and Chebyshev's
methods in d-orthogonality, Numerical Algorithms,
20 (1999) 139-164.

The definitions form, gammaexact and betaexact are
used in the package chv.m."

form::usage="
form[d, alpha, n] is the moment of order n (n>=0)
of the form of order alpha (1<=alpha<=d), with
respect to which the polynomials are orthogonal
of dimension d.
form is used in the package chv.m."

gammaexact::usage="
gammaexact[ d, r, n ], is the exact expression of
the n-th gamma coefficient of order r (n>=1,
0<=r<=d-1), that appears in the (d+1)-order recurrence
relation satisfied by orthogonal polynomials of
dimension d.
gammaexact is used in the package chv.m."

betaexact::usage="
betaexact[ d, n ], n>=0, is the exact expression of
the n-th beta coefficient, that appears in the 
(d+1)-order recurrence relation satisfied by
orthogonal polynomials of dimension d.
betaexact is used in the package chv.m."

Begin["`Private`"]

(*___________________________________________

                Error Messages
 __________________________________________*)
   
form::badarg2=
   "wrong value to the second argument
    alpha=`1` in form;
    alpha must be an integer and verify
    1<=alpha<=`2`;
    try again"

form::badarg3=
   "wrong value to the third argument n=`1` in form;
    n must be an integer;
    try again" 

gammaexact::badarg2=
   "wrong value to the second argument r=`1`
    in gammaexact; r must be an integer and
    verify 0<=r<=`2`;
    try again"

gammaexact::badarg3=
   "wrong value to the third argument n=`1`
    in gammaexact;
    n must be an integer;
    try again" 

betaexact::badarg2=
   "wrong value to the second argument n=`1`
    in betaexact;
    n must be an integer;
    try again" 

(*___________________________________________

             Exported definition
___________________________________________*)
(*_________________________________________________

  the exact expression of the gamma coefficients
_________________________________________________*) 

gammaexact[d_,r_,n_]:=gammaexact[d,r,n]=
 Block[{gamma1,g11,g12},
 
 (*_________ checking the arguments __________*) 
 
  If[Or[r<0,r>=d,Not[IntegerQ[r]]],
  
  Return[Message[gammaexact::badarg2,r,d-1]]
    ];
    
  If[Not[IntegerQ[n]],
  
    Return[Message[gammaexact::badarg3,n]]
    ];
(*____________________________________________*)

     g11=1/2;g12=2/3;
     gamma1={1/(d+1),g11,g12};
     Which[n<=0,Return[0],
           n===1,
            Return[gamma1[[r+1]] ],
           n>1,
            Return[(n-1+d-r)/(n-1)*
                   gammaexact[d,r,n-1]]
     ](*end of Which[n]*)    
        
(*_________________________________________________*)

];(*end of Block*)


(*_________________________________________________

  the exact expression of the beta coefficients
_________________________________________________*) 

betaexact[d_,n_]:=betaexact[d,n]=

 Block[{},
  (*_________ checking the arguments _________*)
      
      If[Not[IntegerQ[n]],
      
     Return[Message[betaexact::badarg3,n]]
        ];   
  (*__________________________________________*)
             
     Return[0] 
 (*_____________________________________________*) 
   
      ];(*end of Block*)          
(*______________________________________________*)

(*______________________________________________

   the moments of the d-dimensional form 
   calculated using the package shfv.m
________________________________________________*)

form[d_,alpha_,n_]:=
 Block[{},
 
 (*_________ checking the arguments _________*)
 
   If[Or[alpha<1,alpha>d,Not[IntegerQ[alpha]]],
    
 	  Return[Message[form::badarg2,alpha,d]]
 	 ];	 
   
   If[Not[IntegerQ[n]],
   
      Return[Message[form::badarg3,n]]
     ];
(*_________________________________________*)
Return[
{{1, 0, 2/3, 1/2, 19/12, 10/3, 85/9, 665/24, 
    40145/432, 980/3, 90335/72, 1468775/288, 
    38046085/1728, 7212205/72, 310665355/648, 
    8297163875/3456, 260394359225/20736, 
    22160813675/324, 12039597695125/31104, 
    31396934644075/13824, 10296439913584825/746496, 
    2691989682507125/31104, 104561691957297575/186624, 
    5579122899023749075/1492992, 
    229736958654546768425/8957952, 
    4220079995530774375/23328, 
    17604093927077850066875/13436928, 
    19373263897290956778125/1990656, 
    884858545599841962524375/11943936, 
    862111380076449315938125/1492992, 
    10312204670740852041036875/2239488, 
    896755463229398888355865625/23887872, 
    134442758605660353687974760625/429981696, 
    2977232425872362405266894375/1119744, 
    4959882653700805778962791651875/214990848, 
    175545711875044049701133437090625/859963392, 
    9498746239274496455014827910721875/5159780352, 
    1211998134720331387371050011028125/71663616, 
    612204351657980366956356781936465625/3869835264, 
    15541698677803354718841299487357615625/10319560704, 
    903079282833503423035107519244066496875/61917364224, 
    138958248783410922687235697574932265625/967458816, 
    133531709049312443527492383170012669403125/
     92876046336, 1810833728622938730661174668522294775\
      934375/123834728448, 
    336726080737397265630934174492396677200759375/
     2229025112064, 49055851178443100034907909945094715\
      973578125/30958682112, 
    1566883360525448479210408043100017726774171875/
     92876046336, 2708460787905589296883666280435615796\
      43420421875/1486016741376, 
    17809428565458612411504409926732289310258087515625/
     8916100448256, 10308834372021173476521144001016509\
      06398364609375/46438023168, 
    33452367739026609241895643623151002132067714953281\
      25/13374150672384, 
    509376074217587673197159242105096068899983450128906\
      25/17832200896512, 
    353671004469102054201972514759161188396853380251792\
      1875/106993205379072, 
    518261149933920126663433963512990814275712423817520\
      3125/13374150672384, 
    369203142993640704653399041553253057479478758053527\
      296875/80244904034304, 
    131517617122668263544514529648787384045395914241014\
      2265625/23776267862016, 
    259311746688972275432605690393349837839429642004711\
      2602265625/3851755393646592, 
    519915315486085424465516727642326983384561501147046\
      0078125/626913312768, 
    199096893010618606631049118584208765259851933791879\
      167447890625/1925877696823296, 
    100437951572456906266323772859955362081646785435827\
      00980823203125/7703510787293184, 
    7687794008276305540479381155773887249289685827579199\
      70819488359375/46221064723759104},
      {0, 1, 0, 2, 2, 95/12, 20, 595/9, 665/3, 40145/48, 
    9800/3, 993685/72, 1468775/24, 494599105/1728, 
    50485435/36, 1553326775/216, 8297163875/216, 
    4426704106825/20736, 22160813675/18, 
    228752356207375/31104, 156984673220375/3456, 
    72075079395093775/248832, 29611886507578375/15552, 
    2404918915017844225/186624, 
    5579122899023749075/62208, 
    5743423966363669210625/8957952, 
    54861039941900066875/11664, 
    17604093927077850066875/497664, 
    135612847281036697446875/497664, 
    25660897822395416913206875/11943936, 
    4310556900382246579690625/248832, 
    319678344792966413272143125/2239488, 
    896755463229398888355865625/746496, 
    1478870344662263890567722366875/143327232, 
    50612951239830160889537204375/559872, 
    173595892879528202263697707815625/214990848, 
    175545711875044049701133437090625/23887872, 
    351453610853156368835548632696709375/5159780352, 
    23027964559686296360049950209534375/35831808, 
    7958656571553744770432638165174053125/1289945088, 
    77708493389016773594206497436788078125/1289945088, 
    37026250596173640344439408289006726371875/
     61917364224, 9727077414838764588106498830245258593\
      75/161243136, 57418634891204350716821724763105447\
      84334375/92876046336, 
    19919171014852326037272921353745242535278125/
     30958682112, 1683630403686986328154670872461983386\
      003796875/247669456896, 
    1128284577104191300802881928737178467392296875/
     15479341056, 7364351794469607852288917802570083315\
      8386078125/92876046336, 
    270846078790558929688366628043561579643420421875/
     30958682112, 8726619997074720081637160864098821762\
      02646288265625/8916100448256, 
    25772085930052933691302860002541272659959115234375/
     23219011584, 5686902515634523571122259415935670362\
      4515115420578125/4458050224128, 
    662188896482863975156307014736624889569978485167578\
      125/4458050224128, 
    187445632368624088727045432822355429850332291533449\
      859375/106993205379072, 
    518261149933920126663433963512990814275712423817520\
      3125/247669456896, 
    203061728646502387559369472854289181613713316929440\
      01328125/80244904034304, 
    920623319858677844811601707541511688317771399687099\
      5859375/2972033482752, 
    492692318709047323321950811747364691894916319808951\
      39443046875/1283918464548864, 
    150775441490964773094999851016274825181522835332643\
      342265625/313456656384, 
    117467166876264977912318979964683171503312640937208\
      70879425546875/1925877696823296, 
    5021897578622845313316188642997768104082339271791350\
      4904116015625/641959232274432},
      {0, 0, 1, 0, 4, 5, 95/4, 70, 2380/9, 1995/2, 
    200725/48, 53900/3, 993685/12, 19094075/48, 
    3462193735/1728, 252427175/24, 1553326775/27, 
    141051785875/432, 4426704106825/2304, 
    421055459825/36, 1143761781036875/15552, 
    1098892712542625/2304, 792825873346031525/248832, 
    681073389674302625/31104, 2404918915017844225/15552, 
    139478072475593726875/124416, 
    74664511562727699738125/8957952, 
    54861039941900066875/864, 
    123228657489544950468125/248832, 
    3932772571150064225959375/995328, 
    128304489111977084566034375/3981312, 
    133627263911849643970409375/497664, 
    319678344792966413272143125/139968, 
    9864310095523387771914521875/497664, 
    25140795859258486139651280236875/143327232, 
    1771453293394055631133802153125/1119744, 
    173595892879528202263697707815625/11943936, 
    6495191339376629838941937172353125/47775744, 
    6677618606209971007875424021237478125/5159780352, 
    299363539275921852680649352723946875/23887872, 
    39793282857768723852163190825870265625/322486272, 
    3186048228949687717362466394908311203125/2579890176, 
    259183754173215482411075858023047084603125/
     20639121408, 41826432883806687728857944970054611953\
      125/322486272, 63160498380324785788503897239415992\
      627678125/46438023168, 
    99595855074261630186364606768726212676390625/
     6879707136, 387234992848006855475574300666256178780\
      87328125/247669456896, 
    53029375123896991137735450650647387967437953125/
     30958682112, 73643517944696078522889178025700833158\
      386078125/3869835264, 
    13271457860737387554729964774134517402527600671875/
     61917364224, 21816549992686800204092902160247054405\
      066157206640625/8916100448256, 
    438125460810899872752148620043201635219304958984375/
     15479341056, 73929732703248806424589372407163714711\
      8696500467515625/2229025112064, 
    3509601151359179068328427178104111914720885971388164\
      0625/8916100448256, 
    1874456323686240887270454328223554298503322915334498\
      59375/3962711310336, 
    2850436324636560696648886799321449478516418330996361\
      71875/495338913792, 
    1421432100525516712915586309980024271295993218506080\
      09296875/20061226008576, 
    1749184307731487905142043244328872207803765659405489\
      21328125/1981355655168, 
    1428807724256237237633657354067357606495257327445959\
      043848359375/1283918464548864, 
    8895751047966921612604991209960214685709847284625957\
      193671875/626913312768, 
    58733583438132488956159489982341585751656320468604354\
      397127734375/320979616137216}}[[alpha]][[n+1]]
  ] (*end of Return*)
(*_____________________________________________*)
   
]; (*end of Block*)  

(*_______________________________________________*) 

   
End[]

EndPackage[]