(* :Title: chv.m   *)

(* :Context: chebyshev`chv` *)

(* :Author:  Zlia da ROCHA *)

(* :Address: 
            Departamento de Matemtica Aplicada da
            Faculdade de Cincias do Porto
            Rua das Taipas, 135
            4050 Porto
            Portugal
  
            Tel: ++351 02 2080313
            Fax: ++351 02 2004109
            Email: mrdioh@fc.up.pt
*)
(* :Summary:
    The chv package implements the Chebyshev's method
    for computing the coefficients that appear in the
    (d+1)-order recurrence relation satisfied by 
    orthogonal polynomials of dimension d.    
*)
(* :Date: July 1999. *)

(* :Available at http://netlib.bell-labs.com/ in the 
    numeralgo library in netlib:       *)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 3.0 *)  
 
(* :Keywords: Chebyshev's method, recurrence relations, 
              orthogonal polynomials of dimension d,
              d-dimensional forms.
*)
(* :Sources:
   Z. da Rocha, Shohat-Favard and Chebyshev's
   methods in d-orthogonality, Numerical Algorithms,
   20 (1999) 139 - 164.
*)

(* :Warnings: How to load this package.

This package must be putted in a directory named 
chebyshev. 
Note that the directory containing the chebyshev
directory must be on the file search path $Path.

Then, do <<chebyshev`chv` or Needs["chebyshev`chv`"] 
in any notebook to load this package. 
*)

(* :Requirements: 

This package requires a data package where are given 
the definitions form and eventually gammaexact and betaexact. 
The corresponding context must have a name like 
chebyshev`something`. For example, in the examples treated
in the paper, we call the packages datachv1.m, datachv2.m, etc,
corresponding to different examples, and the contexts are 
chebyshev`datachv1`, chebyshev`datachv2`, etc, respectively.

In order to test a different example, the user must
write the corresponding context in the BeginPackage 
command of this package, and load again this package.

In order to test a new example, the user must write a 
new data package with the new definitions (form, 
betaexact and gammaexact), write the corresponding 
context in the BeginPackage command of this package,
and load again this package.
*)

(* :Examples:

See the packages datachv1.m, datachv2.m, datachv3.m,
datachv41.m, datachv52.m, datachv53.m and datachv6.m
corresponding to the examples of the paper and the 
following notebooks with the corresponding results
testchv1.nb, testchv2.nb, testchv3.nb, testchv41.nb, 
testchv52.nb, testchv53.nb and testchv6.nb.

In order to test a new example the user can use
the package datachvnew.m and the notebook testchvnew.nb.
*)


BeginPackage["chebyshev`chv`","chebyshev`datachv2`"]

Clear[z, gamma, beta, dorthpoly]
  
chv::usage="is a package that implements the
Chebyshev's method for computing the gamma 
and beta coefficients that appear in the 
(d+1)-order recurrence relation satisfied by 
orthogonal polynomials of dimension d.

This package contains the definitions of the
Chebyshev's relations as they are written in the
paper:

Z. da ROCHA, Shohat-Favard and Chebyshev's
             methods in d-orthogonality,
             Numerical Algorithms 20 (1999) 139 - 164.
            
We use here a notation similar to that it is used
in the article.

This package contains the definitions of:

- the recurrence relation of the Z-elements
  (z). 

- the d gamma coefficients and the beta coefficient
  (gamma, beta), that appear in the (d+1)-order
  recurrence relation satisfied by orthogonal
  polynomials of dimension d. These coefficients are
  calculated from Z-elements, using the Chebyshev's
  relations.

- printgamma and printbeta for printing the
  preceding coefficients in an adequate format,
  eventually calculating a measure of the errors,
  if their exact values are known.
  
- the n-th orthogonal polynomial of dimension d
  (dorthpoly) that is calculated from the 
  (d+1)-order recurrence relation using the
  preceding coefficients gamma and beta, obtained
  by the Chebyshev's method.

- printdorthpoly for printing the first nmax+1 
  orthogonal polynomials of dimension d in an adequate
  format.

Probably, the user needs only to use the
definitions printgamma, printbeta and printdorthpoly.

The user must give the definition form, used in z.
If the exact expressions of the gamma and the beta 
coefficients are known, the user can give the 
corresponding definitions (gammaexact and betaexact) in
order to evaluate the errors committed in printgamma and
printbeta. 
The definitions form, gammaexact and betaexact must be 
written in a package named for example datachvXX.m with
context chebyshev`datachvXX`. This context must be 
written above in the BeginPackage command of this 
package, and then load this package."


z::usage="z[ d, alpha, n, m, optcomp ], 1<=alpha<=d.
By definition, 
z[d,alpha,n,m,optcomp]=< Talpha, x^m*Pn > is
the result of applying the form Talpha to x^m*Pn,
where Pn is the orthogonal polynomial of dimension d
and degree n, with respect to the d-dimensional form
T=(T1,...,Td).
z[d,alpha,n,m,optcomp] is calculated using a Chebyshev's
relation.
z[d,alpha,n,m,optcomp] uses the moments of the forms
T1,...,Td, given in the definition form, that must be
written by the user, in a package named for example
datachvXX.m with context chebyshev`datachvXX`.

The argument optcomp determines if we perform formal
computations (optcomp=Infinity), or numerical 
computations (optcomp is equal to the number of 
significant digits). The default value of optcomp is
Infinity.

See also the definitions beta, gamma, printgamma,
printbeta, printdorthpoly, dorthpoly."


gamma::usage="gamma[ d, r, n, optcomp ], 0<=r<=d-1,
is the coefficient gamma of order r, that appears in
the (d+1)-order recurrence relation satisfied by
orthogonal polynomials of dimension d.
gamma is calculated from the definition z using the
Chebyshev's method.

The argument optcomp determines if we perform formal
computations (optcomp=Infinity), or numerical 
computations (optcomp is equal to the number of 
significant digits). The default value of optcomp is
Infinity.

See also the definitions beta, z, printgamma,
printbeta, printdorthpoly, dorthpoly."


beta::usage="beta[ d, n, optcomp ],
is the coefficient beta, that appears in the 
(d+1)-order recurrence relation satisfied by 
orthogonal polynomials of dimension d.
beta is calculated from the definition z using the
Chebyshev's method.

The argument optcomp determines if we perform formal
computations (optcomp=Infinity), or numerical 
computations (optcomp is equal to the number of
significant digits). The default value of optcomp is
Infinity.

See also the definitions gamma, z, printgamma,
printbeta, printdorthpoly, dorthpoly."


printgamma::usage="
printgamma[ d, nmax, 
            optcomp, optprecw,
            optexaexp, opterror, kinderror,
            opterrorprec, opterrorw ]
  
prints the d gamma coefficients, from order 1
to nmax+r-d+1, calling the definition
gamma[ d, r, n ,optcomp ], 1<= n <=nmax+r-d+1, 0<=r<=d-1. 

We can perform formal computations or numerical
computations according with the moments of the
d-dimensional form (see the form definition in the 
package datachvXX.m) are given in a formal
way or in a numerical way. If they are given in a formal
way, we can perform numerical computations as well.

If we perform formal computations, we must take 
optcomp=Infinity. If we perform numerical
computations, optcomp is the number of significant
digits employed.
The default value of optcomp is Infinity.

If we perform numerical computations, the gamma 
coefficients are writing with optprecw digits.
The default value of optprecw is equal to 
$MachinePrecision.

If the exact expressions of the gamma coefficients
are given in the data package (gammaexact),
taking optexaexp=True, we can write them.
If we perform numerical computations, the exact gamma
coefficients are written with optprecw digits of
precision.

We can also evaluate the errors committed taking 
opterror=True. We can calculate the absolute error,
the number of exact digits, or the relative errors
of the gamma coefficients, taking respectively 
kinderror=\"abs\", kinderror=\"sgd\" or 
kinderror=\"rel\".
If gammaexact===0, we calculate absolute errors.
The evaluation of the errors are made performing 
computations with precision opterrorprec and the
results are written with opterrorw digits.
The default values of optexaexp, opterror and 
kinderror are respectively True, True and \"sgd\".
The default values of opterrorprec and opterrorw 
are respectively $MachinePrecision and 2.

See also the definitions printbeta, printdorthpoly,
dorthpoly, beta, gamma, z."


printbeta::usage="
printbeta[ d, nmax, 
           optcomp, optprecw,
           optexaexp, opterror, kinderror,
           opterrorprec, opterrorw ]

prints the beta coefficients from order 0 to nmax-1,
calling the definition
beta[ d, n, optcomp ], 0<= n <=nmax-1. 

We can do formal computations or numerical
computations according with the moments of the
d-dimensional form (see the form definition in the 
package datachvXX.m) are given in a formal way or in a
numerical way. If they are given in a formal way, we
can perform numerical computations as well.

If we perform formal computations, we must take 
optcomp=Infinity. If we perform numerical
computations, optcomp is the number of significant
digits employed.
The default value of optcomp is Infinity.

If we perform numerical computations, the beta 
coefficients are writing with optprecw digits.
The default value of optprecw is equal to 
$MachinePrecision.

If the exact expressions of the beta coefficients
are given in the data package (betaexact),
taking optexaexp=True, we can write them.
If we perform numerical computations, the exact beta
coefficients are written with optprecw digits of
precision.

We can also evaluate the errors committed taking 
opterror=True. We can calculate the absolute errors,
the number of exact digits, or the relative errors 
of the beta coefficients, taking respectively 
kinderror=\"abs\", kinderror=\"sgd\" or 
kinderror=\"rel\".
If betaexact===0, we calculate absolute errors.
The evaluation of the errors are made performing 
computations with precision opterrorprec and the
results are written with opterrorw digits.
The default values of optexaexp, opterror and 
kinderror are respectively True, True and 2.
The default values of opterrorprec and opterrorw 
are respectively $MachinePrecision and 2.

See also the definitions printgamma, printdorthpoly,
dorthpoly, beta, gamma, z."


dorthpoly::usage="dorthpoly[ d, n, var, optcomp ]
is the orthogonal polynomial of dimension d and 
degree n, calculated using the (d+1)-order recurrence
relation that calls the definitions gamma and beta.

See also the definitions printdorthpoly,
printgamma, printbeta, beta, gamma, z."


printdorthpoly::usage="
printdorthpoly[ d, nmax, var, optcomp, optprecw ] 

prints the first nmax+1 orthogonal polynomials of
dimension d calling the definition
dorthpoly[ d, n, var, optcomp ], 0<= n <=nmax.

We can perform formal computations or numerical
computations according with the moments of the
d-dimensional form (see definitions form in the 
data package) are given in a formal way or in a
numerical way. If they are given in a formal way,
we can perform numerical computations as well.

If we perform formal computations, we must take 
optcomp===Infinity. If we perform numerical
computations, optcomp is the number of significant
digits employed.
The default value of optcomp is Infinity.

If we perform numerical computations, the coefficients
of the polynomials in the canonical base are writing 
with optprecw digits.
The default value of optprecw is the $MachinePrecision.

See also the definitions dorthpoly, printgamma,
printbeta, beta, gamma, z."


Begin["`Private`"]

(*_________________________________________________

                    Error Messages
  ________________________________________________*)
   
z::badarg2=
   "wrong value to the second argument
    alpha=`1` in z;
    alpha must be an integer and verify
    1<=alpha<=`2`;
    try again"
    
z::badarg34=
   "wrong value to the third and/or fourth arguments
    n=`1` and m=`2` in z;
    n and m must be integers;
    try again"
    
z::badarg5=
   "wrong value to the fifth argument
    optcomp= `1` in z;
    optcomp must be a positive integer,
    or be equal to Infinity;
    try again"
    
z::badarg12345=
   "something wrong in z;
    the value of z[`1`,`2`,`3`,`4`,`5`] is undefined;
    try again"

gamma::badarg2=
   "wrong value to the second argument r=`1`
    in gamma, r must be an integer and
    verify 0<=r<=`2`;
    try again"
    
gamma::badarg3=
   "wrong value to the third argument n=`1`
    in gamma, n must be an integer;
    try again"

gamma::badarg4=
   "wrong value to the fourth argument optcomp=`1`
    in gamma; 
    optcomp must be a positive integer,
    or be equal to Infinity;
    try again"
    
gamma::badarg1234=
   "something wrong in gamma;
    the value of gamma[`1`,`2`,`3`,`4`]
    is undefined;
    try again"
    
beta::badarg2=
   "wrong value to the second argument n=`1`
    in beta; n must be an integer;
    try again"

beta::badarg3=
   "wrong value to the third argument optcomp=`1`
    in beta; 
    optcomp must be a positive integer,
    or be equal to Infinity;
    try again"

beta::badarg123=
   "something wrong in beta;
    the value of beta[`1`,`2`,`3`]
    is undefined;
    try again"

printgamma::badarg2=    
   "wrong value to the second argument
    nmax=`1` in printgamma;
    nmax must be a positive integer;
    try again"
    
printgamma::badarg3=
   "wrong value to the third argument
    optcomp= `1` in printgamma;
    optcomp must be a positive integer;
    or be equal to Infinity;
    try again"
   
printgamma::badarg7=
   "wrong value to the seventh argument
    kinderror= `1` in printgamma;
    kinderror must be equal to \"abs\", 
    \"sgd\" or \"rel\";        
    try again"

printbeta::badarg2=    
   "wrong value to the second argument
    nmax=`1` in printbeta;
    nmax must be a non negative integer;
    try again"

printbeta::badarg3=
   "wrong value to the third argument
    optcomp= `1` in printbeta;
    optcomp must be a positive integer;
    or be equal to Infinity;
    try again"

printbeta::badarg7=
   "wrong value to the seventh argument
    kinderror=`1` in printbeta;
    kinderror must be equal to \"abs\", 
    \"sgd\" or \"rel\"; 
    try again"

dorthpoly::badarg2=
   "wrong value to the second argument n=`1`
    in beta; n must be an integer;
    try again"

dorthpoly::badarg4=
   "wrong value to the fourth argument
    optcomp= `1` in dorthpoly;
    optcomp must be a positive integer;
    or be equal to Infinity;   
    try again"
    
dorthpoly::badarg1234=
   "something wrong in dorthpoly;
    the value of dorthpoly[`1`,`2`,`3`,`4`]
    is undefined;   
    try again"
    
printdorthpoly::badarg2=    
   "wrong value to the second argument
    nmax=`1` in printdorthpoly;
    nmax must be a non negative integer;
    try again"
    
printdorthpoly::badarg4=    
   "wrong value to the fourth argument
    optcomp= `1` in printdorthpoly;
    optcomp must be a positive integer;
    or be equal to Infinity;
    try again"
    
(*_______________________________________________

               Private definitions 
________________________________________________*)
(*_______________________________________________

                absolute errors
 _______________________________________________*)
 
abserror[exactvalue_,approxvalue_,
         errorprec_:$MachinePrecision,
         errorprint_:2]:=
 N[N[Abs[exactvalue-approxvalue],
     errorprec],errorprint];

(*_______________________________________________

             exact significant digits
 _______________________________________________*)
   
exactdigits[exactvalue_,approxvalue_,
            errorprec_:$MachinePrecision,
            errorprint_:2]:=       
N[   
  N[-Log[10,Abs[exactvalue-approxvalue]],
    errorprec]+
   MantissaExponent[approxvalue][[2]]-1,
  errorprint];
  
(*_________________________________________________

                   relative errors
 _________________________________________________*)

relerror[exactvalue_,approxvalue_,
         errorprec_:$MachinePrecision,
         errorprint_:2]:=           
N[N[Abs[(exactvalue-approxvalue)/exactvalue],
   errorprec],errorprint];
                 
(*________________________________________________

                 Exported definitions 
__________________________________________________*)
     
           
(*______________________________________________

    The recurrence relation of the Z-elements 
________________________________________________*)

z[d_,alpha_,n_,m_,optcomp_:Infinity]:=
z[d,alpha,n,m,optcomp]=
 
 Block[{},
 (*___________ checking the arguments___________*)
 
  If[ Or[ alpha<1,alpha>d,Not[IntegerQ[alpha]] ],
    
 	Return[Message[z::badarg2,alpha,d]]
 	]; 
  
  If[ Or[ Not[IntegerQ[n]], Not[IntegerQ[m]] ],
  
    Return[Message[z::badarg34,n,m]]
    ];
    
  If[ And[optcomp=!=Infinity,
          Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
    Return[Message[z::badarg5,optcomp]]
    ];
 (*_____________________________________________*)
 
        Which[
              n<0, 
               Return[0],
              n===0,
               Return[ N[form[d,alpha,m],optcomp] ],  
              n>0,
        Return[z[d,alpha,n-1,m+1,optcomp]-
               beta[d,n-1,optcomp]*
               z[d,alpha,n-1,m,optcomp]-
               Sum[ gamma[d,d-1-l,n-1-l,optcomp]*
                    z[d,alpha,n-2-l,m,optcomp],
                  {l,0,d-1}]
              ],
              True,
  Return[Message[z::badarg12345,d,alpha,n,m,optcomp]]
           
             ](*end of Which*)
 (*________________________________________________*) 
     
];(*end of Block*)
   
(*_________________________________________________

                The gamma coefficients
__________________________________________________*)

gamma[d_,r_,n_,optcomp_:Infinity]:=
gamma[d,r,n,optcomp]=
 
 Block[{m,alpha,l},
 (*____________ checking the arguments ___________*)
 
  If[Or[r<0,r>d-1,Not[IntegerQ[r]]],
    
 	Return[Message[gamma::badarg2,r,d-1]]
 	];
 	
  If[ Not[IntegerQ[n]],
   
    Return[Message[gamma::badarg3,n]]
    ];
  
  If[ And[optcomp=!=Infinity,
          Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
    Return[Message[gamma::badarg4,optcomp]]
    ];
(*________________________________________________
    the gamma coefficients are computed from 
    Z-elements using Chebyshev's relations
_________________________________________________*) 
 
        Which[
              n<=0,Return[0],
              r===0,
                m=Floor[n/d]; alpha=n-d*m;
                If[alpha===0, m=m-1;alpha=d];
       Return[z[d,alpha,(m+1)*d+alpha-1,m+1,optcomp]/
              z[d,alpha,m*d+alpha-1,m,optcomp]
             ],
              n===1,
       Return[z[d,1,d-r,1,optcomp]/
              z[d,1,0,0,optcomp]
             ],
              And[2<=n,n<=r],
            Return[(z[d,n,d+n-r-1,1,optcomp]-
                Sum[gamma[d,d-1-l,d-r-1-l+n,optcomp]*
                    z[d,n,d-r+n-l-2,0,optcomp],
                   {l,d-r,d-r+n-2}])/
                   z[d,n,n-1,0,optcomp]
                   ],
               And[1<=r,r<=d-1],  
                 m=Floor[n/d]; alpha=n-d*m;
                 If[alpha===0, m=m-1;alpha=d];
                 If[Or[And[m===0,r+1<=alpha,alpha<=d],
                       And[m>=1,1<=alpha,alpha<=d]],
      Return[(z[d,alpha,(m+1)*d+alpha-r-1,m+1,optcomp]-
        Sum[gamma[d,d-1-l,(m+1)*d+alpha-r-1-l,optcomp]*
            z[d,alpha,(m+1)*d+alpha-r-2-l,m,optcomp],
            {l,d-r,d-1}])/
            z[d,alpha,m*d+alpha-1,m,optcomp]
            ] (*end of Return*)
                   ](*end of If*),
                   
               True,
       Return[Message[gamma::badarg1234,d,r,n,optcomp]]
       
              ](*end of Which*)
(*________________________________________________*)  
           
  ];(*end of Block*)


(*________________________________________________
         
               The beta coefficients
_________________________________________________*)


beta[d_,n_,optcomp_:Infinity]:=beta[d,n,optcomp]=

 Block[{l,m,r},
 (*___________ checking the arguments___________*)
 
  If[ Not[IntegerQ[n]],
  
    Return[Message[beta::badarg2,n]]
    ];
  
  If[ And[optcomp=!=Infinity,
          Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
    Return[Message[beta::badarg3,optcomp]]
    ];
 (*______________________________________________
      the beta coefficients are computed from 
      Z-elements using Chebyshev's relations
________________________________________________*)
 
  Which[
        n<0,Return[0],
        n===0,
        Return[z[d,1,0,1,optcomp]/z[d,1,0,0,optcomp]],
        And[1<=n,n<=d-1],
         Return[(z[d,n+1,n,1,optcomp]-
                Sum[gamma[d,d-1-l,n-l,optcomp]*
                    z[d,n+1,n-1-l,0,optcomp],
                   {l,0,n-1}])/z[d,n+1,n,0,optcomp]
               ],
        n>=d,
           m=Floor[n/d]; r=n-d*m;
           Return[(z[d,r+1,m*d+r,m+1,optcomp]-
            Sum[gamma[d,d-1-l,m*d+r-l,optcomp]*
                z[d,r+1,m*d+r-1-l,m,optcomp],
               {l,0,d-1} ])/z[d,r+1,m*d+r,m,optcomp]
                 ],
        True,
      Return[Message[beta::badarg123,d,n,optcomp]]
       
       ](*end of Which*)
 (*______________________________________________*)
        
 ];(*end of Block*)
    
 (*_______________________________________________
 
           Printing the gamma coefficients
 _________________________________________________*)
 
 printgamma[d_,nmax_,
   optcomp_:Infinity,optprecw_:$MachinePrecision,
   optexaexp_:True,opterror_:True,kinderror_:"sgd",
   opterrorprec_:$MachinePrecision,opterrorw_:2]:=
   
 Block[{r,n},
 (*___________ checking the arguments___________*)
 
 If[ Or[ Not[Integer[nmax]], nmax<=0 ],
 
    Return[Message[printgamma::badarg2,nmax]]
   ];
 
 If[ And[optcomp=!=Infinity,
         Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
    Return[Message[printgamma::badarg3,optcomp]]
    ];
    
 If[
And[kinderror=!="abs",kinderror=!="sgd",
    kinderror=!="rel"],
        
   Return[Message[printgamma::badarg7,kinderror]]
   ];
(*______________________________________________*)   
 Do[
    
    Print["   "];
    Do[  
    (*_____ printing the gamma coefficient____*)
    
       Which[
             optcomp===Infinity,
             (*__ formal computations __*)
             
    Print["gamma[",d,",",r,",",n,"]=",
           gamma[d,r,n,optcomp]],
            
             True, 
             (*__ numerical computations __*)
             
    Print["gamma[",d,",",r,",",n,"]=",
           N[gamma[d,r,n,optcomp],optprecw]]
           
           ];(*end of Which*)        
           
    If[optexaexp===True,
    (*___ printing the exact gamma coefficient____*)
      
      Which[
            optcomp===Infinity,
            (*__ formal computations __*)
    
       Print["gammaexact[",d,",",r,",",n,"]=",
              gammaexact[d,r,n]],
              
            True,
            (*__ numerical computations __*)
            
       Print["gammaexact[",d,",",r,",",n,"]=",
              N[gammaexact[d,r,n],optprecw]]
              
           ];(*end of Which*)
            
    If[ opterror===True, 
   (*_ computing a measure of the errors commited _*)
    
       Which[
       
      Or[kinderror==="abs",gammaexact[d,r,n]===0],
      (*_________ absolute errors _________*)
       
  Print["abserror[",d,",",r,",",n,"]=",
     abserror[gammaexact[d,r,n],gamma[d,r,n,optcomp],
              opterrorprec,opterrorw]
       ](*end of Print*), 
         
     kinderror==="sgd",
     (*_____ exact significant digits ______*)
       
    Print["exactdigits[",d,",",r,",",n,"]=",
    exactdigits[gammaexact[d,r,n],gamma[d,r,n,optcomp],
              opterrorprec,opterrorw]
         ](*end of Print*),
         
     kinderror==="rel",
     (*___________ relative errors __________*)
        
    Print["relerror[",d,",",r,",",n,"]=",
     relerror[gammaexact[d,r,n],gamma[d,r,n,optcomp],
            opterrorprec,opterrorw] 
         ](*end of Print*)
    
    ](*end of Which[kinderror]*)
    
   ](*end of If[opterror]*)
 (*____________________________________________*)
   
  ](*end of If[optexaexp]*);
 (*____________________________________________*)
  
   Print["  "],       
                                       
       {n,1,nmax+r-d+1}], (*end of Do[n]*) 
  (*___________________________________________*)  
  {r,0,d-1}] (*end of Do[r]*)
 (*____________________________________________*) 
];(*end of Block*)
 
 
(*_________________________________________________
 
            Printing the beta coefficients
 _________________________________________________*)
 
 printbeta[d_,nmax_,
   optcomp_:Infinity,optprecw_:$MachinePrecision,
   optexaexp_:True,opterror_:True,kinderror_:"sgd",
   opterrorprec_:$MachinePrecision,opterrorw_:2]:=
   
 Block[{n},
 (*__________ checking the arguments__________*)
 
 If[ Or[ Not[Integer[nmax]], nmax<0 ],
 
    Return[Message[printbeta::badarg2,nmax]]
   ];
 
 If[ And[optcomp=!=Infinity,
         Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
    Return[Message[printbeta::badarg3,optcomp]]
    ];
    
 If[ And[kinderror=!="abs",kinderror=!="sgd",
         kinderror=!="rel"],
        
   Return[Message[printbeta::badarg7,kinderror]]
   ];
   
(*_____________________________________________*) 

    Do[(*_____ printing the beta coefficient____*)
    
       Which[
             optcomp===Infinity,
             (*__ formal computations __*)
             
    Print["beta[",d,",",n,"]=",
           beta[d,n,optcomp]],
           
             True,
             (*__ numerical computations __*)
             
    Print["beta[",d,",",n,"]=",
          N[beta[d,n,optcomp],optprecw]]
          
            ];(*end of Which*)       
           
   If[optexaexp===True,
      (*___ printing the exact beta coefficient____*)
   
     Which[
            optcomp===Infinity,
            (*__ formal computations __*)
    
      Print["betaexact[",d,",",n,"]=",
             betaexact[d,n]],
          
          True,
          (*__ numerical computations __*)
           
      Print["betaexact[",d,",",n,"]=",
           N[betaexact[d,n],optprecw]]
         
         ];(*end of Which*) 
  
  If[opterror===True,       
  (*_ computing a measure of the errors commited _*)
          
     Which[
       Or[kinderror==="abs",betaexact[d,n]===0],
       (*_________ absolute errors _________*)
       
  Print["abserror[",d,",",n,"]=",
     abserror[betaexact[d,n],beta[d,n,optcomp],
              opterrorprec,opterrorw]
       ](*end of Print*),
         
       kinderror==="sgd",
       (*_____ exact significant digits ______*)
       
    Print["exactdigits[",d,",",n,"]=",
     exactdigits[betaexact[d,n],beta[d,n,optcomp],
                 opterrorprec,opterrorw]
         ](*end of Print*),
         
        kinderror==="rel",
        (*_________ relative errors _________*)
        
    Print["relerror[",d,",",n,"]=",
     relerror[betaexact[d,n],beta[d,n,optcomp],
              opterrorprec,opterrorw]
         ](*end of Print*)     
    
   ](*end of Which[kinderror]*)
  (*___________________________________________*)
  ](*end of If[opterror]*)
  (*___________________________________________*)
   
  ](*end of If[optexaexp]*);
  (*___________________________________________*)
  
  Print["  "],
          
 {n,0,nmax-1}] (*end of Do[n]*)
 (*___________________________________________*)
  
];(*end of Block*)

(*_______________________________________________

          The orthogonal polynomial of 
  dimension d and degree n in the variable var
________________________________________________*)

dorthpoly[d_,n_,var_,optcomp_:Infinity]:=
dorthpoly[d,n,var,optcomp]=

 Block[{l},
 (*__________ checking the arguments__________*)
 
 If[ Not[IntegerQ[n]],
   
    Return[Message[dorthpoly::badarg2,n]]
    ];
 
 If[ And[optcomp=!=Infinity,
          Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
   Return[Message[dorthpoly::badarg4,optcomp]]
    ];
(*________________________________________________
       the (d+1)-order recurrence relation for 
       orthogonal polynomials of dimension d, 
       where appear the gamma and beta 
       coefficients       
__________________________________________________*)
  
  Which[n<0,Return[0],
        n===0,Return[1],
        n>0,
  Return[ Collect[(var-beta[d,n-1,optcomp])*
           dorthpoly[d,n-1,var,optcomp]-
           Sum[gamma[d,d-1-l,n-1-l,optcomp]*
               dorthpoly[d,n-2-l,var,optcomp],
              {l,0,d-1}],
           var]
        ],
        True,
 Return[Message[dorthpoly::badarg1234,d,n,var,optcomp]]
       ](*end of Which[n]*)
 (*___________________________________________*) 
      
  ];(*end of Block*)
  

(*_________________________________________________

Printing the orthogonal polynomials of dimension d
__________________________________________________*)


printdorthpoly[d_,nmax_,var_,
   optcomp_:Infinity,optprecw_:$MachinePrecision]:=
   
 Block[{n},
  (*__________ checking the arguments__________*)
  
   If[ Or[nmax<0,Not[IntegerQ[nmax]]],
   
  Return[Message[printdorthpoly::badarg2,nmax]]
     ];
  
   If[ And[optcomp=!=Infinity,
          Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
  Return[Message[printdorthpoly::badarg4,optcomp]]
     ];
  (*_____________________________________________
         printing the orthogonal polynomial of
               dimension d an degree n
  ______________________________________________*)
          
  Which[
        optcomp===Infinity, 
        (*__ formal computations __*)      
        
    Do[
    Print["dorthpoly[",d,",",n,",",var,"]=",
           dorthpoly[d,n,var,optcomp]]
     ,{n,0,nmax}],
     
        True,
        (*__ numerical computations __*) 
              
   Do[
Print["dorthpoly[",d,",",n,",",var,"]=",
      N[dorthpoly[d,n,var,optcomp],optprecw]]
     ,{n,0,nmax}]
     
       ](*end of Which*) 
(*______________________________________________*)
     
 ];(*end of Block*)
      
(*______________________________________________*)


End[]

EndPackage[]