(* :Title: shfv.m   *)

(* :Context: shohatfavard`shfv` *)

(* :Author:  Zlia da ROCHA *)

(* :Address: 
            Departamento de Matemtica Aplicada da
            Faculdade de Cincias do Porto
            Rua das Taipas, 135
            4050 Porto
            Portugal
  
            Tel: ++351 02 2080313
            Fax: ++351 02 2004109
            Email: mrdioh@fc.up.pt
*)
(* :Summary:
    The shfv package implements the Shohat-Favard's 
    method for computing the moments of the
    d-dimensional canonical form, with respect to 
    which the polynomials are orthogonal of dimension
    d.     
*)
(* :Date: July 1999 *)

(* :Available at http://netlib.bell-labs.com/ in the 
    numeralgo library in netlib:       *)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 3.0 *)  
 
(* :Keywords: Shohat-Favard's method, regular forms,
              d-dimensional canonical forms, 
              d-orthogonal polynomial sequences.
*)
(* :Sources:
   Z. da Rocha, Shohat-Favard and Chebyshev's
   methods in d-orthogonality, Numerical Algorithms,
   20 (1999) 139-164.
*)

(* :Warnings: How to load this package.

This package must be putted in a directory named shohatfavard. 
Note that the directory containing the shohatfavard
directory must be on the file search path $Path.

Then, do <<shohatfavard`shfv` or Needs["shohatfavard`shfv`"] 
in any notebook to load this package. 
*)

(* :Requirements: 

This package requires a data package where are given 
the definitions gamma and beta, and eventually formexact. 
The corresponding context must have a name like 
shohatfavard`something`. For example, in the examples 
treated in the paper, we call the packages datashfv2.m,
datashfv3.m, etc, corresponding to different examples, 
and the contexts are shohatfavard`datashfv2`, 
shohatfavard`datashfv3`, etc, respectively.

In order to test a different example, the user must
write the corresponding context in the BeginPackage 
command of this package, and load again this package.

In order to test a new example, the user must write a 
new data package with the new definitions (gamma, beta
and formexact), write the corresponding context in 
the BeginPackage command of this package, and load 
again this package.
*)

(* :Examples:

See the packages datashfv2.m, datashfv3.m, datashfv51.m,
datashfv52.m, datashfv53.m, and datashfv6.m
corresponding to the examples of the paper and the 
following notebooks with the corresponding results
testshfv2.nb, testshfv3.nb, testshfv51.nb, testshfv52.nb, 
testshfv53.nb and testshfv6.nb.

In order to test a new example the user can use the
package datashfvnew.m and the notebook testshfvnew.nb.
*)


BeginPackage["shohatfavard`shfv`",
             "shohatfavard`datashfv2`"]
  
shfv::usage="is a package that implements the
Shohat-Favard's method for computing the moments
of the d-dimensional canonical form, with respect to
which the polynomials are orthogonal of dimension d.
This method requires the gamma and beta coefficients
that appear in the (d+1)-order recurrence relation
satisfied by orthogonal polynomials of dimension d.

Reference:

Z. da ROCHA, Shohat-Favard and Chebyshev's
             methods in d-orthogonality,
             Numerical Algorithms, 20 (1999) 139- 164.
                
We use here a notation similar to that it is used
in the paper.

This package contains the definition of:

- dcanonicalform, that implements the Shohat-Favard's
  method.

- printdcanform, for printing the moments computed
  by dcanonicalform in an adequate format,
  eventually calculating a measure of the errors,
  if the exact values of the moments are known.

Probably, the user needs only to use the
definition printdcanform.

The user must give the definition gamma and beta
of the coefficients that appear in the 
(d+1)-order recurrence relation satisfied by 
the d-orthogonal polynomial sequence.
If the exact expressions of the moments of the 
d-dimensional form are known, the user can give 
the corresponding definition (formexact) in
order to evaluate the errors commited in 
printdcanform.
The definitions gamma, beta and formexact must be
written in a package named, for example, datashfvXX.m
with contex shohatfavard`datashfvXX`. This contex must
be written above in the BeginPackage command of this
package, and then load this package."


dcanonicalform::usage="
 dcanonicalform[d,nmax,optcomp] is a double list
 that contains the nmax+1 first moments of the
 d-dimensional canonical form U=(u0,...,ud-1),
 with respect to which the polynomials are
 orthogonal of dimension d, 
 { {(u0)0,...,(u0)nmax},...,
   {(ud-1)0,...,(ud-1)nmax} }.
   
 The argument optcomp determines if we perform formal
 computations (optcomp=Infinity), or numerical
 computations (optcomp is equal to the number of
 significant digits). The default value of optcomp
 is Infinity."
 
            
printdcanform::usage="
printdcanform[d, nmax,
              optcomp, optprecw,
              optexaexp, opterror, kinderror,
              opterrorprec, opterrorw],
    
prints the nmax+1 first moments of the d-dimensional
canonical form with respect to which the polynomials
are orthogonal of dimension d.

We can perform formal computations or numerical
computations according with the coefficients 
beta and gamma, that appear in the (d+1)-order
recurrence relation satisfied by the orthogonal
polynomials of dimension d (see gamma and beta
definitions in the package datashfvXX.m), are given 
in a formal way or in a numerical way. If they 
are given in a formal way, we can perform numerical
computations as well.

If we perform formal computations, we must take 
optcomp=Infinity. If we perform numerical
computations, optcomp is the number of significant
digits employed.
The default value of optcomp is Infinity.

If we perform numerical computations, the moments
are printing with optprecw digits.
The default value of optprecw is $MachinePrecision.

If the exact expressions of the moments
are given in the package datashfvXX (formexact),
taking optexaexp=True, we can write them.
If we perform numerical computations, the exact 
moments are writen with optprecw digits of precision.

We can also evaluate the errors commited taking 
opterror=True. We can calculate the absolute errors,
the number of exact digits, or the relative errors of
the moments, taking respectively kinderror=\"abs\",
kinderror=\"sgd\" or kinderror=\"rel\".
If the exact value of a moment is zero, we calculate
the absolute error.
The evaluation of the errors are made performing 
computations with precision opterrorprec and the
results are written with opterrorw digits.
The default values of optexaexp, opterror and 
kinderror are respectively True, True and \"sgd\".
The default values of opterrorprec and opterrorw
are respectively $MachinePrecision and 2."


Begin["`Private`"]

(*_________________________________________________

                   Error Messages
  ________________________________________________*)
  
dcanonicalform::badarg2=    
   "wrong value to the second argument
    nmax=`1` in dcanonicalform;
    nmax must be a non negative integer;
    try again"

dcanonicalform::badarg3=    
   "wrong value to the third argument
    optcomp= `1` in dcanonicalform;
    optcomp must be a positive integer;
    or be equal to Infinity;
    try again"

printdcanform::badarg2=    
   "wrong value to the second argument
    nmax=`1` in printdcanform;
    nmax must be a non negative integer;
    try again"
    
printdcanform::badarg3=
   "wrong value to the third argument
    optcomp= `1` in printdcanform;
    optcomp must be a positive integer,
    or be equal to Infinity;
    try again"
   
printdcanform::badarg7=
   "wrong value to the seventh argument
    kinderror= `1` in printdcanform;
    kinderror must be equal to \"abs\", 
    \"sgd\" or \"rel\"; 
    try again"    

(*_________________________________________________

                 Private Definitions
__________________________________________________*)
(*_______________________________________________

                absolute errors
 _______________________________________________*)
 
abserror[exactvalue_,approxvalue_,
         errorprec_:$MachinePrecision,
         errorprint_:2]:=
 N[N[Abs[exactvalue-approxvalue],
     errorprec],errorprint];

(*_______________________________________________

             exact significant digits
 _______________________________________________*)
   
exactdigits[exactvalue_,approxvalue_,
            errorprec_:$MachinePrecision,
            errorprint_:2]:=       
N[   
  N[-Log[10,Abs[exactvalue-approxvalue]],
    errorprec]+
   MantissaExponent[approxvalue][[2]]-1,
  errorprint];
  
(*_________________________________________________

                   relative errors
 _________________________________________________*)

relerror[exactvalue_,approxvalue_,
         errorprec_:$MachinePrecision,
         errorprint_:2]:=           
N[N[Abs[(exactvalue-approxvalue)/exactvalue],
   errorprec],errorprint];
   
(*__________________________________________________

 the orthogonal polynomials of dimension d,
 calculated from the (d+1)-order recurrence relation 
 __________________________________________________*)

Clear[dorthpoly];
                  
dorthpoly[d_,n_,var_,optcomp_:Infinity]:=
dorthpoly[d,n,var,optcomp]=
 Block[{l},
  
  Which[n<0,Return[0],
        n===0,Return[1],
        n>0,
    Return[
     Collect[(var-N[beta[d,n-1],optcomp])*
             dorthpoly[d,n-1,var,optcomp]-
             Sum[N[gamma[d,d-1-l,n-1-l],optcomp]*
                 dorthpoly[d,n-2-l,var,optcomp],
                {l,0,d-1}],
            var]
          ]                                                         
        ](*end of Which[n]*)                
      
  ];(*end of Block*)
                               
(*_________________________________________________

                Exported Definitions
__________________________________________________*)

(*________________________________________________
 
                 dcanonicalform
 ________________________________________________*)
       
 dcanonicalform[d_,nmax_,optcomp_:Infinity]:=
   Block[{u,n,m,r,var},
   
  (*___________ checking the arguments___________*)
    If[ Or[ Not[Integer[nmax]], nmax<=0 ],
 
    Return[Message[dcanonicalform::badarg2,nmax]]
      ];
      
    If[ And[optcomp=!=Infinity,
          Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
    Return[Message[dcanonicalform::badarg3,optcomp]]
    ];
 (*_____________________________________________*) 
   
    u={};
    Do[AppendTo[u,{}],{d}];
    AppendTo[u[[1]],1];
    
    Do[ (*{m,1,nmax}*)
    
      Do[  (*{n,0,Min[d-1,m-1]}*)
      
    (*___computation of the m-th moment of Un___*)
         
      AppendTo[u[[n+1]],
       -Sum[Coefficient[dorthpoly[d,m,var,optcomp],
                        var,r]*
               u[[n+1]][[r+1]],{r,n,m-1}]
              ],
              
        {n,0,Min[d-1,m-1]}](*end of Do*);
       
      If[m<=d-1,
         PrependTo[u[[m+1]],1];
         Do[ PrependTo[u[[m+1]],0],{m} ]
        ], 
        
     {m,1,nmax}](*end of Do*);  
     
     Return[u]
   
 ];(*end of Block*)        
 
 (*______________________________________________
 
    printing the moments of the d-dimensional
    canonical form
 ________________________________________________*)
 
 printdcanform[d_,nmax_,
   optcomp_:Infinity,optprecw_:$MachinePrecision,
   optexaexp_:True,opterror_:True,kinderror_:"sgd",
   opterrorprec_:$MachinePrecision,opterrorw_:2]:=
   
 Block[{u,muan1,alpha,n},
 
 (*____________ checking the arguments ____________*)
 
 If[ Or[ Not[Integer[nmax]], nmax<=0 ],
 
     Return[Message[printdcanform::badarg2,nmax]]
   ];
 
 If[ And[optcomp=!=Infinity,
         Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
     Return[Message[printdcanform::badarg3,optcomp]]
   ];
    
 If[ And[kinderror=!="abs",kinderror=!="sgd",
         kinderror=!="rel"],
        
    Return[Message[printdcanform::badarg7,kinderror]]
   ];
(*__________________________________________________*)
   
   
 u=dcanonicalform[d,nmax,optcomp];
  
 Do[
 
    Print["   "];
    Do[
 (*_________________________________________________ 
        printing the n-th moment of the form of 
                     order alpha 
  ________________________________________________*)
       Which[
             optcomp===Infinity,
             (*__ formal computations __*)
             
    Print["form[",d,",",alpha,",",n,"]=",
      Simplify[ u[[alpha]][[n+1]] ]],
            
             True,
             (*__ numerical computations __*)
             
    Print["form[",d,",",alpha,",",n,"]=",
           N[u[[alpha]][[n+1]],optprecw]]
           
           ];(*end of Which*)        
           
  If[optexaexp===True,
     (*_______ printing the exact moment ________*) 
     
      Which[
            optcomp===Infinity,
            (*__ formal computations __*)
    
       Print["formexact[",d,",",alpha,",",n,"]=",
              formexact[d,alpha,n]],
              
            True,
            (*__ numerical computations __*)
            
       Print["formexact[",d,",",alpha,",",n,"]=",
              N[formexact[d,alpha,n],optprecw]]
              
           ];(*end of Which*)
            
   If[opterror===True, 
   (*_ computing a measure of the errors commited _*)
    
  Which[
    
        Or[kinderror==="abs",formexact[d,alpha,n]===0],
        (*_________ absolute errors _________*)
       
  Print["abserror[",d,",",alpha,",",n,"]=",
     abserror[formexact[d,alpha,n],u[[alpha]][[n+1]],
              opterrorprec,opterrorw]
          ](*end of Print*),
         
        kinderror==="sgd",
        (*_____ exact significant digits ______*)
       
    Print["exactdigits[",d,",",alpha,",",n,"]=",
    exactdigits[formexact[d,alpha,n],u[[alpha]][[n+1]],
                opterrorprec,opterrorw]
         ](*end of Print*),
         
        kinderror==="rel",
        (*___________ relative errors __________*)
        
    Print["relerror[",d,",",alpha,",",n,"]=",    
    relerror[formexact[d,alpha,n],u[[alpha]][[n+1]],
              opterrorprec,opterrorw]
         ](*end of Print*)
    
    ](*end of Which[kinderror]*)
 (*_____________________________________________*)
   
   ](*end of If[opterror]*)
(*_____________________________________________*) 
   
  ](*end of If[optexaexp]*);
(*_____________________________________________*)  
  Print["    "],
                                       
       {n,0,nmax}], (*end of Do[n]*) 
(*_____________________________________________*)    
  {alpha,1,d}] (*end of Do[alpha]*)
(*_____________________________________________*)  
];(*end of Block*)

(*_____________________________________________*)

   
End[]

EndPackage[]