(* :Title: modchv.m   *)

(* :Context: modchebyshev`modchv` *)

(* :Author:  Zlia da ROCHA *)

(* :Address: 
            Departamento de Matemtica Aplicada da
            Faculdade de Cincias do Porto
            Rua das Taipas, 135
            4050 Porto
            Portugal
  
            Tel: ++351 02 2080313
            Fax: ++351 02 2004109
            Email: mrdioh@fc.up.pt
*)
(* :Summary:
    The modchv package implements the Modified 
    Chebyshev's method for computing the
    coefficients that appear in the (d+1)-order
    recurrence relation satisfied by orthogonal
    polynomials of dimension d.    
*)
(* :Date: July 1999. *)

(* :Available at http://netlib.bell-labs.com/ in the 
    numeralgo library in netlib:       *)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 3.0 *)  
 
(* :Keywords: Modified Chebyshev's method, recurrence
              relations, orthogonal polynomials of
              dimension d, d-dimensional forms.
*)
(* :Sources:
   Z. da Rocha, Shohat-Favard and Chebyshev's
   methods in d-orthogonality, Numerical Algorithms,
   20 (1999) 139-164.
*)

(* :Warnings: How to load this package.

This package must be putted in a directory named 
modchebyshev. 
Note that the directory containing the modchebyshev
directory must be on the file search path $Path.

Then, do <<modchebyshev`modchv` or 
Needs["modchebyshev`modchv`"] in any notebook to load
this package. 
*)

(* :Requirements: 

This package requires a data package where are given
the definitions form, auxgamma, auxbeta,
and eventually gammaexact and betaexact.
The corresponding context must have a name like 
modchebyshev`something`.
For example, in the examples treated in the paper, we 
call the packages datamodchv42.m and datamodchv43.m,
corresponding to the examples 4.2 and 4.3, and the
contexts are modchebyshev`datamodchv42` and 
modchebyshev`datamodchv43`, respectively.

In order to test a different example, the user must
write the corresponding context in the BeginPackage 
command of this package, and load again this package.

In order to test a new example, the user must write a 
new data package with the new definitions (form,
auxgamma, auxbeta, betaexact and gammaexact), write 
the corresponding context in the BeginPackage command
of this package, and load again this package.
*)

(* :Examples:

See the packages datamodchv42.m and datamodchv43.m,
corresponding to the examples of the paper and the 
notebooks testmodchv42.nb and testmodchv43.nb with
the corresponding results.

In order to test a new example the user can use
the package datamodchvnew.m and the notebook
testmodchvnew.nb.
*)


BeginPackage["modchebyshev`modchv`",
             "modchebyshev`datamodchv42`"]

Clear[auxp, modform, modz, modgamma, modbeta, 
      moddorthpoly]
  
modchv::usage="is a package that implements the
Modified Chebyshev's method for computing
the gamma and beta coefficients that appear in the 
(d+1)-order recurrence relation satisfied by 
orthogonal polynomials of dimension d.

This package contains the definitions of the
Modified Chebyshev's relations as they are written
in the article:

Z. da ROCHA, Shohat-Favard and Chebyshev's 
             methods in d-orthogonality,
             Numerical Algorithms, 20 (1999) 139 - 164.
            
We use here a notation similar to that it is used in
the paper.

This package contains the definitions of:

- the recurrence relation of the modified Z-elements
  (modz). 

- the d gamma coefficients and the beta coefficient
  (modgamma, modbeta), that appear in the (d+1)-order
  recurrence relation satisfied by orthogonal
  polynomials of dimension d.
  These coefficients are calculated from the modified
  Z-elements, using the Modified Chebyshev's relations.

- printmodgamma and printmodbeta for printing the
  preceding coefficients in an adequate format,
  eventually calculating a measure of the errors,
  if their exact values are known.
  
- the n-th orthogonal polynomial of dimension d
  (moddorthpoly) that is calculated from the 
  (d+1)-order recurrence relation using the
  preceding coefficients modgamma and modbeta,
  obtained by the Modified Chebyshev's method.
  
- printmoddorthpoly for printing the first nmax+1 
  orthogonal polynomials of dimension d in an adequate
  format.

Probably, the user needs only to use the
definitions printmodgamma, printmodbeta and
printmoddorthpoly.

The user must give the definitions form, auxgamma and
auxbeta.
If the exact expressions of the gamma and the beta 
coefficients are known, the user can give the 
corresponding definitions (gammaexact and betaexact) in
order to evaluate the errors committed in printmodgamma
and printmodbeta. 
The definitions form, auxgamma, auxbeta, gammaexact and
betaexact must be written in a package named for example
datamodchvXX.m with context modchebyshev`datamodchvXX`. 
This context must be written above in the BeginPackage
command of this package, and then load this package."


modz::usage="modz[ d, alpha, n, m, optcomp ], 
1<=alpha<=d.
By definition, 
modz[d,alpha,n,m,optcomp]=< Talpha, \[CapitalPi]m*Pn >
is the result of applying the form Talpha to 
\[CapitalPi]m*Pn. {Pn}n>=0 is the sequence of orthogonal
polynomials of dimension d, with respect to the 
d-dimensional form T=(T1,...,Td). 
{\[CapitalPi]m}m>=0 is an auxiliary sequence of 
d-orthogonal polynomials, given by the coefficents
that appear in the (d+1)-order recurrence relation
satisfied by these polynomials (auxgamma, auxbeta).
modz[d,alpha,n,m,optcomp] is calculated using a
Modified Chebyshev's relation.
modz[d,alpha,n,m,optcomp] uses the modified moments 
of the forms T1,...,Td, with respect to the auxiliary 
sequence {\[CapitalPi]m}m>=0.
These modified moments are calculated from the
moments of the forms T1,...,Td, given by the 
definition form, and the auxiliary polynomials 
\[CapitalPi]m, given by the definitions auxgamma
and auxbeta, that must be written by the user, in a
package named for example datamodchvXX.m with context 
modchebyshev`datamodchvXX`.

The argument optcomp determines if we perform formal
computations (optcomp=Infinity), or numerical 
computations (optcomp is equal to the number of 
significant digits). The default value of optcomp is
Infinity.

See also the definitions modbeta, modgamma, 
printmodgamma, printmodbeta, moddorthpoly,
printmoddorthpoly."


modgamma::usage="modgamma[ d, r, n, optcomp ], 0<=r<=d-1,
is the coefficient gamma of order r, that appears in
the (d+1)-order recurrence relation satisfied by
orthogonal polynomials of dimension d.
modgamma is calculated from modz using the Modified
Chebyshev's method.

The argument optcomp determines if we perform formal
computations (optcomp=Infinity), or numerical 
computations (optcomp is equal to the number of 
significant digits). The default value of optcomp is
Infinity.

See also the definitions modbeta, modz, printmodgamma,
printmodbeta, moddorthpoly, printmoddorthpoly."


modbeta::usage="modbeta[ d, n, optcomp ],
is the coefficient beta, that appears in the 
(d+1)-order recurrence relation satisfied by 
orthogonal polynomials of dimension d.
modbeta is calculated from modz using the
Modified Chebyshev's method.

The argument optcomp determines if we perform formal
computations (optcomp=Infinity), or numerical 
computations (optcomp is equal to the number of
significant digits). The default value of optcomp is
Infinity.

See also the definitions modgamma, modz, printmodgamma,
printmodbeta, moddorthpoly, printmoddorthpoly."


printmodgamma::usage="
printmodgamma[ d, nmax, 
               optcomp, optprecw,
               optexaexp, opterror, kinderror,
               opterrorprec, opterrorw ]
  
prints the d gamma coefficients, from order 1
to nmax+r-d+1, calling the definition
modgamma[ d, r, n ,optcomp ], 1<= n <=nmax+r-d+1,
0<=r<=d-1. 

We can perform formal computations or numerical
computations according with the moments of the
d-dimensional form and/or the coefficients of the
auxiliary sequence of d-orthogonal polynomials  
(see the definitions form, auxgamma, auxbeta in the 
package datamodchv.m) are given in a formal
way or in a numerical way. If they are given in a 
formal way, we can perform numerical computations as
well.

If we do formal computations, we must take 
optcomp=Infinity. If we do numerical
computations, optcomp is the number of significant
digits employed.
The default value of optcomp is Infinity.

If we perform numerical computations, the modgamma 
coefficients are writing with optprecw digits.
The default value of optprecw is equal to 
$MachinePrecision.

If the exact expressions of the gamma coefficients
are given in the data package (gammaexact),
taking optexaexp=True, we can write them.
If we perform numerical computations, the exact gamma
coefficients are writen with optprecw digits of
precision.

We can also evaluate the errors committed taking 
opterror=True. We can calculate the absolute error,
the number of exact digits, or the relative errors
of the modgamma coefficients, taking respectively 
kinderror=\"abs\", kinderror=\"sgd\" or 
kinderror=\"rel\".
If gammaexact===0, we calculate absolute errors.
The evaluation of the errors are made performing 
computations with precision opterrorprec and the
results are written with opterrorw digits.
The default values of optexaexp, opterror and 
kinderror are respectively True, True and \"sgd\".
The default values of opterrorprec and opterrorw 
are respectively $MachinePrecision and 2.

See also the definitions printmodbeta, moddorthpoly,
printmoddorthpoly, modbeta, modgamma, modz."


printmodbeta::usage="
printmodbeta[ d, nmax, 
              optcomp, optprecw,
              optexaexp, opterror, kinderror,
              opterrorprec, opterrorw ]

prints the beta coefficients from order 0 to nmax-1,
calling the definition
modbeta[ d, n, optcomp ], 0<= n <=nmax-1. 

We can do formal computations or numerical
computations according with the moments of the
d-dimensional form and/or the coefficients of the
auxiliary sequence of d-orthogonal polynomials  
(see the definitions form, auxgamma, auxbeta in the 
package datamodchv.m) are given in a formal way or in a
numerical way. If they are given in a formal way, we
can perform numerical computations as well.

If we perform formal computations, we must take 
optcomp=Infinity. If we perform numerical
computations, optcomp is the number of significant
digits employed.
The default value of optcomp is Infinity.

If we perform numerical computations, the modbeta 
coefficients are writing with optprecw digits.
The default value of optprecw is equal to 
$MachinePrecision.

If the exact expressions of the beta coefficients
are given in the data package (betaexact),
taking optexaexp=True, we can write them.
If we perform numerical computations, the exact beta
coefficients are written with optprecw digits of
precision.

We can also evaluate the errors committed taking 
opterror=True. We can calculate the number of 
exact digits, or the relative errors of the beta
coefficients, taking respectively kinderror=\"abs\",
kinderror=\"sgd\" or kinderror=\"rel\".
If betaexact===0, we calculate absolute errors.
The evaluation of the errors are made performing 
computations with precision opterrorprec and the
results are written with opterrorw digits.
The default values of optexaexp, opterror and 
kinderror are respectively True, True and 2.
The default values of opterrorprec and opterrorw 
are respectively $MachinePrecision and 2.

See also the definitions printmodgamma, 
printmoddorthpoly, moddorthpoly, modbeta, modgamma,
modz."


moddorthpoly::usage="
moddorthpoly[ d, n, var, optcomp ]
is the orthogonal polynomial of dimension d and 
degree n, calculated using the (d+1)-order recurrence
relation that calls the definitions modgamma and 
modbeta.

See also the definitions printmoddorthpoly,
printmodgamma, printmodbeta, modbeta, modgamma, modz."


printmoddorthpoly::usage="
printmoddorthpoly[ d, nmax, var, optcomp, optprecw ] 

prints the first nmax+1 orthogonal polynomials of
dimension d calling the definition
moddorthpoly[ d, n, var, optcomp ], 0<= n <=nmax.

We can do formal computations or numerical
computations according with the moments of the
d-dimensional form and/or the coefficients of the
auxiliary sequence of d-orthogonal polynomials  
(see the definitions form, auxgamma, auxbeta in the 
package datamodchv.m) are given in a formal
way or in a numerical way. If they are given in a 
formal way, we can perform numerical computations 
as well.

If we perform formal computations, we must take 
optcomp===Infinity. If we perform numerical
computations, optcomp is the number of significant
digits employed.
The default value of optcomp is Infinity.

If we perform numerical computations, the coefficients
of the polynomials in the canonical base are writing 
with optprecw digits.

The default value of optprecw is $MachinePrecision.

See also the definitions moddorthpoly, printmodgamma,
printmodbeta, modbeta, modgamma, modz."


Begin["`Private`"]

(*_________________________________________________

                    Error Messages
  ________________________________________________*)
   
modz::badarg2=
   "wrong value to the second argument
    alpha=`1` in modz;
    alpha must be an integer and verify
    1<=alpha<=`2`;
    try again"
    
modz::badarg34=
   "wrong value to the third and/or fourth arguments
    n=`1` and m=`2` in modz;
    n and m must be integers;
    try again"
    
modz::badarg5=
   "wrong value to the fifth argument
    optcomp= `1` in modz;
    optcomp must be a positive integer,
    or be equal to Infinity;
    try again"
    
modz::badarg12345=
   "something wrong in modz;
    the value of modz[`1`,`2`,`3`,`4`,`5`] is undefined;
    try again"

modgamma::badarg2=
   "wrong value to the second argument r=`1`
    in modgamma, r must be an integer and
    verify 0<=r<=`2`;
    try again"
    
modgamma::badarg3=
   "wrong value to the third argument n=`1`
    in modgamma, n must be an integer;
    try again"

modgamma::badarg4=
   "wrong value to the fourth argument optcomp=`1`
    in modgamma; 
    optcomp must be a positive integer,
    or be equal to Infinity;
    try again"
    
modgamma::badarg1234=
   "something wrong in modgamma;
    the value of modgamma[`1`,`2`,`3`,`4`]
    is undefined;
    try again"
    
modbeta::badarg2=
   "wrong value to the second argument n=`1`
    in modbeta; n must be an integer;
    try again"

modbeta::badarg3=
   "wrong value to the third argument optcomp=`1`
    in modbeta; 
    optcomp must be a positive integer,
    or be equal to Infinity;
    try again"

modbeta::badarg123=
   "something wrong in modbeta;
    the value of modbeta[`1`,`2`,`3`]
    is undefined;
    try again"

printmodgamma::badarg2=    
   "wrong value to the second argument
    nmax=`1` in printmodgamma;
    nmax must be a positive integer;
    try again"
    
printmodgamma::badarg3=
   "wrong value to the third argument
    optcomp= `1` in printmodgamma;
    optcomp must be a positive integer;
    or be equal to Infinity;
    try again"
   
printmodgamma::badarg7=
   "wrong value to the seventh argument
    kinderror= `1` in printmodgamma;
    kinderror must be equal to \"abs\", 
    \"sgd\" or \"rel\";        
    try again"

printmodbeta::badarg2=    
   "wrong value to the second argument
    nmax=`1` in printmodbeta;
    nmax must be a non negative integer;
    try again"

printmodbeta::badarg3=
   "wrong value to the third argument
    optcomp= `1` in printmodbeta;
    optcomp must be a positive integer;
    or be equal to Infinity;
    try again"

printmodbeta::badarg7=
   "wrong value to the seventh argument
    kinderror=`1` in printmodbeta;
    kinderror must be equal to \"abs\", 
    \"sgd\" or \"rel\"; 
    try again"

moddorthpoly::badarg2=
   "wrong value to the second argument n=`1`
    in modbeta; n must be an integer;
    try again"

moddorthpoly::badarg4=
   "wrong value to the fourth argument
    optcomp= `1` in moddorthpoly;
    optcomp must be a positive integer;
    or be equal to Infinity;   
    try again"
    
moddorthpoly::badarg1234=
   "something wrong in moddorthpoly;
    the value of moddorthpoly[`1`,`2`,`3`,`4`]
    is undefined;   
    try again"
    
printmoddorthpoly::badarg2=    
   "wrong value to the second argument
    nmax=`1` in printmoddorthpoly;
    nmax must be a non negative integer;
    try again"
    
printmoddorthpoly::badarg4=    
   "wrong value to the fourth argument
    optcomp= `1` in printmoddorthpoly;
    optcomp must be a positive integer;
    or be equal to Infinity;
    try again"
    
(*_______________________________________________

               Private definitions 
________________________________________________*)
(*_______________________________________________

                absolute errors
 _______________________________________________*)
 
abserror[exactvalue_,approxvalue_,
         errorprec_:$MachinePrecision,
         errorprint_:2]:=
 N[N[Abs[exactvalue-approxvalue],
     errorprec],errorprint];

(*_______________________________________________

             exact significant digits
 _______________________________________________*)
   
exactdigits[exactvalue_,approxvalue_,
            errorprec_:$MachinePrecision,
            errorprint_:2]:=       
N[   
  N[-Log[10,Abs[exactvalue-approxvalue]],
    errorprec]+
   MantissaExponent[approxvalue][[2]]-1,
  errorprint];
  
(*_________________________________________________

                   relative errors
 _________________________________________________*)

relerror[exactvalue_,approxvalue_,
         errorprec_:$MachinePrecision,
         errorprint_:2]:=           
N[N[Abs[(exactvalue-approxvalue)/exactvalue],
   errorprec],errorprint];

(*_________________________________________________

The auxiliary sequence of d-orthogonal polynomials
calculated form the (d+1)-order recurrence relation
using the coefficients auxgamma and auxbeta.
___________________________________________________*)

auxp[d_,n_,var_]:=auxp[d,n,var]=
 Block[{l},
  
  Which[n<0,Return[0],
        n===0,Return[1],
        n>0,
 Return[
  Collect[(var-auxbeta[d,n-1])*auxp[d,n-1,var]-
  Sum[auxgamma[d,d-1-l,n-1-l]*auxp[d,n-2-l,var],
     {l,0,d-1}],var]
       ]
       ](*end of Which[n]*)
       
  ];(*end of Block*)

(*________________________________________________

  The modified moments of the d-dimensional form 
  with respect to the auxiliary sequence of 
  d-orthogonal polynomials.
__________________________________________________*)

modform[d_,alpha_,n_]:=modform[d,alpha,n]=
 Block[{k,var},
 
   Return[Sum[Coefficient[auxp[d,n,var],var,k]*
              form[d,alpha,k],
             {k,0,n}]
         ]
       
      ];(*end of Block*)
                       
(*________________________________________________

                Exported definitions 
__________________________________________________*)
     
           
(*________________________________________________

The recurrence relation of the modified Z-elements 
__________________________________________________*)

modz[d_,alpha_,n_,m_,optcomp_:Infinity]:=
modz[d,alpha,n,m,optcomp]=
 
 Block[{},
 (*___________ checking the arguments___________*)
 
  If[ Or[ alpha<1,alpha>d,Not[IntegerQ[alpha]] ],
    
 	Return[Message[modz::badarg2,alpha,d]]
 	]; 
  
  If[ Or[ Not[IntegerQ[n]], Not[IntegerQ[m]] ],
  
    Return[Message[modz::badarg34,n,m]]
    ];
    
  If[ And[optcomp=!=Infinity,
          Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
    Return[Message[modz::badarg5,optcomp]]
    ];
 (*_____________________________________________*)
 
        Which[
              Or[n<0,m<0], 
               Return[0],
              n===0,
               Return[ N[modform[d,alpha,m],optcomp] ],  
              n>0,
  Return[modz[d,alpha,n-1,m+1,optcomp]+
         auxbeta[d,m]*modz[d,alpha,n-1,m,optcomp]+
         Sum[ auxgamma[d,d-1-l,m-l]*
              modz[d,alpha,n-1,m-1-l,optcomp],
            {l,0,d-1}]-
     modbeta[d,n-1,optcomp]*modz[d,alpha,n-1,m,optcomp]-
         Sum[ modgamma[d,d-1-l,n-1-l,optcomp]*
              modz[d,alpha,n-2-l,m,optcomp],
             {l,0,d-1}]
        ],
              True,
  Return[Message[modz::badarg12345,d,alpha,n,m,optcomp]]
           
             ](*end of Which*)
 (*________________________________________________*) 
     
];(*end of Block*)
   
(*_________________________________________________

           The modified gamma coefficients
__________________________________________________*)

modgamma[d_,r_,n_,optcomp_:Infinity]:=
modgamma[d,r,n,optcomp]=
 
 Block[{m,alpha,l},
 (*____________ checking the arguments ___________*)
 
  If[Or[r<0,r>d-1,Not[IntegerQ[r]]],
    
 	Return[Message[modgamma::badarg2,r,d-1]]
 	];
 	
  If[ Not[IntegerQ[n]],
   
    Return[Message[modgamma::badarg3,n]]
    ];
  
  If[ And[optcomp=!=Infinity,
          Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
    Return[Message[modgamma::badarg4,optcomp]]
    ];
(*________________________________________________
  the modified gamma coefficients are computed from 
  the modified Z-elements using the Modified
  Chebyshev's relations
_________________________________________________*) 
 
        Which[
              n<=0,Return[0],
              r===0,
                m=Floor[n/d]; alpha=n-d*m;
                If[alpha===0, m=m-1;alpha=d];
       Return[modz[d,alpha,(m+1)*d+alpha-1,m+1,optcomp]/
              modz[d,alpha,m*d+alpha-1,m,optcomp]
             ],
              n===1,
       Return[modz[d,1,d-r,1,optcomp]/
              modz[d,1,0,0,optcomp]
             ],
              And[2<=n,n<=r],
            Return[(modz[d,n,d+n-r-1,1,optcomp]-
             Sum[modgamma[d,d-1-l,d-r-1-l+n,optcomp]*
                 modz[d,n,d-r+n-l-2,0,optcomp],
                   {l,d-r,d-r+n-2}])/
                 modz[d,n,n-1,0,optcomp]
                   ],
             And[1<=r,r<=d-1],  
                 m=Floor[n/d]; alpha=n-d*m;
                 If[alpha===0, m=m-1;alpha=d];
                 If[Or[And[m===0,r+1<=alpha,alpha<=d],
                       And[m>=1,1<=alpha,alpha<=d]],
    Return[(modz[d,alpha,(m+1)*d+alpha-r-1,m+1,optcomp]-
     Sum[modgamma[d,d-1-l,(m+1)*d+alpha-r-1-l,optcomp]*
         modz[d,alpha,(m+1)*d+alpha-r-2-l,m,optcomp],
            {l,d-r,d-1}])/
         modz[d,alpha,m*d+alpha-1,m,optcomp]
            ] (*end of Return*)
                   ](*end of If*),
                   
               True,
    Return[Message[modgamma::badarg1234,d,r,n,optcomp]]
       
              ](*end of Which*)
(*________________________________________________*)  
           
  ];(*end of Block*)


(*________________________________________________
         
          The modified beta coefficients
_________________________________________________*)


modbeta[d_,n_,optcomp_:Infinity]:=
modbeta[d,n,optcomp]=

 Block[{l,m,r},
 (*___________ checking the arguments___________*)
 
  If[ Not[IntegerQ[n]],
  
    Return[Message[modbeta::badarg2,n]]
    ];
  
  If[ And[optcomp=!=Infinity,
          Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
    Return[Message[modbeta::badarg3,optcomp]]
    ];
 (*______________________________________________
  The modified beta coefficients are computed from 
  the modified Z-elements using the Modified
  Chebyshev's relations
________________________________________________*)
 
  Which[
        n<0,Return[0],
        n===0,
        Return[auxbeta[d,0]+modz[d,1,0,1,optcomp]/
               modz[d,1,0,0,optcomp]],
        And[1<=n,n<=d-1],
         Return[auxbeta[d,0]+(modz[d,n+1,n,1,optcomp]-
                Sum[modgamma[d,d-1-l,n-l,optcomp]*
                    modz[d,n+1,n-1-l,0,optcomp],
                   {l,0,n-1}])/modz[d,n+1,n,0,optcomp]
               ],
        n>=d,
           m=Floor[n/d]; r=n-d*m;
    Return[auxbeta[d,m]+(modz[d,r+1,m*d+r,m+1,optcomp]-
           Sum[modgamma[d,d-1-l,m*d+r-l,optcomp]*
               modz[d,r+1,m*d+r-1-l,m,optcomp],
              {l,0,d-1} ])/modz[d,r+1,m*d+r,m,optcomp]
          ],
        True,
      Return[Message[modbeta::badarg123,d,n,optcomp]]
       
       ](*end of Which*)
 (*______________________________________________*)
        
 ];(*end of Block*)
    
 (*_______________________________________________
 
           Printing the gamma coefficients
 _________________________________________________*)
 
 printmodgamma[d_,nmax_,
   optcomp_:Infinity,optprecw_:$MachinePrecision,
   optexaexp_:True,opterror_:True,kinderror_:"sgd",
   opterrorprec_:$MachinePrecision,opterrorw_:2]:=
   
 Block[{r,n},
 (*___________ checking the arguments___________*)
 
 If[ Or[ Not[Integer[nmax]], nmax<=0 ],
 
    Return[Message[printmodgamma::badarg2,nmax]]
   ];
 
 If[ And[optcomp=!=Infinity,
         Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
    Return[Message[printmodgamma::badarg3,optcomp]]
    ];
    
 If[
And[kinderror=!="abs",kinderror=!="sgd",
    kinderror=!="rel"],
        
   Return[Message[printmodgamma::badarg7,kinderror]]
   ];
(*______________________________________________*)   
 Do[
    
    Print["   "];
    Do[  
    (*___ printing the modgamma coefficient ____*)
    
       Which[
             optcomp===Infinity,
             (*__ formal computations __*)
             
    Print["modgamma[",d,",",r,",",n,"]=",
           modgamma[d,r,n,optcomp]],
            
             True, 
             (*__ numerical computations __*)
             
    Print["modgamma[",d,",",r,",",n,"]=",
           N[modgamma[d,r,n,optcomp],optprecw]]
           
           ];(*end of Which*)        
           
    If[optexaexp===True,
    (*___ printing the exact gamma coefficient____*)
      
      Which[
            optcomp===Infinity,
            (*__ formal computations __*)
    
       Print["gammaexact[",d,",",r,",",n,"]=",
              gammaexact[d,r,n]],
              
            True,
            (*__ numerical computations __*)
            
       Print["gammaexact[",d,",",r,",",n,"]=",
              N[gammaexact[d,r,n],optprecw]]
              
           ];(*end of Which*)
            
    If[ opterror===True, 
   (*_ computing a measure of the errors commited _*)
    
       Which[
       
      Or[kinderror==="abs",gammaexact[d,r,n]===0],
      (*_________ absolute errors _________*)
       
  Print["abserror[",d,",",r,",",n,"]=",
     abserror[gammaexact[d,r,n],modgamma[d,r,n,optcomp],
              opterrorprec,opterrorw]
       ](*end of Print*), 
         
     kinderror==="sgd",
     (*_____ exact significant digits ______*)
       
    Print["exactdigits[",d,",",r,",",n,"]=",
    exactdigits[gammaexact[d,r,n],modgamma[d,r,n,optcomp],
              opterrorprec,opterrorw]
         ](*end of Print*),
         
     kinderror==="rel",
     (*___________ relative errors __________*)
        
    Print["relerror[",d,",",r,",",n,"]=",
     relerror[gammaexact[d,r,n],modgamma[d,r,n,optcomp],
            opterrorprec,opterrorw] 
         ](*end of Print*)
    
    ](*end of Which[kinderror]*)
    
   ](*end of If[opterror]*)
 (*____________________________________________*)
   
  ](*end of If[optexaexp]*);
 (*____________________________________________*)
  
   Print["  "],       
                                       
       {n,1,nmax+r-d+1}], (*end of Do[n]*) 
  (*___________________________________________*)  
  {r,0,d-1}] (*end of Do[r]*)
 (*____________________________________________*) 
];(*end of Block*)
 
 
(*_________________________________________________
 
            Printing the beta coefficients
 _________________________________________________*)
 
 printmodbeta[d_,nmax_,
   optcomp_:Infinity,optprecw_:$MachinePrecision,
   optexaexp_:True,opterror_:True,kinderror_:"sgd",
   opterrorprec_:$MachinePrecision,opterrorw_:2]:=
   
 Block[{n},
 (*___________ checking the arguments ___________*)
 
 If[ Or[ Not[Integer[nmax]], nmax<0 ],
 
    Return[Message[printmodbeta::badarg2,nmax]]
   ];
 
 If[ And[optcomp=!=Infinity,
         Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
    Return[Message[printmodbeta::badarg3,optcomp]]
    ];
    
 If[ And[kinderror=!="abs",kinderror=!="sgd",
         kinderror=!="rel"],
        
   Return[Message[printmodbeta::badarg7,kinderror]]
   ];
   
(*_______________________________________________*) 

    Do[(*_____ printing the beta coefficient_____*)
    
       Which[
             optcomp===Infinity,
             (*__ formal computations __*)
             
    Print["modbeta[",d,",",n,"]=",
           modbeta[d,n,optcomp]],
           
             True,
             (*__ numerical computations __*)
             
    Print["modbeta[",d,",",n,"]=",
          N[modbeta[d,n,optcomp],optprecw]]
          
            ];(*end of Which*)       
           
   If[optexaexp===True,
      (*___ printing the exact beta coefficient____*)
   
     Which[
            optcomp===Infinity,
            (*__ formal computations __*)
    
      Print["betaexact[",d,",",n,"]=",
             betaexact[d,n]],
          
          True,
          (*__ numerical computations __*)
           
      Print["betaexact[",d,",",n,"]=",
           N[betaexact[d,n],optprecw]]
         
         ];(*end of Which*) 
  
  If[opterror===True,       
  (*_ computing a measure of the errors commited _*)
          
     Which[
       Or[kinderror==="abs",betaexact[d,n]===0],
       (*_________ absolute errors _________*)
       
  Print["abserror[",d,",",n,"]=",
     abserror[betaexact[d,n],modbeta[d,n,optcomp],
              opterrorprec,opterrorw]
       ](*end of Print*),
         
       kinderror==="sgd",
       (*_____ exact significant digits ______*)
       
    Print["exactdigits[",d,",",n,"]=",
     exactdigits[betaexact[d,n],modbeta[d,n,optcomp],
                 opterrorprec,opterrorw]
         ](*end of Print*),
         
        kinderror==="rel",
        (*_________ relative errors _________*)
        
    Print["relerror[",d,",",n,"]=",
     relerror[betaexact[d,n],modbeta[d,n,optcomp],
              opterrorprec,opterrorw]
         ](*end of Print*)     
    
   ](*end of Which[kinderror]*)
   
  ](*end of If[opterror]*)
  (*___________________________________________*)
   
  ](*end of If[optexaexp]*);
  (*___________________________________________*)
  
  Print["  "],
          
 {n,0,nmax-1}] (*end of Do[n]*)
 (*___________________________________________*)
  
];(*end of Block*)

(*_______________________________________________

          The orthogonal polynomial of 
  dimension d and degree n in the variable var
________________________________________________*)

moddorthpoly[d_,n_,var_,optcomp_:Infinity]:=
moddorthpoly[d,n,var,optcomp]=

 Block[{l},
 (*__________ checking the arguments__________*)
 
 If[ Not[IntegerQ[n]],
   
    Return[Message[moddorthpoly::badarg2,n]]
    ];
 
 If[ And[optcomp=!=Infinity,
          Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
   Return[Message[moddorthpoly::badarg4,optcomp]]
    ];
(*________________________________________________
       the (d+1)-order recurrence relation for 
       orthogonal polynomials of dimension d, 
       where appear the modgamma and modbeta 
       coefficients       
__________________________________________________*)
  
  Which[n<0,Return[0],
        n===0,Return[1],
        n>0,
  Return[ Collect[(var-modbeta[d,n-1,optcomp])*
           moddorthpoly[d,n-1,var,optcomp]-
           Sum[modgamma[d,d-1-l,n-1-l,optcomp]*
               moddorthpoly[d,n-2-l,var,optcomp],
              {l,0,d-1}],
           var]
        ],
        True,
 Return[Message[moddorthpoly::badarg1234,
                d,n,var,optcomp]]
       ](*end of Which[n]*)
 (*___________________________________________*) 
      
  ];(*end of Block*)
  

(*_________________________________________________

Printing the orthogonal polynomials of dimension d
__________________________________________________*)


printmoddorthpoly[d_,nmax_,var_,
   optcomp_:Infinity,optprecw_:$MachinePrecision]:=
   
 Block[{n},
  (*__________ checking the arguments__________*)
  
   If[ Or[nmax<0,Not[IntegerQ[nmax]]],
   
  Return[Message[printmoddorthpoly::badarg2,nmax]]
     ];
  
   If[ And[optcomp=!=Infinity,
          Not[And[IntegerQ[optcomp],optcomp>0]] ],
  
  Return[Message[printmoddorthpoly::badarg4,optcomp]]
     ];
  (*_____________________________________________
         printing the orthogonal polynomial of
               dimension d an degree n
  ______________________________________________*)
          
  Which[
        optcomp===Infinity, 
        (*__ formal computations __*)      
        
    Do[
    Print["moddorthpoly[",d,",",n,",",var,
                        ",",optcomp,"]=",
           moddorthpoly[d,n,var,optcomp]]
     ,{n,0,nmax}],
     
        True,
        (*__ numerical computations __*) 
              
   Do[
Print["moddorthpoly[",d,",",n,",",var,"]=",
      N[moddorthpoly[d,n,var,optcomp],optprecw]]
     ,{n,0,nmax}]
     
       ](*end of Which*) 
(*______________________________________________*)
     
 ];(*end of Block*)
      
(*______________________________________________*)


End[]

EndPackage[]