(* :Title: datachv2.m   *)

(* :Context: chebyshev`datachv2` *)

(* :Author:  Zlia da ROCHA *)

(* :Address: 
            Departamento de Matemtica Aplicada da
            Faculdade de Cincias do Porto
            Rua das Taipas, 135
            4050 Porto
            Portugal
  
            Tel: ++351 02 2080313
            Fax: ++351 02 2004109
            Email: mrdioh@fc.up.pt
*)
(* :Summary:
    The datachv2 package contains the definition form,
    gammaexact and betaexact that the package chv needs,
    corresponding to the example 2 of the first paper
    cited bellow.    
*)
(* :Date: July 1999. *)

(* :Available at http://netlib.bell-labs.com/ in the 
    numeralgo library in netlib:       *)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 3.0 *)  
 
(* :Keywords: chebyshev'method, recurrence relations, 
              orthogonal polynomials of dimension d,
              d-dimensional forms.
*)
(* :Sources:

 - Z. da Rocha, Shohat-Favard and Chebyshev's
   methods in d-orthogonality, Numerical Algorithms,
   20 (1999) 139 - 164.
   
 - About the example treated here see pp.90,106 of:
 
   K. Douak, P. Maroni, Les polynmes orthogonaux 
   "classiques" de dimension deux, Analysis 12, 
   71-107 (1992).
*)

(* :Warnings: How to load this package.

This package must be putted in a directory named 
chebyshev. 
Note that the directory containing the chebyshev
directory must be on the file search path $Path.

This package contains the definitions needed in the
package chv.m. 
The context corresponding to this package
(chebyshev`datachv2`) must be written in the 
BeginPackage command of the package chv.m.
Then, when you load the package chv.m doing 
<<chebyshev`chv` or Needs["chebyshev`chv`"] in any
notebook, this package is automatically loaded. 
*)


BeginPackage["chebyshev`datachv2`"]

datachv2::usage="is a package that contains
the definitions of:

- the moments of the d-dimensional form with respect
  to which the polynomials are orthogonal of dimension
  d (form),
  
- the exact expression of the d gamma coefficients
  and the beta coefficient that appear in the 
  (d+1)-order recurrence relation satisfied by 
  orthogonal polynomials of dimension d, in the case
  they are known (gammaexact, betaexact);
  
corresponding to the example 2 of the paper

Z. da Rocha, Shohat-Favard and Chebyshev's
methods in d-orthogonality, Numerical Algorithms,
20 (1999) 139-164.

The definitions form, gammaexact and betaexact are
used in the package chv.m."

form::usage="
form[d, alpha, n] is the moment of order n (n>=0)
of the form of order alpha (1<=alpha<=d), with
respect to which the polynomials are orthogonal
of dimension d.
form is used in the package chv.m."

gammaexact::usage="
gammaexact[ d, r, n ], is the exact expression of
the n-th gamma coefficient of order r (n>=1,
0<=r<=d-1), that appears in the (d+1)-order recurrence
relation satisfied by orthogonal polynomials of
dimension d.
gammaexact is used in the package chv.m."


betaexact::usage="
betaexact[ d, n ], n>=0, is the exact expression of
the n-th beta coefficient, that appears in the 
(d+1)-order recurrence relation satisfied by
orthogonal polynomials of dimension d.
betaexact is used in the package chv.m."

Begin["`Private`"]

(*___________________________________________

                Error Messages
 __________________________________________*)
   
form::badarg2=
   "wrong value to the second argument
    alpha=`1` in form;
    alpha must be an integer and verify
    1<=alpha<=`2`;
    try again"

form::badarg3=
   "wrong value to the third argument n=`1` in form;
    n must be an integer;
    try again" 

gammaexact::badarg2=
   "wrong value to the second argument r=`1`
    in gammaexact; r must be an integer and
    verify 0<=r<=`2`;
    try again"

gammaexact::badarg3=
   "wrong value to the third argument n=`1`
    in gammaexact;
    n must be an integer;
    try again" 

betaexact::badarg2=
   "wrong value to the second argument n=`1`
    in betaexact;
    n must be an integer;
    try again" 

(*___________________________________________

             Exported definition
___________________________________________*)
(*___________________________________________

   the moments of the d-dimensional form
___________________________________________*)

form[d_,alpha_,n_]:=form[d,alpha,n]=

 Block[{ro1,ld,m3,fact},
 
 (*_________ checking the arguments _________*)
 
   If[Or[alpha<1,alpha>d,Not[IntegerQ[alpha]]],
    
 	  Return[Message[form::badarg2,alpha,d]]
 	 ];	 
   
   If[Not[IntegerQ[n]],
   
      Return[Message[form::badarg3,n]]
     ];
(*_________________________________________*)
		
  Which[n<0,Return[0], 
        n>=0,
         ro1=1;ld=1;m3=3*n;fact=n/ro1;
          Which[
                alpha===1,
                If[n===0,Return[1],
 Return[ ro1*(ld+3)*(m3-1)*(m3-2)/2/
         (ld+m3)*form[2,1,n-1] ]
                  ],
                alpha===2,
 Return[fact*form[2,1,n] ]
 
               ](*end of Which[alpha]*)
        ](*end of Which[n]*)  
(*_____________________________________________*)
   
]; (*end of Block*)  
 
   
(*_________________________________________________

  the exact expression of the gamma coefficients
_________________________________________________*) 

gammaexact[d_,r_,n_]:=gammaexact[d,r,n]=
 Block[{ro1,ld,m,s,m3,m6,m9,},
 
 (*_________ checking the arguments __________*) 
 
  If[Or[r<0,r>=d,Not[IntegerQ[r]]],
  
  Return[Message[gammaexact::badarg2,r,d-1]]
    ];
    
  If[Not[IntegerQ[n]],
  
    Return[Message[gammaexact::badarg3,n]]
    ];
(*____________________________________________*)

  Which[n<=0,Return[0],
        n>0,
        ro1=1;ld=1;
        m=Floor[n/2]; s=n-2*m;
        m3=3*m;m6=6*m;m9=9*m;
        Which[r===0,
               Which[s===0,
Return[ ro1^3*(ld+3)^3*(m6-1)*(m6+1)*(m6+3)*(m3-1)*
        m3*(m3+1)/(m9-3+ld)/(m9+ld)/(m9+3+ld) ],
                     s===1,
Return[ ro1^3*(ld+3)^3*(m6+1)*(m6+3)*(m6+5)*(m3+1)*
        (m3+2)*(m3+3)*(m3+ld)*(m3+1+ld)*(m3+2+ld)/
        (m9+ld)/(m9+3+ld)^2/(m9+6+ld)^2/(m9+9+ld) ]
                   ],                  
              r===1,
               Which[s===0,
Return[ ro1^2*(ld+3)^2*(m6-1)*(m3-1)*m3/(m9-3+ld)/
        (m9+ld)*( (m6+1)+(m6-1)*(m3-1+ld)/(m9-3+ld)+
        (m6-3)*(m3-1+ld)*(m3-2+ld)/(m9-6+ld)/
        (m9-3+ld) ) ],
                     And[s===1,m===0],
Return[ 3*ro1^2*(3*ld+8)/(ld+6) ],
                     s===1,
Return[ ro1^2*(ld+3)^2*(m6+1)*(m6+3)*(m3+1)/(m9+ld)/
        (m9+3+ld)*( m3+(m3+ld)*(m3+1)/(m9+3+ld)+
        (m3+1+ld)*(m3+ld)*(m3+2)/(m9+3+ld)/
        (m9+6+ld) ) ]
                  ](*end of Which[s]*)
             ](*end of Which[r]*)     
      ](*end of Which[n]*)  
(*_________________________________________________*)

];(*end of Block*)


(*_________________________________________________

  the exact expression of the beta coefficients
_________________________________________________*) 

betaexact[d_,n_]:=betaexact[d,n]=

 Block[{m,s,ro1,ld,m3,m6,m9},
 
  (*_________ checking the arguments _________*)
      
      If[Not[IntegerQ[n]],
      
     Return[Message[betaexact::badarg3,n]]
        ];   
  (*__________________________________________*)
             
    Which[n<0,Return[0],
          n>=0,      
          ro1=1;ld=1;
          m=Floor[n/2]; s=n-2*m;
          m3=3*m;m6=6*m;m9=9*m;
          Which[n===0,
                 Return[ro1],
                s===0,
      Return[ ro1*(ld+3)/(m9+ld)*( (m6+1)*m3+
     (m6+1)*(m3+1)*(m3+ld)/(m9+3+ld)+
     (m6-1)*m3*(m3-1+ld)/(m9-3+ld) ) ],
                s===1,     
      Return[ ro1*(ld+3)*( (m6+3)*(m3+2)*(m3+1+ld)/
              (m9+3+ld)/(m9+6+ld)+
              (m6+3)*(m3+1)/(m9+3+ld)+
              (m6+5)*(m3+2)/(m9+6+ld) ) ]         
               ](*end of Which[s]*)
             
          ](*end of Which[n]*)   
 (*_____________________________________________*) 
   
      ];(*end of Block*)          
(*_______________________________________________*)

   
End[]

EndPackage[]