/*
 * unpack a netbuild package, sanity check it, and install it
 *
 * netbuild packages are shipped over the net in container files.
 * the container files are in tar format, with these components:
 *
 * "md5sums" - md5 fingerprints for all files (except *) (*)
 * "md5sums.gpg" - gpg detached signature for md5sums (*)
 * "metadata" - description of package
 * "pubkey.gpg" - public key of package signer (*)
 * "post-install.sh" - shell script to execute after install (if present)
 * library files
 *
 * all components must be "regular" files (no directories, symlinks, etc)
 * no component may be stored in other than the current directory
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <libgen.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "concat.h"
#include "copyfile.h"
#include "gpglib.h"
#include "misc.h"
#include "result.h"
#include "tar.h"
#include "tarlib.h"

static int verbosity = 0;

/*
 * callback for tar_rd.  return nonzero iff we want to extract this file.
 */

static int
cb (struct tar_mem *memhdr, void *closure)
{
    int i;
    int verbose = (int) closure;

    /*
     * if it's not a regular file, or if it references a nonlocal
     * directory, fail
     */

    if (memhdr->typeflag != REGTYPE && memhdr->typeflag != AREGTYPE) {
	fprintf (stderr, "unpack: component %s: illegal type %d\n",
		 memhdr->name, memhdr->typeflag);
	return 0;		/* don't extract */
    }

    if (memhdr->name[0] == '/' ||
	strncmp (memhdr->name, "../", 3) == 0 ||
	strncmp (memhdr->name + strlen (memhdr->name) - 3, "/..", 3) == 0 ||
	strstr (memhdr->name, "/../") != NULL) {
	fprintf (stderr,
		 "unpack: component %s references nonlocal directory\n",
		 memhdr->name);
	return 0;
    }

    /*
     * don't extract gpg.status
     */

    if (strstr (memhdr->name, "gpg.status") != NULL)
	return 0;

    if (verbosity > 1)
	printf (" [extracting %s]\n", memhdr->name);

    return 1;
}

/*
 * decode a string of hex characters representing an md5 digest
 * return 0 on success, -1 on error
 */

static int
decode_md5 (unsigned char *md5, char *md5str)
{
    int i;
    char *str = md5str;

    if (md5 == NULL || md5str == NULL)
	return -1;

    for (i = 0; i < MD5_DIGEST_LENGTH; ++i) {
	int foo;

	if (*md5str >= '0' && *md5str <= '9')
	    foo = *md5str - '0';
	else if (*md5str >= 'a' && *md5str <= 'f')
	    foo = *md5str - 'a' + 10;
	else if (*md5str >= 'a' && *md5str <= 'f')
	    foo = *md5str - 'A' + 10;
	else
	    return -1;
	++md5str;
	if (*md5str >= '0' && *md5str <= '9')
	    md5[i] = foo * 16 + (*md5str - '0');
	else if (*md5str >= 'a' && *md5str <= 'f')
	    md5[i] = foo * 16 + (*md5str - 'a' + 10);
	else if (*md5str >= 'a' && *md5str <= 'f')
	    md5[i] = foo * 16 + (*md5str - 'A' + 10);
	else
	    return -1;
	++md5str;
    }
    return 0;
}

/*
 * parse a line from an md5sums file (or output of md5 app)
 * format:
 * MD5 (filename) = 32hexcharacters\n
 * be liberal in parsing white space - accept zero or more spaces
 * before '(', after ')', after '=', and after the hex
 * return 0 on success, -1 on error
 */

static int
parse_md5 (char *linebuf, char *filename, char *md5str)
{
    if (*linebuf++ != 'M')
	return -1;
    if (*linebuf++ != 'D')
	return -1;
    if (*linebuf++ != '5')
	return -1;
    while (*linebuf == ' ')
	++linebuf;
    if (*linebuf++ != '(')
	return -1;
    while (*linebuf != ')') {
	if (*linebuf == '\0')
	    return -1;
	*filename++ = *linebuf++;
    }
    ++linebuf;
    while (*linebuf == ' ')
	++linebuf;
    if (*linebuf++ != '=')
	return -1;
    while (*linebuf == ' ')
	++linebuf;
    while (isxdigit (*linebuf))
	*md5str++ = *linebuf++;
    while (*linebuf == ' ' || *linebuf == '\n' || *linebuf == '\r')
	++linebuf;
    if (*linebuf != '\0')
	return -1;
    *filename = '\0';
    *md5str = '\0';
    return 0;
}

void
free_file_md5 (struct file_md5 *ptr)
{
    if (ptr == NULL)
	return;
    free_file_md5 (ptr->next);
    ptr->next = NULL;
    free (ptr);
    return;
}

/*
 * read $dir/md5sums and compile a list of (filename, md5) pairs
 *
 * return the list on success, NULL on error
 * fill in the result
 */

static struct file_md5 *
read_md5sums (struct result *x, char *dir)
{
    struct file_md5 *list = NULL;
    FILE *fp = NULL;
    int line = 0;
    char *srcfile = NULL;
    char filename[1024];
    char linebuf[1024];
    char md5str[1024];

    success (x);		/* assume success */

    /* open src file */

    srcfile = concat (dir, "/md5sums", NULL);
    if ((fp = fopen (srcfile, "r")) == NULL) {
	failure (x, 1, "unpack:read_md5sums:cannot open %s/md5sums", dir);
	goto done;
    }

    /* read each line */

    line = 0;
    while (fgets (linebuf, sizeof (linebuf), fp) != NULL) {
	struct file_md5 *newfile;
	unsigned char md5bin[MD5_DIGEST_LENGTH];
	
	if (parse_md5 (linebuf, filename, md5str) < 0) {
	    failure (x, 1, "unpack:read_md5sums:%s (line %d): syntax error",
		     srcfile, line);
	    goto done;
	}
#if 0
	printf ("md5sums: %s -> %s\n", filename, md5str);
#endif
	if (decode_md5 (md5bin, md5str) < 0) {
	    failure (x, 1, "unpack:read_md5sums:%s (line %d): illegal hex",
		     srcfile, line);
	    goto done;
	}
	
	newfile = (struct file_md5 *)
	    malloc_or_else (sizeof (struct file_md5));
	newfile->filename = strdup (filename);
	memcpy (newfile->md5, md5bin, sizeof (md5bin));

	newfile->next = list;
	list = newfile;

	++line;
    }

 done:
    if (srcfile)
	free (srcfile);
    if (fp)
	fclose (fp);

    return (x->status == 0) ? list : NULL;
}

static struct file_md5 *
find_filename (struct file_md5 *list, char *filename)
{
    struct file_md5 *ptr;

    for (ptr = list; ptr; ptr=ptr->next)
	if (strcmp (filename, ptr->filename) == 0)
	    return ptr;
    return NULL;
}

/*
 * - copy GPG files to local dir
 * - extract files (collecting md5s)
 * - gpg --import pubkey.gpg (if present)
 * - gpg --verify md5sums md5sums.gpg
 * - check gpg output
 * - check MD5s
 * - cleanup as much as possible (remove everything but problem)
 * - launch problem
 */

static int
unpack_and_verify_tgz (struct result *x, char *dir, char *tarfile, int verbosity)
{
    static char *gpghome = NULL;
    int tar_fd;
    char *sourcefile;
    struct stat home, here;
    struct file_md5 *extract_list;
    struct file_md5 *eptr;
    struct file_md5 *md5sums_list;
    struct file_md5 *mptr;
    int i;

    /* list of files to copy from GPGHOME to current directory */

    static char *files_to_copy[] = {
	"pubring.gpg", "trustdb.gpg", "secring.gpg", NULL
    };



    if (gpghome == NULL)
	gpghome = concat (getenv ("HOME"), "/NetBuild/gnupg", NULL);

    if (gpghome == NULL)
	return failure (x, 1, "unpack: gpghome is NULL");

    if (gpg_check_version () < 0)
	return failure (x, 1, "incompatible gpg version");

    gpg_set_verbosity (verbosity - 2);

    /*
     * copy various gpg files to the appropriate directory
     *
     * XXX concat() core leakage
     */
    if (verbosity > 0)
	printf ("[copying files from %s to %s...]\n", gpghome, dir);

    for (i = 0; files_to_copy[i] != NULL; ++i) {
	char *src, *dst;

	if (verbosity > 1)
	    printf (" [copying %s]\n", files_to_copy[i]);

	src = concat (gpghome, "/", files_to_copy[i], NULL);
	dst = concat (dir, "/", files_to_copy[i], NULL);

	if (copyfile_noisy (src, dst, stderr, "nb: unpack") < 0) {
	    free (src);
	    free (dst);
	    return failure (x, 1, "nb: unpack: copy failed");
	}
	free (src);
	free (dst);
    }    

    /*
     * extract files from tar archive into directory 'dir'
     */
     
    if ((tar_fd = open (tarfile, O_RDONLY, 0)) < 0)
	return failure (x, 1, "unpack: can't open container file %s: %s",
			tarfile, strerror (errno));

    if (verbosity > 1)
	printf ("[extracting %s into %s...]\n", tarfile, dir);

    if ((extract_list = tar_rd (tar_fd, dir, cb, (void *) verbosity)) == NULL)
	return failure (x, 1, "unpack: error extracting from file %s",
			tarfile);
    close (tar_fd);

    /*
     * if "pubkey.gpg" was extracted, import that key
     */
    for (eptr = extract_list; eptr; eptr = eptr->next) {
	if (strcmp (eptr->filename, "pubkey.gpg") == 0) {
	    if (verbosity > 0)
		printf ("[importing pubkey.gpg]\n", gpghome);

	    /*
	     * XXX concat() core leakage
      	     */
	    if (gpg_import_keys (dir, concat (dir, "/", "pubkey.gpg", NULL)) != 0)
		return failure (x, 1, "unpack: cannot import pubkey.gpg");
	    else
		break;
	}
    }

    /*
     * verify signature
     */
    if (verbosity > 0)
	printf ("[verifying signature]\n");
    if (gpg_verify (dir, "md5sums", "md5sums.gpg", "gpg.status") < 0)
	return failure (x, 1, "unpack: signature verification failed");

    if (gpg_check_status (dir, "gpg.status") < 0) {
#if 1
	fprintf (stderr, "==> signature not trusted\n");
#endif
	return failure (x, 1, "unpack: signature not trusted");
    }

    /*
     * check md5s.  make sure every file in the archive
     * (except for md5sums.gpg and pubkey.gpg) is listed
     * in the md5sums file, and that their md5s match
     */

    if (verbosity > 0)
	printf ("[comparing md5s]\n");
    if ((md5sums_list = read_md5sums (x, dir)) == NULL)
	return x->status;

#if 0
    if (verbosity > 1) {
	for (mptr = md5sums_list; mptr; mptr = mptr->next)
	    printf ("md5sums list %s\n", mptr->filename);
    }
#endif

    for (eptr = extract_list; eptr; eptr = eptr->next) {
	if (strcmp (eptr->filename, "md5sums") == 0)
	    continue;
	if (strcmp (eptr->filename, "md5sums.gpg") == 0)
	    continue;
	if (strcmp (eptr->filename, "pubkey.gpg") == 0)
	    continue;
	if (verbosity > 1)
	    printf (" [md5(%s)...", eptr->filename);
	if ((mptr = find_filename (md5sums_list,
				   eptr->filename)) == NULL) {
	    if (verbosity > 1)
		printf ("not in md5sums file]\n");
	    return failure (x, 1, "unpack: %s not found in %s/md5sums file",
			    eptr->filename, dir);
	}
	if (memcmp (eptr->md5, mptr->md5, MD5_DIGEST_LENGTH) != 0) {
	    if (verbosity > 1)
		printf ("mismatch]\n");
	    return failure (x, 1, "unpack: md5 mismatch for %s/%s",
			    dir, eptr->filename);
	}
	if (verbosity > 1)
	    printf ("good]\n");
    }
    
    /*
     * XXX run post-install.sh
     */
    for (eptr = extract_list; eptr; eptr = eptr->next) {
        if (strcmp (eptr->filename, "post-install.sh") == 0) {
	    char **child_argv;

            if (verbosity > 0)
                printf ("[running post-install.sh]\n");

	    if (verbosity > 0) 
		child_argv = make_argv ("/bin/sh", "-x", "post-install.sh", NULL);
	    else
		child_argv = make_argv ("/bin/sh", "post-install.sh", NULL);

	    if (verbosity > 0) {
		int i;

		fprintf (stderr, "+");
		for (i = 0; child_argv[i]; ++i)
		    fprintf (stderr, " %s", child_argv[i]);
		fprintf (stderr, "\n");
	    }

	    if (spawn_in_dir (dir, child_argv, 1, 2, -1) != 0) {
		if (verbosity > 0)
		    printf ("[failed]\n");
		free_argv (child_argv);
	        return failure (x, 1, "unpack: post-install script failed");
	    }
	    free_argv (child_argv);
        }
    }


    return success (x);
}

/*
 * remove a directory and its descendants
 */

static int
rmtree (char *dir)
{
    char *command;

    command = concat ("rm -rf '", dir, "'", NULL);
    system (command);
    if (access (dir, F_OK) == 0)
	return -1;
    return 0;
}

/*
 * unpack a package from 'package_fn' to directory 'dir'
 */

int
unpack_and_verify_package (struct result *x, char *dir, char *package_fn,
			   int verbosity)
{
    char *bn;
    char *suffix;

#if 0
    fprintf (stderr, "unpack_and_verify_package (x, %s, %s)\n",
	     dir, package_fn);
#endif

    bn = basename (package_fn);
    suffix = strrchr (bn, '.');
    if (suffix == NULL)
	suffix = "";

#if 0
    fprintf (stderr, "suffix=%s\n", suffix);
#endif

    if (verbosity > 1)
	fprintf (stderr, "[removing %s]\n", dir);
    rmtree (dir);

    mkdir_recursive (dir, 0700);

    if (strcasecmp (suffix, ".tgz") == 0)
	return unpack_and_verify_tgz (x, dir, package_fn, verbosity);
    else
	return failure (x, 1, "unpack_package: %s: unknown package format\n",
			package_fn);
}

#ifdef TEST
main (int argc, char **argv)
{
    struct result x;
    int verbosity = 0;

    if (argc == 4 && argv[1][0] == '-' && isdigit (argv[1][1])) {
	verbosity = atoi (argv[1] + 1);
	--argc;
	++argv;
    }
    if (argc != 3) {
	fprintf (stderr, "usage: [-#] test-unpack dir tarfile\n");
	fprintf (stderr ,"-# indicates verbosity:\n");
	fprintf (stderr ,"-0 silent, -1 verbose, -2 very verbose, etc.\n");
	exit (1);
    }
    if (unpack_and_verify_tgz (&x, argv[1], argv[2], verbosity) != 0) {
	fprintf (stderr, "%s\n", x.msg);
	exit (1);
    }
    fprintf (stderr, "unpack successful\n");
    exit (0);
}
#endif
