/*
 * interface to sysctl() on *BSD, Darwin, others???
 *
 * sysctl() is a facility on some BSDish systems by which
 * information about the run-time environment can be extracted
 * from the kernel by name.  It's hideously baroque, but 
 * sometimes it's the best way to find out about a platform.
 * (and at least it uses a stable schema)
 * 
 * one problem with sysctl() is that the actual syscall doesn't
 * use names, but an array of integers, and the name-to-integer
 * mappings are bound at compile time.  So you can't lookup a
 * syscall parameter that wasn't known about at compile time.
 *
 * we don't try to support every sysctl() variable here,
 * partially because they're not all useful to netbuild,
 * partially because some of them involve complex data
 * structures, and partially because some of them might even 
 * be a privacy risk by exposing too much information to 
 * netbuild.  so we try to support the variables that would
 * be of use to netbuild.
 *
 * 
 * currently implemented:
 *
 * hw.
 * kern.
 * machdep.
 * net.inet.ip.
 * net.inet.icmp.
 * net.inet.tcp.
 * net.inet6.ip6.
 * net.inet6.icmp6.
 * net.inet6.tcp6.
 * vm.
 *
 * not yet implemented:
 *
 * net.inet.udp.
 * net.inet6.udp6.
 * user.
 *
 * will probably not be implemented:
 *
 * ddb.
 * vfs.
 * proc.
 * emul.
 */

#include "conf.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/sysctl.h>
#include <sys/types.h>
#include <sys/utsname.h>

#include <machine/cpu.h>		/* netbsd: for CTL_MACHDEP_NAMES */

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netinet/ip_var.h>
#include <netinet/icmp_var.h>
#include <netinet/tcp.h>
#include <netinet/tcp_timer.h>
#include <netinet/tcp_var.h>

#include <netinet/ip6.h>
#include <netinet/icmp6.h>
#include <netinet6/ip6_var.h>
#include <netinet6/udp6.h>
#include <netinet6/udp6_var.h>

#include "symtab.h"

#if defined(CTL_HW) && defined(CTL_HW_NAMES)
struct ctlname hw_dir[] = CTL_HW_NAMES;
#endif
#if defined(CTL_KERN) && defined(CTL_KERN_NAMES)
struct ctlname kern_dir[] = CTL_KERN_NAMES;
#endif
#if defined(CTL_MACHDEP) && defined(CTL_MACHDEP_NAMES)
struct ctlname machdep_dir[] = CTL_MACHDEP_NAMES;
#endif
#if defined(CTL_NET) 
#if defined(PF_INET) 
#if defined(IPPROTO_IP) && defined(IPCTL_NAMES)
struct ctlname net_inet_ip_dir[] = IPCTL_NAMES;
#endif
#if defined(IPPROTO_ICMP) && defined(ICMPCTL_NAMES)
struct ctlname net_inet_icmp_dir[] = ICMPCTL_NAMES;
#endif
#if defined(IPPROTO_TCP) && defined(TCPCTL_NAMES)
struct ctlname net_inet_tcp_dir[] = TCPCTL_NAMES;
#endif
#endif /* PF_INET */
#if defined(PF_INET6)
#if defined(IPPROTO_IPV6) && defined(IPV6CTL_NAMES)
struct ctlname net_inet6_ipv6_dir[] = IPV6CTL_NAMES;
#endif
#if defined(IPPROTO_ICMPV6) && defined(ICMPV6CTL_NAMES)
struct ctlname net_inet6_icmp6_dir[] = ICMPV6CTL_NAMES;
#endif
#if defined(IPPROTO_TCP) && defined(TCPCTL_NAMES)
struct ctlname net_inet6_tcp6_dir[] = TCPCTL_NAMES;
#endif
#endif /* PF_INET6 */
#endif /* CTL_NET */
#if defined(CTL_VM) && defined(CTL_VM_NAMES)
struct ctlname vm_dir[] = CTL_VM_NAMES;
#endif

#define STRLEN(x) (sizeof(x) - 1)
#define X(x) (x), STRLEN(x)
#define SIZE(x) (sizeof(x)/sizeof(*x))
#define Y(x) &x[0], SIZE(x)

/*
 * directory tree for sysctl variables
 *
 * directory entry consists of
 * name
 * mib for this name (list of integers)
 * list of sub-entries
 */

struct {
    char *prefix;
    int plen;
    struct ctlname *dir;
    int dirlen;
    int mib[5];
    int miblen;
} mount[] = {
#if defined(CTL_HW) && defined(CTL_HW_NAMES)
    { X("hw."), Y(hw_dir), { CTL_HW }, 1 },
#endif
#if defined(CTL_KERN) && defined(CTL_KERN_NAMES)
    { X("kern."), Y(kern_dir), { CTL_KERN }, 1 },
#endif
#if defined(CTL_MACHDEP) && defined(CTL_MACHDEP_NAMES)
    { X("machdep."), Y(machdep_dir), { CTL_MACHDEP }, 1 },
#endif
#if defined(CTL_NET) 
#if defined(PF_INET)
#if defined(IPPROTO_IP) && defined(IPCTL_NAMES)
    { X("net.inet.ip."), Y(net_inet_ip_dir), 
      { CTL_NET, PF_INET, IPPROTO_IP }, 3 },
#endif
#if defined(IPPROTO_ICMP) && defined(ICMPCTL_NAMES)
    { X("net.inet.icmp."), Y(net_inet_icmp_dir), 
      { CTL_NET, PF_INET, IPPROTO_ICMP }, 3 },
#endif
#if defined(IPPROTO_TCP) && defined(TCPCTL_NAMES)
    { X("net.inet.tcp."), Y(net_inet_tcp_dir), 
      { CTL_NET, PF_INET, IPPROTO_TCP }, 3 },
#endif
#endif /* PF_INET */
#if defined(PF_INET6)
#if defined(IPPROTO_IPV6) && defined(IPV6CTL_NAMES)
    { X("net.inet6.ip6."), Y(net_inet6_ipv6_dir), 
      { CTL_NET, PF_INET6, IPPROTO_IPV6 }, 3 },
#endif
#if defined(IPPROTO_ICMPV6) && defined(ICMPV6CTL_NAMES)
    { X("net.inet6.icmp6."), Y(net_inet6_icmp6_dir), 
      { CTL_NET, PF_INET6, IPPROTO_ICMPV6 }, 3 },
#endif
#if defined(IPPROTO_TCP) && defined(TCPCTL_NAMES)
    { X("net.inet6.tcp6."), Y(net_inet6_tcp6_dir), 
      { CTL_NET, PF_INET6, IPPROTO_TCP }, 3 },
#endif
#endif /* PF_INET6 */
#endif
#if defined(CTL_VM) && defined(CTL_VM_NAMES)
    { X("vm."), Y(vm_dir), { CTL_VM }, 1 } ,
#endif
};

int
sysctl_lookup (int c, char *name, VAL *value)
{
    int i;
    int longest = -1;
    int longest_len = 0;
    int mib[10];
    char *suffix;

    /* find longest matching "mount point" */
    for (i = 0; i < SIZE(mount); ++i) {
	if (longest_len > mount[i].plen)
	    ;
	else if (strncmp (mount[i].prefix, name, mount[i].plen) == 0) {
	    longest = i;
	    longest_len = mount[i].plen;
	}
    }
    if (longest < 0)
	goto error;

    /* now try to look up last component of the name in the table */
    suffix = name + mount[longest].plen;
    for (i = 0; i < mount[longest].dirlen; ++i) {
	if (mount[longest].dir[i].ctl_name == NULL)
	    continue;
	if (strcmp (suffix, mount[longest].dir[i].ctl_name) == 0) {
	    int j;

	    if (mount[longest].dir[i].ctl_type != CTLTYPE_INT &&
		mount[longest].dir[i].ctl_type != CTLTYPE_STRING)
		goto error;

	    for (j = 0; j < mount[longest].miblen; ++j)
		mib[j] = mount[longest].mib[j];
	    mib[j++] = i;

	    if (mount[longest].dir[i].ctl_type == CTLTYPE_INT) {
		int result;
		int rsize = sizeof (result);

		if (sysctl (mib, j, &result, &rsize, NULL, 0) < 0)
		    goto error;
		printf ("%s = %d\n", name, result);
		goto done;
	    }
	    else {
		char *result;
		int rsize;

		if (sysctl (mib, j, NULL, &rsize, NULL, 0) < 0)
		    goto error;
		result = malloc (rsize + 1);
		if (sysctl (mib, j, result, &rsize, NULL, 0) < 0)
		    goto error;
		printf ("%s = \"%s\"\n", name, result);
		goto done;
	    }
	}
    }

error:
    printf ("%s = <error>\n", name);

done:
}

main (int argc, char **argv)
{
    int i;

    if (argc == 1) {
	for (i = 0; i < SIZE(mount); ++i)
	    printf ("%s\n", mount[i].prefix);
    }
    for (i = 1; i < argc; ++i)
	sysctl_lookup ('g', argv[i], NULL);
}

