#ifndef SYMTAB_H
#define SYMTAB_H

/*  
 * symbol table.
 *      
 * a linear list will suffice for now
 *      
 * there are several kinds of symbols:
 * - symbols for ordinary variables that are stored in memory
 * - symbols for pseudo-variables that are treated as if they
 *   were ordinary variables in expressions, but are
 *   set/get using function calls
 * - symbols for functions that are explicitly called in expressions
 *  
 * if funct is null, value is pointed to by 'value'
 * if funct is non-null, this is a pseudo-variable,
 * call funct to set/return value.
 * 
 * funct has arguments: operation, name, value
 * operation is 's' for set and 'g' for get
 * 
 * for now, all values are assumed to be strings
 *
 * return values from set/get functions
 *  
 * -1 no such variable on get
 * -2 read only on set 
 */ 

typedef struct val {
    enum { INT, STR, PSEUDO, FUNC, VERS, ERR } valtype;
    int intval;			/* storage for value if it's an integer */
    char *strval;		/* storage for value if it's a string */
    int (*pseudo)();		/* fn used to set/get value if it's pseudo */
    struct val * (*funval)();	/* fn bound to name if it's a function */
    void *pseudo_arg;		/* pseudo function argument */
    struct val *next;		/* next value in argument list */
} VAL;

#define VAL_PROTO { INT, 0, 0, 0, 0, 0 }

struct symbol {
    struct symbol *next;
    char *name;
    VAL value;
};

struct symbol *symtab_get_symbol (char *name);
int symtab_get_value (char *name, VAL *value);
int symtab_set_value (char *name, VAL *value);
int symtab_set_value_int (char *name, int value);
int symtab_set_value_str (char *name, char *value);
int symtab_set_value_vers (char *name, char *value);
int symtab_bind_pseudo (char *name, int (*function)(), void *arg);
int symtab_bind_function (char *name, VAL *(*function)());
VAL *symtab_append (VAL *a, VAL *b);
VAL *symtab_copyval (VAL *a);
void symtab_printf_val (FILE *fp, VAL *v);
void symtab_print_val (VAL *v);
void symtab_dump ();

#endif
