#include "conf.h"

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_STRING_H
#include <string.h>
#else
char *strdup ();
#endif
#include "symtab.h"

VAL val_proto = VAL_PROTO;

static struct symbol *head;

/*
 * find a symbol in the symbol table.  return NULL if not found
 */

struct symbol *
symtab_get_symbol (char *name)
{
    struct symbol *ptr;
    int len;

    for (ptr = head; ptr != NULL; ptr = ptr->next) {
	if (strcmp (name, ptr->name) == 0)
	    return ptr;

	len = strlen (ptr->name);
	if (ptr->name[len-1] == '*' &&
	    strncmp (name, ptr->name, len-1) == 0)
	    return ptr;
    }
    return NULL;
}


/*
 * get a symbol's value.  if the symbol is a pseudo-variable,
 * invoke its get method 
 */

int 
symtab_get_value (char *name, VAL *value)
{
    struct symbol *ptr;

    if ((ptr = symtab_get_symbol (name)) == NULL)
	return -1;

    switch (ptr->value.valtype) {
    case INT:
	value->valtype = ptr->value.valtype;
	value->intval = ptr->value.intval;
	break;
    case STR:
    case VERS:
	value->valtype = ptr->value.valtype;
	value->strval = ptr->value.strval;
	break;
    case PSEUDO:
	return (*ptr->value.pseudo)('g', name, value, ptr->value.pseudo_arg);
    case FUNC:
	return -1;
    case ERR:
	value->valtype = ptr->value.valtype;
	break;
    default:
	value->valtype = ERR;
	break;
    }
    return -1;
}

/*
 * set a symbol's value.  if the symbol is a pseudo-variable,
 * invoke its set method.
 */

int
symtab_set_value (char *name, VAL *val)
{
    struct symbol *ptr;

    ptr = symtab_get_symbol (name);
    if (ptr == NULL) {
	ptr = (struct symbol *) malloc (sizeof (struct symbol));
	ptr->name = strdup (name);
	ptr->value = *val;
	ptr->next = head;
	head = ptr;
	return 0;
    }
    else if (ptr->value.valtype == PSEUDO)
	return (*ptr->value.pseudo)('s', name, val);
    else if (ptr->value.valtype == FUNC)
	return -1;
    else {
	ptr->value = *val;
	return 0;
    }
}

/*
 * set the value of a symbol to an integer
 */

int
symtab_set_value_int (char *name, int value)
{
    VAL val;

    val.valtype = INT;
    val.intval = value;
    return symtab_set_value (name, &val);
}

/*
 * set the value of a symbol to a string
 */

int
symtab_set_value_str (char *name, char *value)
{
    VAL val;

    val.valtype = STR;
    val.strval = strdup (value);
    return symtab_set_value (name, &val);
}

/*
 * set the value of a symbol to a version
 * XXX probably should deprecate versions
 */

int
symtab_set_value_vers (char *name, char *value)
{
    VAL val;

    val.valtype = VERS;
    val.strval = strdup (value);
    return symtab_set_value (name, &val);
}

/*
 * associate a symbol with a pseudo-variable function
 */

int
symtab_bind_pseudo (char *name, int (*function)(), void *arg)
{
    struct symbol *ptr;
    VAL val;

    val.valtype = PSEUDO;
    val.pseudo = function;
    val.pseudo_arg = arg;

    ptr = symtab_get_symbol (name);

    if (ptr == NULL) {
	ptr = (struct symbol *) malloc (sizeof (struct symbol));
	ptr->name = strdup (name);
	ptr->value = val;
	ptr->next = head;
	head = ptr;
	return 0;
    }
    else {
	ptr->value = val;
	return 0;
    }
}

/*
 * associate a symbol with an explicit function
 */

int
symtab_bind_function (char *name, VAL *(*function)())
{
    struct symbol *ptr;
    VAL val;

    val.valtype = FUNC;
    val.funval = function;

    ptr = symtab_get_symbol (name);

    if (ptr == NULL) {
	ptr = (struct symbol *) malloc (sizeof (struct symbol));
	ptr->name = strdup (name);
	ptr->value = val;
	ptr->next = head;
	head = ptr;
	return 0;
    }
    else {
	ptr->value = val;
	return 0;
    }
}

/*
 * append a value to a list of values (as in an argument list)
 */

VAL *
symtab_append (VAL *a, VAL *b)
{
    VAL *ptr;

    for (ptr = a; ptr->next != NULL; ptr = ptr->next);
    ptr->next = b;
    b->next = NULL;
    return a;
}

/*
 * copy a value to malloc'ed storage
 */

VAL *
symtab_copyval (VAL *a)
{
    VAL *result;

    result = (VAL *) malloc (sizeof (VAL));
    *result = *a;
    return result;
}

void
symtab_printf_val (FILE *fp, VAL *v) 
{                   
    if (v == NULL) {
        fprintf (fp, "(null)");
	return;
    }

    switch (v->valtype) {
    case INT:
        fprintf (fp, "%d", v->intval);
	break;
    case STR:
    case VERS:
        fprintf (fp, "\"%s\"", v->strval);
	break;
    case PSEUDO:
        fprintf (fp, "<pseudo>");
	break;
    case FUNC:
        fprintf (fp, "<function>");
	break;
    case ERR:
        fprintf (fp, "<error>");
	break;
    default:
        fprintf (fp, "<unknown>");
    }
}

void
symtab_print_val (VAL *v)
{
    symtab_printf_val (stdout, v);
}

void
symtab_dump ()
{
    struct symbol *ptr;

    for (ptr = head; ptr; ptr=ptr->next) {
	printf ("%s = ", ptr->name);
	symtab_print_val (&(ptr->value));
	printf ("\n");
    }
}
