/*
 * run a command in a different directory.
 *
 * fd1, fd2, and fd3 set fd's 1, 2, 3 in the child
 * - if -1, close that fd, otherwise copy that fd to the child's fd
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/wait.h>
#include <unistd.h>
#include "spawn.h"

int
spawn_in_dir (char *dir, char **child_argv, int fd1, int fd2, int fd3)
{
    pid_t child_pid;
    struct sigaction intsa, quitsa, sa;
    sigset_t nmask, omask;
    int child_status;
    int err;
    int devnull;

    sa.sa_handler = SIG_IGN;
    sigemptyset (&sa.sa_mask);
    sa.sa_flags = 0;

    if (sigaction (SIGINT, &sa, &intsa) < 0) {
	fprintf (stderr, "unpack: sigaction(SIGINT): %s\n",
		 strerror (errno));
	return -1;
    }
    if (sigaction (SIGQUIT, &sa, &quitsa) < 0) {
	fprintf (stderr, "unpack: sigaction(SIGQUIT): %s\n",
		 strerror (errno));
	return -1;
    }

    sigemptyset (&nmask);
    sigaddset (&nmask, SIGCHLD);
    if (sigprocmask (SIG_BLOCK, &nmask, &omask) < 0) {
	fprintf (stderr, "unpack: sigprocmask(SIG_BLOCK): %s\n",
		 strerror (errno));
	return -1;
    }
    fflush (stdout);
    fflush (stderr);
    devnull = open ("/dev/null", O_RDWR, 0);

    switch (child_pid = fork ()) {
    case -1:			/* error */
	err = errno;
	sigaction (SIGINT, &intsa, NULL);
	sigaction (SIGQUIT, &quitsa, NULL);
	sigprocmask (SIG_SETMASK, &omask, NULL);
	fprintf (stderr, "fork: %s\n", strerror (err));
	return -1;

    case 0:			/* child */
	sigaction (SIGINT, &intsa, NULL);
	sigaction (SIGQUIT, &quitsa, NULL);
	sigprocmask (SIG_SETMASK, &omask, NULL);
	if (chdir (dir) < 0) {
	    fprintf (stderr, "chdir (%s) failed: %s\n", dir, strerror (errno));
	    _exit (127);
	}
	
	switch (fd1) {
	case -1:
	    dup2 (devnull, 1);
	    break;
	case 1:
	    break;
	default:
	    dup2 (fd1, 1);
	    close (fd1);
	}
	switch (fd2) {
	case -1:
	    dup2 (devnull, 2);
	    break;
	case 2:
	    break;
	default:
	    dup2 (fd2, 2);
	    close (fd2);
	}
	switch (fd3) {
	case -1:
	    close (3);
	    break;
	case 3:
	    break;
	default:
	    dup2 (fd3, 3);
	    close (fd3);
	}
	close (devnull);
	execvp (child_argv[0], child_argv);
	_exit (127);
	break;
    }

    close (devnull);
    while (waitpid (child_pid, &child_status, 0) < 0) {
	if (errno != EINTR) {
	    child_status = -1;
	    break;
	}
    }
    sigaction (SIGINT, &intsa, NULL);
    sigaction (SIGQUIT, &quitsa, NULL);
    sigprocmask (SIG_SETMASK, &omask, NULL);

    if (child_status == -1)
	return -1;
    if (WIFSIGNALED (child_status))
	return -1;
    if (WIFEXITED (child_status))
	return WEXITSTATUS(child_status);
}
