#include "conf.h"

#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include "misc.h"
#include "nvpl.h"
#include "result.h"

/*
 * routines for returning results and failure conditions
 *
 * these routines try to make it easy to return moderately complex
 * results from C functions.
 *
 * a 'result' consists of:
 *
 * - a int status code.  0 is success, other codes are failure
 *
 * - an optional list of name,value pairs.  generally these
 *   will be associated with a successful result, but not always.
 *
 * - an optional text error message (sorry, no localization support)
 *
 * - an optional chain of preceding results that precipitated this one
 *   this is mostly so we can display a chain of errors leading up
 *   to the one that we finally indicate to the user, but it could
 *   also (potentially) be used by higher-level routines to look at
 *   chains of status codes.
 *
 * these routines "fill in" a result structure passed to them as
 * an argument, rather than malloc'ing one - since C lacks any
 * kind of automatic storage management for variables returned
 * from a higher activiation layer and it's too much of a pain
 * to deal with free'ing them explicitly.
 *
 * also, these routines for filling in result structures always
 * return the status code - making it easy to just say
 *
 *	return success (x);
 * 	or
 * 	return failure (x, "%s: not found", filename);
 *
 * and then the caller can just check the int return value to see
 * if it's non zero.  it can even do generic handling.  e.g.
 *
 * if (routine1 (x, arg, arg, arg) < 0)
 *	goto fail;
 * if (routine2 (x, arg, arg) < 0)
 *      goto fail;
 * ...
 *
 * it might even be worth adding automatic exception handling,
 * doing a longjmp() to a jmp_buf stored in the result structure.
 * but that's not needed right now.
 */

/*
 * helper routine to generate result structures
 */

int
make_result (struct result *x,
	     int status,
	     struct result *prev,
	     struct nvpl *nvpl)
{
    /*
     * if we're pushing a result onto the 'stack', allocate space
     * for the old result, and copy it into that space.  keep the
     * old msg and nvp list, but zero them out of the result
     * structure on the new top of the stack.
     *
     * if we're not pushing a new result, free the old nvp list
     * and error message.
     */
    if (prev) {
	x->prev = (struct result *) malloc_or_else (sizeof (struct result));
	memcpy ((void *) x->prev, (void *) prev, sizeof (struct result));
	x->nvpl = NULL;
	x->msg = NULL;
    }
    else {
#if 0
	/*
	 * XXX there's really no safe way to free nvpl's, since
	 * we have no idea as to whether there are multiple references
	 * to any nvp or which any of the names or values are
	 * free-able.  and given that we use nvpl's to make lists of
	 * things other than nul-terminated character strings it's
	 * not currently feasible to copy them to heap space either.
	 *
	 * for that matter, we don't even know if they've been initialized.
	 */
	if (x->nvpl)
	    free_nvpl (x->nvpl);
	if (x->msg)
	    free (x->msg);
#endif
	x->prev = NULL;
    }
    
    /*
     * fill in values, format error message
     */
    x->status = status;
    x->nvpl = nvpl;
    return x->status;
}

int
make_result_va (struct result *x, 		/* result to fill in */
		int status,			/* status code */
		struct result *prev,		/* previous result, if any */
		struct nvpl *nvpl,		/* nvp list */
		char *fmt, va_list va)		/* error message */
{
    make_result (x, status, prev, nvpl);
    if (fmt) {
#ifdef HAVE_VASPRINTF
	vasprintf (&(x->msg), fmt, va);
#else
	char temp[10000];	/* XXX */
	vsprintf (temp, fmt, va);
	x->msg = strdup (temp);
#endif
    }
    else
	x->msg = NULL;
    return x->status;
}

/*
 * generate a 'failure' result that is caused by a previous result
 * this way we can represent a chain of failures
 */

int
push_failure (struct result *x,			/* result to fill in */
	      int status,			/* status code */
	      struct result *prev,		/* previous result, if any */
	      char *fmt, ...)			/* error message */
{
    va_list ap;

    va_start (ap, fmt);
    status = make_result_va (x, status, prev, NULL, fmt, ap);
    va_end (ap);
    return status;
}

/*
 * generate a failure result that is not caused by a previous
 * result (at least, not one worth reporting)
 */

int
failure (struct result *x,			/* result to fill in */
	 int status,				/* status code */
	 char *fmt, ...)			/* error message */
{
    va_list ap;

    va_start (ap, fmt);
    (void) make_result_va (x, status, NULL, NULL, fmt, ap);
    va_end (ap);
    return status;
}

/*
 * generate a success result with no nvps
 */

int
success (struct result *x)
{
    return make_result (x, 0, NULL, NULL);
}


/*
 * generate a success result with nvps
 */
int
success_nvpl (struct result *x, ...)
{
    va_list ap;

    va_start (ap, x);
    (void) make_result (x, 0, NULL, make_nvpl_va (ap));
#if 0
    print_nvpl (stderr, x->nvpl);
#endif
    va_end (ap);
    return 0;
}

#ifdef TEST
int verbosity = 1;

main (int argc, char **argv)
{
    struct result x;
    int status;

    status = failure (&x, atoi(argv[1]), argv[2], argv[3], argv[4], argv[5]);

    fprintf (stderr, "status = %d\n", status);
    fprintf (stderr, "x.status = %d\n", x.status);
    fprintf (stderr, "x.msg = %s\n", x.msg);
}
#endif
