/*
 * dig for information about the compile platform, some of which are also
 * used as defaults for the target platform.
 *
 * what we'd like to get includes:
 * cpu architecture
 * os name
 * os version/release (a number that has an ordering relation)
 *
 * XXX
 * this was borrowed from netcompile so that the platform-determining
 * code would be consistent between the two.  but it's a bit confusing,
 * because this code was written to make assumptions about the target
 * platform given characteristics of the compile platform.
 * 
 * this module should eventually be generic for both netcompile and
 * netbuild and perhaps other things.  to do this I would need to separate
 * the code that gets information about the current platform
 * from the code that makes assumptions about the target platform
 * from looking at the current platform.
 */

#include "conf.h"
#include <stdio.h>
#include <string.h>
#include <sys/utsname.h>
#ifdef HAVE_SYS_SYSTEMINFO_H
#include <sys/systeminfo.h>
#endif
#include "global.h"
#include "misc.h"
#include "nvpl.h"

#ifdef OS_aix
#include <sys/types.h>
#include <sys/systemcfg.h>
#endif

struct table {
    char *orig;
    char *new;
};

/*
 * try to have uniform naming for machines across different operating
 * systems.  what we're looking for is the name of an instruction set
 * architecture, not a precise indication of the cpu.
 */

struct table machines[] = {
    { "alpha", "alpha" },
    { "ia64", "ia64" },
    { "i386", "ia32" },
    { "i486", "ia32" },
    { "i586", "ia32" },
    { "i686", "ia32" },
    { "sparc", "sparc" },
    { "sun4", "sparc" },
    { "sun4c", "sparc" },
    { "sun4d", "sparc" },
    { "sun4m", "sparc" },
    { "sun4u", "sparc" },
    { 0, 0 }
};

/*
 * try to have uniform naming for operating systems even though
 * some vendors change this from time to time due to marketing reasons.
 *
 * XXX do we really need this table?  seems like the system name
 * is one of the few things a vendor rarely lies about...
 */

struct table sysnames[] = {
    { "AIX", "aix" },
    { "Darwin", "darwin" },
    { "IRIX64", "irix64" },
    { "Linux", "linux" },
    { "NetBSD", "netbsd" },
    { "OSF1", "tru64" },	/* XXX mapping is only valid on alpha!!! */
    { "SunOS", "sunos" },	/* XXX perhaps sunos should be solaris
				   for versions > 4 */
    { "ULTRIX", "ultrix" },
    { 0, 0 }
};

static char *
translate (char *orig, struct table *table)
{
    int i;

    for (i = 0; table[i].orig; ++i) {
	if (strcmp (orig, table[i].orig) == 0)
	    return table[i].new;
    }
    return orig;
}

static char *
translate_release (char *release)
{
    char *result;
    char *ptr;
   
    /* on some systems the release begins with 'V' */
    if (*release == 'v' || *release == 'V')
	release++;
    result = strdup (release);
    ptr = result;
    while (*ptr) {
	if (*ptr == '-' || *ptr == ' ' || *ptr == ':')
	    *ptr = '.';
	++ptr;
    }
    return result;
}


static char *
get_os_name (struct utsname *u)
{
    if (strcmp (u->machine, "alpha") == 0 && strcmp (u->sysname, "OSF1") == 0)
	return "tru64";
    return translate (u->sysname, sysnames);
}

/*
 * get information about the platform that we are currently using.
 * this also determines some defaults for the target platform.
 *
 * as much as possible detect this at run-time rather than compile-time,
 * since due to emulation it is possible for a program to be run in a
 * different environment from the one for which it was compiled.
 * (even this doesn't always work - sometimes the emulation lies -
 * as when netbsd reports "linux" when running an linux binary)
 *
 * tested on:
 *
 * aix power3 4.3
 * irix64 mips 6.5
 * linux ia32 2.4.18
 * netbsd ia32 1.5.2
 * solaris sparc 5.8
 * tru64 alpha 5.1
 */

void
get_current_platform_info ()
{
    struct utsname foo;
    char *current_platform_os = NULL;
    char *current_platform_os_release = NULL;
    char *current_platform_cpu = NULL;

    uname (&foo);
    current_platform_os = get_os_name (&foo);

    if (strcmp (current_platform_os, "aix") == 0) {
	/*
	 * get architecture name
	 * aix runs on power* (several different versions)
	 * and ia64 architectures 
         * (and perhaps ibm370 arch also?)
	 *
	 * NB we use the word "powerpc" to refer to all power*
	 * architectures even though strictly speaking they are
	 * not all PowerPCs.  they share enough of a common
	 * instruction set to make it feasible to generate a
         * single code that turns on any of them, and that's
	 * a useful enough property to put them all in the same
	 * group for selection of libraries which aren't performance
	 * critical.
	 */

	char buf[100];
#ifdef OS_aix
	char *cputype;

        switch (_system_configuration.architecture) {
            case POWER_RS: cputype="powerpc"; break;
            case POWER_PC: cputype="powerpc"; break;
            case IA64: cputype="ia64"; break;
            default: cputype="unknown"; break;
        }
        current_platform_cpu = cputype;
#endif 
	sprintf (buf, "%s.%s", foo.version, foo.release);
	current_platform_os_release = translate_release (buf);
    }
    /*
     * XXX
     * apparently Cygwin reports different strings on different
     * platforms
     */
    else if (strncmp (current_platform_os, "CYGWIN", 6) == 0) {
	/*
	 * XXX
 	 * need to rationalize this -
	 * perhaps report "real" operating system since there can
	 * be differences from one M$ platform to another - however 
	 * things compiled for cygwin won't necessarily run on another
	 * host without cygwin.  what to do?
	 */
	current_platform_os = "cygwin";
	current_platform_cpu = translate (foo.machine, machines);
	/*
	 * XXX
	 * uname 'version' seems to be release date, which probably
	 * works for our purposes. uname 'release' is in wierd format,
	 * not sure how to translate it.
         */
	current_platform_os_release = translate_release (foo.version);
    }
    else if (strcmp (current_platform_os, "darwin") == 0) {
        current_platform_cpu = "powerpc"; /* XXX */
        current_platform_os_release = translate_release (foo.release);
    }
    else if (strcmp (current_platform_os, "irix64") == 0) {
	/*
	 * XXX
	 * sysinfo returns "mips" but we might want to 
	 * distinguish mips-64 from mips-32 and we probably do 
	 * want to distinguish big- from little-endian
	 *
	 * XXX
	 * does 'irix' (32-bit) run on mips64?
	 */
	char buf[100];

	current_platform_os = "irix";
#ifdef SI_ARCHITECTURE
	sysinfo (SI_ARCHITECTURE, buf, sizeof (buf));
	current_platform_cpu = translate (buf, machines);
	if (strcmp (current_platform_cpu, "mips") == 0)
	    current_platform_cpu = "mips64";
#endif
	current_platform_os_release = translate_release (foo.release);
    }
    else if (strcmp (current_platform_os, "linux") == 0) {
	current_platform_cpu = translate (foo.machine, machines);
	current_platform_os_release = translate_release (foo.release);
    }
    else if (strcmp (current_platform_os, "netbsd") == 0) {
	current_platform_cpu = translate (foo.machine, machines);
	current_platform_os_release = translate_release (foo.release);
    }
    else if (strcmp (current_platform_os, "tru64") == 0) {
	current_platform_cpu = translate (foo.machine, machines);
	current_platform_os_release = translate_release (foo.release);
    }
    else if (strcmp (current_platform_os, "sunos") == 0) {
	current_platform_cpu = translate (foo.machine, machines);
	current_platform_os_release = translate_release (foo.release);
    }
    /*
     * ultrix: we happened to have one of these still around...
     */
    else if (strcmp (current_platform_os, "ultrix") == 0) {
	if (strcmp (foo.machine, "RISC") == 0)
	    current_platform_cpu = "mipsel"; 
        else if (strcmp (foo.machine, "VAX") == 0)
	    current_platform_cpu = "vax";
	current_platform_os_release = translate_release (foo.release);
    }
    
    if (current_platform_cpu)
	global.platform_cpu = current_platform_cpu;
    if (current_platform_os)
	global.platform_os = current_platform_os;
    if (current_platform_os_release)
	global.platform_os_release = current_platform_os_release;
}

char *
get_netbuild_arch ()
{
    char *result = NULL;

    if (global.platform_cpu == NULL)
	get_current_platform_info ();
    result = malloc_or_else (strlen (global.platform_cpu) +
			     strlen (global.platform_os) + 2);
    sprintf (result, "%s-%s", global.platform_cpu, global.platform_os);
    return result;
}

#ifdef TEST
main (int argc, char **argv)
{
    printf ("current arch = %s\n", get_netbuild_arch ());
}
#endif
