/*
 * routines for finding a library on the net
 * and identifying the best matching library for this platform
 *
 * error reporting: return values
 * error messages: print to stderr
 *
 * dependencies: most of netlib
 *
 * todo:
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <errno.h>
#include <ctype.h>
#include "concat.h"
#include "conf.h"
#include "expr.h"
#include "global.h"
#include "httpget.h"
#include "netfind.h"
#include "nvpl.h"
#include "misc.h"
#include "pathnames.h"
#include "platform.h"
#include "str.h"
#include "symtab.h"
#include "version.h"

#if 0
void
print_list (struct nvpl *head)
{
    struct nvpl *ptr;

    for (ptr = head; ptr; ptr=ptr->next) 
	printf ("%s:%s\n", ptr->name, ptr->value);
}

void
print_candidate_list (struct nvpl *head)
{
    struct nvpl *ptr;

    for (ptr = head; ptr; ptr=ptr->next) {
	printf ("[%s]\n", ptr->name);
	print_list ((struct nvpl *) ptr->value);
    }
}

/*
 * parse a description file
 *
 * the description file looks like
 *
 * [candidate-filename1]
 * name1:value1
 * name2:value2
 * name3:value3
 * [candidate-filename2]
 * name1:value1
 * name2:value2
 * [candidate-filename3]
 * name1:value1
 * name2:value2
 *
 * and so forth.
 * the candidate-filename is the name of the library file
 */

struct nvpl *
parse_description_file (FILE *fp, char *url)
{
    struct nvpl *candidate_list = NULL;
    char *current_candidate_fn = NULL;
    struct nvpl *current_candidate = NULL;
    char buf[10240];
    int lineno = 0;

    while (fgets (buf, sizeof (buf), fp) != NULL) {
	++lineno;
	if (*buf == '#' || *buf == '\n' || *buf == '\r')
	    ;
	else if (*buf == ' ' || *buf == '\t') {
	    fprintf (stderr, "malformed description file: line %d in %s\n",
		     lineno, url);
	    return NULL;
	}
	else if (*buf == '[') {
	    /* new candidate */
	    char *ket = strchr (buf+1, ']');

	    if (ket == NULL) {
		/* malformed line */
		fprintf (stderr, "malformed description file: line %d in %s\n",
			 lineno, url);
		return NULL;
	    }
	    *ket = '\0';

	    if (current_candidate_fn != NULL) {
		candidate_list = add_nvp (candidate_list,
					  current_candidate_fn,
					  (char *) current_candidate);
	    }
	    current_candidate_fn = xstrdup (buf+1);
	    current_candidate = NULL;
	}
	else {
	    /* new attribute of current candidate */
	    char *colon = strchr (buf, ':');
	    char *nl = strchr (buf, '\n');

	    if (colon == NULL) {
		fprintf (stderr, "malformed description file: line %d in %s\n",
			 lineno, url);
		return NULL;
	    }
	    *colon++ = '\0';
	    *nl = '\0';
	    while (*colon == ' ')
		++colon;
	    current_candidate = add_nvp (current_candidate, 
					 xstrdup (buf), xstrdup (colon));
#if 0
	    printf ("%x = add_nvp (%s, %s)\n", current_candidate, buf, colon);
#endif
	}
    }
    if (current_candidate_fn != NULL) {
	candidate_list = add_nvp (candidate_list,
				  current_candidate_fn,
				  (char *) current_candidate);
    }
    return candidate_list;
}


/*
 * given a list of candidate libraries on 'fp', pick the best match
 *
 * XXX
 * looking at libraries one at a time is too simple for the real world.
 * some candidate libraries may impose constraints (such as dependencies
 * on other libraries) that keep them from being used.  really we need
 * to get the candidate lists for all libraries, find some number of
 * solution sets that satisfy all of the criteria, and then pick the
 * best of those sets.
 *
 * XXX
 * most criteria should be specified in a config file, rather than
 * having them wired-in here.  also, it would be useful to be able
 * to download some of the criteria - such as which versions and
 * formats of object files were valid on which versions of the OS.
 */

static char *
pick_best_match (FILE *fp, char *url, int verbosity)
{
    struct nvpl *clist = parse_description_file (fp, url);
    struct nvpl *newlist = NULL;
    struct nvpl *cand;
    struct nvpl *best;
    char *best_os_release;

    if (clist == NULL)
	return NULL;
#if 0
    if (verbosity > 0) {
        print_candidate_list (clist);
    }
#endif

    /* scan the list and reject ineligible candidates */

    for (cand = clist; cand != NULL; cand=cand->next) {
	struct nvpl *candidate = (struct nvpl *) cand->value;
	char *candidate_cpu_type;
	char *candidate_os;
	char *candidate_os_release;
        char *candidate_constraint_expr;

	if (verbosity > 0) {
	    fprintf (stderr, "scanning %s...\n", cand->name);

	    fprintf (stderr, "printing list %x\n", candidate);
 	    print_nvpl (stderr, candidate);
	}

	/* if cpu doesn't match our platform, reject it */
	candidate_cpu_type = find_value (candidate, "library.target.cpu.arch");

	if (verbosity > 0)
	    fprintf (stderr, "cpu_type = %s\n", candidate_cpu_type);

	if (candidate_cpu_type == NULL) 
	    continue;

	if (strcmp (candidate_cpu_type, global.platform_cpu) != 0)
	    continue;
	
	/* if OS doesn't match, reject it */
	candidate_os = find_value (candidate, "library.target.os.name");

	if (verbosity > 0)
	   fprintf (stderr, "os = %s\n", candidate_os);

	if (candidate_os == NULL)
	    continue;

	if (strcmp (candidate_os, global.platform_os) != 0)
	    continue;

	/*
	 * if the OS version of the library is later than
	 * the current platform OS version, reject it
	 * however if there is no version specified, don't reject it.
	 *
	 * XXX this is really making unwarranted assumptions - both that
	 * the OS of the platform on which the library was compiled
	 * implies the version of the OS on which the library will run,
	 * and also that earlier versions of the OS cannot use that
	 * library.  however the latter constraint often does hold
	 * in practice, at least for libraries that make system calls
	 * or reference other libraries, and in the absence of other
	 * information it's safer to impose that constraint.
	 */
	candidate_os_release = find_value (candidate,
					   "library.compile.os.release");

	if (verbosity > 0)
	    fprintf (stderr, "release = %s\n", candidate_os_release);

	if (candidate_os_release == NULL)
	    ;
	else if (version_compare (candidate_os_release,
				  global.platform_os_release) > 0)
	    continue;

        /*
         * see if this candidate has a constraint expression, and if
         * so, see if that expression evaluates to true
         */
        candidate_constraint_expr = 
	    find_value (candidate, "library.constraint");

	if (verbosity > 0)
	    fprintf (stderr, "constraint = %s\n", candidate_constraint_expr);

        if (candidate_constraint_expr == NULL) {
	    if (verbosity > 0) 
		fprintf (stderr, "constraint = NULL\n");
        }
	else {
	    VAL *foo = expr_evaluate (candidate_constraint_expr);

	    if (verbosity > 0) {
	    	symtab_printf_val (stderr, foo);
	        fprintf (stderr, " <= %s\n", candidate_constraint_expr);
	    }

	    if (foo->valtype == INT && foo->intval == 0)
		continue;
	    if (foo->valtype != INT) {
		if (verbosity > 0)
		    fprintf (stderr, "constraint expression\n        %s\ndid not have INT result, skipping\n", candidate_constraint_expr);
		continue;
	    }
	}

	/*
	 * otherwise add this candidate to the new list
	 *
	 * XXX moving items from one list to another causes
	 * core leakage.
	 */
	if (verbosity > 0)
	    fprintf (stderr, "%s is eligible\n", cand->name);

	newlist = add_nvp (newlist, cand->name, cand->value);
    }
    clist = newlist;

    if (clist == NULL)
	return NULL;

    /*
     * scan the remaining candidates looking for the OS with the
     * most recent version
     */
    best = NULL;
    best_os_release = "-1";
    
    for (cand = clist; cand != NULL; cand=cand->next) {
	struct nvpl *cand_alist = (struct nvpl *) cand->value;
	char *cand_os_release;

	cand_os_release = find_value (cand_alist,
				      "library.compile.os.release");
	if (cand_os_release == NULL)
	    continue;
	else if (version_compare (cand_os_release, best_os_release) > 0) {
	    best = cand;
	    best_os_release = cand_os_release;

	    if (verbosity > 0) {
		fprintf (stderr, "best so far is %s, %s\n",
			 best->name,
			 find_value ((struct nvpl *) best->value,
				     "library.compile.os.release"));
	    }
	}
    }
    if (best != NULL)
	return best->name;

    /* else pick the first of those remaining */
    return clist->name;
}
#else /* if 0 */
void
print_list (struct nvpl *head)
{
    struct nvpl *ptr;

    for (ptr = head; ptr; ptr=ptr->next) {
	if (strcmp (ptr->name, "#filename") == 0)
	    printf ("[%s]\n", ptr->value);
    }
    for (ptr = head; ptr; ptr=ptr->next) {
	if (strcmp (ptr->name, "#filename") != 0)
	    printf ("%s:%s\n", ptr->name, ptr->value);
    }
}

void
print_candidate_list (struct nvpl *list[])
{
    int i;

    for (i = 0; list[i] != (struct nvpl *) NULL; ++i)
	print_list ((struct nvpl *) list[i]);
}

/*
 * parse a description file
 *
 * the description file looks like
 *
 * [candidate-filename1]
 * name1:value1
 * name2:value2
 * name3:value3
 * [candidate-filename2]
 * name1:value1
 * name2:value2
 * [candidate-filename3]
 * name1:value1
 * name2:value2
 *
 * and so forth.
 * the candidate-filename is the name of the library file
 */

struct nvpl **
parse_description_file (FILE *fp, char *url)
{
    static struct nvpl *list[1000]; /* XXX fixed-length array */
    int n = -1;
    char buf[10240];		/* XXX fixed-length array */
    int lineno = 0;

    while (fgets (buf, sizeof (buf), fp) != NULL) {
	++lineno;
	if (*buf == '#' || *buf == '\n' || *buf == '\r')
	    ;
	else if (*buf == ' ' || *buf == '\t') {
	    fprintf (stderr, "malformed description file: line %d in %s\n",
		     lineno, url);
	    return NULL;
	}
	else if (*buf == '[') {
	    /* new candidate */
	    char *ket = strchr (buf+1, ']');

	    if (ket == NULL) {
		/* malformed line */
		fprintf (stderr, "malformed description file: line %d in %s\n",
			 lineno, url);
		return NULL;
	    }
	    *ket = '\0';

	    ++n;
	    list[n] = add_nvp (NULL, "#filename", xstrdup (buf+1));
	}
	else {
	    /* new attribute of current candidate */
	    char *colon = strchr (buf, ':');
	    char *nl = strchr (buf, '\n');

	    if (colon == NULL) {
		fprintf (stderr, "malformed description file: line %d in %s\n",
			 lineno, url);
		return NULL;
	    }
	    *colon++ = '\0';
	    *nl = '\0';
	    while (*colon == ' ')
		++colon;
	    list[n] = add_nvp (list[n], xstrdup (buf), xstrdup (colon));
#if 0
	    printf ("%x = add_nvp (%s, %s)\n", list[n], buf, colon);
#endif
	}
    }
    list[++n] = NULL;
    return list;
}


/*
 * given a list of candidate libraries on 'fp', pick the best match
 *
 * XXX
 * change this to return an ordered list of candidates
 * 
 * XXX
 * looking at libraries one at a time is too simple for the real world.
 * some candidate libraries may impose constraints (such as dependencies
 * on other libraries) that keep them from being used.  really we need
 * to get the candidate lists for all libraries, find some number of
 * solution sets that satisfy all of the criteria, and then pick the
 * best of those sets.
 *
 * XXX
 * most criteria should be specified in a config file, rather than
 * having them wired-in here.  also, it would be useful to be able
 * to download some of the criteria - such as which versions and
 * formats of object files were valid on which versions of the OS.
 */

struct candidate {
    int precedence;
    struct nvpl *list;
};

/* NB: sort so that higher-valued precedences comes first */

int
compare_candidates (struct candidate *a, struct candidate *b)
{
    return b->precedence - a->precedence;
}

static int
lookup_library_metadata (int c, char *name, VAL *value, void *arg)
{
    struct nvpl *candidate = (struct nvpl *) arg;

    value->valtype = STR;
    value->strval = find_value (candidate, name);
#if 1
    fprintf (stderr, "lookup_library_metadata (%s) => %s\n",
	     name, value->strval);
#endif

    if (value->strval == NULL)
	value->valtype = ERR;
    return 0;
}

static char *
pick_best_match (FILE *fp, char *url, int verbosity, char *libname)
{
    struct nvpl **clist = parse_description_file (fp, url);
    struct candidate new_clist[1000];	 /* XXX */
    struct nvpl *cand;
    struct nvpl *best;
    char *best_os_release;
    int i;
    int n;
    char *lib_specific_constraint = NULL;

#if 0
    if (libname) {
	char *foo = concat ("NB_", libname, "_CONSTRAINT", NULL);
	VAL *bar;

	lib_specific_constraint = getenv (foo);
	if (lib_specific_constraint) {
	    bar = expr_evaluate (lib_specific_constraint);
	    if (bar->valtype == ERR) {
		fprintf (stderr, "evaluating $%s:\n", foo);
		fprintf (stderr, "   ERR <= %s\n", lib_specific_constraint);
		fprintf (stderr, "   constraint will be ignored\n");
		lib_specific_constraint = NULL;
	    }
	}
	free (foo);
    }
#endif

    if (clist == NULL)
	return NULL;
#if 0
    if (verbosity > 0)
        print_candidate_list (clist);
#endif

    /* scan the list and reject ineligible candidates */

    n = -1;
    for (i = 0; clist[i] != (struct nvpl *) NULL; ++i) {
	struct nvpl *candidate = clist[i];

	char *candidate_cpu_type;
	char *candidate_os;
	char *candidate_os_release;
        char *candidate_constraint_expr;
        char *candidate_precedence_expr;
	char *candidate_name = find_value (candidate, "#filename");

	if (verbosity > 0) {
	    fprintf (stderr, "scanning %s...\n", candidate_name);
	    fprintf (stderr, "printing list %x\n", candidate);
 	    print_nvpl (stderr, candidate);
        }

#if 0
	if (lib_specific_constraint) {
	    VAL *bar;

	    symtab_bind_pseudo ("library.*", lookup_library_metadata,
				candidate);
	    bar = expr_evaluate (lib_specific_constraint);
	    symtab_bind_pseudo ("library.*", NULL, 0);
	    if (verbosity > 0) {
		fprintf (stderr, "evalutaing $NB_%s_CONSTRAINT:\n", libname);
		symtab_printf_val (stderr, bar);
		fprintf (stderr, " <= %s\n", lib_specific_constraint);
	    }
	    if (bar->valtype == INT && bar->intval != 0)
		;
	    else 
		continue;
	}
#endif

	/* if cpu doesn't match our platform, reject it */
	candidate_cpu_type = find_value (candidate, "library.target.cpu.arch");

	if (verbosity > 0)
	    fprintf (stderr, "cpu_type = %s\n", candidate_cpu_type);

	if (candidate_cpu_type == NULL) 
	    continue;

	if (strcmp (candidate_cpu_type, global.platform_cpu) != 0)
	    continue;
	
	/* if OS doesn't match, reject it */
	candidate_os = find_value (candidate, "library.target.os.name");

	if (verbosity > 0)
	   fprintf (stderr, "os = %s\n", candidate_os);

	if (candidate_os == NULL)
	    continue;

	if (strcmp (candidate_os, global.platform_os) != 0)
	    continue;

	/*
	 * if the OS version of the library is later than
	 * the current platform OS version, reject it
	 * however if there is no version specified, don't reject it.
	 *
	 * XXX this is really making unwarranted assumptions - both that
	 * the OS of the platform on which the library was compiled
	 * implies the version of the OS on which the library will run,
	 * and also that earlier versions of the OS cannot use that
	 * library.  however the latter constraint often does hold
	 * in practice, at least for libraries that make system calls
	 * or reference other libraries, and in the absence of other
	 * information it's safer to impose that constraint.
	 */
	candidate_os_release = find_value (candidate,
					   "library.compile.os.release");

	if (verbosity > 0)
	    fprintf (stderr, "release = %s\n", candidate_os_release);

	if (candidate_os_release == NULL)
	    ;
	else if (version_compare (candidate_os_release,
				  global.platform_os_release) > 0)
	    continue;

        /*
         * see if this candidate has a constraint expression, and if
         * so, see if that expression evaluates to true
         */
        candidate_constraint_expr = 
	    find_value (candidate, "library.constraint");

	if (verbosity > 0)
	    fprintf (stderr, "constraint = %s\n", candidate_constraint_expr);

        if (candidate_constraint_expr == NULL) {
	    if (verbosity > 0) 
		fprintf (stderr, "constraint = NULL\n");
        }
	else {
	    VAL *foo = expr_evaluate (candidate_constraint_expr);

	    if (verbosity > 0) {
	    	fprintf (stderr, "constraint = ");
	    	symtab_printf_val (stderr, foo);
	    	fprintf (stderr, " <= %s\n", candidate_constraint_expr);
	    }

	    if (foo->valtype == INT && foo->intval == 0)
		continue;
	    if (foo->valtype != INT) {
		if (verbosity > 0) {
		    fprintf (stderr, "constraint expression\n\t%s\ndid not have INT result, skipping\n", candidate_constraint_expr);
		}
		continue;
	    }
	}

	/*
	 * otherwise add this candidate to the new list
	 */
	if (verbosity > 0)
	    fprintf (stderr, "%s is eligible\n", candidate_name);

	new_clist[++n].list = candidate;
	candidate_precedence_expr = find_value (candidate, "library.precedence");
	if (candidate_precedence_expr == NULL) {
	    new_clist[n].precedence = -99999;
	}
	else {
	    VAL *foo = expr_evaluate (candidate_precedence_expr);

	    if (verbosity > 0) {
	    	fprintf (stderr, "precedence = ");
	    	symtab_printf_val (stderr, foo);
	    	fprintf (stderr, " <= %s\n", candidate_precedence_expr);
	    }

	    if (foo->valtype == INT)
		new_clist[n].precedence = foo->intval;
	    else {
		new_clist[n].precedence = -99999;
		if (verbosity > 0) {
		    fprintf (stderr, "precedence expression\n	%s\ndid not have INT result\n", candidate_precedence_expr);
		}
	    }
		
	}
    }
    new_clist[++n].list = (struct nvpl *) NULL;

    /*
     * sort according to precedence value
     */
    qsort (new_clist, n, sizeof (*new_clist), compare_candidates);

#if 0
    for (i = 0; new_clist[i].list != NULL; ++i) {
	printf ("candidate[%d]=\n", i);
	print_list (new_clist[i].list);
	printf ("\n");
    }
#endif

#if 1
    /*
     * now pick the best one
     */
    return find_value (new_clist[0].list, "#filename");
#else
    /*
     * scan the remaining candidates looking for the OS with the
     * most recent version
     */
    best = NULL;
    best_os_release = "-1";
    
    for (i = 0; new_clist[i].list; ++i) {
	char *cand_os_release;

	cand_os_release = find_value (new_clist[i].list,
				      "library.compile.os.release");
	if (cand_os_release == NULL)
	    continue;
	else if (version_compare (cand_os_release, best_os_release) > 0) {
	    best = new_clist[i].list;
	    best_os_release = cand_os_release;

#if 0
	    if (verbosity > 0) {
		fprintf (stderr, "best so far is %s, %s\n",
			 find_value (best, "#filename"),
			 find_value (best, "library.compile.os.release"));
	    }
#endif
	}
    }
    if (best != NULL)
	return find_value (best, "#filename");

    /* else pick the first of those remaining */
    return find_value (new_clist[0].list, "#filename");
#endif
}
#endif


/*
 * find the library named 'libname' on netlib server 'server'
 *
 * method - GET a well-known HTTP URL, defined from the library name,
 * relative to the server root.
 *
 * XXX - this is too primitive a matching algorithm for real-world use.
 */

char *
find_on_server (char *server, char *libname, int verbosity)
{
    FILE *fp;
    char *url;
    char *myarch;
    char *result;

   if (verbosity > 0)
	fprintf (stderr, "[looking for %s on server %s]\n",
		 libname, server);

    /*
     * NB: this forces a call to get_current_platform_info which
     * fills in the relevant global.* variables that describe
     * the current platform
     */
    myarch = get_netbuild_arch ();

#ifdef HAVE_ASPRINTF
    asprintf (&url, "%s/%s/%s/netbuild.index", server, libname, myarch);
#else
    url = concat (server, "/", libname, "/", myarch, "/netbuild.index",
		  NULL);
#endif

    /*
     * XXX why bypass cache?
     */
    if ((fp = http_get_internal (url, 0, NULL)) == NULL) {
	fprintf (stderr, "nb: URL %s not found\n", url);
	return NULL;
    }
    result = pick_best_match (fp, url, verbosity - 1, libname);
    fclose (fp);

#ifdef HAVE_ASPRINTF
    asprintf (&url, "%s/%s/%s/%s", server, libname, myarch, result);
#else
    url = concat (server, "/", libname, "/", myarch, "/", result, NULL);
#endif
    return url; 
}

#ifdef TEST
int verbosity = 1;

test_description_file_parser ()
{
    struct nvpl *clist = parse_description_file (stdin, "stdin");

    if (clist == NULL) {
	printf ("practically empty description file\n");
	exit (0);
    }
    else {
	struct nvpl *candidate;

	for (; clist != NULL; clist=clist->next) {
	    printf ("[%s]\n", clist->name);
	    for (candidate=(struct nvpl *)clist->value;
		 candidate != NULL; candidate=candidate->next) {
		printf ("%s:%s\n", candidate->name, candidate->value);
	    }
	}
    }
    exit (0);
}

test_version_compare (int argc, char **argv)
{
    printf ("version_compare (%s, %s) => %d\n",
	    argv[1], argv[2],
	    version_compare (argv[1], argv[2]));
}

test_find_on_server (int argc, char **argv)
{
    printf ("find_on_server (%s, %s, 100) => %s\n",
	    argv[1], argv[2], find_on_server (argv[1], argv[2], 100));
}

main (int argc, char **argv)
{
    if (argc == 1) {
	printf ("usage:\n");
	printf ("%s -vc version-1 version-2 #version compare\n", argv[0]);
	printf ("%s -df < description-file  #description file parse\n",
		argv[0]);
	exit (0);
    }
    if (strcmp (argv[1], "-vc") == 0)
	test_version_compare (argc-1, argv+1);
    if (strcmp (argv[1], "-df") == 0)
	test_description_file_parser ();
    if (strcmp (argv[1], "-fs") == 0)
	test_find_on_server (argc-1, argv+1);
}
#endif

