#include "conf.h"
#include "pathnames.h"

#ifdef HAVE_LIBGEN_H
#include <libgen.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <dirent.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "basename.h"
#include "cache.h"
#include "concat.h"
#include "custom.h"
#include "download.h"
#include "expr.h"
#include "filedate.h"
#include "gnu_arch.h"
#include "httpget.h"
#include "md5.h"
#include "misc.h"
#include "netfind.h"
#include "nvpl.h"
#include "platform.h"
#include "result.h"
#include "symtab.h"

#ifndef HAVE_STRDUP
char *strdup (char *);
#endif

/*
 * globals
 */

char *new_argv[1000];		/* XXX */
int new_argc;
int linkmode;
int verbosity = 0;

struct string_list *dir_list;	/* list of local directories where
				   libraries can be found */
struct string_list *server_list; /* list of web servers */
struct option *options_list;	/* list of command-line options
				   that we might want to recognize */
struct pattern *lib_pattern_list; /* list of patterns matching library
				     file names */

#if 0
/*
 * options list for gcc
 *
 * XXX should be part of platform/compiler-specific setup
 *
 * XXX add recognition for options that specify or constrain
 * the target platform
 */

struct option f77_opts[] = {
    { "-l*", 0, LIBARG },
    { "-L*", 0, LOCALDIR },
    { "-o",  1 },
    { "-static", 0, STATIC },
    { "-v", 0, VERBOSE },
    { NULL, }
};

struct option gcc_opts[] = {
    { "-b", 1 },
    { "-dynamic", 0, DYNAMIC },
    { "-G", 1 },
    { "-idirafter", 1 },
    { "-include", 1 },
    { "-imacros", 1 },
    { "-iprefix", 1 },
    { "-iwithprefix", 1},
    { "-l*", 0, LIBARG },
    { "-L*", 0, LOCALDIR },
    { "-o",  1 },
    { "-static", 0, STATIC },
    { "-u", 1 },
    { "-v", 0, VERBOSE },
    { "-V", 1 },
    { "-x",  1 },
    { "-Xlinker",  1 },
    { NULL, }
};
#endif

/*
 * extract the argument that goes with this option
 *
 * if the pattern that matched the option ends in '*', then the
 * option and argument are concatenated; return the portion of
 * the option that follows the argument.
 *
 * otherwise, return the subsequent element in argv.
 */

char *
extract_arg (struct option *opt, char **argv, int n)
{
    int len = strlen (opt->pattern);

    if (opt->pattern[len-1] == '*')
	return (argv[n] + len - 1);
    else
	return (argv[n+1]);
}


/*
 * return 1 iff 'str' matches 'pat'
 *
 * very simple pattern matching for now -
 * either pat is a string to be matched exactly,
 * or pat begins or ends with '*'.
 */

int
match_pattern (char *pat, char *str)
{
    int plen, slen;

    if (strcmp (pat, str) == 0)
	return 1;
    plen = strlen (pat);
    slen = strlen (str);
    if (*pat =='*' && (plen - 1 <= slen) &&
	strcmp (str + slen - plen - 1, pat+1) == 0)
	return 1;
    else if (pat[plen-1] == '*' && strncmp (str, pat, plen-1) == 0)
	return 1;
    return 0;
}

/*
 * search directory 'dirname' to see whether a file matching 'pat'
 * appears in that directory
 */

int
find_in_dir (char *dirname, char *pat)
{
    if (strchr (pat, '*')) {
	DIR *dir;
	struct dirent *dp;
    
	if ((dir = opendir (dirname)) == NULL)
	    return 0;
	while (dp = readdir (dir)) {
	    if (match_pattern (pat, dp->d_name)) {
		if (verbosity > 0)
		    fprintf (stderr, "nb: in %s, %s matches %s\n",
			     dirname, dp->d_name, pat);
		closedir (dir);
		return 1;
	    }
	}
	closedir (dir);
	return 0;
    }
    else {
	char str[1024];		/* XXX */

	sprintf (str, "%s/%s", dirname, pat);
	if (access (str, R_OK) == 0) {
	    if (verbosity > 0)
		fprintf (stderr, "nb: found %s in %s\n", pat, dirname);
	    return 1;
	}
	return 0;
    }
}

add_to_argv (char *format, char *arg)
{
    char buf[1024];

    sprintf (buf, format, arg);
#if 0
    fprintf (stderr, "new_argv[%d] = %s\n", new_argc, buf);
#endif
    new_argv[new_argc++] = strdup (buf);
    new_argv[new_argc] = NULL;
}

int
find_local_library (char *libname)
{
    struct string_list *dir;
    struct pattern *pat;
    int i;

    for (dir = dir_list; dir; dir=dir->next) {
	for (pat = lib_pattern_list; pat; pat=pat->next) {
	    char huge[1024];	/* XXX */

	    if (verbosity > 0)
		fprintf (stderr, "nb: searching %s for %s\n",
			 dir, pat);
	    sprintf (huge, pat->pat, libname);
	    if (find_in_dir (dir->this, huge)) {
		if (verbosity > 0)
		    fprintf (stderr, "nb: found %s in %s\n",
			     huge, dir->this);
		add_to_argv ("-l%s", libname);
		return 1;
	    }
	}
    }
    if (verbosity > 0)
	fprintf (stderr, "[library %s not found locally]\n", libname);
    return 0;
}

int
find_net_library (char *libname)
{
    struct string_list *ptr;
    char *instance;
    struct result x;

    for (ptr = server_list; ptr; ptr=ptr->next) {
	if (instance = find_on_server (ptr->this, libname, verbosity-1)) {
	    if (download_library (&x, instance, libname) == 0) {
		char *localdir = find_value (x.nvpl, "localdir");
		char *comment;

#if 0
		printf ("metadata for %s:\n", libname);
		print_nvpl (stdout, x.nvpl);
#endif
		if (localdir == NULL) {
		    fprintf (stderr,
			     "BUG: download_library(%s) no localdir\n",
			     libname);
		}
		else {
		    if (find_value (x.nvpl, "library.linkarg0")) {
			int d;
			char buf[20];
			char *argN;

			for (d = 0; d <= 9; ++d) {
			    sprintf (buf, "library.linkarg%d", d);
			    if (argN = find_value (x.nvpl, buf)) {
				char *tmp = malloc (strlen (argN) + 4 * strlen (localdir) + 1);

				sprintf (tmp, argN, localdir, localdir, localdir, localdir);
				add_to_argv ("%s", tmp);
			    }
			    else
				break;
			}
		    }
		    else {
			add_to_argv ("-L%s", localdir);
			add_to_argv ("-l%s", libname);
		    }
		}
		if (comment = find_value (x.nvpl, "library.comment"))
			printf ("library %s: using %s\n", libname, comment);
		return 1;
	    }
	    else {
		fprintf (stderr, "cannot download library %s\n", libname);
		fprintf (stderr, "reason: %s\n", x.msg);
		return 0;
	    }
	}
    }
    return 0;
}

/*
 * find a library either locally or on the network
 *
 * XXX currently we search the network first, then the local file system.
 * we really should make this consult a settable preference path,
 * and we should be able to search specific local directories or
 * specific base URLs from the network
 */

int
find_library (char *libname)
{
    if (find_net_library (libname)) {
	if (verbosity > 0)
	    fprintf (stderr, "find_library (%s) => network\n", libname);
	return 1;
    }
    if (find_local_library (libname)) {
	if (verbosity > 0)
	    fprintf (stderr, "find_library (%s) => local\n", libname);
	return 1;
    }

    /*
     * add library to argv anyway, just in case we don't have
     * that directory in the search path but the linker will
     * still find it.
     */
    add_to_argv ("-l%s", libname);
    fprintf (stderr, "nb: cannot find library '%s'\n", libname);
    return 0;
}


/*
 * process a command-line argument.  this does several things:
 *
 * - remembers the names of any libraries to be linked
 * - remembers the names of any local search directories
 * - notices whether we insist on statically linking everything
 *   (this affects which libraries we download)
 * - notices any indication of the target platform
 * - processes options that are only for nb
 * - notices whether verbose output is requested
 *
 * returns the number of arguments following this argument
 */

int
arg (struct option *optlist, char **argv, int n)
{
    struct option *ptr;

    for (ptr = options_list; ptr != NULL; ptr=ptr->next) {
	if (match_pattern (ptr->pattern, argv[n])) {
	    int flags = ptr->flags;

	    if (flags & LIBARG) {
		/*
		 * NB: find_library() will call add_to_argv() as
		 * necessary, since it needs to alter the arguments
		 * to reflect the real location of the library
		 * after downloading.
		 */
		find_library (extract_arg (ptr, argv, n));
	    }
	    else {
		if (flags & LOCALDIR)
		    append_string (&dir_list, extract_arg (ptr, argv, n));
		if (flags & VERBOSE)
		    verbosity ++;
		if (flags & STATIC)
		    linkmode = STATIC;
		if (flags & DYNAMIC)
		    linkmode = DYNAMIC;
		if (flags & NETBUILDONLY)
		    ;
		else {
		    int j;
		    
		    for (j = 0; j <= ptr->nargs; ++j)
			add_to_argv ("%s", argv[n+j]);
		}
	    }
	    return ptr->nargs;
	}
    }

    /*
     * not found in options list,
     * assume no arguments and no special handling
     */
    add_to_argv ("%s", argv[n]);
    return 0;
}

#if 0
/*
 * find the option list that goes with the compiler
 */

struct option *
find_optlist (char *s)
{
    int i;
    char *bn = basename (s);

    if (bn == NULL)
	return NULL;

    for (i = 0; compilers[i].compiler_name != NULL; ++i)
	if (strcmp (bn, compilers[i].compiler_name) == 0)
	    return compilers[i].optlist;
    return NULL;
}
#endif

struct string_list *
get_path ()
{
    char *path;
    char *colon;
    static struct string_list *path_list;

    if ((path = getenv ("PATH")) == NULL)
	return NULL;
    while (1) {
	colon = strchr (path, ':');
	if (colon == NULL) {
	    if (strcmp (path, "") == 0)
		append_string (&path_list, ".");
	    else
		append_string (&path_list, path);
	    break;
	}
	else {
	    *colon = '\0';
	    if (strcmp (path, "") == 0)
		append_string (&path_list, ".");
	    else
		append_string (&path_list, path);
	    *colon = ':';
	    path = colon + 1;
	}
    }
    return path_list;
}

static int
issymlink (char *buf)
{
    struct stat st;

    if (lstat (buf, &st) < 0)
	return 0;
    return ((st.st_mode & S_IFMT) == S_IFLNK);
}

static char *
linkval (char *path)
{
    static char buf[1024];
    int count;

    if ((count = readlink (path, buf, sizeof (buf))) < 0 || 
        count >= sizeof (buf))
	return NULL;
    buf[count] = '\0';
    return buf;
}

void
build_new_path (char *exclude_dir)
{
    char *oldpath, *newpath, *npp;
    int led = strlen (exclude_dir);

    oldpath = getenv ("PATH");
    newpath = malloc (strlen (oldpath) + 1);
    npp = newpath;

    while (1) {
        char *colon = strchr (oldpath, ':');
	int cl = colon ? colon - oldpath : strlen (oldpath);

	if (led == cl && strncmp (oldpath, exclude_dir, led) == 0)
	    oldpath += led;
        else {
	    memcpy (npp, oldpath, cl + 1);
	    npp += cl + 1;
	}
	if (colon)
	    oldpath = colon + 1;
	else
	    break;
    }
    *npp = '\0';
    setenv ("PATH", newpath, 1);
#if 0
    if (verbosity > 0) 
        fprintf (stderr, "%s removed; PATH now set to %s\n", 
	         exclude_dir, getenv ("PATH"));
#endif
    free (newpath);
}

char *
find_compiler (char *argv0)
{
    struct string_list *path_list = get_path ();
    struct string_list *ptr;
    char *base = strdup (basename (argv0));	/* XXX core leakage */
    int match = 0;
    int which = strchr (argv0, '/') ? 1 : 2;

    build_new_path (NETBUILD_LIB_DIR);

    for (ptr = path_list; ptr; ptr = ptr->next) {
	char buf[1024];

	sprintf (buf, "%s/%s", ptr->this, base);
	if (strcmp (buf, argv0) == 0)
	    continue;
	else if (access (buf, X_OK) == 0) {
	    /*
	     * if this file is a symlink to something named netbuild,
	     * it's not the compiler we're looking for
	     */
	    if (issymlink (buf)) {
		char *foo = basename (linkval (buf));

		if (strcmp (foo, "nb") == 0 || strcmp (foo, "netbuild") == 0) {
#if 0
		    fprintf (stderr, "skipping %s, linkval is %s\n", buf, linkval (buf));
#endif
		    /* remove this directory from the path */
		    build_new_path (ptr->this);
		    continue;
		}
	    }
	    if (verbosity > 0)
		fprintf (stderr, "nb: using compiler %s\n", buf);
	    return strdup (buf);
	}
    }
    return NULL;
}

#if 0
/*
 * copy argument list
 */

char **
copy_argv (int argc, char **argv)
{
    char **result;
    int i;

    result = (char **) malloc_or_else ((argc + 1) * sizeof (char *));
    for (i = 0; i < argc; ++i)
	result[i] = strdup (argv[i]);
    result[argc] = NULL;
    return result;
}
#endif

int
sanity ()
{
    static char *files_to_check[] = {
	"/NetBuild/gnupg/pubring.gpg",
	"/NetBuild/gnupg/secring.gpg",
	"/NetBuild/gnupg/trustdb.gpg",
	NULL,
    };
    int i;

    for (i = 0; files_to_check[i] != NULL; ++i) {
	char *foo = concat (getenv ("HOME"), files_to_check[i], NULL);

	if (access (foo, R_OK) < 0) {
	    fprintf (stderr, "cannot access required file %s\n", foo);
	    fprintf (stderr, "run 'nb -setup' to install required files\n");
	    exit (1);
	}
	free (foo);
    }
}

/*
 * download a URL into a local file.  used by install scripts
 * via "nb -download url filename" to download files that
 * couldn't be downloaded from netlib.
 */

int
nb_download (char *srcurl, char *dstfile)
{
    FILE *in, *out;
    char buf[8*1024];
    int nread;

    if ((in = http_get_internal (srcurl, 0, NULL)) == NULL) {
	fprintf (stderr, "can't download\n %s\n", srcurl);
	return 1;
    }
    if ((out = fopen (dstfile, "w")) == NULL) {
	perror (concat ("can't open \"", dstfile, "\" for writing", NULL));
	return 1;
    }
    while ((nread = fread (buf, sizeof (char), sizeof (buf), in)) > 0) {
	if (fwrite (buf, sizeof (char), nread, out) != nread)
	    goto fail;
    }
    fclose (in);
    fclose (out);
    return 0;

 fail:
    fclose (in);
    fclose (out);
    unlink (dstfile);
    return 1;
}

/*
 * download a package containing files needed to set up nb,
 * and extract those files.
 *
 * we always exit immediately after returning, so there's no
 * need to clean up memory or file descriptors.
 *
 * XXX really need a way to do integrity checking, to make
 * sure we download the entire file intact.
 */

int
nb_setup ()
{
    FILE *in, *out;
    char tempfilename[1024];
    char *url = "http://www.netlib.org/netbuild/setup/netbuild-setup.tar";
    char *command;
    char buf[8*1024];
    int nread;
    char *home;


    fprintf (stderr, "nb: downloading setup files...\n");

    sprintf (tempfilename, "/tmp/%d.tar", getpid ());
    if ((out = fopen (tempfilename, "w")) == NULL) {
	perror (tempfilename);
	goto fail;
    }
    if ((in = http_get_internal (url, 0, NULL)) == NULL) {
	fprintf (stderr, "nb: download failed:\n %s\n", url);
	goto fail;
    }
    while ((nread = fread (buf, sizeof (char), sizeof (buf), in)) > 0) {
	if (fwrite (buf, sizeof (char), nread, out) != nread)
	    goto fail;
    }
    fclose (in);
    fclose (out);

    if ((home = getenv ("HOME")) == NULL) {
	fprintf (stderr, "no HOME environment variable\n");
	goto fail;
    }

    fprintf (stderr, "nb: installing setup files in %s/NetBuild...\n", home);

    if (chdir (home) < 0) {
	perror (concat ("chdir (", home, ")", NULL));
	goto fail;
    }
    mkdir ("NetBuild", 0711);
    if (chdir ("NetBuild") < 0) {
	perror (concat ("chdir(", home, "/NetBuild)", NULL));
	goto fail;
    }

    /*
     * XXX calling tar makes me nervous
     * 1. tar might not be in the user's PATH
     * 2. syntax might vary slightly from one system to another
     * 3. there's no way to keep tar from scribbling on arbitrary
     *    directories or files (if, for example, the netlib server
     *    got compromised or an evil proxy got in the way)
     * solution: provide our own tar decoder.
     */
    command = concat ("tar xvfp ", tempfilename, NULL);
    if (system (command) != 0) {
	fprintf (stderr, "nb: command failed: %s\n", command);
	goto fail;
    }
    unlink (tempfilename);
    return 0;
    
 fail:
    unlink (tempfilename);
    return 1;
}

#define MD5_DIGEST_LENGTH (128/8)

void
nb_md5 (char *filename)
{
    FILE *fp;
    MD5_CTX c;
    unsigned char md[MD5_DIGEST_LENGTH];
    int i;
    static unsigned char buf[1024*16];

    if ((fp = fopen (filename, "r")) == NULL) {
	perror (filename);
	return;
    }
    MD5Init (&c);
    while (1) {
	i = fread (buf, sizeof (char), sizeof (buf), fp);
	if (i <= 0)
	    break;
	MD5Update (&c, buf, (unsigned long) i);
    }
    MD5Final (md, &c);

    printf ("MD5 (%s) = ", filename);
    for (i = 0; i < MD5_DIGEST_LENGTH; ++i)
	printf ("%02x", md[i]);
    printf ("\n");
    fclose (fp);
}

int
do_nb (char **argv)
{
    char *newpath;

    if (argv[0] == NULL) {
	fprintf (stderr, "usage: nb command\n");
	return 1;
    }
    if (strcmp (argv[0], "-setup") == 0) {
	return nb_setup ();
    }
    else if (strcmp (argv[0], "-download") == 0) {
	if (argv[1] && argv[2])
	    return nb_download (argv[1], argv[2]);
	else {
	    fprintf (stderr, "usage: nb -download source-url dest-filename\n");
	    return 1;
	}
    }
    else if (strcmp (argv[0], "-md5") == 0) {
	int i;

	for (i = 1; argv[i]; ++i)
	    nb_md5 (argv[i]);
	return 0;
    }
    else if (strcmp (argv[0], "-e") == 0) {
	int i;
	int len;
        char *expr;
	VAL *result;
	char hostname[1024];
	char *myarch;
	char *compiler;

	gethostname (hostname, sizeof (hostname));
	myarch = get_netbuild_arch ();
	compiler = "cc";
	read_config (hostname, myarch, compiler);
	cpu_init ();

	/*
	 * handle LD_LIBRARY_PATH
	 * this tries to more-or-less emulate the behavior of the Sloaris
	 * linker, which allows LD_LIBRARY_PATH to separately specify the
	 * paths to search before and after the directories specified by -L.
	 * (by separating these two paths with a semicolon)
	 * note that we don't care what order the searching is done,
	 * all we care about is determining whether the requested library
	 * exists locally 
	 */
	if (getenv ("LD_LIBRARY_PATH")) {
	    char *ld_library_path = getenv ("LD_LIBRARY_PATH");
       	    char *semi = strchr (ld_library_path, ';');
                
	    if (semi) {
	        *semi = '\0'; 
	        process_library_path (ld_library_path);
	        *semi = ';'; 
	        process_library_path (semi+1);
	    }       
	    else { 
	        process_library_path (ld_library_path);
	    }   
	}           

	expr_functions_init ();
        uname_init ();
        len = 0;
	for (i = 1; argv[i]; ++i)
	    len += strlen (argv[i]) + 1;
	expr = malloc (len + 1);
 	*expr = '\0';
	for (i = 1; argv[i]; ++i) {
	    strcat (expr, argv[i]);
	    strcat (expr, " ");
	}
	result = expr_evaluate (expr);
	symtab_print_val (result);
	printf (" = %s\n", expr);
	return 0;
    }
    sanity ();
    newpath = malloc (strlen (getenv ("PATH")) + strlen (NETBUILD_LIB_DIR) + 2);
    sprintf (newpath, "%s:%s", NETBUILD_LIB_DIR, getenv ("PATH"));
#if 0
    if (verbosity > 0)
	fprintf (stderr, "nb: adding %s to PATH\n", NETBUILD_LIB_DIR);
#endif
    setenv ("PATH", newpath, 1);
    execvp (argv[0], argv);
    fprintf (stderr, "nb: %s: program not found\n", argv[1]);
    fprintf (stderr, " (%s)\n", strerror (errno));
    return 1;
}


int
main (int argc, char **argv, char **envp)
{
    int i;
    struct option *optlist;
    struct list *ptr;
    char *compiler;
    char hostname[1024];
    char *myarch;
    char *verbose_str;

    for (i = 1; i < argc; ++i)
	if (strcmp (argv[i], "-v") == 0)
	    verbosity++;
    if (verbose_str = getenv ("NB_VERBOSE")) {
	if (strcasecmp (verbose_str, "yes") == 0)
	    verbosity = 1;
	else if (isdigit (*verbose_str))
	    verbosity = atoi (verbose_str);
    }
    if (verbosity > 0) {
	fprintf (stderr, "\n");
	fprintf (stderr, "nb pre-release (built on %s)\n", BUILD_DATE);
#if 0
	fprintf (stderr, "called as:\n");
	for (i = 0; i < argc; ++i)
	    fprintf (stderr, " %s", argv[i]);
	fprintf (stderr, "\n");
#endif
    }

    append_string (&server_list, "http://www.netlib.org/netbuild/packages");

    /*
     * if we were called with the name 'nb', the rest of the
     * command line is a command to run in a netbuild environment.
     */
    if (strcmp (basename (argv[0]), "nb") == 0 || strcmp (basename (argv[0]), "netbuild") == 0)
 	return do_nb (argv+1);

    /*
     * otherwise, we're a shim for a compiler or linker
     */
    add_to_argv ("%s", argv[0]);

    gethostname (hostname, sizeof (hostname));

    if ((compiler = find_compiler (argv[0])) == NULL) { 
        fprintf (stderr, "nb: unable to find real compiler for %s\n",
                 basename (argv[0]));
        exit (1);
    }
    new_argv[0] = compiler;

    myarch = get_netbuild_arch ();

#if 0
    fprintf (stderr, "reading from %s\n", find_config_file (hostname, myarch));
#endif
    read_config (hostname, myarch, compiler);
    cpu_init ();

    /*
     * handle LD_LIBRARY_PATH
     * this tries to more-or-less emulate the behavior of the Sloaris 
     * linker, which allows LD_LIBRARY_PATH to separately specify the 
     * paths to search before and after the directories specified by -L.  
     * (by separating these two paths with a semicolon)
     * note that we don't care what order the searching is done,
     * all we care about is determining whether the requested library
     * exists locally
     */
    if (getenv ("LD_LIBRARY_PATH")) {
	char *ld_library_path = getenv ("LD_LIBRARY_PATH");
	char *semi = strchr (ld_library_path, ';');
	
	if (semi) {
	    *semi = '\0';
	    process_library_path (ld_library_path);
	    *semi = ';';
	    process_library_path (semi+1);
	}
	else {
	    process_library_path (ld_library_path);
	}
    }
    expr_functions_init ();
    uname_init ();

    for (i = 1; i < argc; ++i)
	i += arg (optlist, argv, i);

#if 0
    for (ptr = dir_list; ptr; ptr=ptr->next)
	printf ("directory: %s\n", ptr->this);
#endif

#if 0
    for (i = 0; new_argv[i]; ++i)
	printf ("new_argv[%d] = %s\n", i, new_argv[i]);
#endif

    if ((compiler = find_compiler (argv[0])) == NULL) {
	fprintf (stderr, "nb: unable to find real compiler for %s\n",
		 basename (argv[0]));
	exit (1);
    }
    new_argv[0] = compiler;

    /* NB: print subprocess command even if not verbose */
    fprintf (stderr, ">>> %s", compiler);
    for (i = 1; i < new_argc; ++i)
	fprintf (stderr, " %s", new_argv[i]);
    fprintf (stderr, "\n");

    execve (compiler, new_argv, envp);
    fprintf (stderr, "nb: unable to exec %s: %s\n", compiler, strerror (errno));
    exit (127);
}
