#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <stdarg.h>
#include <string.h>
#include <sys/stat.h>
#include "conf.h"
#include "misc.h"

/*
 * miscellaneous functions
 */

/*
 * given a NULL-terminated actual parameter list,
 * return a vector to malloc'ed copies of those arguments
 */

char **
make_argv (char *arg0, ...)
{
    int argc;
    char **result;
    char *s;
    va_list ap;

    if (arg0 == NULL) {
	result = (char **) malloc_or_else (sizeof (char *));
	result[0] = NULL;
	return result;
    }

    /* count arguments */	
    argc = 1;			/* for arg0 */
    va_start (ap, arg0);
    while (s = va_arg (ap, char *))
	++argc;
    va_end (ap);

    result = (char **) malloc_or_else ((argc + 2) * sizeof (char *));

    argc = 0;			/* for arg0 */
    result[argc++] = strdup (arg0);
    va_start (ap, arg0);
    while (result[argc] = va_arg (ap, char *)) {
	result[argc] = strdup (result[argc]);
	++argc;
    }
    va_end (ap);
    result[argc] = NULL;

    return result;
}

/*
 * free up a malloc'ed vector (terminated with NULL) of malloc'ed strings.
 *
 * for some reason we are getting garbage in the argv passed from
 * gpg_import_keys
 */

void
free_argv (char **argv)
{
    int i;

    if (argv == NULL)
	return;
    for (i = 0; argv[i]; ++i) {
	free (argv[i]);
	argv[i] = NULL;
    }
    free (argv);
}

/*
 * malloc something or die trying
 */

void *
malloc_or_else (int size)
{
    void *ptr;

    if ((ptr = (void *) malloc (size)) == NULL) {
	fprintf (stderr, "nb: out of memory\n");
	exit (1);
    }
    return ptr;
}

/*
 * create a directory (if it is missing)
 * along with any necessary ancestors
 */

int
mkdir_recursive (char *dir, int mode)
{
    struct stat st;
    extern int verbosity;

#if 0
   if (verbosity > 0)
	fprintf (stderr, "mkdir_recursive (\"%s\", %o)\n", dir, mode);
#endif

    if (*dir == '\0' || strcmp (dir, "/") == 0)
	return 0;
    if (stat (dir, &st) < 0) {
	int status;
	char *parent;

	/*
	 * stat failed.  assume that the directory does not exist, 
 	 * so create any ancestors that are needed and then create 
	 * this directory.  
         * (stat could also fail for other reasons, for instance,
	 * if some ancestor of this directory were not searchable.
	 * if that happens we will take longer to fail, but we will
	 * still fail in a reasonable amount of time.)
	 */
	parent = strdup (my_dirname (dir));
	status = mkdir_recursive (parent, mode);
	if (status == 0)
	    if (status = mkdir (dir, mode) < 0) {
		fprintf (stderr, "mkdir (%s, 0%o) => %s\n", dir, mode,
			 strerror (errno));
	    }
	free (parent);
	return status;
    }
    else if ((st.st_mode & S_IFMT) != S_IFDIR) {
	/*
	 * file exists, but is not a directory
	 */
	fprintf (stderr, "mkdir_recursive (%s) => not a directory\n", dir);
	errno = ENOTDIR;
	return -1;
    }
    else {
	/* directory already exists; do nothing */
#if 0
	fprintf (stderr, "mkdir_recursive (%s) => 0\n", dir);
#endif
	return 0;
    }
}

/*
 *   Supply a private dirname() function.  On some systems, dirname()
 *   returns a pointer to a static buffer.  On others, dirname()
 *   clobbers its argument.  This is a pain because on the first kind of
 *   system you want to do parent = strdup (dirname (child))
 *   while on the others you want to do parent = dirname (strdup (child))
 *   rather than try to write code that works on either kind of system,
 *   we'll just define our own my_dirname() function that uses a static
 *   buffer.
 *
 *   From the NetBSD man page:
 *
 *   The dirname() function takes a pointer to a character string that con-
 *   tains a pathname, path, and returns a pointer to a string that is a path-
 *   name of the parent directory of path.  Trailing '/' characters in path
 *   are not counted as part of the path.
 *
 *   If path does not contain a '/', then dirname() returns a pointer to the
 *   string ".".
 *
 *   If path is a null pointer or points to an empty string, dirname() returns
 *   a pointer to the string ".".
 *
 *   The path is modified in-place, so if you want path to be unchanged
 *   you need to do dirname (strdup (path))
 */

char *
my_dirname (char *path)
{
    char *slash;
    static char mypath[1024];

    if (path == NULL || *path == '\0')
        return ".";
    if (path[0] == '/' && path[1] == '\0')
        return path;
    if (strlen (path) + 1 > sizeof(mypath))
	return NULL;
    strcpy (mypath, path);
    if (slash = strrchr (mypath, '/')) {
        *slash = '\0';
        return mypath;
    }
    return ".";
}


/*
 * lowercase a string in-place
 */

char *
strlower (char *s)
{
    char *p;

    for (p = s; *p; ++p)
        *p = isupper (*p) ? tolower (*p) : (*p);
    return s;
}

/*
 * a version of strdup() that returns NULL when given a NULL argument
 */

char *
xstrdup (char *s)
{
    if (s == NULL)
	return NULL;
    return strdup (s);
}




#if TEST
int verbosity = 1;

main (int argc, char **argv)
{
    mkdir_recursive (argv[1], 0711);
}
#endif
